/*********************************************************************
 Copyright 2000-2003, Princeton University.  All rights reserved. 
 By using this software the USER indicates that he or she has read, 
 understood and will comply with the following:

 --- Princeton University hereby grants USER nonexclusive permission 
 to use, copy and/or modify this software for internal, noncommercial,
 research purposes only. Any distribution, including commercial sale 
 or license, of this software, copies of the software, its associated 
 documentation and/or modifications of either is strictly prohibited 
 without the prior consent of Princeton University.  Title to copyright
 to this software and its associated documentation shall at all times 
 remain with Princeton University.  Appropriate copyright notice shall 
 be placed on all software copies, and a complete copy of this notice 
 shall be included in all copies of the associated documentation.  
 No right is  granted to use in advertising, publicity or otherwise 
 any trademark,  service mark, or the name of Princeton University. 


 --- This software and any associated documentation is provided "as is" 

 PRINCETON UNIVERSITY MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS 
 OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR FITNESS FOR A 
 PARTICULAR PURPOSE, OR THAT  USE OF THE SOFTWARE, MODIFICATIONS, OR 
 ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, 
 TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS OF A THIRD PARTY.  

 Princeton University shall not be liable under any circumstances for 
 any direct, indirect, special, incidental, or consequential damages 
 with respect to any claim by USER or any third party on account of 
 or arising from the use, or inability to use, this software or its 
 associated documentation, even if Princeton University has been advised
 of the possibility of those damages.
*********************************************************************/

/*********************************************************************
 ADDENDUM: Modifications and additions to the original software,
 zChaff version 2003.11.04, have been made by Ashish Sabharwal (the
 "author") to produce this new software, SymChaff version 2005.03.

 --- This new software and any associated documentation is also
     provided "as is"

 THE AUTHOR OR THE UNIVERSITY OF WASHINGTON MAKE NO REPRESENTATIONS OR
 WARRANTIES, EXPRESS OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR
 FITNESS FOR A PARTICULAR PURPOSE, OR THAT USE OF THE SOFTWARE,
 MODIFICATIONS, OR ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY
 PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS
 OF A THIRD PARTY.

 The author or the University of Washington shall not be liable under
 any circumstances for any direct, indirect, special, incidental, or
 consequential damages with respect to any claim by USER or any third
 party on account of or arising from the use of, or inability to use,
 this software or its associated documentation, even if the author or
 the University of Washington have been advised of the possibility of
 those damages.
*********************************************************************/

#include <iterator>
#include "zchaff_dbase.h"

CDatabase::CDatabase()  
{ 	
    _stats.mem_used_up_counts 	= 0;
    _stats.mem_used_up 		= false;

    _stats.init_num_clauses 	= 0;
    _stats.init_num_literals 	= 0;
    _stats.num_added_clauses 	= 0;
    _stats.num_added_literals 	= 0;
    _stats.num_deleted_clauses 	= 0;
    _stats.num_deleted_literals = 0;
    _stats.num_enlarge		= 0;
    _stats.num_compact		= 0;
    _lit_pool_start = (CLitPoolElement *) malloc (sizeof(CLitPoolElement) * STARTUP_LIT_POOL_SIZE); 
    _lit_pool_finish = _lit_pool_start;
    _lit_pool_end_storage = _lit_pool_start + STARTUP_LIT_POOL_SIZE;
    lit_pool_push_back(0); //set the first element as a spacing element

    _params.mem_limit = 1024*1024*1024; //that's 1 G

    variables().resize(1);	//var_id == 0 is never used.
    _allocated_gid    		= 0;
}

CDatabase::~CDatabase()
{ 
    free (_lit_pool_start); 
}

unsigned CDatabase::estimate_mem_usage (void)
{
    unsigned mem_lit_pool = sizeof(CLitPoolElement) * (lit_pool_size() + lit_pool_free_space());

    unsigned mem_vars = sizeof(CVariable) * variables().capacity();

    unsigned mem_cls = sizeof(CClause) * clauses().capacity();

    unsigned mem_cls_queue = sizeof(ConsIdx) * _unused_clause_idx.size();

    unsigned mem_watched = 2 * num_clauses() *	sizeof(CLitPoolElement *);

    // Ashish: estimated mem for pbcons -> very rough estimate of constraint length
    if (num_pbcons() > 0)
      cerr << "WARNING: memory estimate for PBcons very rough!" << endl;
    static const int avg_pbcons_len = 20;
    unsigned mem_pb = sizeof(CPBcons) * pbcons().capacity();
    unsigned mem_pb_queue = sizeof(ConsIdx) * _unused_pbcons_idx.size();
    unsigned mem_pb_coeffs = avg_pbcons_len * num_pbcons() * sizeof(int);
    unsigned mem_pbwatched = avg_pbcons_len * num_pbcons() * sizeof(CLitPoolElement *);

    unsigned mem_lit_cons = 0;
#ifdef KEEP_LIT_CONS
    mem_lit_cons = num_literals() * sizeof(ConsIdx);
#endif

    return (mem_lit_pool + mem_vars + mem_cls + 
	    mem_cls_queue + mem_watched + mem_lit_cons +
            mem_pb + mem_pb_queue + mem_pb_coeffs + mem_pbwatched);
}

unsigned CDatabase::mem_usage(void) {
    unsigned mem_lit_pool = (lit_pool_size() + lit_pool_free_space()) * sizeof(CLitPoolElement);

    unsigned mem_vars = sizeof(CVariable) * variables().capacity();

    unsigned mem_cls = sizeof(CClause) * clauses().capacity();

    unsigned mem_cls_queue = sizeof(int) * _unused_clause_idx.size();

    unsigned mem_watched = 0, mem_lit_cons = 0, mem_pbwatched = 0;

    for (unsigned i=0, sz = variables().size(); i < sz ;  ++i) {
	CVariable & v = variable(i);
	mem_watched += v.watched(0).capacity() + v.watched(1).capacity();
        mem_pbwatched += v.pbwatched(0).capacity() + v.pbwatched(1).capacity();
#ifdef KEEP_LIT_CLAUSES
	mem_lit_cons += v.lit_cons(0).capacity() + v.lit_cons(1).capacity();
#endif
    }
    mem_watched *= sizeof(CLitPoolElement*);
    mem_lit_cons *= sizeof(ConsIdx);

    // Ashish
    unsigned mem_pb = sizeof(CPBcons) * pbcons().capacity();
    unsigned mem_pb_queue = sizeof(ConsIdx) * _unused_pbcons_idx.size();
    unsigned mem_pb_coeffs = 0;
    for (vector<CPBcons>::iterator itr = pbcons().begin(); itr != pbcons().end(); ++itr)
      mem_pb_coeffs += itr->coeffs().capacity();
    mem_pb_coeffs *= sizeof(int);

    return (mem_lit_pool + mem_vars + mem_cls + 
	    mem_cls_queue + mem_watched + mem_lit_cons +
            mem_pb + mem_pb_queue + mem_pb_coeffs + mem_pbwatched);
}

int CDatabase::alloc_gid (void)
{
    for (unsigned i=1; i<= WORD_WIDTH; ++i)
	if (is_gid_allocated(i) == false) {
	    _allocated_gid |= (1 << (i-1));
	    return i;
	}
    warning(_POSITION_, "Not enough GID");
    return VOLATILE_GID;
}

void CDatabase::free_gid (int gid) 
{
    assert (gid > 0 && "Can't free volatile or permanent group");
    assert (gid <= WORD_WIDTH && "gid > WORD_WIDTH?" );
    if (is_gid_allocated(gid)==false) {
	fatal(_POSITION_,"Can't free unallocated GID");
    }
    _allocated_gid &= (~(1<< (gid-1)));
}

bool CDatabase::is_gid_allocated(int gid) 
{
    if (gid==VOLATILE_GID || gid==PERMANENT_GID) 
	return true;
    assert (gid<= WORD_WIDTH && gid > 0);	
    if (_allocated_gid & (1 << (gid -1))) 
	return true;
    return false;
}

int CDatabase::merge_cons_group(int g2, int g1) 
{
    assert (g1 >0 && g2> 0 && "Can't merge with permanent or volatile group");
    assert (g1 != g2);
    assert (is_gid_allocated(g1) && is_gid_allocated(g2));
    for (vector<CClause>::iterator itr = clauses().begin(); itr != clauses().end(); ++itr)
	if (itr->status() != DELETED_CONS)
	    if (itr->gid(g1) == true) {
		itr->clear_gid(g1);
		itr->set_gid(g2);	
	    }
    for (vector<CPBcons>::iterator itr = pbcons().begin(); itr != pbcons().end(); ++itr)
	if (itr->status() != DELETED_CONS)
	    if (itr->gid(g1) == true) {
		itr->clear_gid(g1);
		itr->set_gid(g2);	
	    }
    free_gid (g1);
    return g2;
}

void CDatabase::mark_cons_deleted(CConstraint & cons) {
    assert(cons.cons_type() == CONS_CL);
    ++_stats.num_deleted_clauses;
    _stats.num_deleted_literals += cons.num_lits();
    cons.set_status(DELETED_CONS);
    for (unsigned i=0; i< cons.num_lits(); ++i) {
	CLitPoolElement & l = cons.literal(i);
	--variable(l.var_index()).lits_count(l.var_sign());
	//   --variable(l.var_index()).score(l.var_sign());
	l.val() = 0;
    }
    _unused_clause_idx.push( (&cons - (CConstraint*) &(*clauses().begin())) << 1);
}

bool CDatabase::is_conflicting(ConsIdx idx)
{
  if ((idx & 0x01) == 1) {        // PBcons
    CPBcons & pb = pbcons(idx);
    int real_slack = pb.watch_slack() + pb.watch_max();
    int i = 0;
    CLitPoolElement * ptr = pb.literals();
    while (ptr->is_literal()) {
      if (ptr->is_watched() && pb.coeff(i) > real_slack) {
        int the_value = literal_value(*ptr);
        if (literal_value(*ptr) == 0)
          return true;
      }
      ++ptr;
      ++i;
    }
    return false;
  }

  // Boolean clause
  CLitPoolElement * lits = clause(idx).literals();
  for (int i=0, sz= clause(idx).num_lits(); i<sz;  ++i)
    if (literal_value(lits[i]) != 0)
      return false;
  return true;
}

bool CDatabase::is_satisfied(ConsIdx idx)
{
  if ((idx & 0x01) == 1) {        // PBcons
    CPBcons & pb = pbcons(idx);
    int lhs = 0;
    int i = 0;
    CLitPoolElement * ptr = pb.literals();
    while (ptr->is_literal()) {
      if (literal_value(*ptr) == 1)
        lhs += pb.coeff(i);
      ++ptr;
      ++i;
    }
    return (lhs >= pb.rhs());
  }

  // Boolean clause
  CLitPoolElement * lits = clause(idx).literals();
  for (int i=0, sz= clause(idx).num_lits(); i<sz; ++i) 
    if (literal_value(lits[i]) == 1) 
      return true;
  return false;
}

bool CDatabase::is_unit (ConsIdx idx)
{
  if ((idx & 0x01) == 1)         // PBcons
    return pbcons(idx).watch_slack() < 0;

  // Boolean clause
  int num_unassigned = 0;
  CLitPoolElement * lits = clause(idx).literals();
  for (int i=0, sz= clause(idx).num_lits(); i<sz;  ++i) {
    int value = literal_value(lits[i]);
    if (value == 1)
      return false;
    else if (value != 0)
      ++ num_unassigned; 
  }
  return ( num_unassigned == 1);
}

int CDatabase::find_unit_literal(ConsIdx cl) 
//will return 0 if not unit
{
  assert((cl & 0x01) == 0);
    int unit_lit = 0;
    for (int i=0, sz= clause(cl).num_lits(); i<sz;  ++i) {
	int value =literal_value(clause(cl).literal(i));
	if ( value == 1)
	    return 0;
	else if (value != 0) {
	    if (unit_lit == 0)
		unit_lit = clause(cl).literals()[i].s_var();
	    else 
		return 0;
	}
    }
    return unit_lit;
}

CLitPoolElement * CDatabase::lit_pool_begin(void) 
{ 
    return _lit_pool_start; 
}

CLitPoolElement * CDatabase::lit_pool_end(void) 
{ 
    return _lit_pool_finish; 
}

void CDatabase::lit_pool_incr_size( int size) 
{
    _lit_pool_finish += size; 
    assert (_lit_pool_finish <= _lit_pool_end_storage);
}

void CDatabase::lit_pool_push_back(int value) 
{ 
    assert (_lit_pool_finish <= _lit_pool_end_storage );
    _lit_pool_finish->val() = value;
    ++ _lit_pool_finish;
}

int CDatabase::lit_pool_size(void) 
{ 
    return _lit_pool_finish - _lit_pool_start; 
}

int CDatabase::lit_pool_free_space(void) 
{ 
    return _lit_pool_end_storage - _lit_pool_finish; 
}

double CDatabase::lit_pool_utilization(void) 
{
    //minus num_clauses() is because of spacing (i.e. clause indices)
    // Ashish: same for pbcons
    return (double)num_literals()/((double) (lit_pool_size() - num_clauses() - num_pbcons())) ; 
}

CLitPoolElement & CDatabase::lit_pool(int i) 
{
    return _lit_pool_start[i];
}

void CDatabase::compact_lit_pool(void)
{
    unsigned i, sz;
    DBG1(cout << "Begin Compaction...... " << 
	 endl << "***Beforeclauses Compaction "<< endl;  
	 output_lit_pool_stats(););
    int new_index = 1;
    //first do the compaction for the lit pool
    for (i=1, sz= lit_pool_size(); i<sz;  ++i){ //begin with 1 because 0 position is always 0
	if (!lit_pool(i).is_literal() && !lit_pool(i-1).is_literal())
	    continue;
	else {
	    lit_pool(new_index) = lit_pool(i);
	    ++new_index;
	}
    }
    _lit_pool_finish = lit_pool_begin() + new_index;
    //update all the pointers to the literals;
    //1. clean up the watched pointers from variables
    for (i=1, sz = variables().size(); i< sz;  ++i) {
	variable(i).watched(0).clear();
	variable(i).watched(1).clear();
    }
    for (i=1, sz = lit_pool_size(); i<sz;  ++i) {
	CLitPoolElement & lit = lit_pool(i);
        //2. reinsert the watched pointers
	if (lit.is_literal()) {
	    if (lit.is_watched()) {
		int var_idx = lit.var_index();
		int sign = lit.var_sign();
		variable(var_idx).watched(sign).push_back(& lit_pool(i));
	    }
	}
	//3. update the clauses' first literal pointer
	else { //lit is not literal
	    int cons_idx = lit.get_cons_index();
	    constraint(cons_idx).first_lit() = &lit_pool(i) - constraint(cons_idx).num_lits();
	}
    }
    DBG1(cout << "***After Compaction " << endl; 
	 output_lit_pool_stats(); 
	 cout << endl << endl;);
    ++_stats.num_compact;
}

bool CDatabase::enlarge_lit_pool(void) //will return true if successful, otherwise false.
{
    unsigned i, sz;
    //if memory efficiency < 2/3, we do a compaction
    if ( lit_pool_utilization() < 0.67) {
	compact_lit_pool();
	return true;
    }
    //otherwise we have to enlarge it.
    DBG1(cout << "Begin Enlarge Lit Pool" << endl;);
    //first, check if memory is running out
    int current_mem = estimate_mem_usage();
    float grow_ratio = 1;
    if (current_mem < _params.mem_limit /4 ) 
	grow_ratio = 2;
    else if (current_mem < _params.mem_limit /2 ) 
	grow_ratio = 1.5;
    else if (current_mem < _params.mem_limit * 0.8) 
	grow_ratio = 1.2;
    if (grow_ratio < 1.2) {
	if ( lit_pool_utilization() < 0.9) {  //still has some garbage
	    compact_lit_pool();
	    return true;
	}
	else 
	    return false;
    }
    //second, make room for new lit pool.
    CLitPoolElement * old_start = _lit_pool_start;
    CLitPoolElement * old_finish = _lit_pool_finish;
    int old_size = _lit_pool_end_storage - _lit_pool_start;
    int new_size = (int)(old_size * grow_ratio);
    _lit_pool_start = (CLitPoolElement *) realloc( _lit_pool_start, sizeof(CLitPoolElement) * new_size);
    _lit_pool_finish = _lit_pool_start + (old_finish - old_start);
    _lit_pool_end_storage = _lit_pool_start + new_size;

    //update all the pointers
    int displacement = _lit_pool_start - old_start;
    for (vector<CClause>::iterator itr = clauses().begin(); itr != clauses().end(); ++itr)
	if (itr->status()!=DELETED_CONS) 
	    itr->first_lit() += displacement; 

    // Ashish
    for (vector<CPBcons>::iterator itr = pbcons().begin(); itr != pbcons().end(); ++itr)
	if (itr->status()!=DELETED_CONS) 
	    itr->first_lit() += displacement; 

    for (i=0, sz = variables().size(); i < sz ;  ++i) {
	CVariable & v = variable(i);
	for (int j=0; j< 2 ; ++j) {
	    int k, sz1;
	    vector<CLitPoolElement *> & watched = v.watched(j);
	    for (k=0, sz1 = watched.size(); k< sz1 ; ++k) {
		watched[k] += displacement; 
	    }
            // Ashish
	    vector<CLitPoolElement *> & pbwatched = v.pbwatched(j);
	    for (k=0, sz1 = pbwatched.size(); k< sz1 ; ++k) {
		pbwatched[k] += displacement; 
	    }
	}
    }
    DBG1(output_lit_pool_stats();
	 cout << endl << endl;);
    ++_stats.num_enlarge;
    return true;
}

ConsIdx CDatabase::get_free_clause_idx(void)
{
    ConsIdx new_cl;
    if (_unused_clause_idx.empty()) {
	new_cl = _clauses.size();
	_clauses.resize(new_cl + 1);
        new_cl = new_cl << 1;
    }
    else {
	new_cl = _unused_clause_idx.front();
	_unused_clause_idx.pop();
    }
    clause(new_cl).set_id(_stats.num_added_clauses << 1);
    // Note that for the above line to work correctly, num_added_clauses
    // must be update *after* get_free_clause_idx is called
    return new_cl;
}

// Ashish;
ConsIdx CDatabase::get_free_pbcons_idx(void)
{
    ConsIdx new_pb;
    if (_unused_pbcons_idx.empty()) {
	new_pb = _pbcons.size();
	_pbcons.resize(new_pb + 1);
        new_pb = (new_pb << 1) | 0x01;
    }
    else {
	new_pb = _unused_pbcons_idx.front();
	_unused_pbcons_idx.pop();
    }
    pbcons(new_pb).set_id((_stats.num_added_pbcons << 1) | 0x01);
    // Note that for the above line to work correctly, num_added_pbcons
    // must be update *after* get_free_pbcons_idx is called
    return new_pb;
}

ConsIdx CDatabase::add_clause(vector<int> & lits, int gflag, bool is_symclause)  {
  int new_cl;
  int n_lits = lits.size();

  //a. do we need to enlarge lits pool?
  while (lit_pool_free_space() <= n_lits + 1) { 
    if (enlarge_lit_pool()==false) 
      return -1; //mem out, can't enlarge lit pool, because 
    //ConsIdx can't be -1, so it shows error.
  }

  //b. get a free cl index;
  new_cl = get_free_clause_idx();

  //c. add the clause lits to lits pool
  CClause & cl = clause(new_cl);
  cl.init(lit_pool_end(), n_lits, gflag);
  lit_pool_incr_size(n_lits + 1);
  for (int i=0; i< n_lits; ++i){
    int var_idx = lits[i]>>1;
    assert((unsigned)var_idx < variables().size());
    int var_sign = lits[i]&0x1;
    cl.literal(i).set(var_idx, var_sign);
    if (!is_symclause)  // Ashish: don't change lits_count for symclauses
      ++variable(var_idx).lits_count(var_sign);
//     if (!symmetry_enabled()) {
//       ++variable(var_idx).lits_count(var_sign);
//     }
//     else {
//       CSymindexMapping* sm = var_findsymindex(var_idx);
//       if (!sm || symmetry_exhausted())
//         ++variable(var_idx).lits_count(var_sign);
//       else {
//         // Increment lits_count for ALL vars originally symmetric 
//         // to var_idx
//         for (int i=0; i<sm->indices.size(); i++) {
//           /// ???????????
//         }
        
//         ++variable(var_idx).lits_count(var_sign);
//       }
//     }
#ifdef KEEP_LIT_CLAUSES
    variable(var_idx).lits_clauses(var_sign).push_back(new_cl);
#endif
  }
  //the element after the last one is the spacing element
  cl.literal(n_lits).set_cons_index(new_cl); 

  //d. set the watched pointers
  if (cl.num_lits() > 1) {
    //add the watched literal. note: watched literal must be the last free var
    int max_idx = -1, max_dl = -1;
    int i, sz = cl.num_lits();
    //set the first watched literal
    for (i=0; i< sz; ++i) {
      int v_idx = cl.literal(i).var_index();
      int v_sign = cl.literal(i).var_sign();
      CVariable & v = variable(v_idx);
      if (literal_value(cl.literal(i)) != 0 ){
        v.watched(v_sign).push_back( & cl.literal(i));
        cl.literal(i).set_watch(1);
        break;
      }
      else {
        if (v.dlevel() > max_dl) {
          max_dl = v.dlevel();
          max_idx = i;
        }
      }
    }
    if (i >= sz) { //no unassigned literal. so watch literal with max dlevel
      int v_idx = cl.literal(max_idx).var_index();
      int v_sign = cl.literal(max_idx).var_sign();
      variable(v_idx).watched(v_sign).push_back( & cl.literal(max_idx));
      cl.literal(max_idx).set_watch(1);
    }
    //set the second watched literal
    max_idx = -1; max_dl = -1;
    for (i=sz-1; i>=0; --i) {
      if (cl.literal(i).is_watched())
        continue; // need to watch two different literals
      int v_idx = cl.literal(i).var_index();
      int v_sign = cl.literal(i).var_sign();
      CVariable & v = variable(v_idx);
      if (literal_value(cl.literal(i)) != 0) {
        v.watched(v_sign).push_back( & cl.literal(i));
        cl.literal(i).set_watch(-1);
        break;
      }
      else {
        if (v.dlevel() > max_dl) {
          max_dl = v.dlevel();
          max_idx = i;
        }
      }
    }
    if (i < 0) {
      int v_idx = cl.literal(max_idx).var_index();
      int v_sign = cl.literal(max_idx).var_sign();
      variable(v_idx).watched(v_sign).push_back( & cl.literal(max_idx));
      cl.literal(max_idx).set_watch(-1);
    }
  }
  //update some statistics
  ++_stats.num_added_clauses; 
  _stats.num_added_literals += n_lits;
  return new_cl;
}

ConsIdx CDatabase::add_symclause (vector<int> & lits) 
{
  ConsIdx new_cl = add_clause(lits, 0, true);

  // update statistics for symmetry
  // note that both added_clauses and added_symclauses increase
  ++_stats.num_added_symclauses; 
  _stats.num_added_symliterals += lits.size();
  return new_cl;
}

ConsIdx CDatabase::add_pbcons(vector<int> & lits, vector<int> & coeffs, int rhs)  { 
  int new_pb;
  int n_lits = coeffs.size();
  //a. do we need to enlarge lits pool?
  while (lit_pool_free_space() <= n_lits + 1) { 
    if (enlarge_lit_pool()==false) 
      return -1; //mem out, can't enlarge lit pool, because 
    //PBconsIdx can't be -1, so it shows error.
  }

  //b. get a free pb index;
  new_pb = get_free_pbcons_idx();

  //c. add the pb lits to lits pool
  CPBcons & pb = pbcons(new_pb);
  pb.init(lit_pool_end(), coeffs, rhs);
  lit_pool_incr_size(n_lits + 1);
  for (int i=0; i< n_lits; ++i){
    int var_idx = lits[i]>>1;
    assert((unsigned)var_idx < variables().size());
    int var_sign = lits[i]&0x1;
    pb.literal(i).set(var_idx, var_sign);
    ++variable(var_idx).lits_count(var_sign);
#ifdef KEEP_LIT_CLAUSES
    variable(var_idx).lits_clauses(var_sign).push_back(new_cl);
#endif
  }
  //the element after the last one is the spacing element
  pb.literal(n_lits).set_cons_index(new_pb);

  //d. set the watched pointers
  // For now assume that we have enough unassigned/true literals to form
  // a watch set; otherwise there is a conflict and we will have to look
  // for max dlevel false literals

  int i=0;
  bool max_is_set = false;
  CLitPoolElement * ptr = pb.literals();
  while (ptr->is_literal() && pb.watch_slack()<0) {
    if (!ptr->is_watched() && literal_value(*ptr) != 0) {
      int v_idx = ptr->var_index();
      int v_sign = ptr->var_sign();
      CVariable & v = variable(v_idx);
      v.pbwatched(v_sign).push_back(ptr);
      ptr->set_watch();
      pb.incr_watch_slack(pb.coeff(i));
    }
    if (!max_is_set && ptr->is_watched()) {   // the first one must be the max since sorted
      pb.set_watch_max(pb.coeff(i));
      max_is_set = true;
    }
    ptr++;
    i++;
  }

  if (pb.watch_slack() < 0) {
    // Conflict!
    // Need to look for max dlevel false literals to fill the watch set
    cerr << "ERROR 14: adding pbcons under conflict not yet implemented!" << endl;
    exit(14);
  }

  //e. update statistics
  ++_stats.num_added_pbcons;
  _stats.num_added_literals += n_lits;
  return new_pb;
}

void CDatabase::output_lit_pool_stats (void) 
{
    cout << "Lit_Pool Used " << lit_pool_size() << " Free " << lit_pool_free_space()
	 << " Total " << lit_pool_size() + lit_pool_free_space()
	 << " Num. Cl " << num_clauses() << " Num. Lit " << num_literals(); 
    cout << " Efficiency " <<  lit_pool_utilization() << endl;
}


void CDatabase::detail_dump_cl(ConsIdx cl_idx, ostream & os ) 
{
  assert ((cl_idx & 0x01) == 0);
  os << "CL : " << (cl_idx >> 1);
  CClause & cl = clause(cl_idx);
  if (cl.status()==DELETED_CONS) 
    os << "\t\t\t======removed=====";
  char value;
  for (unsigned i=0; i< cl.num_lits(); ++i) {
    if (literal_value(cl.literal(i))==0) value = '0';
    else if (literal_value(cl.literal(i))==1) value = '1';
    else value = 'X';
    os << cl.literal(i) << "(" << value << "@" << variable(cl.literal(i).var_index()).dlevel()<< ")  ";
  }
  os << endl;
}

void CDatabase::detail_dump_pb(ConsIdx pb_idx, ostream & os ) 
{
  assert ((pb_idx & 0x01) == 1);
  os << "PB : " << (pb_idx >> 1);
  CPBcons & pb = pbcons(pb_idx);
  if (pb.status()==DELETED_CONS) 
    os << "\t\t\t======removed=====";
  char value;
  for (unsigned i=0; i<pb.num_lits(); ++i) {
    if (literal_value(pb.literal(i))==0) value = '0';
    else if (literal_value(pb.literal(i))==1) value = '1';
    else value = 'X';
    os << pb.literal(i) << "(" << value << "@" << variable(pb.literal(i).var_index()).dlevel();
    os << ",c" << pb.coeff(i) << ")  ";
  }
  os << " >= " << pb.rhs();
  os << " (watch-slack = " << pb.watch_slack() << ')' << endl;
}

void CDatabase::dump(ostream & os) {
  unsigned i;
  os << "Dump Database: " << endl;
  for( i=0; i<_clauses.size(); ++i) 
    detail_dump_cl((i<<1), os);
  for( i=0; i<_pbcons.size(); ++i) 
    detail_dump_pb((i<<1)|0x01, os);
  for( i=1; i<_variables.size(); ++i)
    os << "VID " << i << ":\t" << variable(i);
  if (symindex_sets().size() > 0)
    dump_symmetry_info(os);
}

// Ashish
void CDatabase::del_symindex_high(int sym_high) {
  // Delete only one occurrence
  multiset<int>::iterator itr = _symindex_sets.find((CSymindexSet) -(sym_high+1));
  assert(itr != _symindex_sets.end());    // must exist in the multiset
  _symindex_sets.erase(itr);
  bool originally_nosymmetry;
  DBG0(originally_nosymmetry = symmetry_exhausted();)
  if (_symindex_sets.count((CSymindexSet) -(sym_high+1)) == 0)
    -- _symindex_distinct;
  DBG0(if (originally_nosymmetry && !symmetry_exhausted()) 
       cout << "===> Symmetry is back! <===" << endl;)

  multiset<int>::iterator itr2 = _symindex_sets_reverse.find((CSymindexSet) sym_high);
  assert(itr2 != _symindex_sets_reverse.end());
  _symindex_sets_reverse.erase(itr2);
}

void CDatabase::compute_varclass_details() {
  for (vector<CVarclass>::iterator itr = _varclasses.begin(); itr != _varclasses.end(); itr++) {
    int size = 1;
    CVarclass & vc = *itr;
    for (unsigned i=0; i < vc.idx_start.size(); i++) {
      CSymindexSet cs = idx_symindex_set(vc.idx_start[i]);
      vc.idx_size[i] = vc.idx_start[i] - cs + 1;
      vc.idx_start[i] = cs;
      size *= vc.idx_size[i];
    }
    vc.varmap.resize(size);
  }
}

void CDatabase::add_symindex_mapping(int var_idx, int varclass_idx, int* index, unsigned num_indices) {
  // Insert into symindex_map hash table
  _symindex_map.insert(pair<int,CSymindexMapping> (var_idx, CSymindexMapping(varclass_idx, vector<int> (index, index + num_indices))));

  // Insert into varmap of the corresponding varclass
  CVarclass & vc = _varclasses[varclass_idx];
  assert(num_indices == vc.idx_start.size());
  int idx = 0;
  unsigned i;
  for (i=0; i<num_indices-1; i++)
    idx = (idx + index[i] - vc.idx_start[i]) * vc.idx_size[i+1];
  idx += index[i] - vc.idx_start[i];
  vc.varmap[idx] = var_idx;
}

void CDatabase::dump_symmetry_info(ostream & os) {
  os << "Symmetry info dump: " << endl;
  dump_symindex_sets(os);
  dump_varclasses(os);
  dump_symindex_map(os);
}

void CDatabase::dump_varclasses(ostream & os) {
  unsigned i;
  os << "VARCLASSES:" << endl;
  for (i=1; i<=num_varclasses(); i++) {
    CVarclass & vc = _varclasses[i];
    os << "\tvarclass " << i << ": " << endl
       << "\t  idx_start = ";
    copy(vc.idx_start.begin(), vc.idx_start.end(), ostream_iterator<const int>(os, " "));
    os << endl;
    os << "\t  idx_size  = ";
    copy(vc.idx_size.begin(), vc.idx_size.end(), ostream_iterator<const int>(os, " "));
    os << endl;
    os << "\t  varmap    = ";
    copy(vc.varmap.begin(), vc.varmap.end(), ostream_iterator<const int>(os, " "));
    os << endl;
  }
  os << endl;
}

void CDatabase::dump_symindex_sets(ostream & os) {
  os << "SYMINDEX SETS (last index is one more than the end):" << endl << "\t";
  transform(_symindex_sets.rbegin(), _symindex_sets.rend(), ostream_iterator<const CSymindexSet>(os, " "), negate<CSymindexSet>());
  os << endl << endl;
}

void CDatabase::dump_symindex_map(ostream & os) {
  os << "SYMINDEX MAP:" << endl;
  for (hash_map<int,CSymindexMapping>::iterator sm = _symindex_map.begin(); sm != _symindex_map.end(); sm++) {
    os << "\tvar " << sm->first << ": varclass " << sm->second.varclass_idx
       << ", indices ";
    copy(sm->second.indices.begin(), sm->second.indices.end(), ostream_iterator<const int>(os, " "));
    os << endl;
  }
  os << endl;
}











