/*********************************************************************
 Copyright 2000-2003, Princeton University.  All rights reserved. 
 By using this software the USER indicates that he or she has read, 
 understood and will comply with the following:

 --- Princeton University hereby grants USER nonexclusive permission 
 to use, copy and/or modify this software for internal, noncommercial,
 research purposes only. Any distribution, including commercial sale 
 or license, of this software, copies of the software, its associated 
 documentation and/or modifications of either is strictly prohibited 
 without the prior consent of Princeton University.  Title to copyright
 to this software and its associated documentation shall at all times 
 remain with Princeton University.  Appropriate copyright notice shall 
 be placed on all software copies, and a complete copy of this notice 
 shall be included in all copies of the associated documentation.  
 No right is  granted to use in advertising, publicity or otherwise 
 any trademark, service mark, or the name of Princeton University. 


 --- This software and any associated documentation is provided "as is" 

 PRINCETON UNIVERSITY MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS 
 OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR FITNESS FOR A 
 PARTICULAR PURPOSE, OR THAT  USE OF THE SOFTWARE, MODIFICATIONS, OR 
 ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, 
 TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS OF A THIRD PARTY.  

 Princeton University shall not be liable under any circumstances for 
 any direct, indirect, special, incidental, or consequential damages 
 with respect to any claim by USER or any third party on account of 
 or arising from the use, or inability to use, this software or its 
 associated documentation, even if Princeton University has been advised
 of the possibility of those damages.
*********************************************************************/

/*********************************************************************
 ADDENDUM: Modifications and additions to the original software,
 zChaff version 2003.11.04, have been made by Ashish Sabharwal (the
 "author") to produce this new software, SymChaff version 2005.03.

 --- This new software and any associated documentation is also
     provided "as is"

 THE AUTHOR OR THE UNIVERSITY OF WASHINGTON MAKE NO REPRESENTATIONS OR
 WARRANTIES, EXPRESS OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR
 FITNESS FOR A PARTICULAR PURPOSE, OR THAT USE OF THE SOFTWARE,
 MODIFICATIONS, OR ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY
 PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS
 OF A THIRD PARTY.

 The author or the University of Washington shall not be liable under
 any circumstances for any direct, indirect, special, incidental, or
 consequential damages with respect to any claim by USER or any third
 party on account of or arising from the use of, or inability to use,
 this software or its associated documentation, even if the author or
 the University of Washington have been advised of the possibility of
 those damages.
*********************************************************************/


#ifndef __BASIC_CLASSES__
#define __BASIC_CLASSES__

#include <iostream>
#include <vector>
#include <set>

#include "zchaff_header.h"

#define UNKNOWN   	2
#define NULL_CONS  	-1

#define VOLATILE_GID   -1
#define	PERMANENT_GID 	0

typedef int ConsIdx; // used to refer a constraint. Because of dynamic 
// allocation of vector storage, no pointer is allowed
// Ashish: first bit of the id is used to identify constraint type
// 0 -> clause, 1 -> pb-constraint

enum CONS_TYPE
{
  CONS_CL,
  CONS_PB
};

enum CONS_STATUS
{
  ORIGINAL_CONS,
  CONFLICT_CONS,
  DELETED_CONS,
  PROBE_CONS,
  UNKNOWN_CONS
};


/**Class**********************************************************************

  Synopsis    [Definition of a literal]

  Description [A literal is a variable with phase. Two things specify a literal: 
               its "sign", and its variable index. 

	       Each clause that has more than 1 literal contains two special literals.
	       They are being "watched". A literal is marked with 2 bits: 
	       00->not watched; 11->watched, direction = 1;  01->watched, dir = -1; 
	       10 is not valid. These two bits occupy the least significant bits
	       of the literal. 

	       Each literal is represented by a 32 bit signed integer. The higher 29
	       bits represent the variable index. At most 2**28 varialbes are
	       allowed. If the sign of this integer is negative, it means that it is
	       not a valid literal. It could be a clause index or a deleted literal 
	       pool element. The 3rd least significant bit is used to mark its sign. 
	       0->positive, 1->negative. 

	       The literals are collected in a storage space called literal
	       pool. An element in a literal pool can be a literal or a special
	       spacing element to indicate the termination of a clause. The 
	       spacing elements has negative value of the clause index.

               // Ashish
               For pesudo-Boolean constraints, 'direction' of a watched literal
               is not used. Instead, a new watch literal to replace the current
               one is searched in a linear manner starting from the highest
               coefficient literal in the constraint.]

  SeeAlso     [CDatabase, CClause, CPBcons]

******************************************************************************/

class CLitPoolElement
{
protected:
    int32 _val;

public:
    //constructors & destructors
    CLitPoolElement(void):_val(0)	{}

    ~CLitPoolElement() 			{}

    //member access function
    int & val(void) {
	return _val;
    }

    int s_var(void) { //stands for signed variable, i.e. 2*var_idx + sign
	return _val>>2;
    }

    unsigned var_index(void) {
	return _val>>3; 
    }

    unsigned var_sign(void) { 
	return ( (_val>> 2)& 0x1); 
    }

    void set (int s_var) {
	_val = (s_var << 2);
    }

    void set(int vid, int sign) { 
	_val = (((vid<<1) + sign)<< 2); 
    }

    //followings are for manipulate watched literals
    int direction (void) {
	return ((_val&0x03) - 2);
    }

    bool is_watched(void) {
	return ((_val&0x03) != 0);
    }

    void unwatch(void) {
	_val = _val & (~0x3);
    }

    void set_watch(int dir = 1) {
	_val = _val + dir + 2;
    }

    //following are used for spacing (e.g. indicate clause's end)
    bool is_literal(void) {
	return _val > 0;
    }

    // Ashish
    void set_cons_index(ConsIdx cons_idx) {
	_val = - (int) cons_idx;
    }

    ConsIdx get_cons_index(void) {
	assert (_val <= 0);
	return -_val; 
    }

    // misc functions
    unsigned find_cons_index(void) {
	CLitPoolElement * ptr;
	for (ptr = this; ptr->is_literal(); ++ ptr);
	return ptr->get_cons_index();
    }

    //every class should have a dump function and a self check function
    void dump(ostream & os= cout);

    friend ostream & operator << ( ostream & os, CLitPoolElement & l) { 
	l.dump(os); 
	return os;
    }
};


// Ashish:::::::::::::
/**Class**********************************************************************

  Synopsis    [Definition of a constraint]

  Description [A constraint consists of a certain number of
               literals. Depending on whether it is a Boolean clause
               or a pseudo-Boolean constraint, it may have other
               fields as well, such as coefficients and rhs.  All
               literals are collected in a single large vector, called
               literal pool. Each constraint has a pointer to the
               beginning position of its literals in the pool.
	       
	       Zchaff support incremental SAT. Constraints can be
	       added or deleted from the database during search. To
	       accomodate this feature, some modifications are needed.

	       Constraints can be generated during search by conflict
	       driven analysis. Conflict constraints are generated by
	       a resolution process.  Therefore, if after one search
	       some constraints get deleted, then some of the learned
	       conflict clause may be invalidated. To maintain the
	       integrity of the constraint database, it is necessary
	       to keep track of the constraints that are involved in
	       the resolution process for a certain conflict
	       constraint so that when those constraints are deleted,
	       the conflict constraint can also be deleted.
	       
	       The scheme we implement is similar to the scheme
	       described in: Ofer Strichman, Pruning techniques for
	       the SAT-based Bounded Model Checking Problems, in
	       Proc. 11th Advanced Research Working Conference on
	       Correct Hardware Design and Verification Methods
	       (CHARME'01)

               Subclasses CClause and CPBcons inherit from this base
               class.]

  SeeAlso     [CDatabase, CConstraint, CClause, CPBcons]

******************************************************************************/
class CConstraint
{
 protected:
  CLitPoolElement *	_first_lit;	// pointer to the first literal in literal pool
  unsigned 		_num_lits ;	// number of literals
  CONS_STATUS 	_status;	// indicate if this constraint has been deleted or not
  unsigned		_id;    	// the unique ID of a constraint
                                        // the first bit identifies cons_type: 0-clause, 1-pbcons
  unsigned		_gflag;		// the constraint group id flag, maximum allow WORD_WIDTH groups

 public:
  // constructors & destructors
  CConstraint(void){
    _status = UNKNOWN_CONS;
    _id = 0;
  }

  virtual ~CConstraint() {}

  // member access function
  CLitPoolElement * literals(void) 	{ 	//literals()[i] is it's the i-th literal
    return _first_lit; 
  }	

  CLitPoolElement & literal(int idx) 	{ 	//return the idx-th literal
    return *(_first_lit + idx); 
  }

  CLitPoolElement * & first_lit(void) {	//use it only if you want to modify _first_lit
    return _first_lit; 
  }

  unsigned & num_lits(void) { 
    return _num_lits; 
  }

  CONS_TYPE cons_type(void) {
    return (_id & 0x01) ? CONS_PB : CONS_CL;
  }

  ConsIdx id(void) {
    return _id >> 1;
  }

  void set_id(int id) {
    _id = id;
  }

  CONS_STATUS status(void) { 
    return _status; 
  }

  void set_status(CONS_STATUS st) { 
    _status = st;
  }

  // manipulate the group flag
  unsigned & gflag(void) {
    return _gflag;
  }

  bool gid (int i) {
    assert (i>=1 && i<= WORD_WIDTH);
    return (( _gflag & (1<<(i-1))) != 0);
  }

  void set_gid (int i) {
    assert (i>=1 && i<= WORD_WIDTH);
    _gflag |= (1<<(i-1));
  }

  void clear_gid(int i) {
    assert (i>=1 && i<= WORD_WIDTH);
    _gflag &= ~(1<<(i-1));
  }

  // misc functions
  virtual bool self_check(void) { return false; }

  virtual void dump(ostream & os = cout) {}

  friend ostream & operator << ( ostream & os, CConstraint & cons) { 
    cons.dump(os); 
    return os;
  }
};


/**Class**********************************************************************

  Synopsis    [Definition of a clause]

  Description [Class CClase is a subclass of CConstraint (Ashish)

               A clause is consisted of a certain number of literals. 
               All literals are collected in a single large vector, called
	       literal pool. Each clause has a pointer to the beginning position
	       of it's literals in the pool.]

  SeeAlso     [CDatabase, CConstraint]

******************************************************************************/
class CClause : public CConstraint
{
 protected:

 public:
  // constructors & destructors
  CClause(void) {}

  ~CClause() {}

  // initialization & clear up
  void init(CLitPoolElement * head, unsigned num_lits, unsigned gflag) { //initialization of a clause
    _first_lit = head;
    _num_lits = num_lits;
    _gflag = gflag;
  }

  // misc functions
  bool self_check(void);

  void dump(ostream & os = cout); 
};


// Ashish:::::::::::::
/**Class**********************************************************************

  Synopsis    [Definition of a pseudo-Boolean constraint]

  Description [Class CPBcons inherits from CConstraint.

               A pseudo-Boolean constraint consists of some literals along
               with their integer coefficients, an inequality symbol, and
               a right-hand-side. In their normal form, PB constraints look
               like 
                     c1 x1 + c2 x2 + ... + ck xk <= b
               where the x's are literals and the c's and b are _positive_
               coefficients. All PB constraints are stored in normal form.

               All literals are collected in a single large vector, called
	       literal pool. Each PB constraint has a pointer to the beginning
               position of its literals in the pool.

               'watch_max' is the max of the coeffs of watched literals
               'watch_slack' denotes the slack in the watched set for the
               constraint, i.e. sum of watched values minus watch_max minus RHS.
               This is used to optimize finding new watch literals when a 
               current one is assigned 0.]

  SeeAlso     [CDatabase, CConstraint]

******************************************************************************/
class CPBcons : public CConstraint
{
 protected:
  vector<int> 	        _coeffs ;	// coefficients
  int                   _rhs;           // the right-hand-side of the constraint
  int                   _watch_max;     // Max coeff of a watched literal
  int                   _watch_slack;   // Slack in watch variables under current assignment

 public:
  // constructors & destructors
  CPBcons(void) {}

  ~CPBcons() {}

  // initialization & clear up
  void init(CLitPoolElement * head, vector<int> & coeffs, int rhs) {
    _first_lit   = head;
    _coeffs      = coeffs;
    _rhs         = rhs;
    _num_lits    = coeffs.size();
    _watch_max   = 0;  // since nothing is watched so far
    _watch_slack = - rhs;
  }

  vector<int> & coeffs(void) {
    return _coeffs;
  }

  int coeff(int i) {
    return _coeffs[i];
  }

  int coeff(CLitPoolElement * lit) {
    assert (lit != NULL);
    assert (lit - _first_lit < _num_lits);
    return _coeffs[lit - _first_lit];
  }

  int & rhs(void) {
    return _rhs;
  }

  int watch_max(void) {
    return _watch_max;
  }

  void set_watch_max(int max) {
    _watch_slack = _watch_slack + _watch_max - max;    // update watch_slack
    _watch_max = max;
  }

  int watch_slack(void) {
    return _watch_slack;
  }

  void incr_watch_slack(int incr) {
    _watch_slack += incr;
  }

  // misc functions
  bool self_check(void);

  void dump(ostream & os = cout); 
};


/**Class**********************************************************************

  Synopsis    [Definition of a variable]

  Description [CVariable contains the necessary information for a variable.]

  SeeAlso     [CDatabase]

******************************************************************************/
class CVariable 
{
protected:
    unsigned _value		: 2;	//it can take 3 values, 0, 1 and UNKNOWN

    bool _marked		: 1;	//used in conflict analysis.
    
    unsigned _new_cl_phase	: 2;	//it can take 3 value 0: pos phase, 
    //1: neg phase, UNKNOWN : not in new clause;
    //used to keep track of literals appearing
    //in newly added clause so that a. each
    //variable can only appearing in one phase
    //b. same literal won't appear more than once.
    bool _enable_branch 	: 1;	//if this variable is enabled in branch selection

    int _implied_sign		: 1;	//when a var is implied, here is the sign (1->negative, 0->positive)

    ConsIdx _antecedent	;       	//used in conflict analysis. 

    int _dlevel; 			//decision level this variable being assigned

    int _assgn_stack_pos;		//the position where it is in the assignment stack

    int _lits_count[2];			//how many literals are there with this variable. (two phases)

    vector<CLitPoolElement *> _watched[2];	//watched literals of this var. 0: pos phase, 1: neg phase

    // Ashish
    vector<CLitPoolElement *> _pbwatched[2];	// watched literals of this var. 0: pos phase, 1: neg phase
                                                // for pseudo-Boolean constraints

#ifdef KEEP_LIT_CONS
    vector<ConsIdx> _lit_cons[2];	//this will keep track of ALL the appearance of the variable in constraints
    //note this will increase the database size by upto a factor of 2
#endif
    int _scores[2];			//the score used for decision making

    int _var_score_pos;			//keep track of this variable's position in the sorted score array

public:
//constructors & destructors
    CVariable(void) {
	init();
	_lits_count[0] = _lits_count[1] = 0;
    }

    ~CVariable() {}

    void init(void) {
	_value = UNKNOWN; 
	_antecedent=NULL_CONS; 
	_marked = false;
	_dlevel = -1; 
	_assgn_stack_pos = -1;
	_new_cl_phase = UNKNOWN;	
	_scores[0] = _scores[1] = 0;
	_enable_branch = true;
    }
//member access function
    int & score(int i) 	{ 
	return _scores[i]; 
    }

    int score(void)	{ 
//	return 1;	//this will make a fixed order branch heuristic
	return score(0)>score(1)?score(0):score(1); 
    }

    int & var_score_pos(void) {
	return _var_score_pos; 
    }
    
    void set_var_score_pos(int pos) {
	_var_score_pos = pos;
    }

    unsigned value(void) { 
	return _value;
    }

    void set_value(unsigned v) {
	_value = v;
    }

    int & dlevel(void) { 
	return _dlevel;
    }

    int get_dlevel(void) {
	return _dlevel;
    }

    void set_dlevel(int dl) {
	_dlevel = dl;
    }

    int & assgn_stack_pos(void) {
	return _assgn_stack_pos;
    }

    int & lits_count(int i) { 
	return _lits_count[i];
    }

    bool is_marked(void) { 
	return _marked; 
    }    
    
    int get_implied_sign(void) {
	return _implied_sign;
    }
    
    void set_implied_sign(int sign) {
	_implied_sign = sign;
    }

    unsigned new_cl_phase(void) { 
	return _new_cl_phase; 
    } 

    void set_new_cl_phase(unsigned phase) { 
	_new_cl_phase = phase; 
    }

    void set_marked(void) { 
	_marked = true; 
    }

    void clear_marked(void) { 
	_marked = false; 
    }

    ConsIdx & antecedent(void) { 
	return _antecedent; 
    }
    
    ConsIdx get_antecedent(void) { 
	return _antecedent; 
    }
    
    void set_antecedent(ConsIdx idx) {
	_antecedent = idx;
    }

    vector<CLitPoolElement *> & watched(int i) { 
	return _watched[i]; 
    }

    vector<CLitPoolElement *> & pbwatched(int i) { 
	return _pbwatched[i]; 
    }

    void enable_branch(void) {
	_enable_branch = true;
    }

    void disable_branch(void) {
	_enable_branch = false;
    }

    bool is_branchable(void) {
	return _enable_branch;
    }
#ifdef KEEP_LIT_CLAUSES
    vector<ConsIdx> & lit_cons(int i) {
	return _lit_cons[i]; 
    }
#endif    

//misc function
    bool self_check(void);

    void  dump(ostream & os=cout);

    friend ostream & operator << ( ostream & os, CVariable & v) { 
	v.dump(os); 
	return os;
    }
};


// Ashish: symmetry stuff

/*
 CVarclass: a structure for storing variable classes. 

            Each variable class is indexed by several symindex
            sets. The beginnings of these symindex sets are stored in
            idx_start, and their sizes in idx_size.

            varmap, along with locate_var, allows one to find which
            variable in this class is indexed by such and such vector
            of symindices. The symindex_map, defined shorty, provides
            the reverse mapping.
*/
struct CVarclass {
  // NOTE: Varclass is sometimes also referred to as Vartype
  vector<int> idx_start, idx_size;
  vector<int> varmap;

  CVarclass() {}
  CVarclass(int *index, int num_indices) {
    // store object info temporarily; 
    // fields will be updated later when all varclasses are available
    // (method compute_varclass_details())
    idx_start = vector<int> (index, index+num_indices);
    idx_size.resize(num_indices);
  }
  int locate_var(vector<int> & index) {
    assert(index.size() == idx_start.size());
    int var_idx = 0;
    unsigned i;
    for (i=0; i<idx_start.size()-1; i++)
      var_idx = (var_idx + index[i] - idx_start[i]) * idx_size[i+1];
    var_idx += index[i] - idx_start[i];
    return varmap[var_idx];
  }
};


/*
 A symindex set is represented by its starting point. Note that in the
   .sym input file, the symindex sets are represented by their high
   ending point instead, which is somewhat more natural (e.g. 20
   packages, 30 trucks, etc. would be represented as (20,30)). For
   internal computation with the STL library, it's a little easier to
   manipulate instead (the negative of) the starting points (e.g. 20
   packages, 30 trucks would be represented as (-1,-21,-51), where 51
   is the implicit starting point of the hypothetical next symindex
   set). All this is taken care of by the add_symindex_high() method
   in zchaff_dbase.h.
*/
typedef int CSymindexSet;


/*
 CSymindexMap: a structure for storing a mapping from each variable to
               the varclass that it belongs to and the symindices
               within that class that it corresponds to. A hash table
               of such structures is created later to facilitate a
               quick lookup.
*/
struct CSymindexMapping {
  int         varclass_idx;
  vector<int> indices;
  CSymindexMapping() {}
  CSymindexMapping(int vc_idx, const vector<int> & idx) :
    varclass_idx(vc_idx), indices (idx) {}
};


/*
 CBranchInfo: a structure for storing all necessary information related
              to a (symmetric or otherwise) branch. 
*/
struct CBranchInfo {
  int  branch_idx;          // stores which of the many symindices of a
                            //   varclass is used for multiway branching

  int  branch_idx_start;    // stores the start of the symindex set that,
                            //   along with the varclass, defines this branch

  int  branch_point;        // stores where we currently are in the 
                            //   multiway branching process; starts at 1
                            //   and goes up to branch_size+1 (modulo that)

  int  branch_idx_to_flip;  // stores info for skipping redundant branches

  bool fast_backjumped;     // indicates whether fastbackjump already done

  int  sign;                // defines the polarity of multiway branches;
                            //   first branch_point are (1-sign), the rest 
                            //   have polarity = sign

  bool redo_symbranch;      // indicates whether one should re-do the branch
                            //   in finish_add_conflict_clause

  vector<int> vids;         // stores the variables involed in this branch

  set<int>    symconflict_lits;   // slowly develops the symconflict clause

  int         max_symconflict_dl; // helps with symconflict clause computation

  CBranchInfo() : branch_idx(-1), branch_idx_start(0), branch_point(1),
       branch_idx_to_flip(-1), fast_backjumped(false), sign(0),
       redo_symbranch(false), max_symconflict_dl(-1) {}

  void clear() {
    branch_idx = -1;
    branch_idx_start = 0;
    branch_point = 1;
    branch_idx_to_flip = -1;
    fast_backjumped = false;
    sign = 0;
    redo_symbranch = false;
    vids.clear();
    symconflict_lits.clear();
    max_symconflict_dl = -1;
  }

  bool inprogress() {return !vids.empty();}
  bool nosymmetry() {return vids.size() == 1;}
  int  branch_size() {return vids.size();}
  void add_symconflict_lit(int lit, int dl) {
    symconflict_lits.insert(lit);
    if (dl > max_symconflict_dl)
      max_symconflict_dl = dl;
  }
};


#endif
