/* =========FOR INTERNAL USE ONLY. NO DISTRIBUTION PLEASE ========== */

/*********************************************************************
 Copyright 2000-2001, Princeton University.  All rights reserved. 
 By using this software the USER indicates that he or she has read, 
 understood and will comply with the following:

 --- Princeton University hereby grants USER nonexclusive permission 
 to use, copy and/or modify this software for internal, noncommercial,
 research purposes only. Any distribution, including commercial sale 
 or license, of this software, copies of the software, its associated 
 documentation and/or modifications of either is strictly prohibited 
 without the prior consent of Princeton University.  Title to copyright
 to this software and its associated documentation shall at all times 
 remain with Princeton University.  Appropriate copyright notice shall 
 be placed on all software copies, and a complete copy of this notice 
 shall be included in all copies of the associated documentation.  
 No right is  granted to use in advertising, publicity or otherwise 
 any trademark,  service mark, or the name of Princeton University. 


 --- This software and any associated documentation is provided "as is" 

 PRINCETON UNIVERSITY MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS 
 OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR FITNESS FOR A 
 PARTICULAR PURPOSE, OR THAT  USE OF THE SOFTWARE, MODIFICATIONS, OR 
 ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, 
 TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS OF A THIRD PARTY.  

 Princeton University shall not be liable under any circumstances for 
 any direct, indirect, special, incidental, or consequential damages 
 with respect to any claim by USER or any third party on account of 
 or arising from the use, or inability to use, this software or its 
 associated documentation, even if Princeton University has been advised
 of the possibility of those damages.
*********************************************************************/

/*********************************************************************
 ADDENDUM: Modifications and additions to the original software,
 zChaff version 2003.11.04, have been made by Ashish Sabharwal (the
 "author") to produce this new software, SymChaff version 2005.03.

 --- This new software and any associated documentation is also
     provided "as is"

 THE AUTHOR OR THE UNIVERSITY OF WASHINGTON MAKE NO REPRESENTATIONS OR
 WARRANTIES, EXPRESS OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR
 FITNESS FOR A PARTICULAR PURPOSE, OR THAT USE OF THE SOFTWARE,
 MODIFICATIONS, OR ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY
 PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS
 OF A THIRD PARTY.

 The author or the University of Washington shall not be liable under
 any circumstances for any direct, indirect, special, incidental, or
 consequential damages with respect to any claim by USER or any third
 party on account of or arising from the use of, or inability to use,
 this software or its associated documentation, even if the author or
 the University of Washington have been advised of the possibility of
 those damages.
*********************************************************************/

using namespace std;

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <cstdio>
#include <cassert>

#include <set>
#include <vector>

// Ashish: To use getopt() for handling options neatly
#include <getopt.h>
extern int _global_debug_level;

#include "SAT.h"

#define MAX_LINE_LENGTH    65536
#define MAX_WORD_LENGTH    64

enum INEQ_TYPE {
  INEQ_GE,
  INEQ_G,
  INEQ_E,
  INEQ_L,
  INEQ_LE,
  INEQ_UNKNOWN
};

INEQ_TYPE get_ineq_type(char* s) {
  if (s[0] == '>') {
    if (s[1] == '\0')
      return INEQ_G;
    else if (s[1] == '=' && s[2] == '\0')
      return INEQ_GE;
    else
      return INEQ_UNKNOWN;
  }
  else if (s[0] == '<') {
    if (s[1] == '\0')
      return INEQ_L;
    else if (s[1] == '=' && s[2] == '\0')
      return INEQ_LE;
    else
      return INEQ_UNKNOWN;
  }
  else if (s[0] == '=' && s[1] == '\0')
    return INEQ_E;
  else
    return INEQ_UNKNOWN;
}

//This cnf parser function is based on the GRASP code by Joao Marques Silva
void read_cnf(SAT_Manager mng, char * filename )
{
  char line_buffer[MAX_LINE_LENGTH];
  char word_buffer[MAX_WORD_LENGTH];
  set<int> clause_vars;
  set<int> clause_lits;
  int line_num = 0;

  ifstream inp (filename, ios::in);
  if (!inp) {
    cerr << "ERROR 1: Can't open input file " << filename << endl;
    exit(1);
  }
  while (inp.getline(line_buffer, MAX_LINE_LENGTH)) {
    ++ line_num;
    if (line_buffer[0] == 'c') {
      continue; 
    }
    else if (line_buffer[0] == 'p') {
      int var_num;
      int cl_num;

      int arg = sscanf (line_buffer, "p cnf %d %d", &var_num, &cl_num);
      if( arg < 2 ) {
	cerr << "ERROR3: Unable to read number of variables and clauses"
	     << "at line " << line_num << " of input file " << filename << endl;
	exit(3);
      }
      SAT_SetNumVariables(mng, var_num); //first element not used.
    }
    else {                             // Clause definition or continuation
      char *lp = line_buffer;
      do {
	char *wp = word_buffer;
	while (*lp && ((*lp == ' ') || (*lp == '\t'))) {
	  lp++;
	}
	while (*lp && (*lp != ' ') && (*lp != '\t') && (*lp != '\n')) {
	  *(wp++) = *(lp++);
	}
	*wp = '\0';                                 // terminate string

	if (strlen(word_buffer) != 0) {     // check if number is there
	  int var_idx = atoi (word_buffer);
	  int sign = 0;

	  if( var_idx != 0) {
	    if( var_idx < 0)  { var_idx = -var_idx; sign = 1; }
	    clause_vars.insert(var_idx);
	    clause_lits.insert( (var_idx << 1) + sign);
	  } 	
	  else {
	    //add this clause
	    if (clause_vars.size() != 0 && (clause_vars.size() == clause_lits.size())) { //yeah, can add this clause
	      vector <int> temp;
	      for (set<int>::iterator itr = clause_lits.begin();
		   itr != clause_lits.end(); ++itr)
		temp.push_back (*itr);
	      SAT_AddClause(mng, temp);
	    }
	    else {} //it contain var of both polarity, so is automatically satisfied, just skip it
	    clause_lits.clear();
	    clause_vars.clear();
	  }
	}
      }
      while (*lp);
    }
  }
  if (!inp.eof()) {
    cerr << "ERROR 2: Input line " << line_num <<  " too long. Unable to continue..." << endl;
    exit(2);
  }
  //    assert (clause_vars.size() == 0); 	//some benchmark has no 0 in the last clause
  if (clause_lits.size() && clause_vars.size()==clause_lits.size() ) {
    vector <int> temp;
    for (set<int>::iterator itr = clause_lits.begin();
	 itr != clause_lits.end(); ++itr)
      temp.push_back (*itr);
    SAT_AddClause(mng, temp);
  }
  clause_lits.clear();
  clause_vars.clear();
}

// Ashish: method to read a pbcnf file
void read_pbcnf(SAT_Manager mng, char * filename )
{
  char line_buffer[MAX_LINE_LENGTH];
  char word_buffer[MAX_WORD_LENGTH];
  int  line_num = 0;

  vector<int> elements, lits, coeffs;
  set<int>    vars;
  int file_section = 0, cl_idx = 0, pb_idx = 0;
  int var_num, cl_num, pb_num;
  int n=0;
  int coeffs_sum;
  INEQ_TYPE ineq;
  bool is_clause;

  ifstream inp (filename, ios::in);
  if (!inp) {
    cerr << "ERROR 1: Can't open input file " << filename << endl;
    exit(1);
  }
  while (inp.getline(line_buffer, MAX_LINE_LENGTH)) {
    ++ line_num;
    if (line_buffer[0] == 'c') { 
      continue; 
    }
    else if (line_buffer[0] == 'p') {
      int arg = sscanf (line_buffer, "p pbcnf %d %d %d", &var_num, &cl_num, &pb_num);
      if(arg < 3) {
	cerr << "ERROR 3: Unable to read number of variables, clauses and pb-constraints\n"
	     << "at line " << line_num << " of input file " << filename << endl;
	exit(3);
      }
      SAT_SetNumVariables(mng, var_num);
      file_section++;
    }
    else {
      char *lp = line_buffer;
      do {
	char *wp = word_buffer;
	while (*lp && ((*lp == ' ') || (*lp == '\t'))) {
	  lp++;
	}
	while (*lp && (*lp != ' ') && (*lp != '\t') && (*lp != '\n')) {
	  *(wp++) = *(lp++);
	}
	*wp = '\0';                                 // terminate string

	if (strlen(word_buffer) != 0) {
	  int i = atoi (word_buffer);
	  if( i != 0) {
	    elements.push_back(i);
	  } 	
	  else { // i == 0
	    if (elements.empty()) {
	      file_section++;   // move to next section of input file
	      break;  // get next line
	    }
	    switch (file_section) {
	    case 1:
	      // add this clause
	      ++cl_idx;
	      if (cl_idx > cl_num) {
		cerr << "ERROR 13: More clauses than indicated in header found at line " <<
		  line_num << " of input file " << filename << endl;
		exit(13);
	      }
              // check for duplicate variables and create literals
              for (vector<int>::iterator itr = elements.begin(); itr != elements.end(); ++itr) {
                int vid = *itr, sign = 0;
                if (vid < 0)  { vid = -vid; sign = 1; }
                *itr = (vid << 1 ) + sign;  // replace signed var with lit
                vars.insert(vid);
              }
              if (vars.size() == elements.size()) {
                SAT_AddClause(mng, elements);
              } else {} // assume duplicate vars are of opposite polarity. => satisfied!
              vars.clear();
	      elements.clear();
	      break;

	    case 2:
	      // add this pb-constraint
	      ++pb_idx;
	      if (pb_idx > pb_num) {
		cerr << "ERROR 13: More pb-constraints than indicated in header found at line " <<
		  line_num << " of input file " << filename << endl;
		exit(13);
	      }
              for (int i=0; i<elements.size(); i+=2) {
		if (elements[i+1] == 0)
		  continue;  // discard 0-coefficient lits
                int vid = elements[i], sign = 0;
                if (vid < 0)  { vid = -vid; sign = 1; }
                lits.push_back((vid << 1) + sign);
                coeffs.push_back(elements[i+1]);
              }

	      ineq = get_ineq_type(word_buffer);
	      if (ineq == INEQ_UNKNOWN) {
		cerr << "ERROR 12: Invalid ineqality at line " <<
		  line_num << " of input file " << filename << endl;
		exit(12);
	      }

              int rhs;
              sscanf(lp, "%d", &rhs);

	      // get rid of inequalities other than INEQ_GE and INEQ_E
	      if (ineq == INEQ_L)
		{ --rhs; ineq = INEQ_LE; }
	      if (ineq == INEQ_G)
		{ ++rhs; ineq = INEQ_GE; }
	      if (ineq == INEQ_LE) {
		for (vector<int>::iterator itr = coeffs.begin(); itr != coeffs.end(); ++itr)
		  *itr = -(*itr);
		rhs = -rhs;
		ineq = INEQ_GE;
	      }

	      // Make all coeffs positive (they are already non-zero)
              // Also check if the constraint is a simple clause
	      coeffs_sum = 0;
              is_clause = true;
	      for (int i=0; i<coeffs.size(); i++) {
		if (coeffs[i] < 0) {
		  coeffs[i] = -coeffs[i];
		  lits[i] = lits[i] ^ 0x01;
		  rhs += coeffs[i];
		}
		coeffs_sum += coeffs[i];
                if (coeffs[i] != 1)
                  is_clause = false;
	      }
              if (rhs != 1)
                is_clause = false;

	      if (coeffs_sum < rhs) {
		cout << "Unsatisfiable constraint at line " << line_num
		     << " of input file " << filename << endl;
		cout << "<<<<<< UNSAT >>>>>>" << endl;
		exit(0);
	      }

	      if (!is_clause && (rhs > 0 || ineq == INEQ_E)) {  
                // is_clause => no sorting needed
                // rhs<=0    => constraint trivially satisfied
		// sort lits and coeff in decreasing order of coeffs
		for (int i=0; i<coeffs.size()-1; i++) {
		  int max_coeff = coeffs[i], max_idx = i;
		  for (int j=i+1; j<coeffs.size(); j++)
		    if (coeffs[j] > max_coeff)
		      { max_coeff = coeffs[j]; max_idx = j; }
		  if (max_idx != i) {
		    int tmp         = coeffs[i]; 
		    coeffs[i]       = coeffs[max_idx]; 
		    coeffs[max_idx] = tmp; 
		  tmp           = lits[i]; 
		  lits[i]       = lits[max_idx]; 
		  lits[max_idx] = tmp; 
		  }
		}

		if (rhs > 0)
		  SAT_AddPBConstraint(mng, lits, coeffs, rhs);
	      }
              if (is_clause)
                SAT_AddClause(mng, lits);

	      if (ineq == INEQ_E) {
		// INEQ_GE constraint already added
		// Also add the INEQ_LE version of the constraint
		for (vector<int>::iterator itr = lits.begin(); itr != lits.end(); ++itr)
		  *itr = (*itr) ^ 0x01;
		rhs = coeffs_sum - rhs;
		SAT_AddPBConstraint(mng, lits, coeffs, rhs);
	      }

              lits.clear();
              coeffs.clear();
	      elements.clear();
              break;

            default:
              cerr << "ERROR 12: " << filename << " has illegal file structure; extra sections\n";
              exit(12);
	    }
            break;  // get next line
	  }
	}
      }
      while (*lp);
    }
  }

  if (file_section < 3) {
    cerr << "ERROR 12: " << filename << " has illegal file structure; missing sections\n";
    exit(12);
  }

  if (!inp.eof()) {
    cerr << "ERROR 2: Input line " << line_num <<  " too long. Unable to continue..." << endl;
    exit(2);
  }
}

// Ashish: method to read the optional symmetry file
void read_sym(SAT_Manager mng, char * filename, int & num_varclasses)
{
  char line_buffer[MAX_LINE_LENGTH];
  char word_buffer[MAX_WORD_LENGTH];
  int  line_num = 0;

  vector<int> elements;
  int file_section = 0, sym_idx = 0, varclass_idx = 0;
  int var_num, sym_num, varclass_num, symindex_highest=-1;
  int n=0;

  ifstream inp (filename, ios::in);
  if (!inp) {
    cerr << "ERROR 1: Can't open input file " << filename << endl;
    exit(1);
  }
  while (inp.getline(line_buffer, MAX_LINE_LENGTH)) {
    ++ line_num;
    if (line_buffer[0] == 'c') { 
      continue; 
    }
    else if (line_buffer[0] == 'p') {
      int arg = sscanf (line_buffer, "p sym %d %d %d", &var_num, &sym_num, &varclass_num);
      if(arg < 3) {
	cerr << "ERROR 3: Unable to read number of variables, symmetries and varclasses\n"
	     << "at line " << line_num << " of input file " << filename << endl;
	exit(3);
      }
      if (var_num > SAT_NumVariables(mng)) {
	cerr << "ERROR 13: More symmetric variables than all variables together!" << endl;
	exit(13);
      }
      SAT_SetNumSym(mng, sym_num);
      SAT_SetNumVarclasses(mng, varclass_num);
      num_varclasses = varclass_num;
      file_section++;
    }
    else {
      char *lp = line_buffer;
      do {
	char *wp = word_buffer;
	while (*lp && ((*lp == ' ') || (*lp == '\t'))) {
	  lp++;
	}
	while (*lp && (*lp != ' ') && (*lp != '\t') && (*lp != '\n')) {
	  *(wp++) = *(lp++);
	}
	*wp = '\0';                                 // terminate string

	if (strlen(word_buffer) != 0) {     // check if number is there
	  int i = atoi (word_buffer);
	  if (i<0) {
	    cerr << "ERROR 12: Unexpected negative number at line " << line_num <<
	      "\nof input file " << filename << endl;
	    exit(12);
	  }

	  if( i != 0) {
	    elements.push_back(i);
	  } 	
	  else { // i == 0
	    if (elements.empty()) {
	      file_section++;   // move to next section of input file
	      if (file_section == 3)       // finished reading varclasses
		SAT_ComputeVarclassDetails(mng);
	      break;  // get next line
	    }
	    switch (file_section) {
	    case 1:
	      // add this symindex set
	      ++sym_idx;
	      if (sym_idx > sym_num) {
		cerr << "ERROR 13: More symindex sets than indicated in header found at line " <<
		  line_num << " of input file " << filename << endl;
		exit(13);
	      }
	      if (elements.front() != sym_idx) {
		cerr << "ERROR 12: Symindex sets not in consecutive order at line" <<
		  line_num << " of input file " << filename << endl;
		exit(12);
	      }
	      if (elements.size() != 2) {
		cerr << "ERROR 12: Illegal symindex set specification at line" <<
		  line_num << " of input file " << filename << endl;
		exit(12);
	      }
	      SAT_AddSymindexHigh(mng, elements.back());
              if (symindex_highest < elements.back())
                symindex_highest = elements.back();
	      elements.clear();
	      break;

	    case 2:
	      // add this varclass
	      ++varclass_idx;
	      if (varclass_idx > varclass_num) {
		cerr << "ERROR 13: More varclasses than indicated in header found at line " <<
		  line_num << " of input file " << filename << endl;
		exit(13);

	      }
	      if (elements.front() != varclass_idx) {
		cerr << "ERROR 12: Varclasses not in consecutive order at line" <<
		  line_num << " of input file " << filename << endl;
		exit(12);
	      }
	      SAT_AddVarclass(mng, varclass_idx, & elements.begin()[1], elements.size()-1);
	      elements.clear();
	      break;

	    case 3:
	      // add this symindex mapping
              ++n;
	      if (n > var_num) {
		cerr << "ERROR 13: More symindex mappings than variables\nfound at line " <<
		  line_num << " of input file " << filename << endl;
		exit(13);

	      }
	      SAT_AddSymindexMapping(mng, elements[0], elements[1], 
				     & elements.begin()[2], elements.size()-2);
	      elements.clear();
              break;

            default:
              cerr << "ERROR 12: " << filename << " has illegal file structure; extra sections\n";
              exit(12);
	    }
	  }
	}
      }
      while (*lp);
    }
  }

  assert(symindex_highest > 0);
  SAT_SetSymindexHighest(mng, symindex_highest);

  if (file_section < 4) {
    cerr << "ERROR 12: " << filename << " has illegal file structure; missing sections\n";
    exit(12);
  }

  if (!inp.eof()) {
    cerr << "ERROR 2: Input line " << line_num <<  " too long. Unable to continue..." << endl;
    exit(2);
  }
}

// Ashish: method to read the optional order file
void read_ord(SAT_Manager mng, char * filename, const int num_varclasses)
{
  char line_buffer[MAX_LINE_LENGTH];
  char word_buffer[MAX_WORD_LENGTH];
  int  line_num = 0;

  vector<int> elements;
  int file_section = 0, varclass_count = 0, priority_group_count = 0;
  int varclass_num, priority_group_num;
  int n=0;

  ifstream inp (filename, ios::in);
  if (!inp) {
    cerr << "ERROR 1: Can't open input file " << filename << endl;
    exit(1);
  }
  while (inp.getline(line_buffer, MAX_LINE_LENGTH)) {
    ++ line_num;
    if (line_buffer[0] == 'c') { 
      continue; 
    }
    else if (line_buffer[0] == 'p') {
      int arg = sscanf (line_buffer, "p ord %d %d", &varclass_num, &priority_group_num);
      if(arg < 1) {
	cerr << "ERROR 3: Unable to read number of varclasses and priority groups\n"
	     << "at line " << line_num << " of input file " << filename << endl;
	exit(3);
      }
      if (varclass_num != num_varclasses) {
	cerr << "ERROR 13: Inconsistent number of varclasses between sym and ord files!" << endl;
	exit(13);
      }
      file_section++;
    }
    else {
      char *lp = line_buffer;
      do {
	char *wp = word_buffer;
	while (*lp && ((*lp == ' ') || (*lp == '\t'))) {
	  lp++;
	}
	while (*lp && (*lp != ' ') && (*lp != '\t') && (*lp != '\n')) {
	  *(wp++) = *(lp++);
	}
	*wp = '\0';                                 // terminate string

	if (strlen(word_buffer) != 0) {     // check if number is there
	  int i = atoi (word_buffer);
	  if (i<0) {
	    cerr << "ERROR 12: Unexpected negative number at line " << line_num <<
	      "\nof input file " << filename << endl;
	    exit(12);
	  }

	  if( i != 0) {
	    elements.push_back(i);
	  } 	
	  else { // i == 0
	    if (elements.empty()) {
              file_section++;
              break;
            }
            switch (file_section) {
            case 1:
              // process index order for this varclass
              ++varclass_count;
              if (varclass_count > varclass_num) {
                cerr << "ERROR 13: More varclasses than indicated in header found at line " <<
                  line_num << " of input file " << filename << endl;
                exit(13);
              }
              SAT_StoreVarclassIdxOrder(mng, elements[0], & elements.begin()[1], elements.size()-1);
              elements.clear();
              break;

            case 2:
              // process this varclass priority group
              ++priority_group_count;
              if (priority_group_count > priority_group_num) {
                cerr << "ERROR 13: More priority groups than indicated in header found at line " <<
                  line_num << " of input file " << filename << endl;
                exit(13);
              }
              SAT_StoreVarclassPriorityGroup(mng, elements);
              elements.clear();
              break;

            default:
              cerr << "ERROR 12: " << filename << " has illegal file structure; extra sections\n";
              exit(12);
	    }
	  }
	}
      }
      while (*lp);
    }
  }

  if (!inp.eof()) {
    cerr << "ERROR 2: Input line " << line_num <<  " too long. Unable to continue..." << endl;
    exit(2);
  }
}

void handle_result(SAT_Manager mng, int outcome, char * filename, bool printsolution, bool usesymmetry, bool usepb)
{
  char * result = "UNKNOWN";
  switch (outcome) {
  case SATISFIABLE:
    cout << "Instance satisfiable" << endl;

    // Ashish: print solution only if that option is given
    // Following lines will print out a solution if a solution exist
    if (printsolution) {
      for (int i=1, sz = SAT_NumVariables(mng); i<= sz; ++i) {
	switch(SAT_GetVarAsgnment(mng, i)) {
	case -1:	
	  cout <<"("<< i<<")"; break;
	case 0:
	  cout << "-" << i; break;
	case 1:
	  cout << i ; break;
	default:
	  cerr << "ERROR 4: Unknown variable value state"<< endl;
	  exit(4);
	}
	cout << " ";
      }
    }
    result  = "SAT";
    cout << endl;
    break;
  case UNSATISFIABLE:
    result  = "UNSAT";
    cout << "Instance unsatisfiable" << endl;
    break;
  case TIME_OUT:
    result  = "ABORT : TIME OUT"; 
    cout << "Time out, unable to determing the satisfiablility of the instance";
    cout << endl;
    break;
  case MEM_OUT:
    result  = "ABORT : MEM OUT"; 
    cout << "Memory out, unable to determing the satisfiablility of the instance";
    cout << endl;
    break;
  default:
    cerr << "Unknown outcome" << endl;
    exit (5);
  }
	
  cout << "Max Decision Level                      " << SAT_MaxDLevel(mng) << endl;
  cout << "Num. of Decisions                       " << SAT_NumDecisions(mng)<< endl;
  if (usesymmetry)
    cout << "Num. of SymDecisions                    " << SAT_NumSymDecisions(mng)<< endl;
  cout << "Num. of Variables                       " << SAT_NumVariables(mng) << endl;
  cout << "Original Num Clauses                    " << SAT_InitNumClauses(mng) << endl;
  if (usepb)
    cout << "Original Num PB Constraints             " << SAT_InitNumPBcons(mng) << endl;
  cout << "Original Num Literals                   " << SAT_InitNumLiterals(mng) << endl;
  cout << "Added Usual Conflict Clauses            " << SAT_NumAddedClauses(mng) - SAT_NumAddedSymClauses(mng) - SAT_InitNumClauses(mng)<< endl;
  cout << "Added Usual Conflict Literals           " << SAT_NumAddedLiterals(mng) - SAT_NumAddedSymLiterals(mng) - SAT_InitNumLiterals(mng) << endl;
  if (usesymmetry)
    cout << "Added SymConflict Clauses               " << SAT_NumAddedSymClauses(mng) << endl;
  if (usesymmetry)
    cout << "Added SymConflict Literals              " << SAT_NumAddedSymLiterals(mng) << endl;
  if (usepb)
    cout << "Added PB Conflict Clauses               " << SAT_NumAddedPBcons(mng) - SAT_InitNumPBcons(mng)<< endl;
  cout << "Deleted Irrelevant clauses              " << SAT_NumDeletedClauses(mng) <<endl;
  cout << "Deleted Irrelevant literals             " << SAT_NumDeletedLiterals(mng) <<endl;
  cout << "Number of Implications                  " << SAT_NumImplications(mng)<< endl;
  cout << "Number of Restarts                      " << SAT_NumRestarts(mng)<< endl;
  //other statistics come here
  cout << "Total Run Time (excluding I/O)          " << SAT_GetCPUTime(mng) << endl;
  //    cout << "RESULT:\t" << filename << " " << result << " RunTime: " << SAT_GetCPUTime(mng)<< endl;
  cout << "<<<<<< " << result << " >>>>>>" << endl;

}
void output_status(SAT_Manager mng, bool usesymmetry)
{
    cout << "Dec: " << SAT_NumDecisions(mng)<< "\t ";
    if (usesymmetry)
      cout << "SymDec: " << SAT_NumSymDecisions(mng)<< "\t ";
    cout << "AddCl: " << SAT_NumAddedClauses(mng) <<"\t";
    cout << "AddLit: " << SAT_NumAddedLiterals(mng)<<"\t";
    cout << "DelCl: " << SAT_NumDeletedClauses(mng) <<"\t";
    cout << "DelLit: " << SAT_NumDeletedLiterals(mng)<<"\t";
    cout << "NumImp: " << SAT_NumImplications(mng) <<"\t";
    cout << "AveBubbleMove: " << SAT_AverageBubbleMove(mng) <<"\t";
    //other statistics comes here
    cout << "RunTime:" << SAT_GetElapsedCPUTime(mng) << endl;
}

void verify_solution(SAT_Manager mng, bool usepb)
{
  int num_verified = 0;
  for ( int cl_idx = SAT_GetFirstClause (mng); cl_idx >= 0; 
	cl_idx = SAT_GetNextClause(mng, cl_idx)) {
    int len = SAT_GetClauseNumLits(mng, cl_idx);
    int * lits = new int[len+1];
    SAT_GetClauseLits( mng, cl_idx, lits);
    int i;
    for (i=0; i< len; ++i) {
      int v_idx = lits[i] >> 1;
      int sign = lits[i] & 0x1;
      int var_value = SAT_GetVarAsgnment( mng, v_idx);
      if( (var_value == 1 && sign == 0) ||
	  (var_value == 0 && sign == 1) ) break;
    }
    if (i >= len) {
      cerr << "ERROR 6: Verify Satisfiable solution failed, please file a bug report, thanks. " << endl;
      exit(6);
    }
    delete [] lits;
    ++ num_verified;
  }
  cout << num_verified << " Clauses are true" << endl;
  if (usepb)
    cout << "WARNING: PBcons not verified" << endl;
  cout << "Verify Solution successful";
}

int main(int argc, char ** argv)
{
  //     SAT_Manager mng = SAT_InitManager();
  //     if (argc < 2) {
  // 	cerr << "Z-Chaff: Accelerated Sat solver from Princeton. " << endl;
  // 	cerr << "Copyright 2000-2001, Princeton Univ." << endl << endl;;
  // 	cerr << "Usage: "<< argv[0] << " cnf_file [time_limit]" << endl;
  // 	return 2;
  //     }
  //     cout << "Z-Chaff Version: " << SAT_Version(mng) << endl;
  //     cout <<"Solving " << argv[1] << " ......" << endl;
  //     if (argc == 2) {
  // 	read_cnf (mng, argv[1] );
  //     }
  //     else {
  // 	read_cnf (mng, argv[1] );
  // 	SAT_SetTimeLimit(mng, atoi(argv[2]));
  //     }

  // /* if you want some statistics during the solving, uncomment following line */
  // //    SAT_AddHookFun(mng,output_status, 5000);

  // /* you can set all your parameters here, following values are the defaults*/ 
  // //    SAT_SetMaxUnrelevance(mng, 20);
  // //    SAT_SetMinClsLenForDelete(mng, 100);
  // //    SAT_SetMaxConfClsLenAllowed(mng, 5000);

  // /* randomness may help sometimes, by default, there is no randomness */
  // //    SAT_SetRandomness (mng, 10);
  // //    SAT_SetRandSeed (mng, -1);
  //     int result = SAT_Solve(mng);
  //     if (result == SATISFIABLE) 
  // 	verify_solution(mng);
  //     handle_result (mng, result,  argv[1]);
  //     return 0;

  char base_file[128], cnf_file[128], pbcnf_file[128], sym_file[128], ord_file[128], given_ord_file[128];
  bool nofastbackjumping=false, printsolution=false, usesymmetry=false, useorder=false, usepb=false;
  int  debuglevel=0, num_varclasses;
  SAT_Manager mng = SAT_InitManager();
  given_ord_file[0] = '\0';

  cout << endl << SAT_Version(mng) << endl;

  // Ashish: Using getopt() for handling options neatly
  char usage[1024];
  usage[0] = '\0';
  strcat(usage, "symchaff [options] cnf_file[.cnf]");
  strcat(usage, "\n");
  strcat(usage, "\n\tFeature options");
  strcat(usage, "\n\t  [--sym]"); 
  strcat(usage, "\n\t  [--pb]");
  strcat(usage, "\n\t  [-f <order_file>, --ord]");
  strcat(usage, "\n");
  strcat(usage, "\n\tGeneral options");
  strcat(usage, "\n\t  [--nofb]"); 
  strcat(usage, "\n\t  [--nolrn]"); 
  strcat(usage, "\n\t  [--printsoln]"); 
  strcat(usage, "\n\t  [-r, --restarts]");
  strcat(usage, "\n");
  strcat(usage, "\n\t  [-m <memory_limit>]");
  strcat(usage, "\n\t  [-t <time_limit>]"); 
  strcat(usage, "\n");
  strcat(usage, "\n\t  [-h, --help]"); 
  strcat(usage, "\n\t  [-v, --version]");
  strcat(usage, "\n");
  strcat(usage, "\n\tDebugging and tracing options");
  strcat(usage, "\n\t  [-d <debug_level>]"); 
  strcat(usage, "\n");
  strcat(usage, "\nPlease refer to the file README-SymChaff for a detailed description.\n\n");

  int opt_c;
  
  while (1) {
    int option_index = 0;
    static struct option long_options[] = {
      {"help",       0, 0, 0},
      {"restarts",   0, 0, 0},
      {"nofb",       0, 0, 0},
      {"nolrn",      0, 0, 0},
      {"ord",        0, 0, 0},
      {"printsoln",  0, 0, 0},
      {"pb",         0, 0, 0},
      {"sym",        0, 0, 0},
      {"version",    0, 0, 0},
      {0, 0, 0, 0}
    };

    opt_c = getopt_long (argc, argv, "d:f:hm:rt:v", long_options, &option_index);
    if (opt_c == -1)
      break;

    int debug_level;
    switch (opt_c) {

    case 0:
      char optionName[32];
      strcpy(optionName, long_options[option_index].name);
      if (!strcmp(optionName, "help"))
	{ printf("\nUsage: %s", usage); exit(0); }
      else if (!strcmp(optionName, "restarts"))
	{ SAT_SetAllowRestart(mng, true); }
      else if (!strcmp(optionName, "nofb"))
	{ nofastbackjumping=true; SAT_SetAllowFastBackjumping(mng, false); }
      else if (!strcmp(optionName, "nolrn"))
	{ SAT_SetAllowLearning(mng, false); 
	cerr << "WARNING: Not sure how well '--nolrn' is implemented!"; }
      else if (!strcmp(optionName, "printsoln"))
	{ printsolution=true; }
      else if (!strcmp(optionName, "sym"))
	{ usesymmetry=true; SAT_SetAllowSymmetry(mng, true); }
      else if (!strcmp(optionName, "ord"))
	{ useorder=true; SAT_SetAllowOrder(mng, true); }
      else if (!strcmp(optionName, "pb"))
	{ cerr << "ERROR 14: The work on pseudo-Boolean part is IN PROGRESS!";
//        exit(14);
        usepb=true; SAT_SetAllowPB(mng, true); }
      else if (!strcmp(optionName, "version"))
        { cout << SAT_Version(mng) << endl; exit(0); }

      break;

    case 'd':
      debug_level = atoi(optarg);
      if (debug_level < 0 || debug_level > 3) {
	cerr << "ERROR 11: debug_level must be between 0 and 3 (inclusive)" << endl;
	exit(11);
      }
      _global_debug_level = debug_level;
      break;

    case 'f':
      strcpy(given_ord_file, optarg);
      break;

    case 'h':
      printf("\nUsage: %s", usage);
      exit(0);

    case 'm':
      SAT_SetMemLimit(mng, atoi(optarg));
      cout << "Memory limit set to " << atoi(optarg) << " MB.\n";
      break;

    case 'r':
      SAT_SetAllowRestart(mng, true);
      break;

    case 't':
      SAT_SetTimeLimit(mng, atoi(optarg));
      cout << "Time limit set to " << atoi(optarg) << " seconds.\n";
      break;

    case 'v':
      cout << SAT_Version(mng) << endl;
      exit(0);

    case ':':
    case '?':
      // getopt found an nvalid option
      cerr << endl << "Usage: " << usage;
      exit(11);

    default:
      // This should never really happen with getopt
      cerr << "sat_solver.cpp: ERROR 15: switch statement in getopt failed" 
           << endl;;
      exit(15);
    }
  }

  if (optind != argc-1) {
    cerr << endl << "Usage: " << usage;
    exit(11);
  }

  if (nofastbackjumping && !usesymmetry) {
    cerr << "ERROR 14: Fastbackjumping can currently be disabled only with --sym"
	 << endl << "Aborting." << endl;
    exit(14);
  }

  if (useorder && !usesymmetry) {
    cerr << "ERROR 11: .ord file can be specified only with a .sym file"
	 << endl << "Aborting." << endl;
    exit(11);
  }

  if (!usesymmetry) {
    cout << "NOTE: Not using symmetry information; --sym, -s not specified." 
	 << endl;
  }

  // Create file names with proper extensions.
  // Will use them later only if the corresponding options are set.
  strcpy(base_file, argv[argc-1]);
  char* file_extension = strrchr(base_file, '.');
  if (file_extension) {
    if (!strcmp(file_extension, ".cnf")) {
      if (usepb) {
        cerr << "ERROR 11: can't use .cnf file with --pb option" << endl;
        exit(11);
      }
      base_file[ strlen(base_file) - 4 ] = '\0';
    }
    if (!strcmp(file_extension, ".pbcnf")) {
      if (!usepb) {
        cerr << "ERROR 11: must use .pbcnf file with --pb option" << endl;
        exit(11);
      }
      base_file[ strlen(base_file) - 6 ] = '\0';
    }
  }
  strcpy(cnf_file,   base_file);
  strcpy(pbcnf_file, base_file);
  strcpy(sym_file,   base_file);
  strcpy(ord_file,   base_file);
  strcat(cnf_file,   ".cnf");
  strcat(pbcnf_file, ".pbcnf");
  strcat(sym_file,   ".sym");
  strcat(ord_file,   ".ord");

  if (given_ord_file[0] != '\0') {
    if (!useorder) {
      cerr << "ERROR 11: must use --ord option when specifying -f <file>" << endl;
      exit(11);
    }
    strcpy(ord_file, given_ord_file);
  }    

  cout << endl;

  if (!usepb) {
    printf("Reading cnf file %s\n", cnf_file);
    read_cnf(mng, cnf_file);
  }
  else {
    printf("Reading pbcnf file %s\n", pbcnf_file);
    read_pbcnf(mng, pbcnf_file);
  }
  if (usesymmetry) {
    printf("Reading sym file %s\n", sym_file);
    read_sym(mng, sym_file, num_varclasses);
  }
  if (useorder) {
    printf("Reading ord file %s\n", ord_file);
    read_ord(mng, ord_file, num_varclasses);
  }

  int result = SAT_Solve(mng);
  if (result == SATISFIABLE) 
    verify_solution(mng, usepb);

  cout << endl;

  handle_result (mng, result, cnf_file, printsolution, usesymmetry, usepb);
  return 0;
}


