#!/usr/bin/perl

###################################################
# Script for generating CNF formulas encoding
# the pigeonhole principle (PHP). It supports
# symmetry information as well as pseudo-Boolean
# representations. Can also generate formulas for
# the 'functional' pigeonhole principle (easier).
#
# Created by Ashish Sabharwal, University of Washington, Seattle
# March 2005
#
# Usage example:
#   1) phpgen.pl -b -n 10
#   2) phpgen.pl -bs -n 20 -l 17
#   3) phpgen.pl -pbs -n 15
###################################################

## use strict;
use Getopt::Std;

my $usage = 
    "\nUsage: phpgen.pl -n <numPigeons> [-k <numHoles>] [-fcpsh]".
    "\n".
    "\n  -n pigeons     number of pigeons".
    "\n  -k holes       number of holes; default is n-1".
    "\n  -f             generate FUNCTIONAL php".
    "\n  -b             generate Boolean formula (cnf)".
    "\n  -p             generate pseudo-Boolean formula (pbcnf)".
    "\n  -s             generate symmetry information (sym)".
    "\n  -h             help".
    "\n".
    "\nNote: must specify at least one of -b, -p, -s".
    "\n\n";


######################
## Global variables ##
######################
my $n;
my $k;
my $numVars      = 0;
my $numClauses   = 0;
my $cnffilename;

#############################
# Get current date and time #
#############################
my ($dd, $mm, $yy) = (localtime)[3,4,5];
$yy += 1900;
$mm += 1;
my $fileheader = "c Generated on $mm/$dd/$yy by Ashish Sabharwal, ashish\@cs.washington.edu\n";

#############################
## Handle input parameters ##
#############################

getopts('n:k:fbpsh') || die "$usage";
die "\nError: extra arguments $usage" if ($#ARGV != -1);  ## extra unwanted stuff specified
die "\nError: n must be specified $usage" if (!$opt_n);  ## n must be specified

if (!$opt_b && !$opt_p && !$opt_s) {print $usage; exit(1);}

if ($opt_h) {print $usage; exit(0);}

$n = $opt_n;
$k = $n - 1;
if ($opt_k) {$k = $opt_k;}

($n > 0)  || die "n must be positive";
($k >= 0) || die "k must be non-negative";
# ($n > $k)  || die "n must be larger than k";   # otherwise sat instances

my $basefilename = substr((1000+$n),1) . "-" . substr((1000+$k),1);
$cnffilename = "php-" . $basefilename . ".cnf";
$pbcnffilename = "php-" . $basefilename . ".pbcnf";
$symfilename = "php-" . $basefilename . ".sym";
if ($opt_f) {
  $cnffilename = "f" . $cnffilename;
  $pbcnffilename = "f" . $pbcnffilename;
}

!$opt_b or open (CNFFILEHANDLE, ">$cnffilename") or die "ERROR: Can't open file \"$cnffilename\": $!";

!$opt_p or open (PBCNFFILEHANDLE, ">$pbcnffilename") or die "ERROR: Can't open file \"$pbcnffilename\": $!";

!$opt_s or open (SYMFILEHANDLE, ">$symfilename") or die "ERROR: Can't open file \"$symfilename\": $!";


########################################
## Generate and print PHP_n_k formula ##
########################################


## X(a,i) computes the variable index for x_{a,i}
sub X {
  my ($a,$i) = @_;
  return $k*$a + $i + 1;
}

$numVars = $n * $k;


#################################
## Generate CNF file if needed ##
#################################

if ($opt_b) {
  print "Generating $cnffilename\n";

  $numClauses = $n + $k*($n*($n-1))/2;
  if ($opt_f) {
    $numClauses += $n*($k*($k-1))/2;
  }

  print CNFFILEHANDLE $fileheader;
  if ($opt_f) {
    print CNFFILEHANDLE "c Functional Pigeonhole Principle FPHP";
  }
  else {
    print CNFFILEHANDLE "c Pigeonhole Principle PHP";
  }
  print CNFFILEHANDLE "($n,$k)\n",
    "c $n pigeons, $k holes, $numVars variables, $numClauses clauses\n",
    "c\n",
    "p cnf $numVars $numClauses\n",
    "c\n";

  ## Pigeon clauses: every pigeon must go into a hole
  print CNFFILEHANDLE "c Pigeon clauses\n";
  for (my $a=0; $a<$n; $a++) {
    for (my $i=0; $i<$k; $i++) {
      print CNFFILEHANDLE &X($a,$i), ' ';
    }
    print CNFFILEHANDLE "0\n";
  }

  ## Hole clauses: no hole can take two pigeons
  print CNFFILEHANDLE "c\nc Hole clauses\n";
  for (my $i=0; $i<$k; $i++) {
    for (my $a=0; $a<$n-1; $a++) {
      for (my $b=$a+1; $b<$n; $b++) {
        print CNFFILEHANDLE '-', &X($a,$i), " -", &X($b,$i), " 0\n";
      }
    }
  }

  if ($opt_f) {
  ## Functional clauses: no pigeon can go into two holes
    print CNFFILEHANDLE "c\nc Functional clauses\n";
    for (my $a=0; $a<$n; $a++) {
      for (my $i=0; $i<$k-1; $i++) {
        for (my $j=$i+1; $j<$k; $j++) {
          print CNFFILEHANDLE '-', &X($a,$i), " -", &X($a,$j), " 0\n";
        }
      }
    }
  }

  print CNFFILEHANDLE "0\n";

  close(CNFFILEHANDLE);
}


###################################
## Generate PBCNF file if needed ##
###################################

if ($opt_p) {
  print "Generating $pbcnffilename\n";

  $numPB = $n + $k;

  if ($opt_f) {
    print PBCNFFILEHANDLE "c Functional Pigeonhole Principle FPHP";
  }
  else {
    print PBCNFFILEHANDLE "c Pigeonhole Principle PHP";
  }
  print PBCNFFILEHANDLE "($n,$k) in PB format\n",
    "c $n pigeons, $k holes, $numVars variables, 0 clauses, $numPB PB constraints\n",
    "c\n",
    "p pbcnf $numVars 0 $numPB\n",
    "c\n",
    "c No Boolean clauses\n",
    "0\n";

  ## Pigeon constraints: every pigeon must go into a hole
  ## Functional pigeon constraints: every pigeon must go into exactly one hole
  if ($opt_f) {
    print PBCNFFILEHANDLE "c Functional pigeon PB constraints\n";
  }
  else {
    print PBCNFFILEHANDLE "c Pigeon PB constraints\n";
  }
  for (my $a=0; $a<$n; $a++) {
    for (my $i=0; $i<$k; $i++) {
      print PBCNFFILEHANDLE &X($a,$i), " 1 ";
    }
    print PBCNFFILEHANDLE '>' if (!$opt_f);
    print PBCNFFILEHANDLE "= 1\n";
  }

  ## Hole constraints: a hole can take at most one pigeon
  print PBCNFFILEHANDLE "c\nc Hole PB constraints\n";
  for (my $i=0; $i<$k; $i++) {
    for (my $a=0; $a<$n; $a++) {
      print PBCNFFILEHANDLE &X($a,$i), " 1 ";
    }
    print PBCNFFILEHANDLE "<= ", 1, "\n";
  }

  print PBCNFFILEHANDLE "0\n";

  close(PBCNFFILEHANDLE);
}


###############################
# Generate SYM file if needed #
###############################

if ($opt_s) {
  print "Generating $symfilename\n";

  print SYMFILEHANDLE "c Symmetry file for ", substr($cnffilename,1),
    " and $cnffilename\n",
    "c $n pigeons, $k holes, $numVars symmetric variables\n",
    "c 2 symindex sets, 1 varclass\n",
    "c\n",
    "p sym $numVars 2 1\n",
    "c\n";

  ## symindex sets
  print SYMFILEHANDLE "c symindex sets\n",
    "1 $n 0\n",
    "2 ", $n+$k, " 0\n",
    "0\n";
  "c\n";

  ## varclasses
  print SYMFILEHANDLE "c varclasses\n",
    "1 $n ", $n+$k, " 0\n",
    "0\n",
    "c\n";

  ## symindex mappings
  print SYMFILEHANDLE "c symindex mappings\n";
  for (my $a=0; $a<$n; $a++) {
    for (my $i=0; $i<$k; $i++) {
      print SYMFILEHANDLE &X($a,$i), " 1 ", $a+1, ' ', $n+$i+1, " 0\n";
    }
  }
  print SYMFILEHANDLE "0\n";

  close(SYMFILEHANDLE);

  if ($opt_f) {
    print "Linking $symfilename to f$symfilename\n";
    system "ln -sf $symfilename f$symfilename";
  }
}
