#!/usr/bin/perl

###################################################
# Script for generating CNF formulas encoding the clique coloring
# problem. It supports symmetry information for Symchaff as well as
# pseudo-Boolean representations. Can also generate formulas for the
# 'functional' version of the clique coloring problem (easier).
#
# Created by Ashish Sabharwal, University of Washington, Seattle
# March 2005
#
# Usage example:
#   1) clqcolorgen.pl -bs -n 5 -m 4
#   2) clqcolorgen.pl -pbs -n 10 -m 6 -k 7
#   3) clqcolorgen.pl -h
###################################################


## use strict;
use Getopt::Std;

my $usage = 
    "\nUsage: clqcolorgen.pl -n <numNodes> -m <numColors> [-k <clqSize>] [-fbpsh]".
    "\n".
    "\n  -n nodes       number of nodes".
    "\n  -m colors      number of colors".
    "\n  -k size        clique size; default is m+1".
    "\n  -f             generate FUNCTIONAL clqcolor".
    "\n  -b             generate Boolean formula (.cnf)".
    "\n  -p             generate pseudo-Boolean formula (.pbcnf)".
    "\n  -s             generate symmetry information (.sym)".
    "\n  -h             help".
    "\n".
    "\nNote: at least one of -b, -p, -s must be specified".
    "\n\n";


######################
## Global variables ##
######################
my $n;
my $m;
my $k;
my $numVars      = 0;
my $numClauses   = 0;
my $numPBcons    = 0;
my $basefilename;
my $cnffilename;
my $pbcnffilename;
my $symfilename;

#############################
# Get current date and time #
#############################
my ($dd, $mm, $yy) = (localtime)[3,4,5];
$yy += 1900;
$mm += 1;
my $fileheader = "c Generated on $mm/$dd/$yy by Ashish Sabharwal, ashish\@cs.washington.edu\n";

#############################
## Handle input parameters ##
#############################

getopts('n:m:k:fbpsh') || die "$usage";
die "\nError: extra arguments $usage" if ($#ARGV != -1);  ## extra unwanted stuff specified
die "\nError: n must be positive $usage" if (!$opt_n);  ## n must be specified
die "\nError: m must be positive $usage" if (!$opt_m);  ## m must be specified

if ($opt_h) {print $usage; exit(1);}

if (!$opt_b && !$opt_p && !$opt_s) {print $usage; exit(0);}

$m = $opt_m;
$n = $opt_n;
$k = $m + 1;
if ($opt_k) {$k = $opt_k;}

($k >= 0) || die "k must be non-negative";
# ($m > $k)  || die "m must be larger than k";   # otherwise sat instance

$basefilename = "clqcolor-" . substr((100+$n),1) . "-" . substr((100+$m),1) . "-" . substr((100+$k),1);
$symfilename = $basefilename . ".sym";  # no f<name>.sym for functional symfiles

if ($opt_f) {$basefilename = "f" . $basefilename;}

$cnffilename = $basefilename . ".cnf";
$pbcnffilename = $basefilename . ".pbcnf";

!$opt_b or open (CNFFILEHANDLE, ">$cnffilename") or die "ERROR: Can't open file \"$cnffilename\": $!";
!$opt_p or open (PBCNFFILEHANDLE, ">$pbcnffilename") or die "ERROR: Can't open file \"$pbcnffilename\": $!";
!$opt_s or open (SYMFILEHANDLE, ">$symfilename") or die "ERROR: Can't open file \"$symfilename\": $!";



#####################################
## Generate CLQCOLOR_n_m_k formula ##
#####################################

## Nodes : a, b, ... in [0..n-1]
## Colors: s, t, ... in [0..m-1]
## Clique: i, j, ... in [0..k-1]

## Edge vars  : E(a,b)
## Color vars : C(a,s)
## Clique vars: Q(a,i)

## E(a,b) computes the variable index for e_{a,b}
sub E {
  my ($a,$b) = @_;
  if ($a > $b) {
    my $c = $a; $a = $b; $b = $c;
  }
  return $n*$a + $b + 1;
}

## C(a,s) computes the variable index for c_{a,s}
sub C {
  my ($a,$s) = @_;
  my $prev = $n*$n;
  return $prev + $m*$a + $s + 1;
}

## Q(a,i) computes the variable index for q_{a,i}
sub Q {
  my ($a,$i) = @_;
  my $prev = $n*$n + $n*$m;
  return $prev + $k*$a + $i + 1;
}

$numVars = $n*$n + $n*$m + $n*$k;



####################################
## Generate CNF formula if needed ##
####################################

if ($opt_b) {
  $numClauses = $n + $n + $m*($n*($n-1))/2 + $k + ($n*($n-1))/2*$k*($k-1) + $n*($k*($k-1))/2;
  if ($opt_f) {
    $numClauses += $n*($m*($m-1))/2 + ($n*($n-1))/2*$k;
  }

  print "Generating $cnffilename\n";

  print CNFFILEHANDLE $fileheader;
  if ($opt_f) {
    print CNFFILEHANDLE "c Functional Clique Coloring Principle FCLQCOLOR";
  }
  else {
    print CNFFILEHANDLE "c Clique Coloring Principle CLQCOLOR";
  }
  print CNFFILEHANDLE "($n,$m,$k)\n",
  "c $n nodes, $m colors, clique of size $k\n",
  "c $numVars variables, $numClauses clauses\n",
  "c\n",
  "p cnf $numVars $numClauses\n",
  "c\n";

  ## Color clauses: every node gets a color
  print CNFFILEHANDLE "c Color clauses\n";
  for (my $a=0; $a<$n; $a++) {
    for (my $s=0; $s<$m; $s++) {
      print CNFFILEHANDLE &C($a,$s), ' ';
    }
    print CNFFILEHANDLE "0\n";
  }
  print CNFFILEHANDLE "c\n";

  ## Valid coloring clauses: no two nodes sharing an edge get the same color
  print CNFFILEHANDLE "c Valid coloring clauses\n";
  for (my $a=0; $a<$n-1; $a++) {
    for (my $b=$a+1; $b<$n; $b++) {
      for (my $s=0; $s<$m; $s++) {
        print CNFFILEHANDLE '-', &E($a,$b), " -", &C($a,$s), " -", &C($b,$s), " 0\n";
      }
    }
  }
  print CNFFILEHANDLE "c\n";

  ## Clique clauses: some node is the i^th one in the clique
  print CNFFILEHANDLE "c Clique clauses\n";
  for (my $i=0; $i<$k; $i++) {
    for (my $a=0; $a<$n; $a++) {
      print CNFFILEHANDLE &Q($a,$i), ' ';
    }
    print CNFFILEHANDLE "0\n";
  }
  print CNFFILEHANDLE "c\n";

  ## Valid clique clauses: all nodes in the clique are connected
  print CNFFILEHANDLE "c Valid clique clauses\n";
  for (my $a=0; $a<$n-1; $a++) {
    for (my $b=$a+1; $b<$n; $b++) {
      for (my $i=0; $i<$k; $i++) {
        for (my $j=0; $j<$k; $j++) {
          if ($i != $j) {
            print CNFFILEHANDLE &E($a,$b), " -", &Q($a,$i), " -", &Q($b,$j), " 0\n";
          }
        }
      }
    }
  }
  print CNFFILEHANDLE "c\n";

  ## Clique size clauses: no node is counted twice in the clique
  print CNFFILEHANDLE "c Clique size clauses\n";
  for (my $a=0; $a<$n; $a++) {
    for (my $i=0; $i<$k-1; $i++) {
      for (my $j=$i+1; $j<$k; $j++) {
        print CNFFILEHANDLE '-', &Q($a,$i), " -", &Q($a,$j), " 0\n";
      }
    }
  }

  if ($opt_f) {
    print CNFFILEHANDLE "c\n";

    ## Functional color clauses: no node gets two colors
    print CNFFILEHANDLE "c\nc Functional color clauses\n";
    for (my $a=0; $a<$n; $a++) {
      for (my $s=0; $s<$m-1; $s++) {
        for (my $t=$s+1; $t<$m; $t++) {
	  print CNFFILEHANDLE '-', &C($a,$s), " -", &C($a,$t), " 0\n";
        }
      }
    }
    print CNFFILEHANDLE "c\n";

    ## Functional clique clauses: no two nodes are i^th in the clique
    print CNFFILEHANDLE "c\nc Functional clique clauses\n";
    for (my $a=0; $a<$n-1; $a++) {
      for (my $b=$a+1; $b<$n; $b++) {
        for (my $i=0; $i<$k; $i++) {
	  print CNFFILEHANDLE '-', &Q($a,$i), " -", &Q($b,$i), " 0\n";
        }
      }
    }
  }

  ## No loop clauses
  print CNFFILEHANDLE "c No loop clauses\n";
  for (my $a=0; $a<$n; $a++) {
      print CNFFILEHANDLE '-', &E($a,$a), " 0\n";
  }

  print CNFFILEHANDLE "0\n";

  close(CNFFILEHANDLE);
}



######################################
## Generate PBCNF formula if needed ##
######################################

if ($opt_p) {
  $numPBcons = $n + $m*($n*($n-1))/2 + $k + ($n*($n-1))/2 + $n;

  print "Generating $pbcnffilename\n";

  print PBCNFFILEHANDLE $fileheader;
  if ($opt_f) {
    print PBCNFFILEHANDLE "c Functional Clique Coloring Principle FCLQCOLOR";
  }
  else {
    print PBCNFFILEHANDLE "c Clique Coloring Principle CLQCOLOR";
  }
  print PBCNFFILEHANDLE "($n,$m,$k)\n",
  "c $n nodes, $m colors, clique of size $k\n",
  "c $numVars variables, $n clauses, $numPBcons pb-constraints\n",
  "c\n",
  "p pbcnf $numVars $n $numPBcons\n",
  "c\n",
  "c $n Boolean clauses\n";

  ## No loop clauses
  print PBCNFFILEHANDLE "c No loop clauses\n";
  for (my $a=0; $a<$n; $a++) {
      print PBCNFFILEHANDLE '-', &E($a,$a), " 0\n";
  }

  print PBCNFFILEHANDLE "0\n",
  "c\n";

  ## Color constraints: every node gets at least one color
  ## Functional: every node gets exactly one color
  print PBCNFFILEHANDLE "c Color PB constraints";
  print PBCNFFILEHANDLE " (functional)" if ($opt_f);
  print PBCNFFILEHANDLE "\n";
  for (my $a=0; $a<$n; $a++) {
    for (my $s=0; $s<$m; $s++) {
      print PBCNFFILEHANDLE &C($a,$s), " 1 ";
    }
    print PBCNFFILEHANDLE '>' if (!$opt_f);
    print PBCNFFILEHANDLE "= 1\n";
  }
  print PBCNFFILEHANDLE "c\n";

  ## Valid coloring constraints: no two nodes sharing an edge get the same color
  print PBCNFFILEHANDLE "c Valid coloring PB constraints\n";
  for (my $a=0; $a<$n-1; $a++) {
    for (my $b=$a+1; $b<$n; $b++) {
      for (my $s=0; $s<$m; $s++) {
        print PBCNFFILEHANDLE '-', &E($a,$b), " 1 -", &C($a,$s), " 1 -", &C($b,$s), " 1 >= 1\n";
      }
    }
  }
  print PBCNFFILEHANDLE "c\n";

  ## Clique constraints: some node is the i^th one in the clique
  ## Functional: exactly one node is the i^th one in the clique
  print PBCNFFILEHANDLE "c Clique PB constraints";
  print PBCNFFILEHANDLE " (functional)" if ($opt_f);
  print PBCNFFILEHANDLE "\n";
  for (my $i=0; $i<$k; $i++) {
    for (my $a=0; $a<$n; $a++) {
      print PBCNFFILEHANDLE &Q($a,$i), " 1 ";
    }
    print PBCNFFILEHANDLE '>' if (!$opt_f);
    print PBCNFFILEHANDLE "= 1\n";
  }
  print PBCNFFILEHANDLE "c\n";

  ## Valid clique constraints: all nodes in the clique are connected
  print PBCNFFILEHANDLE "c Valid clique PB constraints\n";
  for (my $a=0; $a<$n-1; $a++) {
    for (my $b=$a+1; $b<$n; $b++) {
      for (my $i=0; $i<$k; $i++) {
	print PBCNFFILEHANDLE &Q($a,$i), " 1 ";
      }
      for (my $i=0; $i<$k; $i++) {
	print PBCNFFILEHANDLE &Q($b,$i), " 1 ";
      }
      print PBCNFFILEHANDLE &E($a,$b), " -1 <= 1\n";
    }
  }
  print PBCNFFILEHANDLE "c\n";

  ## Clique size constraints: no node is counted twice in the clique
  print PBCNFFILEHANDLE "c Clique size PB constraints\n";
  for (my $a=0; $a<$n; $a++) {
    for (my $i=0; $i<$k; $i++) {
      print PBCNFFILEHANDLE &Q($a,$i), " 1 ";
    }
    print PBCNFFILEHANDLE "<= 1\n";
  }

  print PBCNFFILEHANDLE "0\n";

  close(PBCNFFILEHANDLE);
}




##################################
# create symmetry file if needed #
##################################

if ($opt_s) {
  print "Generating $symfilename\n";

  ## Nodes : a, b, ... in [0..m-1]
  ## Colors: s, t, ... in [0..n-1]
  ## Clique: i, j, ... in [0..k-1]

  ## Edge vars  : E(a,b)
  ## Color vars : C(a,s)
  ## Clique vars: Q(a,i)

  if (!$opt_f) {
    # make name consistent
    $cnffilename = "f" . $cnffilename;
  }
  my $numSymVars = $numVars;
  print SYMFILEHANDLE $fileheader;
  print SYMFILEHANDLE "c Symmetry file for ", substr($cnffilename,1),
  " and $cnffilename\n",
  "c $n nodes, $m colors, clique size $k, $numSymVars symmetric variables\n",
  "c 3 symindex sets, 3 varclasses\n",
  "c\n",
  "p sym $numSymVars 3 3\n",
  "c\n";

  ## symindex sets
  print SYMFILEHANDLE "c symindex sets\n",
  "1 $n 0\n",
  "2 ", $n+$m, " 0\n",
  "3 ", $n+$m+$k, " 0\n",
  "0\n";
  "c\n";

  ## varclasses
  print SYMFILEHANDLE "c varclasses\n",
  "1 $n $n 0\n",
  "2 $n ", $n+$m, " 0\n",
  "3 $n ", $n+$m+$k, " 0\n",
  "0\n",
  "c\n";

  ## symindex mappings
  print SYMFILEHANDLE "c symindex mappings\n";
  for (my $a=0; $a<$n; $a++) {
    for (my $b=0; $b<$n; $b++) {
      print SYMFILEHANDLE &C($a,$s), " 2 ", $a+1, ' ', $n+$s+1, " 0\n";
    }
  }
  for (my $a=0; $a<$n; $a++) {
    for (my $s=0; $s<$m; $s++) {
      print SYMFILEHANDLE &C($a,$s), " 2 ", $a+1, ' ', $n+$s+1, " 0\n";
    }
  }
  for (my $a=0; $a<$n; $a++) {
    for (my $i=0; $i<$k; $i++) {
      print SYMFILEHANDLE &Q($a,$i), " 3 ", $a+1, ' ', $n+$m+$i+1, " 0\n";
    }
  }
  print SYMFILEHANDLE "0\n";

  close(SYMFILEHANDLE);

  if ($opt_f) {
    print "Linking $symfilename to f$symfilename\n";
    system "ln -sf $symfilename f$symfilename";
  }
}
