#!/usr/bin/perl

###################################################
# Script for generating CNF formulas encoding
# the channel routing problem. It supports
# symmetry information for SymChaff.
#
# Created by Ashish Sabharwal, University of Washington, Seattle
# March 2005
#
# Usage example:
#   1) chnlgen.pl -t 10
#   2) chnlgen.pl -s -t 11 -n 13
#   3) chnlgen.pl -s -t 11 -n 20
###################################################


## use strict;
use Getopt::Std;

my $usage = 
    "\nUsage: chnlgen.pl -t <numTracks> [-n <numNets>] [-sh]".
    "\n".
    "\n  -t tracks      number of available tracks".
    "\n  -n nets        number of nets or connections; default is t+1".
    "\n  -s             generate symmetry file".
    "\n  -h             help".
    "\n\n";


######################
## Global variables ##
######################
my $n;
my $t;
my $numVars      = 0;
my $numClauses   = 0;
my $basefilename, $cnffilename, $symfilename;


#############################
## Handle input parameters ##
#############################

getopts('n:t:sh') || die "$usage";
die "\nError: extra arguments $usage" if ($#ARGV != -1);  ## extra unwanted stuff specified
die "\nError: t must be specified $usage" if (!$opt_t);  ## t must be specified

if ($opt_h) {print $usage; exit(1);}

$t = $opt_t;
$n = $t + 1;
if ($opt_n) {$n = $opt_n;}

($n > 0)  || die "n must be positive";
($t >= 0) || die "t must be non-negative";
# ($n > $t)  || die "n must be larger than t";   # otherwise sat instances

$basefilename = "chnl-" . substr((1000+$t),1) . "x" . substr((1000+$n),1);
$cnffilename  = "$basefilename.cnf";
open (CNFFILEHANDLE, ">$cnffilename") or die "ERROR: Can't open file \"$cnffilename\": $!";

if ($opt_s) {
    $symfilename = "$basefilename.sym";
  open (SYMFILEHANDLE, ">$symfilename") or die "ERROR: Can't open file \"$symfilename\": $!";
}


#########################################
## Generate and print the chnl formula ##
#########################################


## XOne(a,i) computes the variable index for x_{a,i} on first C-block
sub XOne {
  my ($a,$i) = @_;
  return $t*$a + $i + 1;
}

## XTwo(a,i) computes the variable index for x_{a,i} on second C-block
sub XTwo {
  my ($a,$i) = @_;
  return $n*$t + ($t*$a + $i + 1);
}

$numVars = $n * $t * 2;
$numClauses = 2*$n + 2*$t* ($n*($n-1))/2;

print "Generating $cnffilename\n";

print CNFFILEHANDLE "c Channel routing formula $basefilename\n",
  "c Route-based encoding\n",
  "c $n nets, $t tracks, $numVars variables, $numClauses clauses\n",
  "c\n",
  "p cnf $numVars $numClauses\n",
  "c\n";

## Liveness constraints: each net gets assigned a track in each C-block 
print CNFFILEHANDLE "c Liveness clauses\n";
# first C-block
for (my $a=0; $a<$n; $a++) {
  for (my $i=0; $i<$t; $i++) {
    print CNFFILEHANDLE &XOne($a,$i), ' ';
  }
  print CNFFILEHANDLE "0\n";
}
# second C-block
for (my $a=0; $a<$n; $a++) {
  for (my $i=0; $i<$t; $i++) {
    print CNFFILEHANDLE &XTwo($a,$i), ' ';
  }
  print CNFFILEHANDLE "0\n";
}
print CNFFILEHANDLE "c\n";

## Exclusivity constraints
print CNFFILEHANDLE "c Exclusivity clauses\n";
for (my $i=0; $i<$t; $i++) {
  for (my $a=0; $a<$n-1; $a++) {
    for (my $b=$a+1; $b<$n; $b++) {
      print CNFFILEHANDLE '-', &XOne($a,$i), " -", &XOne($b,$i), " 0\n";
    }
  }
}
for (my $i=0; $i<$t; $i++) {
  for (my $a=0; $a<$n-1; $a++) {
    for (my $b=$a+1; $b<$n; $b++) {
      print CNFFILEHANDLE '-', &XTwo($a,$i), " -", &XTwo($b,$i), " 0\n";
    }
  }
}

print CNFFILEHANDLE "0\n";

close(CNFFILEHANDLE);



##################################
# create symmetry file if needed #
##################################

exit(0) if (!$opt_s);

print "Generating $symfilename\n";

## Note that all nets are symmetric in this case, but in general,
##  only tracks are symmetric, nets are not.
## For generality, I'll create symmetry only for tracks.
## This increases the number of vartypes, but reduces the load on 
##  zChaff to decide which symmetry to use!

my $numSymindexSets = 2;   # one for connecting each end of the nets
my $numVartypes = 2 * $n;

print SYMFILEHANDLE "c Symmetry file for $cnffilename\n",
  "c $n nets, $t tracks\n",
  "c $numVars symmetric variables, $numSymindexSets symindex sets, $numVartypes vartypes\n",
  "c\n",
  "p sym $numVars $numSymindexSets $numVartypes\n",
  "c\n";

## symindex sets
print SYMFILEHANDLE "c symindex sets\n",
  "1 $t 0\n",
  "2 ", 2*$t, " 0\n",
  "0\n";
  "c\n";

## vartypes
print SYMFILEHANDLE "c vartypes\n";
for my $j (1 .. $n) {
    print SYMFILEHANDLE "$j $t 0\n";    # beginning of nets
}
for my $j ($n+1 .. 2*$n) {
    print SYMFILEHANDLE "$j ", 2*$t, " 0\n";  # end of nets
}
print SYMFILEHANDLE "0\nc\n";

## symindex mappings
print SYMFILEHANDLE "c symindex mappings\n";
for (my $a=0; $a<$n; $a++) {
    for (my $i=0; $i<$t; $i++) {
        print SYMFILEHANDLE &XOne($a,$i), " ", $a+1, " ", $i+1, " 0\n";
    }
}
for (my $a=0; $a<$n; $a++) {
    for (my $i=0; $i<$t; $i++) {
        print SYMFILEHANDLE &XTwo($a,$i), " ", $n+$a+1, " ", $t+$i+1, " 0\n";
    }
}
print SYMFILEHANDLE "0\n";

close(SYMFILEHANDLE);
