/* Lowerbound, equivalences, sampleonly, printsamples, and picksafeuneven added */
/*                    by Ashish Sabharwal <sabhar@cs.cornell.edu> */

/* SampleSat and ApproxCount implementation by Wei Wei <weiwei@cs.cornell.edu> */
/* MPI code by Jordan Erenrich <erenrich@cs.cornell.edu> */
/* Original WalkSAT code by Henry Kautz <kautz@cs.washington.edu> */
#define VERSION "walksat-v42MPI"

/************************************/
/* Compilation flags				*/
/************************************/

/* If the constant DYNAMIC is set, then dynamic arrays are used instead of static arrays.
This allows very large or very small problems to be handled, without
having to adjust the constants MAXATOM and/or MAXCLAUSE.  However, on some
architectures the compiled program is about 25% slower, because not
as many optimizations can be performed. */

/* #define DYNAMIC 1 */

/***************************************************************/
/* Please select only one of following flags: UNIX, ANSI or NT */
/* Description: 											   */
/*	  UNIX:  uses some non-POSIX unix time routines, compile   */
/*			 successful under SunOS 5.6, time accuracy 1/60sec */
/*	  ANIS:  use POSIX time routines, can compile under all    */
/*			 unix and NT, time accuracy is 1sec 			   */ 
/*	  NT:	 use NT/Win32 multimedia routines, compile under   */
/*			 NT only, time accuracy is 1ms					   */
/*			 Uses NT "timeGetTime" function:				   */
/*			   Header: Decl. in Mmsystem.h; include Windows.h. */
/*			   Library: Use Winmm.lib.						   */
/***************************************************************/

#define UNIX 1
/* #define ANSI 1 */
/* #define NT 1 */

/*Distribute the WalkSAT trials over a cluster with MPI*/
//#define MPICODE 1 //JSE


/* Define BIGINT to be the type used for the "cutoff" variable.
Under gcc "long long int" gives a 64 bit integer.
Program will still function using a 32-bit value, but it
limit size of cutoffs that can be specified. */

#ifdef UNIX
#define BIGINT long long int
#endif
#ifdef NT
#define BIGINT __int64
#endif
#ifdef ANSI
#define BIGINT long int
#endif

#ifdef MPICODE
#ifdef UNIX
#define MPI_myBIGINT MPI_LONG_LONG
#endif
#ifdef NT
#define MPI_myBIGINT MPI_LONG_LONG
#endif
#ifdef ANSI
#define MPI_myBIGINT long int
#endif
#endif


/************************************/
/* Standard includes				*/
/************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <sys/types.h>
#include <limits.h>
#include <signal.h>
#include <unistd.h>

// Ashish
#include <vector>
using namespace std;
typedef pair<int, int> intPair;   // pair of integers

#ifdef UNIX
#include <sys/times.h>
#include <sys/time.h>
#endif
#ifdef NT
#include <time.h>
#include <windows.h>
#include <mmsystem.h>
#endif
#ifdef ANSI
#include <sys/time.h>
#endif

#ifndef UNIX   /* ANSI or NT */
#define random() rand()
#define srandom(seed) srand(seed)
#endif

/* Sometimes CLK_TCK is defined in a weird place */
#ifndef CLK_TCK
#define CLK_TCK 60
#endif

#define CURRENTFILENAME "currentfile1.cnf"
#define TEMPFILENAME "tempfile1.cnf"
#define OUTPUTFILENAME "outputfile1.cnf"
#define CUTOFF 10000
#define NUMSAMPLE 1000

/************************************/
/* Constant parameters				*/
/************************************/

#define MAXATOM 150000		/* maximum possible number of atoms */

#ifdef DYNAMIC
#define STOREBLOCK 2000000	/* size of block to malloc each time */
#else
#define STOREBLOCK 2000000	/* size of block to malloc each time */
#define MAXCLAUSE 500000	/* maximum possible number of clauses */
#endif

#define MAXSTORE 5000

#define TRUE 1
#define FALSE 0

#define MAXLENGTH 500			/* maximum number of literals which can be in any clause */

/************************************/
/* Internal constants				*/
/************************************/

enum heuristics { RANDOM, BEST, TABU, NOVELTY, RNOVELTY, SA, SARANDOM };

#define NOVALUE -1
#define INIT_PARTIAL 1
#define HISTMAX 101 	/* length of histogram of tail */

#define Var(CLAUSE, POSITION) (ABS(clause[CLAUSE][POSITION]))

static int scratch;
#define ABS(x) ((scratch=(x))>0?(scratch):(-scratch))

#define BIG 100000000

int numatom2;
int numclause2;
int lits[MAXLENGTH+1];
double currentcount = 1.0;
char currentcount_str[1<<20];     // Ashish
FILE * infile = stdin;
char  currentfilestr[1024]= {0};
char  tempfilestr[1024]={0};
char  tempfilestr2[1024]={0};
int unitpropagation =1;
int * origin;
int * pos_array;
int * neg_array;
int ** same_array;   // Ashish: for variable equivalence (2-xors)
int ** diff_array;   // Ashish: for variable equivalence (2-xors)
int storerecnum;
int * storerec[MAXSTORE];

/************************************/
/* Main data structures 			*/
/************************************/

/* Atoms start at 1 */
/* Not a is recorded as -1 * a */
/* One dimensional arrays are statically allocated. */
/* Two dimensional arrays are dynamically allocated in */
/* the second dimension only.  */

int numatom;
int numclause;
int numliterals;

int counter=0;

#ifdef DYNAMIC
int ** clause;			/* clauses to be satisfied */
/* indexed as clause[clause_num][literal_num] */
int * size; 		/* length of each clause */
int * falseC;			/* clauses which are false */
int * lowfalse;
int * wherefalse;		/* where each clause is listed in false */
int * numtruelit;		/* number of true literals in each clause */
#else
int * clause[MAXCLAUSE];	/* clauses to be satisfied */
/* indexed as clause[clause_num][literal_num] */
int size[MAXCLAUSE];		/* length of each clause */
int falseC[MAXCLAUSE];		/* clauses which are false */
int lowfalse[MAXCLAUSE];
int wherefalse[MAXCLAUSE];	/* where each clause is listed in false */
int numtruelit[MAXCLAUSE];	/* number of true literals in each clause */
#endif

int *occurence[2*MAXATOM+1];	/* where each literal occurs */
/* indexed as occurence[literal+MAXATOM][occurence_num] */

int numoccurence[2*MAXATOM+1];	/* number of times each literal occurs */


int atom[MAXATOM+1];		/* value of each atom */ 
int lowatom[MAXATOM+1];
int solution[MAXATOM+1];

BIGINT changed[MAXATOM+1]; 	/* step at which atom was last flipped */ /*JSE Mod - Changed int to BigInt*/

int breakcount[MAXATOM+1];	/* number of clauses that become unsat if var if flipped */
int makecount[MAXATOM+1];	/* number of clauses that become sat if var if flipped */

int numfalse;			/* number of false clauses */

/************************************/
/* Global flags and parameters		*/
/************************************/

int status_flag = 0;		/* value returned from main procedure */
int abort_flag;

// Ashish
bool lowerbound_flag = false;
bool sampleonly_flag = false;
bool printsamples_flag = false;
bool equiv_flag = false;
bool equivonly_flag = false;
bool knuth_flag = false;
double lb_slack = 1;

int heuristic = BEST;		/* heuristic to be used */
//int countvar = 1;
int numerator = NOVALUE;	/* make random flip with numerator/denominator frequency */
int denominator = 100;		
int novelty_numerator = 0;
int novelty_denominator = 100;	/* used by novelty+ */
int tabu_length;		/* length of tabu list */

BIGINT numflip; 	/* number of changes so far */
BIGINT numnullflip; 	/*	number of times a clause was picked, but no  */
/*	variable from it was flipped  */
int numrun = 10;
BIGINT cutoff = 100000;
BIGINT base_cutoff = 100000;
int target = 0;
int numtry = 0; 		/* total attempts at solutions */
int numsol = NOVALUE;		/* stop after this many tries succeeds */
int superlinear = FALSE;

int makeflag = FALSE;		/* set to true by heuristics that require the make values to be calculated */

/* Histogram of tail */

long int tailhist[HISTMAX]; /* histogram of num unsat in tail of run */
long histtotal;
int tail = 3;
int tail_start_flip;

/* Printing options */

int printonlysol = FALSE;
int printsolcnf = FALSE;
int printfalse = FALSE;
int printlow = FALSE;
int printhist = FALSE;
int printtrace = FALSE;
int trace_assign = FALSE;
int printEveryN = 1;

char outfile[1024] = { 0 };
char outfileAll[1024] = { 0 };
char numsolfile[1024] = { 0 };

/* Initialization options */

char initfile[100] = { 0 };
int initoptions = FALSE;

/* Randomization */

unsigned int seed, seed2;	/* Sometimes defined as an unsigned long int */

#ifdef UNIX
struct timeval tv;
struct timezone tzp;
#endif
#ifdef NT
DWORD win_time; 	/* elapsed time in ms, since windows boot up */
#endif

/* Statistics */

double expertime;
BIGINT flips_this_solution;
long int lowbad;		/* lowest number of bad clauses during try */
BIGINT totalflip = 0;		/* total number of flips in all tries so far */
BIGINT totalsuccessflip = 0;	/* total number of flips in all tries which succeeded so far */
int numsuccesstry = 0;		/* total found solutions */

BIGINT x;
BIGINT integer_sum_x = 0;
double sum_x = 0.0;
double sum_x_squared = 0.0;
double mean_x;
double second_moment_x;
double variance_x;
double std_dev_x;
double std_error_mean_x;
double seconds_per_flip;
int r;
int sum_r = 0;
double sum_r_squared = 0.0;
double mean_r;
double variance_r;
double std_dev_r;
double std_error_mean_r;

double avgfalse;
double sumfalse;
double sumfalse_squared;
double second_moment_avgfalse, variance_avgfalse, std_dev_avgfalse, ratio_avgfalse;
//double std_dev_avgfalse //JSE - Unused variable
double f;
double sample_size;

double sum_avgfalse = 0.0;
double sum_std_dev_avgfalse = 0.0;
double mean_avgfalse;
double mean_std_dev_avgfalse;
int number_sampled_runs = 0;
double ratio_mean_avgfalse;

double suc_sum_avgfalse = 0.0;
double suc_sum_std_dev_avgfalse = 0.0;
double suc_mean_avgfalse;
double suc_mean_std_dev_avgfalse;
int suc_number_sampled_runs = 0;
double suc_ratio_mean_avgfalse;

double nonsuc_sum_avgfalse = 0.0;
double nonsuc_sum_std_dev_avgfalse = 0.0;
double nonsuc_mean_avgfalse;
double nonsuc_mean_std_dev_avgfalse;
int nonsuc_number_sampled_runs = 0;
double nonsuc_ratio_mean_avgfalse;

/* Hamming calcualations */

char hamming_target_file[512] = { 0 };
char hamming_data_file[512] = { 0 };
int hamming_sample_freq;
int hamming_flag = FALSE;
int hamming_distance;
int hamming_target[MAXATOM+1];
void read_hamming_file(char initfile[]);
void open_hamming_data(char initfile[]);
int calc_hamming_dist(int atom[], int hamming_target[], int numatom);
FILE * hamming_fp;

/* Noise level */
int samplefreq = 1;

int sa_ratio;
int temperature;
int dont_care_fract;
int exact_count_nvar= 50;
int verify_flag = 0;
int latesa = 1;
int doublecheck = 0;
int exact_time_limit=100;
int pickeven = 0;
// int pickuneven = 1;    // Ashish: doing this after parsing parameters
int pickuneven = 0;
int picksafeuneven = 0;   // Ashish: added this to avoid multiplier of infinity
int pickrand = 0;
int picknatural = 0;
int inclnonsoln = 0;
int imbalance = 0;

/************************************/
/* Forward declarations 			*/
/************************************/

void readatom(FILE * infile);
void sample(char * filename);
int updatefile(int next, int iteration);
int updatefile_e(intPair litpair, int iteration);   // Ashish
void updateorigin(int next, int iteration);
void updatefile_nu(int next);
void updatefile_e_nu(intPair litpair);   // Ashish
void parse_param(int argc, char* argv[]);
void print_param();
void parse_parameters(char * _filename);
void print_parameters(int argc, char * argv[]);
int findlit(int next);
double retrievesolution(char * tempfilestr, char * resultsolutionstr);
double retrievesolution2(char * tempfilestr, char * resultsolutionstr);
double retrievesolution3(char * tempfilestr, char * resultsolutionstr);
double countexact(int change, int exact_time_limit);
int fpickeven(int numvar);
int fpickuneven(int numvar);
int fpicksafeuneven(int numvar);    // Ashish
int fpickrand(int numvar);

// Ashish
intPair fpickeven_e(int numvar);
intPair fpickuneven_e(int numvar);
intPair fpicksafeuneven_e(int numvar);
intPair fpickrand_e(int numvar);


int pickrandom(void);
int pickbest(void);
int picksa(void);
int picksarandom(void);
int picktabu(void);
int picknovelty(void);
int pickrnovelty(void);
int picknoveltyplus(void);
char * heuristic_names[] = { "random", "best", "tabu", "novelty", "rnovelty", "sa", "sarandom" };
static int (*pickcode[])(void) = {pickrandom, pickbest, picktabu, picknovelty, pickrnovelty, picksa, picksarandom };

double elapsed_seconds(void);
int countunsat(void);
void scanone(int argc, char *argv[], int i, int *varptr);
void scanone(int argc, char *argv[], int i, unsigned int *varptr); //JSE MOD - Compiler complained without unsigned type
void scanone(int argc, char *argv[], int i, BIGINT *varptr);
void scanone(int argc, char *argv[], int i, double *varptr);   // Ashish
void init(char initfile[], int initoptions);
void initprob(void); 
void flipatom(int toflip);

void print_false_clauses(long int lowbad);
void save_false_clauses(long int lowbad);
void print_low_assign(long int lowbad);
void save_low_assign(void);
void save_solution(void);

void print_current_assign(void);
void handle_interrupt(int sig);

long super(int i);

void print_sol_file(char * filename);
void print_sol_fileAll(char * filename); //JSE
void print_statistics_header(void);
void initialize_statistics(void);
void update_statistics_start_try(void);
void print_statistics_start_flip(void);
void update_and_print_statistics_end_try(void);
void update_statistics_end_flip(void);
void print_statistics_final(void);
void print_sol_cnf(void);
void freethings(void);

/************************************/
/* JSE vars 						*/
/************************************/
#include <fstream>
#include "Solutions.h"
Solutions solutionMap;
SATSolution satSoln;
const int MAXSTRING = 500;
char inputCNFFile[MAXSTRING] = ""; //Loads the SAT instance from this file
char nextChar(FILE *fp);  //Misc. helper function for file input

#ifdef MPICODE	//JSE MPI Code
#include "mpi.h"
const int SOLNTAG = 1;
const int STARTTAG = 2;
const int STOPTAG = 3;
const int SEEDTAG = 4;

const int MAXBUFF = 2*MAXATOM+MAXCLAUSE+HISTMAX+500;
char commBuf[MAXBUFF]; //Byte array - Next version will include dynamic allocation

int mpi_id;
int numSlavesProcs = 0, procRecv;
const int ROOT_PROCID = 0;
bool rootProc;

bool mpiTagRecv(int tag);
void mpiRecvTry(void);
void mpiSendTry(void);
void mpiWaitForAll(void);

unsigned int hash(unsigned int);
#else
const int procRecv = 0;
#endif


/************************************/
/* Main 							*/
/************************************/
int main(int argc, char * argv[])
{
	int pos_appear = 0, neg_appear = 0;
	int same_appear = 0, diff_appear = 0;   // Ashish
	int next =1;
	intPair nextpair(1,2);   // Ashish
	int unitprop = 0;
	double multiplier;
	int i;
	int k;
	double resultsolution, resultsolution2;
    char resultsolutionstr[1<<20], resultsolutionstr2[1<<20];   // Ashish: for larger than MAX_DOUBLE!
	char scache[100];
	double pos_exact = -1, neg_exact = -1;
	double exactcount =1.0;
	double exactmultiplier = 0;
	
	int num_unitprop = 0;
	int num_correct = 0;
	int num_incorrect = 0;
	int num_determined = 0;
	
	// Ashish
	int lowerbound_log2 = 0;
	double lowerbound_residual = 1.0;
	char lowerbound_residual_str[1<<20] = "1";

#ifdef UNIX
	gettimeofday(&tv,&tzp);
 	seed = (unsigned int)((( tv.tv_sec & 0177 ) * 1000000) + tv.tv_usec);
#endif
#ifdef NT
  	seed = (unsigned int)(timeGetTime());
#endif
#ifdef ANSI
  	seed = (unsigned int)(time());
#endif
	sprintf(currentfilestr, "/tmp/%s_%u", CURRENTFILENAME, seed);
	sprintf(tempfilestr, "/tmp/%s_%u", TEMPFILENAME, seed);
	sprintf(tempfilestr2, "/tmp/%s_%u_02", TEMPFILENAME, seed);
	// printf("approxcount version 1.0\ncommand line: ");
	// Ashish: looks like this is version 1.2!
  //         with lowerbound: 1.2L
	printf("Approxcount version 1.2L (with lowerbounds)\n");
	printf("Copyright Cornell University, 2007\n");
	printf("See approxcount -h and README-approxcount for help.\n");
	printf("\nCommand line: ");
  	for (i=0;i<argc;i++) printf(" %s", argv[i]);
  	printf("\n\n");

	parse_param(argc, argv);
	print_param();
	readatom(infile);
	if (exact_count_nvar <=0) //exact count when N var remain
		exact_count_nvar = numatom2 + exact_count_nvar + 1;
	origin = (int *)malloc((numatom2+1) * sizeof(int));
	if (!equivonly_flag) {  // Ashish
	  pos_array = (int *)malloc((numatom2+1) * sizeof(int));
	  neg_array = (int *)malloc((numatom2+1) * sizeof(int));
    }
	// Ashish: initialize same_array and diff_array
	if (equiv_flag) {
	  same_array = (int **)malloc((numatom2+1) * sizeof(int *));
	  diff_array = (int **)malloc((numatom2+1) * sizeof(int *));
	  for (i=1; i<=numatom2; i++) {
		same_array[i] = (int *)malloc((numatom2+1) * sizeof(int));
		diff_array[i] = (int *)malloc((numatom2+1) * sizeof(int));
	  }
	}
	for (i = 1; i <= numatom2; i++)
		origin[i] = i;
	for (i = 1; i<=numatom2; i++)
	{
		printf("iteration %i\n",i);
		printf("current formula has %i variables and %i clauses\n", numatom2-i+1, numclause2);
		if ((i >= exact_count_nvar) && unitpropagation && !unitprop) {
			printf("invoking exact count\n");
			if (doublecheck) {
				sprintf(scache, "./exactcount %s %i %s %s", currentfilestr, exact_time_limit, tempfilestr, tempfilestr2);
				system(scache);
				resultsolution = retrievesolution2(tempfilestr, resultsolutionstr);
				resultsolution2 = retrievesolution3(tempfilestr2, resultsolutionstr2);
				remove(tempfilestr);
				remove(tempfilestr2);
				if (resultsolution >= 0.0) {
					printf("cachet finds current formula has %.0f solutions.\n", resultsolution);
					printf(" (as a string: %s solutions)\n", resultsolutionstr);
				}
				else {
					printf("cachet fails in %i seconds. continue approx count.\n", exact_time_limit);
					if (resultsolution2 >=0.0) {
						resultsolution = resultsolution2; //take relsat value
						strcpy(resultsolutionstr, resultsolutionstr2);
					}
				}
				if (resultsolution2 >= 0.0) {
					printf("relsat finds current formula has %.0f solutions.\n", resultsolution);
					printf(" (as a string: %s solutions)\n", resultsolutionstr);
				}
				else 
					printf("relsat fails in %i seconds. continue approx count.\n", exact_time_limit);
			}	
			else {
				//sprintf(scache, "./cachet %s -t %i >%s", currentfilestr, exact_time_limit, tempfilestr);
				sprintf(scache, "cachet %s -t %i >%s", currentfilestr, exact_time_limit, tempfilestr);
				system(scache);
				resultsolution = retrievesolution(tempfilestr, resultsolutionstr);
				remove(tempfilestr);
				if (resultsolution >= 0.0) {
					printf("cachet finds current formula has %.0f solutions.\n", resultsolution);
					printf(" (as a string: %s solutions)\n", resultsolutionstr);
				}
				else 
					printf("cachet fails in %i seconds. continue approx count.\n", exact_time_limit);
			}
			// Ashish
			if (lowerbound_flag) {
				lowerbound_residual = resultsolution;
				strcpy(lowerbound_residual_str, resultsolutionstr);
			}
			if (resultsolution >= 0.0) {
				sprintf(currentcount_str, "%s x %lf", resultsolutionstr, currentcount);
				currentcount *= resultsolution;
				exactcount *= resultsolution;
				break;
			}
		} 
		if (!unitpropagation) {
			next = i;
			unitprop = 0;
		}
		if (unitprop) {

			printf("unit propagating %i, (%i in original formula)\n", next, next/abs(next)*origin[abs(next)]);
			next = updatefile(next,i );
			// printf("multiplier: 1.0\n");   // Ashish: commented out
			if (verify_flag) {
				num_unitprop++;
				// printf("exact product so far: %f\n", exactcount);
				// Ashish: easier to read in scientific notation; 
				//         changed %f to %.2e or %.3e everywhere for partial counts!
				// printf("exact product so far: %.2e\n", exactcount);   // Ashish: commented out
			}
			// printf("product so far: %.2e\n", currentcount);  // Ashish: commented out
// 			if (lowerbound_flag) {   // Ashish: commented out
//               if (!knuth_flag)
//                 printf("lowerbound so far: 2^(%i-%.1lf) = %.2e\n", 
//                        lowerbound_log2, lb_slack,
//                        pow(2.0,(lowerbound_log2 - lb_slack)));
//               else
//                 printf("knuth-based lowerbound so far: %.6e\n",
//                        currentcount/pow(2.0,lb_slack));
//             }
			printf("__________________________________________________\n");
			}
		else if (!findlit(next)) {
			printf("variable %i irrelevant, (%i in original formula)\n", abs(next), origin[abs(next)]);
			printf("asserting literal %i, (%i in original formula)\n", abs(next), origin[abs(next)]);
			printf("multiplier: 2.0\n");
			if (unitpropagation) 
				next = updatefile(next,i);
			else 
				updatefile_nu(next);
			currentcount *= 2.0;
			if (lowerbound_flag) ++lowerbound_log2;   // Ashish
			if (verify_flag) exactcount *= 2.0;
			if (verify_flag) printf("exact product so far: %.2e\n", exactcount);
			printf("product so far: %.2e\n", currentcount);
			if (lowerbound_flag) {     // Ashish
              if (!knuth_flag)
                printf("lowerbound so far: 2^(%i-%.1lf) = %.2e\n", 
                       lowerbound_log2, lb_slack,
                       pow(2.0,(lowerbound_log2 - lb_slack)));
              else
                printf("knuth-based lowerbound so far: %.6e\n",
                       currentcount/pow(2.0,lb_slack));
            }

			printf("__________________________________________________\n");
			}
		else {
			if (!equivonly_flag) {   // Ashish
			  for (k=1; k<=numatom2; k++) {
				pos_array[k]=0;
				neg_array[k]=0;
			  }
			}
			if (equiv_flag) {        // Ashish
			  for (k=1; k<numatom2; k++) {
				for (int ell=k+1; ell<=numatom2; ell++) {
				  same_array[k][ell]=0;
				  diff_array[k][ell]=0;
				}
			  }
			}
			sample(currentfilestr);
			if ((!inclnonsoln) && numsuccesstry == 0) {
				printf("No solution found. Consider increasing cutoff value\n");
				exit(1);
			}
			if (sampleonly_flag) {   // Ashish: exit after sampling
			  cout << solutionMap.getSolnCount() << " total solutions sampled" << endl;
			  cout << solutionMap.getUniqueSolnCount() << " unique solutions sampled" << endl;
			  cout << "Sampled solutions sorted lexicographically:" << endl;
			  solutionMap.printSolnsWithFreq(cout);

			  cout << solutionMap.getSolnCount() << " total solutions sampled" << endl;
			  cout << solutionMap.getUniqueSolnCount() << " unique solutions sampled" << endl;
// 			  cout << "Sampled solutions sorted by Hamming distance from previous solution:" << endl;
// 			  solutionMap.printSolnsWithFreqHammDistSort(cout);
			  exit(0);
			}

            // Ashish
			bool use_single_lit = true;;
			bool is_already_perfect = false;
			int value_single_lit = 0;

			if (!equivonly_flag) {   // Ashish
			  if (unitpropagation)
				if (pickuneven) next = fpickuneven(numatom2-i+1);
				else if (picksafeuneven) next = fpicksafeuneven(numatom2-i+1);
				else if (pickeven) next = fpickeven(numatom2-i+1);
				else if (pickrand) next = fpickrand(numatom2-i+1);
				else next= 1; //picknatural;
			  pos_appear = pos_array[next];
			  neg_appear = neg_array[next];
			  if (equiv_flag) {
				// check if single lit is already as good as it gets
				value_single_lit = abs(pos_appear - neg_appear);
				is_already_perfect = (pickuneven && (pos_appear == 0 || neg_appear == 0))
				  || (picksafeuneven && (pos_appear == 1 || neg_appear == 1))
				  || (pickeven && value_single_lit == 0);
			  }
			}

			// Ashish: check if 2-xors are better
			if (!is_already_perfect && equiv_flag) {
			  if (unitpropagation)
				if (pickuneven) nextpair = fpickuneven_e(numatom2-i+1);
				else if (picksafeuneven) nextpair = fpicksafeuneven_e(numatom2-i+1);
				else if (pickeven) nextpair = fpickeven_e(numatom2-i+1);
				else if (pickrand) nextpair = fpickrand_e(numatom2-i+1);
				else nextpair = intPair(1,2); //picknatural;
			  same_appear = same_array[nextpair.first][nextpair.second];
			  diff_appear = diff_array[nextpair.first][nextpair.second];
			  if (equivonly_flag)
				use_single_lit = false;
			  else {
				int value_equiv = abs(same_appear - diff_appear);
				if (pickuneven) 
				  use_single_lit = (value_single_lit >= value_equiv);
				else if (picksafeuneven) 
				  use_single_lit = (value_equiv == numsuccesstry || (value_single_lit >= value_equiv && value_single_lit != numsuccesstry));
				else if (pickeven) 
				  use_single_lit = (value_single_lit <= value_equiv);
				else // pickrand
				  use_single_lit = (random()%2 == 0);
			  }
			}

			// Ashish: handle equivalences (2-xors) separately
			if (use_single_lit) {
				counter = 0;
				printf("sampling on variable %i, (%i in original formula)\n", abs(next), origin[abs(next)]);
				
				if (verify_flag) {
					pos_exact = countexact(abs(next), exact_time_limit);
					neg_exact = countexact(-abs(next), exact_time_limit);
					if ((pos_exact <0.0) || (neg_exact <0.0)) {
						pos_exact = pos_appear;
						neg_exact = neg_appear;
					}
				}
			
				printf("number of successful tries: %i\n", numsuccesstry);
				printf("positive appearence: %i   negative appearence: %i\n", pos_appear, neg_appear); 
				// Ashish: for lowerbound, pick pos or neg uniformly at random
				bool force_pos = false;
				if (lowerbound_flag) {
				  // Do not rely on variables that did not sample both positively
				  // and negatively; could make the formula unsat with 50% chance
				  if (pos_appear == 0 || neg_appear == 0) {
					printf("not flipping random coin for lowerbound\n");
					--lowerbound_log2;
					force_pos = (pos_appear > 0);
				  }
				  else {
                    if (!knuth_flag)
                      force_pos = (random() % 2 == 0);
                    else
                      force_pos = (random() % (pos_appear+neg_appear) < pos_appear);
                  }
				}
				if (lowerbound_flag && force_pos || 
				 	(!lowerbound_flag && (pos_appear > neg_appear || ((pos_appear == neg_appear) && (random()%2))))) { 
					if (verify_flag) exactmultiplier = (pos_exact + neg_exact)/pos_exact;
					// Ashish: make the following work also when pos_appear < neg_appear
					// if (pos_appear > (pos_appear + neg_appear)* (50+dont_care_fract) /100)
					int majority_appear = (pos_appear > neg_appear) ? pos_appear : neg_appear;
					if (majority_appear > (pos_appear + neg_appear) * (50+dont_care_fract) /100)
						multiplier = ((pos_appear + neg_appear)*1.0)/pos_appear;
					else {
						multiplier = 2.0;
						printf("don't care variable\n");
					}
					if (lowerbound_flag) ++lowerbound_log2;   // Ashish
					printf("asserting literal %i, (%i in original formula)\n", abs(next), origin[abs(next)]);
					if (unitpropagation) 
						next = updatefile(next,i);
					else 
						updatefile_nu(next);
					}
				else {
					if (verify_flag) exactmultiplier = (pos_exact + neg_exact)/neg_exact;
					// Ashish: make the following work also when neg_appear < pos_appear
					// if (neg_appear > (pos_appear + neg_appear)* (50+dont_care_fract) /100)
					int majority_appear = (pos_appear > neg_appear) ? pos_appear : neg_appear;
					if (majority_appear > (pos_appear + neg_appear)* (50+dont_care_fract) /100)
						multiplier = ((pos_appear + neg_appear)*1.0)/neg_appear;
					else {
						multiplier = 2.0;
						printf("don't care variable\n");
						}
					if (lowerbound_flag) ++lowerbound_log2;   // Ashish
					printf("asserting literal %i, (%i in original formula)\n", -abs(next), -origin[abs(next)]);
					if (unitpropagation) 
						next = updatefile(-next,i);
					else 
						updatefile_nu(-next);
				}
			}
            else {   // Ashish: use variable equivalences (2-xors)
				if (verify_flag)
                  exit(fprintf(stderr, "ERROR: verify_flag not supported with -equiv\n"));
				counter = 0;
				printf("sampling on variable pair (%i,%i), (originally (%i,%i))\n", abs(nextpair.first), abs(nextpair.second), origin[abs(nextpair.first)], origin[abs(nextpair.second)]);
				printf("number of successful tries: %i\n", numsuccesstry);
				printf("same appearence: %i   different appearence: %i\n", same_appear, diff_appear); 
				// for lowerbound, pick same or diff uniformly at random
				bool force_same = false;
				if (lowerbound_flag) {
				  // Do not rely on variables that did not sample both same
				  // and different; could make the formula unsat with 50% chance
				  if (same_appear == 0 || diff_appear == 0) {
					printf("not flipping random coin for lowerbound\n");
					--lowerbound_log2;
					force_same = (same_appear > 0);
				  }
				  else {
                    if (!knuth_flag)
                      force_same = (random() % 2 == 0);
                    else
                      force_same = (random() % (same_appear+diff_appear) < same_appear);
                  }
				}
				if (lowerbound_flag && force_same || 
				 	(!lowerbound_flag && (same_appear > diff_appear || ((same_appear == diff_appear) && (random()%2))))) { 
					int majority_appear = (same_appear > diff_appear) ? same_appear : diff_appear;
					if (majority_appear > (same_appear + diff_appear) * (50+dont_care_fract) /100)
						multiplier = ((same_appear + diff_appear)*1.0)/same_appear;
					else {
						multiplier = 2.0;
						printf("don't care equivalence\n");
					}
					if (lowerbound_flag) ++lowerbound_log2;
					printf("replacing literal %i with %i, (originally %i with %i)\n", abs(nextpair.second), abs(nextpair.first), origin[abs(nextpair.second)], origin[abs(nextpair.first)]);
					if (unitpropagation) 
						next = updatefile_e(nextpair, i);
					else 
						updatefile_e_nu(nextpair);
					}
				else {
					int majority_appear = (same_appear > diff_appear) ? same_appear : diff_appear;
					if (majority_appear > (same_appear + diff_appear)* (50+dont_care_fract) /100)
						multiplier = ((same_appear + diff_appear)*1.0)/diff_appear;
					else {
						multiplier = 2.0;
						printf("don't care equivalence\n");
					}
					if (lowerbound_flag) ++lowerbound_log2;
					printf("replacing literal %i with %i, (originally %i with %i)\n", abs(nextpair.second), -abs(nextpair.first), origin[abs(nextpair.second)], -origin[abs(nextpair.first)]);
					nextpair.second = -nextpair.second;
					if (unitpropagation) 
						next = updatefile_e(nextpair, i);
					else 
						updatefile_e_nu(nextpair);
				}
			}

			if (verify_flag) printf("exact multiplier: %f\n",exactmultiplier);
			printf("multiplier: %f\n",multiplier);
			currentcount *= multiplier;
			if (verify_flag) {
				exactcount *= exactmultiplier;
				if (exactmultiplier >2.0) num_incorrect++;
				else if (exactmultiplier == 1.0) num_determined++;
				else num_correct++;
				printf("exact product so far: %.2e\n", exactcount);
			}
			printf("product so far: %.2e\n", currentcount);
			if (lowerbound_flag) {
              if (!knuth_flag)
                printf("lowerbound so far: 2^(%i-%.1lf) = %.2e\n", 
                       lowerbound_log2, lb_slack,
                       pow(2.0,(lowerbound_log2 - lb_slack)));   // Ashish
              else
                printf("knuth-based lowerbound so far: %.6e\n",
                       currentcount/pow(2.0,lb_slack));
            }
			printf("__________________________________________________\n");
			fflush(stdout);
		}
		if (next == MAXATOM +1) {
			printf("contradiction found\n");
			exit(1);
		}
		else if (next ==0) {
			unitprop = 0;
			next =1;
		}
		else unitprop = 1;
	} 
	if (verify_flag) {
		printf("exact product: %.6e\n", exactcount);
		printf("#correct majority choice: %i\n", num_correct);
		printf("#incorrect majority choice: %i\n", num_incorrect);
		printf("#determined: %i\n", num_determined);
		printf("#unit propagation: %i\n", num_unitprop);
	}
	printf("\n");
	printf("random seed used: %i\n", seed);
	printf("last random number generated: %li\n", random());
	printf("total estimate: %.6e\n",currentcount);
	printf(" (as a string: %s)\n",currentcount_str);   // Ashish
	if (lowerbound_flag) {  // Ashish
      if (!knuth_flag) {
        // reporting 2^{t - lb_slack} when t vars are set
        printf("lowerbound: %.0f x 2^(%i-%.2lf) = %.6e\n", 
               lowerbound_residual,
               lowerbound_log2, lb_slack, 
               pow(2.0,(lowerbound_log2 - lb_slack))*lowerbound_residual
               );
        printf(" (as a string: %s x 2^(%i-%.2lf))\n", 
               lowerbound_residual_str,
               lowerbound_log2, lb_slack 
               );
      }
      else {
        // reporting product of multipliers divided by 2^lb_slack
        printf("knuth-based lowerbound: %.6e\n", currentcount/pow(2.0,lb_slack));
        printf(" (as a string: %s / 2^%.2lf)\n", currentcount_str, lb_slack);
      }
	}
	
	return 1;
}


int fpickeven(int numvar)
{
	int i;
	int best;
	int best_pos = 0;
	
	best = numsuccesstry + 1;
	for (i=1; i<=numvar; i++) {
		if (abs(pos_array[i]-neg_array[i]) < best) {
			best = abs(pos_array[i]-neg_array[i]);
			best_pos = i;
			if (best == 0)   // Ashish
				break;
		}
	}
	return best_pos;
}
		
int fpickuneven(int numvar)
{
	int i;
	int best;
	int best_pos = 0;
	int n_samples = pos_array[1] + neg_array[1];   // Ashish
	
	best = -1;
	for (i=1; i<=numvar; i++) {
		if (abs(pos_array[i]-neg_array[i]) > best) {
			best = abs(pos_array[i]-neg_array[i]);
			best_pos = i;
			if (best == n_samples)   // Ashish
				break;
		}
	}
	return best_pos;
}

// Ashish: same as pickuneven, but prefers to pick variables that
//         do occur both positively and negatively
int fpicksafeuneven(int numvar)
{
	int i;
	int best;
	int best_pos = 0;
	int n_samples = pos_array[1] + neg_array[1];
	
	best = -1;
	for (i=1; i<=numvar; i++) {
	  if (pos_array[i] > 0 && neg_array[i] > 0) { // make sure both are non-zero
		if (abs(pos_array[i]-neg_array[i]) > best) {
		  best = abs(pos_array[i]-neg_array[i]);
		  best_pos = i;
			if (best == n_samples-1)
				break;
		}
	  }
	}
	if (best > -1)
	  return best_pos;
	else
	  return fpickuneven(numvar);
}

int fpickrand(int numvar)
{
	int i;
	int j = 0;
	int max;
	int success = 0;
	do {
		i= random()%numvar + 1;
		j++;
		max = ((pos_array[i] > neg_array[i])? pos_array[i] : neg_array[i]);
		success = (max * 100 >= (pos_array[i]+neg_array[i]) * (50 + imbalance));
	} 
	while ((j<=10) && (!success));
	if (success)
		return i;
	else 
		return fpickuneven(numvar);
}

// Ashish: pick the most balanced equivalence
intPair fpickeven_e(int numvar)
{
	int i, j;
	int best;
	int best_pos1 = 0, best_pos2 = 0;
	
	best = numsuccesstry + 1;
	for (i=1; i<numvar; i++) {
	  for (j=i+1; j<=numvar; j++) {
		if (abs(same_array[i][j]-diff_array[i][j]) < best) {
		  best = abs(same_array[i][j]-diff_array[i][j]);
		  best_pos1 = i;
		  best_pos2 = j;
			if (best == 0)
				break;
		}
	  }
	}
	return intPair (best_pos1, best_pos2);
}
		
// Ashish: pick the most unbalanced equivalence
intPair fpickuneven_e(int numvar)
{
	int i, j;
	int best;
	int best_pos1 = 0, best_pos2 = 0;
	int n_samples = pos_array[1] + neg_array[1];
	
	best = -1;
	for (i=1; i<numvar; i++) {
	  for (j=i+1; j<=numvar; j++) {
		if (abs(same_array[i][j]-diff_array[i][j]) > best) {
		  best = abs(same_array[i][j]-diff_array[i][j]);
		  best_pos1 = i;
		  best_pos2 = j;
			if (best == n_samples)
				break;
		}
	  }
	}
	return intPair (best_pos1, best_pos2);
}

// Ashish: pick the most unbalanced equivalence, preferring
// variables that occur both as same and as different
intPair fpicksafeuneven_e(int numvar)
{
	int i, j;
	int best;
	int best_pos1 = 0, best_pos2 = 0;
	int n_samples = pos_array[1] + neg_array[1];
	
	best = -1;
	for (i=1; i<numvar; i++) {
	  for (j=i+1; j<=numvar; j++) {
		if (same_array[i][j] > 0 && diff_array[i][j] > 0) {   // make sure both are positive
		  if (abs(same_array[i][j]-diff_array[i][j]) > best) {
			best = abs(same_array[i][j]-diff_array[i][j]);
			best_pos1 = i;
			best_pos2 = j;
			if (best == n_samples-1)
				break;
		  }
		}
	  }
	}
	if (best > -1)
	  return intPair (best_pos1, best_pos2);
	else
	  return fpickuneven_e(numvar);
}

// Ashish: pick a random pair of variables
// satisfying the "imbalance" parameter
intPair fpickrand_e(int numvar)
{
	int i, j;
	int k = 0;
	int max;
	int success = 0;
	do {
		i = random()%(numvar-1) + 1;
		j = random()%(numvar-i) + i+1;
		k++;
		max = ((same_array[i][j] > diff_array[i][j])? same_array[i][j] : diff_array[i][j]);
		success = (max * 100 >= (same_array[i][j]+diff_array[i][j]) * (50 + imbalance));
	} 
	while ((k<=20) && (!success));
	if (success)
		return intPair (i,j);
	else 
		return fpickuneven_e(numvar);
}

double retrievesolution(char * tempfilestr, char * resultsolutionstr) //cachet with -t
{
	// Ashish: changed this to also return the actual solution string read
	double resultsolution;
	FILE * tempfile;
	char currentword[1024];
	int timeout = 0;
	
	tempfile = fopen(tempfilestr, "r");
	
	do {
		fscanf(tempfile, " %s", currentword);
		if (strcmp(currentword, "out,")==0) timeout = 1;
	} while (strcmp( currentword, "solutions") !=0);
	
	fscanf(tempfile, " %s", resultsolutionstr);
	resultsolution = atof(resultsolutionstr);
	
	fclose(tempfile);
	
	if (timeout)
		return -1.0;
	else
		return resultsolution;
}

double retrievesolution2(char * tempfilestr, char * resultsolutionstr) //cachet with limit
{
	// Ashish: changed this to also return the actual solution string read
	double resultsolution;
	FILE * tempfile;
	char currentword[1024];
	int timeout;
	
	tempfile = fopen(tempfilestr, "r");
	
	do {
		timeout = fscanf(tempfile, " %s", currentword);
	} while ((strcmp( currentword, "solutions")) !=0 && (timeout != EOF));
	
	
	if (timeout ==EOF) {
		fclose(tempfile);
		return -1.0;
	}
	
	fscanf(tempfile, " %s", resultsolutionstr);
	resultsolution = atof(resultsolutionstr);
		
	fclose(tempfile);
	
	return resultsolution;
}

double retrievesolution3(char * tempfilestr, char * resultsolutionstr) //relsat with limit
{
	// Ashish: changed this to also return the actual solution string read
	double resultsolution;
	FILE * tempfile;
	char currentword[1024];
	int timeout;
	
	tempfile = fopen(tempfilestr, "r");
	
	do {
		timeout = fscanf(tempfile, " %s", currentword);
	} while ((strcmp( currentword, "solutions:")) !=0 && (timeout != EOF));
	
	
	if (timeout ==EOF) {
		fclose(tempfile);
		return -1.0;
	}
	
	fscanf(tempfile, " %s", resultsolutionstr);
	resultsolution = atof(resultsolutionstr);
	
	fclose(tempfile);
	
	return resultsolution;
}

int findlit(int next)
{
	FILE * current;
	int i, lit;
	
	current = fopen(currentfilestr,"r");
	if (fscanf(current, "p cnf %i %i",&numatom,&numclause) != 2)
	{
		fprintf(stderr,"Bad input file\n");
		exit(-1);
	}
	
	if (numclause ==0 ) {
		fclose(current);
		return 0;
	}

	for(i = 0;i < numclause;i++)
	{
		do {
			fscanf(current, "%i ",&lit);
			}
		while ((lit !=0) && (abs(lit) != abs(next)));
		if (abs(lit) == abs(next)) break;
	}
	fclose(current);
	if (abs(lit) == abs(next)) 
		return 1;
	else 
		return 0;
}

void print_param()
{
	printf("-seed %u\n", seed);
	
	if (!unitpropagation)
		printf("-nounitprop\n");
	else 
		printf("with unit propogation\n");
	printf("-cutoff %lli\n", cutoff);
	printf("-tries %i\n", numrun);
	if (numsol != NOVALUE)
		printf("-samples %i\n", numsol);
	else 
		printf("-samples %i\n", numrun);
	printf("-sa %i %i\n", sa_ratio, temperature);
	if (numerator != NOVALUE)
		printf("-noise %i %i\n", numerator, denominator);
	else 
		printf("-noise %i %i\n", 50, 100);
	printf("-dont_care_fract %i\n", dont_care_fract);
	printf("-exact_count %i\n", exact_count_nvar);
	printf("-exact_time_limit %i\n", exact_time_limit);
	if (lowerbound_flag) {  // Ashish
		printf("-lowerbound\n");
		printf("-lb_slack %lf\n", lb_slack);
	}
	else 
		printf("no lowerbound \n");
	if (equivonly_flag)   // Ashish
		printf("-equivonly\n");
	else if (equiv_flag)   // Ashish
		printf("-equiv\n");
	else 
		printf("no equiv \n");
	if (knuth_flag)   // Ashish
		printf("-knuth\n");
	if (verify_flag)
		printf("-verify\n");
	else 
		printf("no verify \n");
	if (inclnonsoln)
		printf("-inclnonsoln\n");
	else 
		printf("no inclnonsoln \n");
	if (doublecheck)
		printf("-doublecheck\n");
	else 
		printf("no doublecheck \n");
	if (!latesa)
		printf("-early_sa\n");
	else 
		printf("no early sa\n");
	if (pickuneven)
		printf("-pickuneven\n");
	if (picksafeuneven)             // Ashish
		printf("-picksafeuneven\n");
	if (pickeven)
		printf("-pickeven\n");
	if (picknatural)
		printf("-picknatural\n");
	if (pickrand) {
		printf("-pickrand\n");
		printf("-imbalance %i\n",imbalance);
	}
	printf("__________________________________________________");
}	

void parse_param(int argc, char * argv[])
{
	int i;
	int temp;
	cutoff = CUTOFF;
	numrun = NUMSAMPLE;
	heuristic = SA;
	makeflag = TRUE;
	sa_ratio = 50;
	temperature = 10;
	dont_care_fract = 0;
	
	for (i=1;i < argc;i++)
	{
	  if (strcmp(argv[i],"-seed") == 0) {
			unsigned int tmpseed;
			scanone(argc,argv,++i,&tmpseed);
			if (tmpseed != 0)    // Ashish
			  seed = tmpseed;
		}
		else if (strcmp(argv[i],"-cutoff") == 0) {
			scanone(argc,argv,++i,&cutoff);
		}
		else if (strcmp(argv[i],"-dont_care_fract") == 0) {
			scanone(argc,argv,++i,&dont_care_fract);
		}
		else if (strcmp(argv[i],"-exact_count") == 0) {
			scanone(argc,argv,++i,&exact_count_nvar);
		}
		else if (strcmp(argv[i],"-exact_time_limit") == 0) {
			scanone(argc,argv,++i,&exact_time_limit);
		}
		else if (strcmp(argv[i],"-nounitprop")==0)
			unitpropagation = 0;
		else if (strcmp(argv[i],"-verify")==0)
			verify_flag = 1;
		else if (strcmp(argv[i],"-doublecheck")==0)
			doublecheck = 1;
		else if (strcmp(argv[i],"-early_sa")==0)
			latesa = 0;
		else if (strcmp(argv[i],"-inclnonsoln")==0)
			inclnonsoln = 1;
		else if (strcmp(argv[i],"-pickeven")==0) {
			pickuneven = 0;
			picksafeuneven = 0;   // Ashish
			pickrand = 0;
			picknatural = 0;
			pickeven = 1;
			unitpropagation = 1;
		}
		else if (strcmp(argv[i],"-pickuneven")==0) {
			pickuneven = 1;
			picksafeuneven = 0;   // Ashish
			pickrand = 0;
			picknatural = 0;
			pickeven = 0;
			unitpropagation = 1;
		}
		else if (strcmp(argv[i],"-picksafeuneven")==0) {   // Ashish
			pickuneven = 0;
			picksafeuneven = 1;
			pickrand = 0;
			picknatural = 0;
			pickeven = 0;
			unitpropagation = 1;
		}
		else if (strcmp(argv[i],"-pickrand")==0) {
			pickuneven = 0;
			picksafeuneven = 0;   // Ashish
			pickrand = 1;
			picknatural = 0;
			pickeven = 0;
			unitpropagation = 1;
		}
		else if (strcmp(argv[i],"-picknatural")==0) {
			pickuneven = 0;
			picksafeuneven = 0;   // Ashish
			pickrand = 0;
			picknatural = 1;
			pickeven = 0;
			unitpropagation = 1;
		}
		else if (strcmp(argv[i],"-imbalance") == 0) {
			scanone(argc,argv,++i,&imbalance);
			pickuneven = 0;
			picksafeuneven = 0;   // Ashish
			pickrand = 1;
			picknatural = 0;
			pickeven = 0;
			unitpropagation = 1;
		}
			
		
/*		else if (strcmp(argv[i],"-random") == 0)
			heuristic = RANDOM;
		else if (strcmp(argv[i],"-novelty") == 0){
			heuristic = NOVELTY;
			makeflag = TRUE;
		}
		else if (strcmp(argv[i],"-rnovelty") == 0){
			heuristic = RNOVELTY;
			makeflag = TRUE;
		}
		else if (strcmp(argv[i],"-novelty+") == 0){
			scanone(argc, argv, ++i, &novelty_numerator);
			scanone(argc, argv, ++i, &novelty_denominator);
			heuristic = NOVELTY;
			makeflag = TRUE;
		}
		else if (strcmp(argv[i],"-best") == 0)
			heuristic = BEST;
*/		else if (strcmp(argv[i],"-sa") == 0){
			heuristic = SA;
			makeflag = TRUE;
			scanone(argc, argv, ++i, &sa_ratio);
			scanone(argc, argv, ++i, &temperature);
		}
		else if (strcmp(argv[i],"-noise") == 0){
			scanone(argc,argv,++i,&numerator);
			if (i < argc-1 && sscanf(argv[i+1],"%i",&temp)==1){
				denominator = temp;
				i++;
			}
		}
		else if (strcmp(argv[i],"-tries") == 0 || strcmp(argv[i],"-restart") ==0)
			scanone(argc,argv,++i,&numrun);
		else if (strcmp(argv[i],"-samples") ==0)
			scanone(argc,argv,++i,&numsol);
		else if (strcmp(argv[i],"-sampleonly") ==0)     // Ashish
			sampleonly_flag = true;
		else if (strcmp(argv[i],"-printsamples") ==0)     // Ashish
			printsamples_flag = true;
		else if (strcmp(argv[i],"-lowerbound") ==0) {   // Ashish
			lowerbound_flag = true;
			if (pickeven == 0 && pickuneven == 0 && picksafeuneven == 0 && pickrand == 0 && picknatural == 0) {   // use pickeven by default with -lowerbound
			  pickeven = 1;
			  pickuneven = 0;
			  picksafeuneven = 0;
			  pickrand = 0;
			  picknatural = 0;
			}
			unitpropagation = 1;
        }
		else if (strcmp(argv[i],"-lb_slack") ==0) {   // Ashish
			scanone(argc,argv,++i,&lb_slack);
			if (lb_slack < 0) {
			  fprintf(stderr, "ERROR: lowerbound slack must be non-negative\n");
			  exit(1);
			}
//      else if (lb_slack == 0)
//			  fprintf(stderr, "WARNING: lowerbound slack is zero\n");
		}
		else if (strcmp(argv[i],"-equiv") ==0)        // Ashish
			equiv_flag = true;
		else if (strcmp(argv[i],"-equivonly") ==0)    // Ashish
		    { equivonly_flag = true; equiv_flag = true; }
		else if (strcmp(argv[i],"-noequiv") ==0) {    // Ashish
			if (equiv_flag == true)
              fprintf(stderr, "WARNING: overriding previous -equiv flag\n");
			equiv_flag = false;
        }
		else if (strcmp(argv[i],"-knuth") ==0)        // Ashish
			knuth_flag = true;
		else 
		{
			fprintf(stderr, "USAGE: approxcount [options] < yourcnf.cnf\n\n");
			fprintf(stderr, "  -seed N\n");
			fprintf(stderr, "  -dont_care_fract N   consider variables that are positive in \n               (50-N)%c to (50+N)%c samples to be dont_cares,\n               and set the multiplier to be 2. (default N = 0)\n", '%', '%');
			fprintf(stderr, "  -exact_count N   start exact count from the Nth iteration (default N = 50)\n");
			fprintf(stderr, "                       to disable exact count, use -exact_count 0\n");
			fprintf(stderr, "                       to start exact count when N variables remain, use -exact_count -N\n");
			fprintf(stderr, "  -exact_time_limit N  try exact count for N seconds in each iteration (defualt N = 100)\n");
			fprintf(stderr, "  -verify  use cachet to veify the accuracy of multipliers\n");
			fprintf(stderr, "  -doublecheck  use both cachet and relsat to count the sulutions exactly\n");
			fprintf(stderr, "  -nounitprop do not use unitpropagation\n");			
			fprintf(stderr, "  -pickuneven  branch on the most unevenly sampled variable (default)\n");
			fprintf(stderr, "  -picksafeuneven  branch on the most unevenly sampled variable, preferably\n");
			fprintf(stderr, "                   one that occurs both positively and negatively\n");   // Ashish
			fprintf(stderr, "  -pickeven  branch on the most evenly sampled variable\n");
			fprintf(stderr, "  -pickrand  branch on a random variable\n");
			fprintf(stderr, "  -picknatural  branch on the lowest numbered variable\n");
			fprintf(stderr, "  -imbalance N  choose pickrandom, and assert a literal only if (50+N)%c \n                of the sample agree with the assertion (default: N = 0)\n", '%');
			fprintf(stderr, "        if more than one of the above 6 options selected, only the last one is effective\n");
			fprintf(stderr, "\nSampling specific options:\n");   // Ashish
			fprintf(stderr, "  -cutoff N\n");
			fprintf(stderr, "  -tries N number of tries taken at each step\n");
			fprintf(stderr, "  -samples N number of samples (successful tries) taken at each step\n");
			fprintf(stderr, "  -sa SA_RATIO TEMPERATURE walksat with SA_RATIO percent SA\n               moves added at temp TEMPERAURE/100. (default: -sa 50 10)\n");
			fprintf(stderr, "  -noise N or -noise N M (default M = 100)\n");
			fprintf(stderr, "  -inclnonsoln  include non-solution states in sampling counts\n");
			fprintf(stderr, "  -early_sa  use mixed simulated annealing moves from the start of search\n");
			fprintf(stderr, "  -sampleonly  exit right after sampling solutions\n");   // Ashish
			fprintf(stderr, "  -printsamples  print solution samples as they are computed\n");   // Ashish
			fprintf(stderr, "\nLowerbound specific options:\n");   // Ashish
			fprintf(stderr, "  -lowerbound  produce a probabilistically guaranteed lowerbound\n");   // Ashish
			fprintf(stderr, "  -lb_slack X  lowerbound slack factor (positive real, default: X = 1)\n");   // Ashish
			fprintf(stderr, "  -equiv  allow variable equivalences (2-xors)\n");   // Ashish
			fprintf(stderr, "  -equivonly  use only variable equivalences (2-xors)\n");   // Ashish
			fprintf(stderr, "  -knuth  use generalized random coins for lowerbound\n");   // Ashish
			exit(-1);
		}
	}

    if (pickeven == 0 && pickuneven == 0 && picksafeuneven == 0 && pickrand == 0 && picknatural == 0)
      pickuneven = 1;   // Ashish: the default for approxcount moved here
}

void readatom(FILE *infile)
{
	int lastc;
	int nextc;
	FILE * current;
	
	current = fopen(currentfilestr,"w");
	if (current == NULL)
	{
		fprintf(stderr,"Bad open file\n");
		exit(-1);
	}
	while ((lastc = nextChar(infile)) == 'c')
	{
		while ((nextc = nextChar(infile)) != EOF && nextc != '\n');
	}
	ungetc(lastc, infile);
	if (fscanf(infile, "p cnf %i %i",&numatom2,&numclause2) != 2)
	{
		fprintf(stderr,"Bad input file\n");
		exit(-1);
	}
	fprintf(current, "p cnf %i %i", numatom2, numclause2);
	while ((lastc = fgetc(infile)) != EOF)
	{
		fputc(lastc, current);
	}
	fclose(current);
}

double countexact(int change, int exact_time_limit)
{
	int lastc;
	int nextc;
	int numatom;
	int numclause;
	char scache[100];
	char tempfilestr2[100];
	char tempfilestr3[100];
	double resultsolution;
	double resultsolution2;
	char resultsolutionstr[1<<20];   // Ashish
	char resultsolutionstr2[1<<20];
	FILE * current;
	FILE * newfile;
	
	current = fopen(currentfilestr,"r");
	newfile = fopen(tempfilestr,"w");
	if (current == NULL || newfile == NULL)
	{
		fprintf(stderr,"Bad open file\n");
		exit(-1);
	}
	while ((lastc = nextChar(current)) == 'c')
	{
		while ((nextc = nextChar(current)) != EOF && nextc != '\n');
	}
	ungetc(lastc, current);
	if (fscanf(current, "p cnf %i %i",&numatom,&numclause) != 2)
	{
		fprintf(stderr,"Bad input file\n");
		exit(-1);
	}
	numclause2 = numclause + 1;
	fprintf(newfile, "p cnf %i %i", numatom, numclause+1);
	while ((lastc = fgetc(current)) != EOF)
	{
		fputc(lastc, newfile);
	}
	fprintf(newfile, "%i 0\n", change);
	fclose(current);
	fclose(newfile);
	sprintf(tempfilestr2, "%s_2", tempfilestr);
	sprintf(tempfilestr3, "%s_3", tempfilestr);
	if (doublecheck) {
		sprintf(scache, "./exactcount %s %i %s %s", tempfilestr, exact_time_limit, tempfilestr2, tempfilestr3);
		system(scache);
		resultsolution = retrievesolution2(tempfilestr2, resultsolutionstr);
		resultsolution2 = retrievesolution3(tempfilestr3, resultsolutionstr2);
		remove(tempfilestr2);
		remove(tempfilestr3);
		if (resultsolution >= 0.0) 
			if (change >0) 
				printf("pos_exact by cachet: %f\n", resultsolution);
			else
				printf("neg_exact by cachet: %f\n", resultsolution);
		else {
			if (change >0) 
				printf("pos_exact by cachet: time out\n");
			else
				printf("neg_exact by cachet: time out\n");
			if (resultsolution2 >=0.0)
				resultsolution = resultsolution2; //take relsat value
		}
		if (resultsolution2 >= 0.0) 
			if (change >0) 
				printf("pos_exact by relsat: %f\n", resultsolution2);
			else
				printf("neg_exact by relsat: %f\n", resultsolution2);
		else 
			if (change >0) 
				printf("pos_exact by relsat: time out\n");
			else
				printf("neg_exact by relsat: time out\n");
	}
	else {
		//sprintf(scache, "./cachet %s -t %i > %s", tempfilestr, exact_time_limit, tempfilestr2);
		sprintf(scache, "cachet %s -t %i > %s", tempfilestr, exact_time_limit, tempfilestr2);
		system(scache);
		resultsolution = retrievesolution(tempfilestr2, resultsolutionstr);
		if (change >0) 
			if (resultsolution >=0)
				printf("pos_exact by cachet: %f\n", resultsolution);
			else
				printf("pos_exact by cachet: time out\n");
		else 
			if (resultsolution >=0)
				printf("neg_exact by cachet: %f\n", resultsolution);
			else
				printf("neg_exact by cachet: time out\n");
			
		remove(tempfilestr2);
	}
	
	remove(tempfilestr);
	return resultsolution;
}

void updatefile_nu(int change)
{
	int lastc;
	int nextc;
	int numatom;
	int numclause;
	FILE * current;
	FILE * newfile;
	
	current = fopen(currentfilestr,"r");
	newfile = fopen(tempfilestr,"w");
	if (current == NULL || newfile == NULL)
	{
		fprintf(stderr,"Bad open file\n");
		exit(-1);
	}
	while ((lastc = nextChar(current)) == 'c')
	{
		while ((nextc = nextChar(current)) != EOF && nextc != '\n');
	}
	ungetc(lastc, current);
	if (fscanf(current, "p cnf %i %i",&numatom,&numclause) != 2)
	{
		fprintf(stderr,"Bad input file\n");
		exit(-1);
	}
	numclause2 = numclause + 1;
	fprintf(newfile, "p cnf %i %i", numatom, numclause+1);
	while ((lastc = fgetc(current)) != EOF)
	{
		fputc(lastc, newfile);
	}
	fprintf(newfile, "%i 0\n", change);
	fclose(current);
	fclose(newfile);
	remove(currentfilestr);
	rename(tempfilestr, currentfilestr);
}


// Ashish: for variable equivalence with no unit propagation;
// adds the constraint lit1 == lit2
void updatefile_e_nu(intPair litpair)
{
	int lit1 = litpair.first, lit2 = litpair.second;
	int lastc;
	int nextc;
	int numatom;
	int numclause;
	FILE * current;
	FILE * newfile;
	
	current = fopen(currentfilestr,"r");
	newfile = fopen(tempfilestr,"w");
	if (current == NULL || newfile == NULL)
	{
		fprintf(stderr,"Bad open file\n");
		exit(-1);
	}
	while ((lastc = nextChar(current)) == 'c')
	{
		while ((nextc = nextChar(current)) != EOF && nextc != '\n');
	}
	ungetc(lastc, current);
	if (fscanf(current, "p cnf %i %i",&numatom,&numclause) != 2)
	{
		fprintf(stderr,"Bad input file\n");
		exit(-1);
	}
	numclause2 = numclause + 1;
	fprintf(newfile, "p cnf %i %i", numatom, numclause+2);
	while ((lastc = fgetc(current)) != EOF)
	{
		fputc(lastc, newfile);
	}
	fprintf(newfile, "-%i %i 0\n", lit1, lit2);
	fprintf(newfile, "%i -%i 0\n", lit1, lit2);
	fclose(current);
	fclose(newfile);
	remove(currentfilestr);
	rename(tempfilestr, currentfilestr);
}


void updateorigin(int change, int iteration)
{
	change = abs(change);
	for (int j = change; j <= numatom2-iteration + 1; j++)
		origin[j]= origin[j+1];
}
	

int updatefile(int change, int iteration)
{
	int lastc;
	int nextc;
	int numatom;
	int numclause;
	int i,j;
	int size;
	int flag;
	int removedclause;
	int returnvalue;
	int curlit;
	FILE * current;
	FILE * newfile;
	
	updateorigin(change, iteration);
	
	returnvalue = 0;
	flag = 0;
	removedclause =0;
	current = fopen(currentfilestr,"r");
	newfile = fopen(tempfilestr,"w");
	if (current == NULL || newfile == NULL)
	{
		fprintf(stderr,"Bad open file\n");
		exit(-1);
	}
	while ((lastc = nextChar(current)) == 'c')
	{
		while ((nextc = nextChar(current)) != EOF && nextc != '\n');
	}
	ungetc(lastc, current);
	if (fscanf(current, "p cnf %i %i",&numatom,&numclause) != 2)
	{
		fprintf(stderr,"Bad input file\n");
		exit(-1);
	}
	fprintf(newfile, "p cnf %i %i\n", numatom -1 , numclause);


	for(i = 0;i < numclause;i++)
	{
		size = -1;
		do
		{
			size++;
			if(size > MAXLENGTH)
			{
				printf("ERROR - clause too long\n");
				exit(-1);
			}
			fscanf(current, "%i ",&(lits[size]));
			curlit = lits[size];
			if(curlit != 0)
			{
				if (curlit == change) {
					flag = 1;
				}
				else if (curlit == -change) 
					size--;
				else if (curlit < -abs(change))
					lits[size]++;
				else if (curlit > abs(change))
					lits[size]--;
			}
		}
		while(curlit != 0);
		if (flag) {
			removedclause++;
			flag =0;
		}
		else {
			for (j=0; j<size; j++)
				fprintf(newfile, "%i ", lits[j]);
			fprintf(newfile,"0\n");
			if (size == 0)
				returnvalue = MAXATOM + 1;
			if ((returnvalue >= 0) && size==1) 
				returnvalue = lits[0];
		}
	}
	fclose(current);
	fclose(newfile);
	remove(currentfilestr);
	rename(tempfilestr, currentfilestr); 
	if (removedclause >0) {
		current = fopen(currentfilestr,"r");
		newfile = fopen(tempfilestr,"w");
		if (current == NULL || newfile == NULL)
		{
			fprintf(stderr,"Bad open file\n");
			exit(-1);
		}
		while ((lastc = nextChar(current)) == 'c')
		{
			while ((nextc = nextChar(current)) != EOF && nextc != '\n');
		}
		ungetc(lastc, current);
		if (fscanf(current, "p cnf %i %i",&numatom,&numclause) != 2)
		{
			fprintf(stderr,"Bad input file\n");
			exit(-1);
		}
		numclause2 = numclause-removedclause;
		fprintf(newfile, "p cnf %i %i", numatom, numclause2);
		while ((lastc = fgetc(current)) != EOF)
		{
			fputc(lastc, newfile);
		}
		fclose(current);
		fclose(newfile);
		remove(currentfilestr);
		rename(tempfilestr, currentfilestr);
	}

	return returnvalue;
}


// Ashish: for variable equivalence with unit propagation;
// replaces lit2 with lit1; 
// must have abs(lit1) < abs(lit2)
int updatefile_e(intPair litpair, int iteration)
{
	int lit1 = litpair.first, lit2 = litpair.second;
	int lastc;
	int nextc;
	int numatom;
	int numclause;
	int i,j;
	int size;
	int flag;
	int removedclause;
	int returnvalue;
	int curlit;
	FILE * current;
	FILE * newfile;
	
	updateorigin(lit2, iteration);
	
	returnvalue = 0;
	flag = 0;
	removedclause =0;
	current = fopen(currentfilestr,"r");
	newfile = fopen(tempfilestr,"w");
	if (current == NULL || newfile == NULL)
	{
		fprintf(stderr,"Bad open file\n");
		exit(-1);
	}
	while ((lastc = nextChar(current)) == 'c')
	{
		while ((nextc = nextChar(current)) != EOF && nextc != '\n');
	}
	ungetc(lastc, current);
	if (fscanf(current, "p cnf %i %i",&numatom,&numclause) != 2)
	{
		fprintf(stderr,"Bad input file\n");
		exit(-1);
	}
	fprintf(newfile, "p cnf %i %i\n", numatom -1 , numclause);


	for(i = 0;i < numclause;i++)
	{
		size = -1;
		int pos_occurrences = 0, neg_occurrences = 0;   // for lit1, lit2
		do
		{
			size++;
			if(size > MAXLENGTH)
			{
				printf("ERROR - clause too long\n");
				exit(-1);
			}
			fscanf(current, "%i ",&(lits[size]));
			curlit = lits[size];
			if(curlit != 0)
			{
			  if (curlit == lit1)
				pos_occurrences++;
			  else if (curlit == -lit1)
				neg_occurrences++;
			  else if (curlit == lit2) {
				pos_occurrences++;
				lits[size] = lit1;
			  }
			  else if (curlit == -lit2) {
				neg_occurrences++;
				lits[size] = -lit1;
			  }
			  else if (curlit < -abs(lit2))
				lits[size]++;
			  else if (curlit > abs(lit2))
				lits[size]--;
			}
		}
		while(curlit != 0);
		if (pos_occurrences > 0 && neg_occurrences > 0)
		  removedclause++;
		else {
		  int size_reduction = 0;
		  for (j=0; j<size; j++) {
			curlit = lits[j];
			if (curlit == lit1 && pos_occurrences > 1) {
			  pos_occurrences--;
			  size_reduction++;
			}
			else if (curlit == -lit1 && neg_occurrences > 1) {
			  neg_occurrences--;
			  size_reduction++;
			}
			else
			  fprintf(newfile, "%i ", curlit);
		  }
		  fprintf(newfile,"0\n");
		  size -= size_reduction;
		  if (size == 0)
			returnvalue = MAXATOM + 1;
		  if ((returnvalue >= 0) && size==1) 
			returnvalue = lits[0];
		}
	}
	fclose(current);
	fclose(newfile);
	remove(currentfilestr);
	rename(tempfilestr, currentfilestr); 
	if (removedclause >0) {
		current = fopen(currentfilestr,"r");
		newfile = fopen(tempfilestr,"w");
		if (current == NULL || newfile == NULL)
		{
			fprintf(stderr,"Bad open file\n");
			exit(-1);
		}
		while ((lastc = nextChar(current)) == 'c')
		{
			while ((nextc = nextChar(current)) != EOF && nextc != '\n');
		}
		ungetc(lastc, current);
		if (fscanf(current, "p cnf %i %i",&numatom,&numclause) != 2)
		{
			fprintf(stderr,"Bad input file\n");
			exit(-1);
		}
		numclause2 = numclause-removedclause;
		fprintf(newfile, "p cnf %i %i", numatom, numclause2);
		while ((lastc = fgetc(current)) != EOF)
		{
			fputc(lastc, newfile);
		}
		fclose(current);
		fclose(newfile);
		remove(currentfilestr);
		rename(tempfilestr, currentfilestr);
	}

	return returnvalue;
}

void sample(char * _filename)
{
#ifdef MPICODE	//JSE MPI Code
  int done = 0, n, numprocs, i, temp;
  int namelen, stop = FALSE;
  char processor_name[MPI_MAX_PROCESSOR_NAME];
	MPI_Status status;
	
  MPI_Init(&argc, &argv);
  MPI_Comm_size(MPI_COMM_WORLD, &numprocs);
  MPI_Comm_rank(MPI_COMM_WORLD, &mpi_id);
  MPI_Get_processor_name(processor_name, &namelen);
	
  if (mpi_id == ROOT_PROCID) {
		rootProc = true;
  } else {
    rootProc = false;
    temp = 1;
    MPI_Send(&temp, 1, MPI_INT, ROOT_PROCID, STARTTAG, MPI_COMM_WORLD);  //Tell root I'm starting
  }
  
/*  fprintf(stderr, "Process %d/%d on %s\n", mpi_id, numprocs, processor_name);*/
#endif	
 
/*
#ifdef UNIX
  gettimeofday(&tv,&tzp);
  seed = (unsigned int)((( tv.tv_sec & 0177 ) * 1000000) + tv.tv_usec);
#endif
#ifdef NT
  seed = (unsigned int)(timeGetTime());
#endif
#ifdef ANSI
  seed = (unsigned int)(time());
#endif
*/

  parse_parameters( _filename);
#ifdef MPICODE
	if (! rootProc)
		MPI_Recv(&seed, 1, MPI_UNSIGNED, ROOT_PROCID, SEEDTAG, MPI_COMM_WORLD, &status);
		cout << "Seed(" << processor_name << ") = " << seed << endl;
#endif
	srandom(seed);
  initprob();
  initialize_statistics();

#ifdef MPICODE 
	if (rootProc) { //Only print headers once if running through MPI
#endif
//	print_parameters(argc, argv);
//	print_statistics_header();
#ifdef MPICODE	
	}
#endif
  signal(SIGINT, handle_interrupt);
  abort_flag = FALSE;
  (void) elapsed_seconds();
numtry =0;
numsuccesstry =0;

#ifdef MPICODE
  while (! abort_flag && numsuccesstry < numsol && numtry < numrun && !mpiTagRecv(STOPTAG)) {
		if (rootProc && mpiTagRecv(STARTTAG)) {
			MPI_Probe(MPI_ANY_SOURCE, STARTTAG, MPI_COMM_WORLD, &status);
			MPI_Recv(commBuf, MAXBUFF, MPI_INT, MPI_ANY_SOURCE, STARTTAG, MPI_COMM_WORLD, &status);
			numSlavesProcs += *commBuf;

			if (*commBuf == 1) { //Send out a random seed
				seed2 = hash(seed+(status.MPI_SOURCE*7));
				MPI_Send(&seed2, 1, MPI_UNSIGNED, status.MPI_SOURCE, SEEDTAG, MPI_COMM_WORLD);
			}
		} else if (rootProc && mpiTagRecv(SOLNTAG)) {
			mpiRecvTry();
			numtry++;
			update_and_print_statistics_end_try();
		} else {
			procRecv = 0;
#else
  while (! abort_flag && numsuccesstry < numsol && numtry < numrun) {
#endif
			//MAIN SOLUTON FINDING CODE
			numtry++;
			init(initfile, initoptions);
			update_statistics_start_try();
			numflip = 0;
			
			if (superlinear) cutoff = base_cutoff * super(numtry);
			
			while(((numfalse > target) || (heuristic == SA) || (heuristic == SARANDOM)) && (numflip < cutoff)) {
				print_statistics_start_flip();
				numflip++;
				flipatom((pickcode[heuristic])());
				update_statistics_end_flip();
			}
			
#ifdef MPICODE
			if (rootProc) {
				update_and_print_statistics_end_try();
			} else {
				mpiSendTry();
			}
		}
#else
			update_and_print_statistics_end_try();

#endif
		}
		
#ifdef MPICODE
	if (! rootProc) { //Tell root that I'm stopping
		temp = -1;
	  MPI_Send(&temp, 1, MPI_INT, ROOT_PROCID, STARTTAG, MPI_COMM_WORLD);
	} else {
		for(int i = 1; i <= numSlavesProcs; i++)
			MPI_Send(&temp, 1, MPI_INT, i, STOPTAG, MPI_COMM_WORLD);
		mpiWaitForAll(); //Wait for other proccesses to finish
#endif
	expertime = elapsed_seconds();
	print_statistics_final();
	freethings();


#ifdef MPICODE
	}
	MPI_Finalize();
#endif
}

void freethings() {
	int i;
	//STOREBLOCK
	for (i=0; i<storerecnum; i++)
		free(storerec[i]);
}

#ifdef MPICODE   //JSE

//Wait for all proccesses to finish
void mpiWaitForAll(void) {
  MPI_Status status;
  int flag; 

  MPI_Iprobe(MPI_ANY_SOURCE, MPI_ANY_TAG, MPI_COMM_WORLD, &flag, &status);   
  while (flag || (numSlavesProcs > 0)) {
    MPI_Probe(MPI_ANY_SOURCE, MPI_ANY_TAG, MPI_COMM_WORLD, &status);  //Blocking call
    if (status.MPI_TAG == STARTTAG) { //Track the number of slave procceses running
      MPI_Recv(commBuf, MAXBUFF, MPI_INT, MPI_ANY_SOURCE, STARTTAG, MPI_COMM_WORLD, &status);
      numSlavesProcs += *commBuf;
    } else if (status.MPI_TAG == SOLNTAG) {
      mpiRecvTry();
      numtry++;
      update_and_print_statistics_end_try();
    }

    MPI_Iprobe(MPI_ANY_SOURCE, MPI_ANY_TAG, MPI_COMM_WORLD, &flag, &status);
  }
  
}


//Retruns true if a tag has been received
bool mpiTagRecv(int tag) {
	MPI_Status status;
	int flag; 
	
	MPI_Iprobe(MPI_ANY_SOURCE, tag, MPI_COMM_WORLD, &flag, &status);  //Tag received
	if (flag)
		return true;
	else
		return false;
}

void mpiRecvTry(void) { //JSE
	MPI_Status status;
	int pos = 0; 

	MPI_Recv(commBuf, MAXBUFF, MPI_PACKED, MPI_ANY_SOURCE, SOLNTAG, MPI_COMM_WORLD, &status);
	MPI_Unpack(commBuf, MAXBUFF, &pos, &numflip, 1, MPI_myBIGINT, MPI_COMM_WORLD);
	MPI_Unpack(commBuf, MAXBUFF, &pos, &sumfalse, 1, MPI_DOUBLE, MPI_COMM_WORLD);
	MPI_Unpack(commBuf, MAXBUFF, &pos, &sumfalse_squared, 1, MPI_DOUBLE, MPI_COMM_WORLD);
	MPI_Unpack(commBuf, MAXBUFF, &pos, &sample_size, 1, MPI_DOUBLE, MPI_COMM_WORLD);
	MPI_Unpack(commBuf, MAXBUFF, &pos, atom, numatom+1, MPI_INT, MPI_COMM_WORLD);
	MPI_Unpack(commBuf, MAXBUFF, &pos, &numfalse, 1, MPI_INT, MPI_COMM_WORLD);
	MPI_Unpack(commBuf, MAXBUFF, &pos, tailhist, HISTMAX, MPI_LONG, MPI_COMM_WORLD);
	MPI_Unpack(commBuf, MAXBUFF, &pos, &lowbad, 1, MPI_LONG, MPI_COMM_WORLD);
	MPI_Unpack(commBuf, MAXBUFF, &pos, lowfalse, numclause+1, MPI_INT, MPI_COMM_WORLD);
	MPI_Unpack(commBuf, MAXBUFF, &pos, lowatom, numatom+1, MPI_INT, MPI_COMM_WORLD);

  procRecv = status.MPI_SOURCE;
}


void mpiSendTry(void) { //JSE
	int pos = 0;
	
	MPI_Pack(&numflip, 1, MPI_myBIGINT, commBuf, MAXBUFF, &pos, MPI_COMM_WORLD);
	MPI_Pack(&sumfalse, 1, MPI_DOUBLE, commBuf, MAXBUFF, &pos, MPI_COMM_WORLD);
	MPI_Pack(&sumfalse_squared, 1, MPI_DOUBLE, commBuf, MAXBUFF, &pos, MPI_COMM_WORLD);
	MPI_Pack(&sample_size, 1, MPI_DOUBLE, commBuf, MAXBUFF, &pos, MPI_COMM_WORLD);
	MPI_Pack(atom, numatom+1, MPI_INT, commBuf, MAXBUFF, &pos, MPI_COMM_WORLD);
	MPI_Pack(&numfalse, 1, MPI_INT, commBuf, MAXBUFF, &pos, MPI_COMM_WORLD);
	MPI_Pack(tailhist, HISTMAX, MPI_LONG, commBuf, MAXBUFF, &pos, MPI_COMM_WORLD);
	MPI_Pack(&lowbad, 1, MPI_LONG, commBuf, MAXBUFF, &pos, MPI_COMM_WORLD);
	MPI_Pack(lowfalse, numclause+1, MPI_INT, commBuf, MAXBUFF, &pos, MPI_COMM_WORLD);
	MPI_Pack(lowatom, numatom+1, MPI_INT, commBuf, MAXBUFF, &pos, MPI_COMM_WORLD);
	MPI_Ssend(commBuf, pos, MPI_PACKED, ROOT_PROCID, SOLNTAG, MPI_COMM_WORLD); /*Syncronous send*/
}

//Robert Jenkins Hash function from http://www.concentric.net/~Ttwang/tech/inthash.htm
unsigned int hash(unsigned int key) {
  key += (key << 12);
  key ^= (key >> 22);
  key += (key << 4);
  key ^= (key >> 9);
  key += (key << 10);
  key ^= (key >> 2);
  key += (key << 7);
  key ^= (key >> 12);
  return key;
}


#endif


void parse_parameters(char * _filename)
{
//	cutoff = _cutoff;
//	countvar = _variable;
	//printf("countvar: %i\n", countvar);
//	numrun = _numsample;
	strcpy(inputCNFFile, _filename);
	
/*	for (i=1;i < argc;i++)
	{
		if (strcmp(argv[i],"-seed") == 0)
			scanone(argc,argv,++i,&seed);
		else if (strcmp(argv[i],"-out") == 0 && i<argc-1)
			strcpy(outfile, argv[++i]);
		else if (strcmp(argv[i],"-outAll") == 0 && i<argc-1)  //JSE
			strcpy(outfileAll, argv[++i]);
		else if (strcmp(argv[i],"-outnum") == 0 && i<argc-1) 
			strcpy(numsolfile, argv[++i]);
		else if (strcmp(argv[i],"-hist") == 0)
			printhist = TRUE;
		else if (strcmp(argv[i],"-status") == 0)
			status_flag = 1;
		else if (strcmp(argv[i],"-cutoff") == 0) {
			scanone(argc,argv,++i,&cutoff);
		}
		else if (strcmp(argv[i],"-variable") == 0) {
			scanone(argc,argv,++i,&countvar);
		}
		else if (strcmp(argv[i],"-random") == 0)
			heuristic = RANDOM;
		else if (strcmp(argv[i],"-novelty") == 0){
			heuristic = NOVELTY;
			makeflag = TRUE;
		}
		else if (strcmp(argv[i],"-rnovelty") == 0){
			heuristic = RNOVELTY;
			makeflag = TRUE;
		}
		else if (strcmp(argv[i],"-novelty+") == 0){
			scanone(argc, argv, ++i, &novelty_numerator);
			scanone(argc, argv, ++i, &novelty_denominator);
			heuristic = NOVELTY;
			makeflag = TRUE;
		}
		else if (strcmp(argv[i],"-best") == 0)
			heuristic = BEST;
		else if (strcmp(argv[i],"-sa") == 0){
			heuristic = SA;
			makeflag = TRUE;
			scanone(argc, argv, ++i, &sa_ratio);
			scanone(argc, argv, ++i, &temperature);
		}
		else if (strcmp(argv[i],"-sarandom") == 0){
			heuristic = SARANDOM;
			makeflag = TRUE;
			scanone(argc, argv, ++i, &sa_ratio);
			scanone(argc, argv, ++i, &temperature);
		}
		else if (strcmp(argv[i],"-noise") == 0){
			scanone(argc,argv,++i,&numerator);
			if (i < argc-1 && sscanf(argv[i+1],"%i",&temp)==1){
				denominator = temp;
				i++;
			}
		}
		else if (strcmp(argv[i],"-init") == 0  && i < argc-1)
			sscanf(argv[++i], " %s", initfile);
		else if (strcmp(argv[i],"-hamming") == 0  && i < argc-3){
			sscanf(argv[++i], " %s", hamming_target_file);
			sscanf(argv[++i], " %s", hamming_data_file);
			sscanf(argv[++i], " %i", &hamming_sample_freq);
			hamming_flag = TRUE;
			numrun = 1;
		}
		else if (strcmp(argv[i],"-partial") == 0)
			initoptions = INIT_PARTIAL;
		else if (strcmp(argv[i],"-super") == 0)
			superlinear = TRUE;
		else if (strcmp(argv[i],"-tries") == 0 || strcmp(argv[i],"-restart") == 0)
			scanone(argc,argv,++i,&numrun);
		else if (strcmp(argv[i],"-target") == 0)
			scanone(argc,argv,++i,&target);
		else if (strcmp(argv[i],"-tail") == 0)
			scanone(argc,argv,++i,&tail);
		else if (strcmp(argv[i],"-sample") == 0)
			scanone(argc,argv,++i,&samplefreq);
		else if (strcmp(argv[i],"-tabu") == 0)
		{
			scanone(argc,argv,++i,&tabu_length);
			heuristic = TABU;
		}
		else if (strcmp(argv[i],"-low") == 0)
			printlow = TRUE;
		else if (strcmp(argv[i],"-sol") == 0)
		{
			printonlysol = TRUE;
			printlow = TRUE;
		}
		else if (strcmp(argv[i],"-solcnf") == 0)
		{
			printsolcnf = TRUE;
			if (numsol == NOVALUE) numsol = 1;
		}
		else if (strcmp(argv[i],"-bad") == 0)
			printfalse = TRUE;
		else if (strcmp(argv[i],"-printEveryN") == 0) {
			scanone(argc,argv,++i, &printEveryN); 
		} else if (strcmp(argv[i],"-numsol") == 0)
			scanone(argc,argv,++i,&numsol);
		else if (strcmp(argv[i],"-trace") == 0)
			scanone(argc,argv,++i,&printtrace);
		else if (strcmp(argv[i],"-assign") == 0){
			scanone(argc,argv,++i,&printtrace);
			trace_assign = TRUE;
		} else if (strcmp(argv[i],"-infile") == 0){ //JSE
			sscanf(argv[++i], " %s", inputCNFFile);
		} else 
		{
			fprintf(stderr, "General parameters:\n");
			fprintf(stderr, "  -seed N -cutoff N -restart N\n");
			fprintf(stderr, "  -numsol N = stop after finding N solutions\n");
			fprintf(stderr, "  -super = use the Luby series for the cutoff values\n");
			fprintf(stderr, "  -init FILE = set vars not included in FILE to false\n");
			fprintf(stderr, "  -partial FILE = set vars not included in FILE randomly\n");
			fprintf(stderr, "  -status = return fail status if solution not found\n");
			fprintf(stderr, "  -target N = succeed if N or fewer clauses unsatisfied\n");
			fprintf(stderr, "  -infile FILE = use FILE to input the instance (uses STDIN by default)\n");  //JSE
			fprintf(stderr, "Heuristics:\n");
			fprintf(stderr, "  -random -best -tabu N -novelty -rnovelty\n");
			fprintf(stderr, "  -novelty+ RN RM = modify novelty by RN/RM random walk\n");
			fprintf(stderr, "  -sa SA_RATIO TEMPERATURE walksat with SA_RATIO% SA moves added at temp TEMPERAURE/100\n");
			fprintf(stderr, "  -sarandom SA_RATIO TEMPERATURE RW with SA_RATIO% SA moves added at temp TEMPERAURE/100\n");
			fprintf(stderr, "  -noise N or -noise N M (default M = 100)\n");
			fprintf(stderr, "Printing:\n");
			fprintf(stderr, "  -out FILE = print solution as a list of literals to FILE\n");
			fprintf(stderr, "  -outAll FILE = print solution as a list of literals to FILE\n");
			fprintf(stderr, "  -trace N = print statistics every N flips\n");
			fprintf(stderr, "  -assign N = print assignments at flip N, 2N, ...\n");
			fprintf(stderr, "  -sol = print satisfying assignments to stdout\n");
			fprintf(stderr, "  -solcnf = print sat assign to stdout in DIMACS format, and exit\n");
			fprintf(stderr, "  -low = print lowest assignment each try\n");
			fprintf(stderr, "  -bad = print unsat clauses each try\n");
			fprintf(stderr, "  -hist = print histogram of tail\n");
			fprintf(stderr, "  -tail N = assume tail begins at nvars*N\n");
			fprintf(stderr, "  -sample N = sample noise level every N flips\n");
			fprintf(stderr, "  -hamming TARGET_FILE DATA_FILE SAMPLE_FREQUENCY\n");
			exit(-1);
		}
	}
*/
	base_cutoff = cutoff;
	if (numsol==NOVALUE || numsol>numrun) numsol = numrun;
	if (numerator==NOVALUE){
		switch(heuristic) {
		case SA:
		case BEST:
		case NOVELTY:
		case RNOVELTY:
			numerator = 50;
			break;
		default:
			numerator = 0;
			break;
		}
	}
}


void print_parameters(int argc, char * argv[])
{
	/*
	int i;
	
	printf("%s\n", VERSION);
	printf("command line =");
	for (i=0;i < argc;i++){
		printf(" %s", argv[i]);
	}
	printf("\n");
	printf("seed = %u\n",seed);
	printf("cutoff = %lli\n",cutoff);
	printf("tries = %i\n",numrun);
	
	printf("heuristic = ");
	switch(heuristic)
	{
	case TABU:
		printf("tabu %d", tabu_length);
		break;
	case NOVELTY:
		if (novelty_numerator)
			printf("novelty+ %d %d", novelty_numerator, novelty_denominator);
		else 
			printf("novelty");
		break;
	default:
		printf("%s", heuristic_names[heuristic]);
		break;
	}
	if (numerator>0){
		printf(", noise %d / %d", numerator, denominator);
	}
	printf("\n");
	*/
}

void print_statistics_header(void)
{
/*	printf("numatom = %i, numclause = %i, numliterals = %i\n",numatom,numclause,numliterals);
	printf("wff read in\n\n");

	printf("        lowest     final       avg     noise     noise     total                                      avg          mean        mean\n");
	printf("        #unsat    #unsat     noise   std dev     ratio     flips             unique     total      length         flips       flips\n");
	printf("          this      this      this      this      this      this   success   solns.     succ.     success         until         std\n");
	printf("CPU        try       try       try       try       try       try      rate    found     tries       tries        assign         dev\n\n");
	

	fflush(stdout);	
*/
}

void initialize_statistics(void)
{
	x = 0; r = 0;
	if (hamming_flag) {
		read_hamming_file(hamming_target_file);
		open_hamming_data(hamming_data_file);
	}
	tail_start_flip = tail * numatom;
#ifdef MPICODE
	if (rootProc)
#endif
		//printf("tail starts after flip = %i\n", tail_start_flip);
	numnullflip = 0;
	
	
}

void update_statistics_start_try(void)
{
	int i;
	
	lowbad = numfalse;
	
	sample_size = 0;
	sumfalse = 0.0;
	sumfalse_squared = 0.0;
	
	for (i=0; i<HISTMAX; i++)
		tailhist[i] = 0;
	if (tail_start_flip == 0){
		tailhist[numfalse < HISTMAX ? numfalse : HISTMAX - 1] ++;
	}
	
	if (printfalse) save_false_clauses(lowbad);
	if (printlow) save_low_assign();
}

void print_statistics_start_flip(void)
{
	if (printtrace && (numflip % printtrace == 0)){
		//printf(" %9i %9i                     %9lli\n", lowbad,numfalse,numflip);
		if (trace_assign)
			print_current_assign();
		fflush(stdout);
	}
}


void update_and_print_statistics_end_try(void)
{
	totalflip += numflip;
	x += numflip;
	r ++;


	if (sample_size > 0){
		avgfalse = sumfalse/sample_size;
		second_moment_avgfalse = sumfalse_squared / sample_size;
		variance_avgfalse = second_moment_avgfalse - (avgfalse * avgfalse);
		if (sample_size > 1) { variance_avgfalse = (variance_avgfalse * sample_size)/(sample_size - 1); }
		std_dev_avgfalse = sqrt(variance_avgfalse);
		
		ratio_avgfalse = avgfalse / std_dev_avgfalse;
		
		sum_avgfalse += avgfalse;
		sum_std_dev_avgfalse += std_dev_avgfalse;
		number_sampled_runs += 1;
		
		if (numfalse <= target){
			suc_number_sampled_runs += 1;
			suc_sum_avgfalse += avgfalse;
			suc_sum_std_dev_avgfalse += std_dev_avgfalse;
		}
		else {
			nonsuc_number_sampled_runs += 1;
			nonsuc_sum_avgfalse += avgfalse;
			nonsuc_sum_std_dev_avgfalse += std_dev_avgfalse;
		}
	}
	else{
		avgfalse = 0;
		variance_avgfalse = 0;
		std_dev_avgfalse = 0;
		ratio_avgfalse = 0;
	}
	
	if (inclnonsoln) save_solution();
	
	if(numfalse <= target){
		
		status_flag = 0;
		
		if (!inclnonsoln) save_solution();
		numsuccesstry++;
		
		totalsuccessflip += numflip;
		integer_sum_x += x;
		sum_x = (double) integer_sum_x;
		sum_x_squared += ((double)x)*((double)x);
		mean_x = sum_x / numsuccesstry;
		if (numsuccesstry > 1){
			second_moment_x = sum_x_squared / numsuccesstry;
			variance_x = second_moment_x - (mean_x * mean_x);
			/* Adjustment for small small sample size */
			variance_x = (variance_x * numsuccesstry)/(numsuccesstry - 1);
			std_dev_x = sqrt(variance_x);
			std_error_mean_x = std_dev_x / sqrt((double)numsuccesstry);
		}
		sum_r += r;
		mean_r = ((double)sum_r)/numsuccesstry;
		sum_r_squared += ((double)r)*((double)r);
		
		x = 0;
		r = 0;
	}
	
/*	if ((numtry % printEveryN) == 0) {
		printf("%4i %9i %9i %9.2f %9.2f %9.2f %9lli %9li",
			procRecv, lowbad, numfalse, avgfalse, std_dev_avgfalse, ratio_avgfalse, numflip, (numsuccesstry*100)/numtry);
 		printf("%9i %9i", solutionMap.getUniqueSolnCount(), solutionMap.getSolnCount());  //JSE - Print Number of unique solutions and total successes
		if (numsuccesstry > 0){
			printf(" %11lli", totalsuccessflip/numsuccesstry);
			printf(" %13.2f", mean_x);
			if (numsuccesstry > 1){
				printf(" %11.2f", std_dev_x);
			}
		}
		printf("\n");
	
		if (printhist){
			printf("histogram: ");
			for (j=HISTMAX-1; tailhist[j] == 0; j--);
			for (i=0; i<=j; i++){
				printf(" %i(%i)", tailhist[i], i);
				if ((i+1) % 10 == 0) printf("\n           ");
			}
			if (j==HISTMAX-1) printf(" +++");
			printf("\n");
		}
	
		if (numfalse>0 && printfalse)
			print_false_clauses(lowbad);
		if (printlow && (!printonlysol || numfalse >= target))
			print_low_assign(lowbad);
	}
	
	if(numfalse == 0 && countunsat() != 0){
		fprintf(stderr, "Program error, verification of solution fails!\n");
		
		exit(-1);
	}
*/	
}

void update_statistics_end_flip(void)
{
	if (numfalse < lowbad){
		lowbad = numfalse;
		if (printfalse) save_false_clauses(lowbad);
		if (printlow) save_low_assign();
	}
	if (numflip >= tail_start_flip){
		tailhist[(numfalse < HISTMAX) ? numfalse : (HISTMAX - 1)] ++;
		if ((numflip % samplefreq) == 0){
			sumfalse += numfalse;
			sumfalse_squared += numfalse * numfalse;
			sample_size ++;
		}
	}
}

void print_statistics_final(void)
{
/*	seconds_per_flip = expertime / totalflip;
	printf("\ntotal elapsed seconds = %f\n", expertime);
	printf("average flips per second = %d\n", (long)(totalflip/expertime));
	if (heuristic == TABU)
		printf("proportion null flips = %f\n", ((double)numnullflip)/totalflip);
	printf("number solutions found = %d\n", numsuccesstry);
  	printf("number of tries = %d\n", numtry);
	printf("final success rate = %f\n", ((double)numsuccesstry * 100.0)/numtry);
	printf("average length successful tries = %lli\n", numsuccesstry ? (totalsuccessflip/numsuccesstry) : 0);
	if (numsuccesstry > 0)
	{
		printf("average flips per assign (over all runs) = %f\n", ((double)totalflip)/numsuccesstry);
		printf("average seconds per assign (over all runs) = %f\n", (((double)totalflip)/numsuccesstry)*seconds_per_flip);
		printf("mean flips until assign = %f\n", mean_x);
		if (numsuccesstry>1){
			printf("  variance = %f\n", variance_x);
			printf("  standard deviation = %f\n", std_dev_x);
			printf("  standard error of mean = %f\n", std_error_mean_x);
		}
		printf("mean seconds until assign = %f\n", mean_x * seconds_per_flip);
		if (numsuccesstry>1){
			printf("  variance = %f\n", variance_x * seconds_per_flip * seconds_per_flip);
			printf("  standard deviation = %f\n", std_dev_x * seconds_per_flip);
			printf("  standard error of mean = %f\n", std_error_mean_x * seconds_per_flip);
		}
		printf("mean restarts until assign = %f\n", mean_r);
		if (numsuccesstry>1){
			variance_r = (sum_r_squared / numsuccesstry) - (mean_r * mean_r);
			if (numsuccesstry > 1) variance_r = (variance_r * numsuccesstry)/(numsuccesstry - 1);	   
			std_dev_r = sqrt(variance_r);
			std_error_mean_r = std_dev_r / sqrt((double)numsuccesstry);
			printf("  variance = %f\n", variance_r);
			printf("  standard deviation = %f\n", std_dev_r);
			printf("  standard error of mean = %f\n", std_error_mean_r);
		}
	}
	
	if (number_sampled_runs){
		mean_avgfalse = sum_avgfalse / number_sampled_runs;
		mean_std_dev_avgfalse = sum_std_dev_avgfalse / number_sampled_runs;
		ratio_mean_avgfalse = mean_avgfalse / mean_std_dev_avgfalse;
		
		if (suc_number_sampled_runs){
			suc_mean_avgfalse = suc_sum_avgfalse / suc_number_sampled_runs;
			suc_mean_std_dev_avgfalse = suc_sum_std_dev_avgfalse / suc_number_sampled_runs;
			suc_ratio_mean_avgfalse = suc_mean_avgfalse / suc_mean_std_dev_avgfalse;
		}
		else {
			suc_mean_avgfalse = 0;
			suc_mean_std_dev_avgfalse = 0;
			suc_ratio_mean_avgfalse = 0;
		}
		
		if (nonsuc_number_sampled_runs){
			nonsuc_mean_avgfalse = nonsuc_sum_avgfalse / nonsuc_number_sampled_runs;
			nonsuc_mean_std_dev_avgfalse = nonsuc_sum_std_dev_avgfalse / nonsuc_number_sampled_runs;
			nonsuc_ratio_mean_avgfalse = nonsuc_mean_avgfalse / nonsuc_mean_std_dev_avgfalse;
		}
		else {
			nonsuc_mean_avgfalse = 0;
			nonsuc_mean_std_dev_avgfalse = 0;
			nonsuc_ratio_mean_avgfalse = 0;
		}
		
		printf("final noise level statistics\n");
		printf("    statistics over all runs:\n");
		printf("      overall mean average noise level = %f\n", mean_avgfalse);
		printf("      overall mean noise std deviation = %f\n", mean_std_dev_avgfalse);
		printf("      overall ratio mean noise to mean std dev = %f\n", ratio_mean_avgfalse);
		printf("    statistics on successful runs:\n");
		printf("      successful mean average noise level = %f\n", suc_mean_avgfalse);
		printf("      successful mean noise std deviation = %f\n", suc_mean_std_dev_avgfalse);
		printf("      successful ratio mean noise to mean std dev = %f\n", suc_ratio_mean_avgfalse);
		printf("    statistics on nonsuccessful runs:\n");
		printf("      nonsuccessful mean average noise level = %f\n", nonsuc_mean_avgfalse);
		printf("      nonsuccessful mean noise std deviation = %f\n", nonsuc_mean_std_dev_avgfalse);
		printf("      nonsuccessful ratio mean noise to mean std dev = %f\n", nonsuc_ratio_mean_avgfalse);
	}
	
	if (hamming_flag){
		fclose(hamming_fp);
		printf("Final distance to hamming target = %i\n", calc_hamming_dist(atom, hamming_target, numatom));
		printf("Hamming distance data stored in %s\n", hamming_data_file);
	}
	
	if (numsuccesstry > 0){
		printf("ASSIGNMENT FOUND\n");
		if (printsolcnf == TRUE) {print_sol_cnf();}
		if (outfile[0]) {print_sol_file(outfile);}
		if (outfileAll[0]) {print_sol_fileAll(outfileAll);}
	}
	else
		printf("ASSIGNMENT NOT FOUND\n");
	
*/}


long super(int i)
{
	long power;
	int k;
	
	if (i<=0){
		fprintf(stderr, "bad argument super(%d)\n", i);
		exit(1);
	}
	/* let 2^k be the least power of 2 >= (i+1) */
	k = 1;
	power = 2;
	while (power < (i+1)){
		k += 1;
		power *= 2;
	}
	if (power == (i+1)) return (power/2);
	return (super(i - (power/2) + 1));
}

void handle_interrupt(int sig)
{
#ifdef MPICODE
	fprintf(stderr, "SIGINT: %i", mpi_id);
#endif
	if (abort_flag) exit(-1);
	abort_flag = TRUE;
}

void scanone(int argc, char *argv[], int i, int *varptr)
{
	if (i>=argc || sscanf(argv[i],"%i",varptr)!=1){
		fprintf(stderr, "Bad argument %s\n", i<argc ? argv[i] : argv[argc-1]);
		exit(-1);
	}
}

void scanone(int argc, char *argv[], int i, unsigned int *varptr)
{
	if (i>=argc || sscanf(argv[i],"%u",varptr)!=1){
		fprintf(stderr, "Bad argument %s\n", i<argc ? argv[i] : argv[argc-1]);
		exit(-1);
	}
}

void scanone(int argc, char *argv[], int i, BIGINT *varptr)
{
	if (i>=argc || sscanf(argv[i],"%lli",varptr)!=1){
		fprintf(stderr, "Bad argument %s\n", i<argc ? argv[i] : argv[argc-1]);
		exit(-1);
	}
}

// Ashish
void scanone(int argc, char *argv[], int i, double *varptr)
{
	if (i>=argc || sscanf(argv[i],"%lf",varptr)!=1){
		fprintf(stderr, "Bad argument %s\n", i<argc ? argv[i] : argv[argc-1]);
		exit(-1);
	}
}

int calc_hamming_dist(int atom[], int hamming_target[], int numatom)
{
	int i;
	int dist = 0;
	
	for (i=1; i<=numatom; i++){
		if (atom[i] != hamming_target[i]) dist++;
	}
	return dist;
}

void open_hamming_data(char initfile[])
{
	if ((hamming_fp = fopen(initfile, "w")) == NULL){
		fprintf(stderr, "Cannot open %s for output\n", initfile);
		exit(1);
	}
}


void read_hamming_file(char initfile[])
{
	int i;			/* loop counter */
	FILE * infile;
	int lit;	
	
	printf("loading hamming target file %s ...", initfile);
	
	if ((infile = fopen(initfile, "r")) == NULL){
		fprintf(stderr, "Cannot open %s\n", initfile);
		exit(1);
	}
	i=0;
	for(i = 1;i < numatom+1;i++)
		hamming_target[i] = 0;
	
	while (fscanf(infile, " %d", &lit)==1){
		if (ABS(lit)>numatom){
			fprintf(stderr, "Bad hamming file %s\n", initfile);
			exit(1);
		}
		if (lit>0) hamming_target[lit]=1;
	}
	printf("done\n");
	fclose(infile);    // Ashish
}


void init(char initfile[], int initoptions)
{
	int i;
	int j;
	int thetruelit = 0;
	FILE * infile;
	int lit;
	
	for(i = 0;i < numclause;i++)
		numtruelit[i] = 0;
	numfalse = 0;
	
	for(i = 1;i < numatom+1;i++)
	{
		changed[i] = -BIG;
		breakcount[i] = 0;
		makecount[i] = 0;
	}
	
	if (initfile[0] && initoptions!=INIT_PARTIAL){
		for(i = 1;i < numatom+1;i++)
			atom[i] = 0;
	}
	else {
		for(i = 1;i < numatom+1;i++)
			atom[i] = random()%2;
	}
	
	if (initfile[0]){
		if ((infile = fopen(initfile, "r")) == NULL){
			fprintf(stderr, "Cannot open %s\n", initfile);
			exit(1);
		}
		i=0;
		while (fscanf(infile, " %d", &lit)==1){
			i++;
			if (ABS(lit)>numatom){
				fprintf(stderr, "Bad init file %s\n", initfile);
				exit(1);
			}
			if (lit<0) atom[-lit]=0;
			else atom[lit]=1;
		}
		if (i==0){
			fprintf(stderr, "Bad init file %s\n", initfile);
			exit(1);
		}
		fclose(infile);
	}
	
	/* Initialize breakcount and makecount in the following: */
	for(i = 0;i < numclause;i++)
	{
		for(j = 0;j < size[i];j++)
		{
			if((clause[i][j] > 0) == atom[ABS(clause[i][j])])
			{
				numtruelit[i]++;
				thetruelit = clause[i][j];
			}
		}
		if(numtruelit[i] == 0)
		{
			wherefalse[i] = numfalse;
			falseC[numfalse] = i;
			numfalse++;
			for(j = 0;j < size[i];j++){
				makecount[ABS(clause[i][j])]++;
			}
		}
		else if (numtruelit[i] == 1)
		{
			breakcount[ABS(thetruelit)]++;
		}
	}
	if (hamming_flag){
		hamming_distance = calc_hamming_dist(atom, hamming_target, numatom);
		fprintf(hamming_fp, "0 %i\n", hamming_distance);
	}
}

void
print_false_clauses(long int lowbad)
{
	int i, j;
	int cl;
	
	printf("Unsatisfied clauses:\n");
	for (i=0; i<lowbad; i++){
		cl = lowfalse[i];
		for (j=0; j<size[cl]; j++){
			printf("%d ", clause[cl][j]);
		}
		printf("0\n");
	}
	printf("End unsatisfied clauses\n");
}

void
save_false_clauses(long int lowbad)
{
	int i;
	
	for (i=0; i<lowbad; i++)
		lowfalse[i] = falseC[i];
}

//JSE - Helper function
char nextChar(FILE *fp) {
	char ch;
	fscanf(fp, "%c", &ch);
	return ch;
}

void initprob(void)
{
	int i;
	int j;
	int lastc;
	int nextc;
	int *storeptr = NULL;
	int freestore;
	int lit;

	//JSE - Allows CNF input from command line
	FILE *fp;
	if (strlen(inputCNFFile) > 0)
		fp = fopen(inputCNFFile, "r");
	else
		fp = stdin;

	while ((lastc = nextChar(fp)) == 'c')
	{
		while ((nextc = nextChar(fp)) != EOF && nextc != '\n');
	}
	ungetc(lastc, fp);
	if (fscanf(fp, "p cnf %i %i",&numatom,&numclause) != 2)
	{
		fprintf(stderr,"Bad input file\n");
		exit(-1);
	}
	if(numatom > MAXATOM)
	{
		fprintf(stderr,"ERROR - too many atoms\n");
		exit(-1);
	}
	
#ifdef DYNAMIC
	clause = (int **) malloc(sizeof(int *)*(numclause+1));
	size = (int *) malloc(sizeof(int)*(numclause+1));
	falseC = (int *) malloc(sizeof(int)*(numclause+1));
	lowfalse = (int *) malloc(sizeof(int)*(numclause+1));
	wherefalse = (int *) malloc(sizeof(int)*(numclause+1));
	numtruelit = (int *) malloc(sizeof(int)*(numclause+1));
#else
	if(numclause > MAXCLAUSE)					  
	{									   
		fprintf(stderr,"ERROR - too many clauses\n"); 
		exit(-1);							   
	}										 
#endif
	freestore = 0;
	numliterals = 0;
	storerecnum = 0;
	for(i = 0;i < 2*MAXATOM+1;i++)
		numoccurence[i] = 0;
	for(i = 0;i < numclause;i++)
	{
		size[i] = -1;
		if (freestore < MAXLENGTH)
		{
			storeptr = (int *) malloc( sizeof(int) * STOREBLOCK );
			storerec[storerecnum++] = storeptr;
			if (storerecnum >= MAXSTORE) {
				printf("MAXSTORE too small\n");
				fflush(stdout);
				exit(1);
			}
			
			if (storeptr == NULL) {
				printf("not enough memory\n");
				fflush(stdout);
				exit(1);
			}
			freestore = STOREBLOCK;
//			fprintf(stderr,"allocating memory...\n");
		}
		clause[i] = storeptr;
		do
		{
			size[i]++;
			if(size[i] > MAXLENGTH)
			{
				printf("ERROR - clause too long\n");
				exit(-1);
			}
			if (fscanf(fp, "%i ",&lit) != 1)
			{
				fprintf(stderr, "Bad input file\n");
				exit(-1);
			}
			if(lit != 0)
			{
				*(storeptr++) = lit; /* clause[i][size[i]] = j; */
				freestore--;
				numliterals++;
				numoccurence[lit+MAXATOM]++;
			}
		}
		while(lit != 0);
	}

	if(size[0] == 0)
	{
		fprintf(stderr,"ERROR - incorrect problem format or extraneous characters\n");
		exit(-1);
	}
	
	for(i = 0;i < 2*MAXATOM+1;i++)
	{
		if (freestore < numoccurence[i])
		{
			storeptr = (int *) malloc( sizeof(int) * STOREBLOCK );
			storerec[storerecnum++] = storeptr;
			if (storerecnum >= MAXSTORE) {
				printf("MAXSTORE too small\n");
				fflush(stdout);
				exit(1);
			}
			freestore = STOREBLOCK;
			fprintf(stderr,"allocating memory...\n");
		}
		occurence[i] = storeptr;
		freestore -= numoccurence[i];
		storeptr += numoccurence[i];
		numoccurence[i] = 0;
	}

	
	for(i = 0;i < numclause;i++)
	{
		for(j = 0;j < size[i];j++)
		{
			occurence[clause[i][j]+MAXATOM]
				[numoccurence[clause[i][j]+MAXATOM]] = i;
			numoccurence[clause[i][j]+MAXATOM]++;
		}
	}

	// Ashish
	if (fp != stdin)
	  fclose(fp);
}


void flipatom(int toflip)
{
	int i, j;			
	int toenforce;		
	register int cli;
	register int lit;
	int numocc;
	register int sz;
	register int * litptr;
	int * occptr;
	
	/* printf("flipping %i\n", toflip); */
	
	if (toflip == NOVALUE){
		numnullflip++;
		return;
	}
	
	changed[toflip] = numflip;
	if(atom[toflip] > 0)
		toenforce = -toflip;
	else
		toenforce = toflip;
	atom[toflip] = 1-atom[toflip];
	
	if (hamming_flag){
		if (atom[toflip] == hamming_target[toflip])
			hamming_distance--;
		else
			hamming_distance++;
		if ((numflip % hamming_sample_freq) == 0)
			fprintf(hamming_fp, "%lli %i\n", numflip, hamming_distance);
	}
	
	numocc = numoccurence[MAXATOM-toenforce];
	occptr = occurence[MAXATOM-toenforce];
	for(i = 0; i < numocc ;i++)
	{
		/* cli = occurence[MAXATOM-toenforce][i]; */
		cli = *(occptr++);
		
		if (--numtruelit[cli] == 0){
			falseC[numfalse] = cli;
			wherefalse[cli] = numfalse;
			numfalse++;
			/* Decrement toflip's breakcount */
			breakcount[toflip]--;
			
			if (makeflag){
				/* Increment the makecount of all vars in the clause */
				sz = size[cli];
				litptr = clause[cli];
				for (j=0; j<sz; j++){
					/* lit = clause[cli][j]; */
					lit = *(litptr++);
					makecount[ABS(lit)]++;
				}
			}
		}
		else if (numtruelit[cli] == 1){
			/* Find the lit in this clause that makes it true, and inc its breakcount */
			sz = size[cli];
			litptr = clause[cli];
			for (j=0; j<sz; j++){
				/* lit = clause[cli][j]; */
				lit = *(litptr++);
				if((lit > 0) == atom[ABS(lit)]){
					breakcount[ABS(lit)]++;
					break;
				}
			}
		}
	}
	
	numocc = numoccurence[MAXATOM+toenforce];
	occptr = occurence[MAXATOM+toenforce];
	for(i = 0; i < numocc; i++)
	{
		/* cli = occurence[MAXATOM+toenforce][i]; */
		cli = *(occptr++);
		
		if (++numtruelit[cli] == 1){
			numfalse--;
			falseC[wherefalse[cli]] =
				falseC[numfalse];
			wherefalse[falseC[numfalse]] =
				wherefalse[cli];
			/* Increment toflip's breakcount */
			breakcount[toflip]++;
			
			if (makeflag){
				/* Decrement the makecount of all vars in the clause */
				sz = size[cli];
				litptr = clause[cli];
				for (j=0; j<sz; j++){
					/* lit = clause[cli][j]; */
					lit = *(litptr++);
					makecount[ABS(lit)]--;
				}
			}
		}
		else if (numtruelit[cli] == 2){
		/* Find the lit in this clause other than toflip that makes it true,
			and decrement its breakcount */
			sz = size[cli];
			litptr = clause[cli];
			for (j=0; j<sz; j++){
				/* lit = clause[cli][j]; */
				lit = *(litptr++);
				if( ((lit > 0) == atom[ABS(lit)]) &&
					(toflip != ABS(lit)) ){
					breakcount[ABS(lit)]--;
					break;
				}
			}
		}
	}
}

int pickrandom(void)
{
	int tofix;
	
	tofix = falseC[random()%numfalse];
	return Var(tofix, random()%size[tofix]);
}


int pickbest(void)
{
	int numbreak;
	int tofix;
	int clausesize;
	int i;		
	int best[MAXLENGTH];
	register int numbest;
	register int bestvalue;
	register int var;
	
	tofix = falseC[random()%numfalse];
	clausesize = size[tofix];
	numbest = 0;
	bestvalue = BIG;
	
	for (i=0; i< clausesize; i++){
		var = ABS(clause[tofix][i]);
		numbreak = breakcount[var];
		if (numbreak<=bestvalue){
			if (numbreak<bestvalue) numbest=0;
			bestvalue = numbreak;
			best[numbest++] = var;
		}
	}
	
	if (bestvalue>0 && (random()%denominator < numerator))
		return ABS(clause[tofix][random()%clausesize]);
	
	if (numbest == 1) return best[0];
	return best[random()%numbest];
}

int picksa(void)
{
	int change;
	int toflip;
	if (random()%100 >=sa_ratio || ((numfalse > 0) && latesa))
		if (numfalse >0)
			return pickbest();
		else 
			return NOVALUE;
	else {
		toflip = random()%numatom;
		change = makecount[toflip]-breakcount[toflip];
		if (change >= 0)
			return toflip;
		else if (random()<=exp(change/(temperature/100.0))*RAND_MAX)
			return toflip;
		else return NOVALUE;
	}

}

int picksarandom(void)
{
	int change;
	int toflip;
	if (random()%100 >=sa_ratio || ((numfalse > 0) && latesa))
		if (numfalse >0)
			return pickrandom();
		else 
			return NOVALUE;
	else {
		toflip = random()%numatom;
		change = makecount[toflip]-breakcount[toflip];
		if (change >= 0)
			return toflip;
		else if (random()<=exp(change/(temperature/100.0))*RAND_MAX)
			return toflip;
		else return NOVALUE;
	}

}

int picknovelty(void)
{
	int var, diff, birthdate;
	int youngest = 0, youngest_birthdate, best = 0, second_best = 0, best_diff, second_best_diff;
	int tofix, clausesize, i;
	
	tofix = falseC[random()%numfalse];
	clausesize = size[tofix];  
	
	if (clausesize == 1) return ABS(clause[tofix][0]);
	
	if (novelty_numerator && ((random()%novelty_denominator)<novelty_numerator))
		return ABS(clause[tofix][random()%clausesize]);
	
	youngest_birthdate = -1;
	best_diff = -BIG;
	second_best_diff = -BIG;
	best = -1;
	
	for(i = 0; i < clausesize; i++){
		var = ABS(clause[tofix][i]);
		diff = makecount[var] - breakcount[var];
		birthdate = changed[var];
		if (birthdate > youngest_birthdate){
			youngest_birthdate = birthdate;
			youngest = var;
		}
		if (diff > best_diff || (diff == best_diff && changed[var] < changed[best])) {
			/* found new best, demote best to 2nd best */
			second_best = best;
			second_best_diff = best_diff;
			best = var;
			best_diff = diff;
		}
		else if (diff > second_best_diff || (diff == second_best_diff && changed[var] < changed[second_best])){
			/* found new second best */
			second_best = var;
			second_best_diff = diff;
		}
	}
	if (best != youngest) return best;
	if (second_best<0) {
		fprintf(stderr, "Program error in picknovelty!\n");
		exit(-1);
	}
	if ((random()%denominator < numerator)) return second_best;
	return best;
}


int pickrnovelty(void)
{
	int var, diff, birthdate;
	int diffdiff;
	int youngest = 0, youngest_birthdate, best = 0, second_best = 0, best_diff, second_best_diff;
	int tofix, clausesize, i;
	
	tofix = falseC[random()%numfalse];
	clausesize = size[tofix];  
	
	if (clausesize == 1) return ABS(clause[tofix][0]);
	if ((numflip % 100) == 0) return ABS(clause[tofix][random()%clausesize]);
	
	youngest_birthdate = -1;
	best_diff = -BIG;
	second_best_diff = -BIG;
	
	for(i = 0; i < clausesize; i++){
		var = ABS(clause[tofix][i]);
		diff = makecount[var] - breakcount[var];
		birthdate = changed[var];
		if (birthdate > youngest_birthdate){
			youngest_birthdate = birthdate;
			youngest = var;
		}
		if (diff > best_diff || (diff == best_diff && changed[var] < changed[best])) {
			/* found new best, demote best to 2nd best */
			second_best = best;
			second_best_diff = best_diff;
			best = var;
			best_diff = diff;
		}
		else if (diff > second_best_diff || (diff == second_best_diff && changed[var] < changed[second_best])){
			/* found new second best */
			second_best = var;
			second_best_diff = diff;
		}
	}
	if (best != youngest) return best;
	
	diffdiff = best_diff - second_best_diff;
	
	if (numerator < 50 && diffdiff > 1) return best;
	if (numerator < 50 && diffdiff == 1){
		if ((random()%denominator) < 2*numerator) return second_best;
		return best;
	}
	if (diffdiff == 1) return second_best;
	if ((random()%denominator) < 2*(numerator-50)) return second_best;
	return best;
}


int picktabu(void)
{
	int numbreak[MAXLENGTH];
	int tofix;
	int clausesize;
	int i;			/* a loop counter */
	int best[MAXLENGTH];	/* best possibility so far */
	int numbest;		/* how many are tied for best */
	int bestvalue;		/* best value so far */
	int noisypick;
	
	tofix = falseC[random()%numfalse];
	clausesize = size[tofix];
	for(i = 0;i < clausesize;i++)
		numbreak[i] = breakcount[ABS(clause[tofix][i])];
	
	numbest = 0;
	bestvalue = BIG;
	
	noisypick = (numerator > 0 && random()%denominator < numerator); 
	for (i=0; i < clausesize; i++) {
		if (numbreak[i] == 0) {
			if (bestvalue > 0) {
				bestvalue = 0;
				numbest = 0;
			}
			best[numbest++] = i;
		}
		else if (tabu_length < numflip - changed[ABS(clause[tofix][i])]) {
			if (noisypick && bestvalue > 0) { 
				best[numbest++] = i; 
			}
			else {
				if (numbreak[i] < bestvalue) {
					bestvalue = numbreak[i];
					numbest = 0;
				}
				if (numbreak[i] == bestvalue) {
					best[numbest++] = i; 
				}
			}
		}
	}
	if (numbest == 0) return NOVALUE;
	if (numbest == 1) return Var(tofix, best[0]);
	return (Var(tofix, best[random()%numbest]));
}

int countunsat(void)
{
	int i, j, unsat, bad, lit, sign;
	
	unsat = 0;
	for (i=0;i < numclause;i++)
	{
		bad = TRUE;
		for (j=0; j < size[i]; j++)
		{
			lit = clause[i][j];
			sign = lit > 0 ? 1 : 0;
			if ( atom[ABS(lit)] == sign )
			{
				bad = FALSE;
				break;
			}
		}
		if (bad)
			unsat++;
	}
	return unsat;
}


double elapsed_seconds(void) 
{ 
	double answer;
	
#ifdef ANSI
	static long prev_time = 0;
	time( &long_time );
	/* Note:  time(&t) returns t in seconds, so do not need to /CLK_TCK */
	answer = long_time - prev_time;
	prev_time = long_time;
#endif
#ifdef NT
	static DWORD prev_time = 0;
	win_time = timeGetTime();
	/* Note:  return value of timeGetTime() is ms, so divide by 1000*/
	answer = (double)(win_time - prev_time) / (double)1000; 
	prev_time = win_time;
#endif
#ifdef UNIX
	static struct tms prog_tms;
	static long prev_times = 0;
	(void) times(&prog_tms);
	answer = ((double)(((long)prog_tms.tms_utime)-prev_times))/((double) CLK_TCK);
	prev_times = (long) prog_tms.tms_utime;
#endif
	return answer; 
}




void print_sol_cnf(void)
{
	int i;
	for(i = 1;i < numatom+1;i++)
		printf("v %i\n", solution[i] == 1 ? i : -i);
}


void print_sol_file(char * filename)
{
	FILE * fp;
	int i;
	
	if ((fp = fopen(filename, "w"))==NULL){
		fprintf(stderr, "Cannot open output file\n");
		exit(-1);
	}
	for(i = 1;i < numatom+1;i++){
		fprintf(fp, " %i", solution[i] == 1 ? i : -i);
		if (i % 10 == 0) fprintf(fp, "\n");
	}
	if ((i-1) % 10 != 0) fprintf(fp, "\n");
	fclose(fp);
}


//JSE - Prints all unique solutions to a file
//along with the number of flips to hit each soltution
void print_sol_fileAll(char * filename)  {
  ofstream file (filename);
  if (file.is_open())
  {
    solutionMap.printData(&file);
    file.close();
  } else {
		fprintf(stderr, "Cannot open output file: %s\n", filename);
	}
}


void
print_low_assign(long int lowbad)
{
	int i;
	
	printf("Begin assign with lowest # bad = %ld\n", lowbad);
	for (i=1; i<=numatom; i++){
		printf(" %d", lowatom[i]==0 ? -i : i);
		if (i % 10 == 0) printf("\n");
	}
	if ((i-1) % 10 != 0) printf("\n");
	printf("End assign\n");
}

void
print_current_assign(void)
{
	int i;
	
	printf("Begin assign at flip = %lld\n", numflip);
	for (i=1; i<=numatom; i++){
		printf(" %d", atom[i]==0 ? -i : i);
		if (i % 10 == 0) printf("\n");
	}
	if ((i-1) % 10 != 0) printf("\n");
	printf("End assign\n");
}

void
save_low_assign(void)
{
	int i;
	
	for (i=1; i<=numatom; i++)
		lowatom[i] = atom[i];
}

void
save_solution(void)
{
	int i, j;
  
	for (i=1; i<=numatom; i++) {
	  //if (i==countvar && atom[i] >0) counter++;
		if (!equivonly_flag) {    // Ashish
		  if (atom[i] >0) pos_array[i]++;
		  else neg_array[i]++;
		}
		solution[i] = atom[i];
	}
	
	// Ashish: save variable-pair statistics for equivalences (2-xors)
	if (equiv_flag) {
		for (i=1; i<numatom; i++) {
		  for (j=i+1; j<=numatom; j++) {
			if (atom[i] == atom[j]) same_array[i][j]++;
			else diff_array[i][j]++;
		  }
		}
	}
	
	//JSE - Save unique solutions
	//ww	satSoln.fromArray(solution+1, numatom);
	//ww	solutionMap.addSoln(satSoln, numflip);
	
	// Ashish: save unique solutions for sampleonly
	if (sampleonly_flag) {
		satSoln.fromArray(solution+1, numatom);
		solutionMap.addSoln(satSoln, numflip);
    if (printsamples_flag)
      cout << satSoln.c_str() << endl;
	}
}

