/*********************************************************************
 Copyright 2002-2003, Princeton University.  All rights reserved. 
 By using this software the USER indicates that he or she has read, 
 understood and will comply with the following:

 --- Princeton University hereby grants USER nonexclusive permission 
 to use, copy and/or modify this software for internal, noncommercial,
 research purposes only. Any distribution, including commercial sale 
 or license, of this software, copies of the software, its associated 
 documentation and/or modifications of either is strictly prohibited 
 without the prior consent of Princeton University.  Title to copyright
 to this software and its associated documentation shall at all times 
 remain with Princeton University.  Appropriate copyright notice shall 
 be placed on all software copies, and a complete copy of this notice 
 shall be included in all copies of the associated documentation.  
 No right is  granted to use in advertising, publicity or otherwise 
 any trademark, service mark, or the name of Princeton University. 


 --- This software and any associated documentation is provided "as is" 

 PRINCETON UNIVERSITY MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS 
 OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR FITNESS FOR A 
 PARTICULAR PURPOSE, OR THAT  USE OF THE SOFTWARE, MODIFICATIONS, OR 
 ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, 
 TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS OF A THIRD PARTY.  

 Princeton University shall not be liable under any circumstances for 
 any direct, indirect, special, incidental, or consequential damages 
 with respect to any claim by USER or any third party on account of 
 or arising from the use, or inability to use, this software or its 
 associated documentation, even if Princeton University has been advised
 of the possibility of those damages.
*********************************************************************/

using namespace std;

#include <cstdio>
#include <cstdlib>
#include <cassert>
#include <iostream>
#include <fstream>
#include <stack>
#include <vector>
#include <queue>
#include <set>

#define UNKNOWN         -1
#define NULL_CLAUSE     -1
#define NULL_CUBE       -1
#define QLEVEL_SHIFT    8
#define QLEVEL_MASK     ((1<<QLEVEL_SHIFT)-1)
#define CUBE_MASK       (1<<30)

#define DBG0(x)
#define DBG1(x)
#define DBG2(x)
#define CHECK(x)

enum DEDUCTION_RESULT {
  CONFLICT,
  SATISFIED,   // Ashish
  // NO_CONFLICT
  NO_DEDUCTION
};

enum SOLVER_RESULT {
  UNSATISFIABLE,
  SATISFIABLE,
  TIME_OUT,
  MEM_OUT,
  UNDECIDED
};

// Ashish
enum COMB_OP {
  OR,
  AND
};

class CVariable;

long get_cpu_time(void);
bool comp_qlevel (const int l1, const int l2);
bool comp_var_priority (const CVariable * v1, const CVariable * v2);


class CLitPoolElement {
 private:
  int _val;
 public:
  CLitPoolElement(void) {}
  ~CLitPoolElement() {}
  int & val(void) { return _val;   }
  int s_var(void) { return _val>>2;    }
  int var_index(void) { return _val>>3;   }
  int var_sign(void) { return ( (_val>> 2)& 0x1);  }
  void set (int s_var) {_val = (s_var << 2);    }
  void set(int vid, int sign) { _val = (((vid<<1) + sign)<< 2);   }

  //followings are for manipulate watched literals
  // Ashish: watched literals not used at all
  /*
    int direction (void) {return ((_val&0x03) - 2);  }
    bool is_watched(void) {return (_val&0x03 != 0);   }
    void unwatch(void) {_val = _val & (~0x3);   }
    void set_watch(int dir) {_val = _val + dir + 2;   }
  */

  //following are used for spacing (e.g. indicate clause's end)
  bool is_literal(void) {return _val > 0;   }
  void set_index(int cl_idx) {_val = - cl_idx;  }
  int get_index(void) { assert (_val <= 0); return -_val; }
  void dump(ostream & os= cout);
  friend ostream & operator << ( ostream & os, CLitPoolElement & l) { l.dump(os); return os; }
};

enum CLAUSE_STATUS {
  ORIGINAL_CL,
  CONFLICT_CL,
  CONF_GEN_CL,
  DELETED_CL,
  UNKNOWN_CL
};

enum CUBE_STATUS {
  ORIGINAL_CUBE,
  SAT_CUBE,
  SAT_GEN_CUBE,
  DELETED_CUBE,
  UNKNOWN_CUBE
};

//each clause has three literals being watched: 2 existential literals, just as 
//zchaff, and one universal literals which has the lowest q_level that is unassigned.
class CClause {
 public:
  int num_exist_lits;
  int num_univ_lits;
  //first and last element is the negative of clause index. 
  CLitPoolElement * exist_lits;
  //first element and last element is 0. 
  CLitPoolElement * univ_lits;
  //the unit existential literal, the corresponding CLitPoolElement will have
  //a value of negative of the original.
  int unit_lit;
  //a clause can be a conflict clause or a original clause or deleted.
  int status;
 public:
  CClause(void);
  ~CClause() {};
  void init(int num_exist, int num_univ, int cl_idx);
  void dump(ostream & os = cout);
  friend ostream & operator << ( ostream & os, CClause & c) { c.dump(os); return os; }
};

class CCube {
 public:
  int num_exist_lits;
  int num_univ_lits;
  //first and last element is the negative of clause index. 
  CLitPoolElement * exist_lits;
  //first element and last element is 0. 
  CLitPoolElement * univ_lits;
  //the unit existential literal, the corresponding CLitPoolElement will have
  //a value of negative of the original.
  int unit_lit;
  //a clause can be a conflict clause or a original clause or deleted.
  int status;
 public:
  CCube(void);
  ~CCube() {};
  void init(int num_exist, int num_univ, int cl_idx);
  void dump(ostream & os = cout);
  friend ostream & operator << ( ostream & os, CCube & c) { c.dump(os); return os; }
};


class CVariable {
 public:
  int antecedence;
  int value           :2;
  bool is_universal   :1; 
  bool is_flipped     :1;
  bool is_in_cube     :1; 
  int q_level         :QLEVEL_SHIFT;
  int dlevel;
  int asgn_stack_pos;
  int score_pos;
  unsigned scores[2];
  //if it's existential, the elements are CLitPoolElement *, pointing to the actual
  //literal. However, if it's universal, it is actually the ClauseID of the clause
  //that has this variable as watched. The actual watched literal is stored in 
  //clause->univ_watch
  vector<CLitPoolElement*> watched[2];
  vector<int> lit_clauses[2];
  vector<int> lit_cubes[2];
 public:    
  CVariable(void) { init(); }
  ~CVariable() {};
  void init(void);
  int score(void) { return scores[0] > scores[1] ? scores[0]:scores[1]; }
  void set_univ(void) { is_universal = true; }
  void set_exist(void) { is_universal = false; }
  bool is_univ (void) const { return is_universal; }
  bool is_exist(void) const { return !is_universal; }
  void dump(ostream & os = cout);
  friend ostream & operator << ( ostream & os, CVariable & v) { v.dump(os); return os; }
};

struct SolverStats {
  int num_added_clauses;
  int num_added_cubes;   // Ashish
  int num_added_exist_literals;
  int num_added_univ_literals;
  int num_original_clauses;
  int num_original_cubes;   // Ashish
  int num_original_exist_literals;
  int num_original_univ_literals;
  int num_deleted_clauses;
  int num_deleted_cubes;
    
  long long num_implications;
  int num_free_variables;
  int num_unsat_clauses;
  int num_sat_cubes;  // Ashish
  int num_decisions;
  long start_cpu_time;
  int num_backtracks;
  int num_conflicts;
  int num_sat_leaves;
  int max_dlevel;
  int outcome;
  bool is_mem_out;
};

struct SolverParams {
  float  time_limit;
  int bubble_init_step;
  int decay_period;
  int max_cl_size;
  int deletion_period;
  int cl_min_del_length;
  int cl_max_length;
  int cl_max_unrelevance;
  int cube_min_del_length;
  int cube_max_length;
  int cube_max_unrelevance;
};


struct comp_qlevel_lits {
  bool operator () (const int l1, const int l2) {
    if ( (l1 & QLEVEL_MASK) < (l2 & QLEVEL_MASK) ) return true;
    else if ((l1 & QLEVEL_MASK) > (l2 & QLEVEL_MASK) ) return false;
    if ( (l1>>QLEVEL_SHIFT) < (l2>>QLEVEL_SHIFT)) return true;
    return false;
  }
};

struct comp_assigned_vars {
  bool operator() (const CVariable * v1, const CVariable * v2) {
    if (v1->dlevel > v2->dlevel)
      return true;
    else if (v1->dlevel < v2->dlevel)
      return false;
    else if (v1->asgn_stack_pos > v2->asgn_stack_pos)
      return true;
    return false;
  }
};


typedef set<CVariable *, comp_assigned_vars> AsgnVarSet;
typedef set<int, comp_qlevel_lits> QLevelLitsSet ;

class CSolver {
 public:
  SolverStats stats;
  SolverParams params;
  vector<CVariable> variables;
  vector<vector<int> > q_groups;
  vector<int> q_unassigned_counts;
  vector<CClause> clauses;
  vector<CCube> cubes;

  // Ashish
  COMB_OP comb_op;

  // Ashish:
  vector<int> cls_count_1_n0;
  vector<int> cube_count_0_n1;
  //for clauses: the higher 16 bits are number of 1's in ALL it's literals
  //the lower 16 bits is the  number of non_zero's in it's EXISTENTIAL literals
  //    vector<int> cls_count_1_exist_n0;
  //for cubes: the higher 16 bits are number of 0's in ALL it's literals
  //the lower 16 bits is the number of non_one's in it's UNIVERSAL literals
  //vector<int> cube_count_0_univ_n1;

  vector<vector<int> > assignment_stack;
  vector<vector<int> > unit_lit_stack;
    
  queue<pair<int, int> > implication_queue;
  vector<int> unused_clause_idx;
  vector<int> unused_cube_idx;
  int dlevel;
  vector<int> conflicts;
  AsgnVarSet conf_exist;
  QLevelLitsSet conf_lits_univ;

  vector<int> sat_cubes;
  AsgnVarSet sat_univ;
  QLevelLitsSet sat_lits_exist;

  vector<pair<CVariable *,int> > ordered_vars;
  int max_score_pos;
  int num_marked;
  int num_in_new_cl;
 private:
  void init (void);
  int svar_value(int lit) { return (variables[lit>>1].value ^ (lit&0x1)); }
  int literal_value(CLitPoolElement * lit) { return svar_value((*lit).val() >> 2); }

  void decide_next_branch(void);
  void make_decision(int s_var);
  int branch_VSIDS_SLOW(void);
  int branch_VSIDS_FAST(void);
  int branch_BOEHM(void);
  int branch_DLIS(void);
  int branch_MOM(void);
  int branch_JW(void);
    
  int last_unflipped_univ_dl(void);
  int last_unflipped_exist_dl(void);
  void flip_decision_at_dl(int dl);

  void set_var_value(int v, int value, int ante);
  void set_univ_var_value(int v, int value);
  void set_exist_var_value(int v, int value);

  void unset_var_value(int v);
  void unset_univ_var_value(int v);
  void unset_exist_var_value(int v);

  int get_free_clause_idx(void);
  int get_free_cube_idx(void);
  void mark_watched_lits(int cl_idx);
  bool time_out(void);
  void run_periodic_functions(void);

  void queue_implication(int lit, int ante, bool is_cube) {
    DBG1 ( cout << "\t\t\tQueuing " << ((lit&1)?"-":"+") << (lit>>1) 
           << "  because of " << (is_cube?"Cube ":"Cls ") <<  ante << endl;);
    assert ((lit>>1) != 0);
    assert (ante < CUBE_MASK);
    if (is_cube)
      ante += CUBE_MASK;
    implication_queue.push(pair<int, int> (lit, ante));
  }

  bool is_exist_dlevel(int dl);

  int add_conf_clause(void);
  void set_up_resolve(int cl_idx);
  void resolve_one_lit(void);
  int analyze_conflicts(void);
  int analyze_conflicts_smart(void);
  int analyze_conflicts_naive(void);
  void dump_conf_cls(void);
  bool is_asserting_clause(void);
  bool is_conflicting_clause(int cl);
  bool is_sat_clause(int cl) {  return (cls_count_1_n0[cl] >= (1<<16)); }
  // Ashish: not needed
  // int num_free_lits(int cl) { return cls_count_1_exist_n0[cl] & 0x0ffff; }

  int add_sat_cube(void);
  void gen_sat_induced_cube(void);
  void set_up_consensus(int cube_idx);
  void consensus_one_lit(void);
  int analyze_satisfiability(void);
  int analyze_satisfiability_naive(void);
  int analyze_satisfiability_smart(void);
  bool is_asserting_cube(void);
  bool is_sat_cube(int cube_idx);

  void cls_adjust_variable_order(int cl);
  void cube_adjust_variable_order(int cube);
  void adjust_one_lit(int lit);
  void update_var_score(void);
  void decay_variable_score(void) ;
  void verify_integrity(void);
  void verify_integrity_one_cl(int clid);

  void deduce(void);
  void back_track(int level);
  void init_solve(void);
  DEDUCTION_RESULT preprocess(void);

  // Ashish: the following method never used!
  // int first_level_invert_quantifier(void);

  bool can_branching(void);
  bool is_conflicting(void);
  bool is_satisfied(void);
  void delete_unrelevant_clauses_and_cubes(void);

  void read_rintanen(char * filename);
  void read_qdimacs(char * filename);
 public:
  CSolver(void);
  ~CSolver(void);
  int add_orig_clause(vector<int> & lits);
  int add_orig_cube(vector<int> & lits);   // Ashish
  int num_variables(void) { return variables.size() - 1; }

  void set_variable_number(int n) { variables.resize(n + 1); }

  int solve(void);
  void set_combining_op(char * opname);  // Ashish
  COMB_OP get_combining_op(void) { return comb_op; }  // Ashish
  void read_cnfdnf(char * filename);
  void print_solution_e0(void);   // Ashish
  void detail_dump_clause(int cl, ostream & os = cout);
  void detail_dump_cube(int cube, ostream & os = cout);
  void dump_assignment_stack(ostream & os=cout);
  void dump_implication_queue(ostream & os= cout);
  void dump(ostream & os= cout);
  friend ostream & operator << ( ostream & os, CSolver & s) { s.dump(os); return os; }
} ;
