/*********************************************************************
 Copyright 2002-2003, Princeton University.  All rights reserved. 
 By using this software the USER indicates that he or she has read, 
 understood and will comply with the following:

 --- Princeton University hereby grants USER nonexclusive permission 
 to use, copy and/or modify this software for internal, noncommercial,
 research purposes only. Any distribution, including commercial sale 
 or license, of this software, copies of the software, its associated 
 documentation and/or modifications of either is strictly prohibited 
 without the prior consent of Princeton University.  Title to copyright
 to this software and its associated documentation shall at all times 
 remain with Princeton University.  Appropriate copyright notice shall 
 be placed on all software copies, and a complete copy of this notice 
 shall be included in all copies of the associated documentation.  
 No right is  granted to use in advertising, publicity or otherwise 
 any trademark, service mark, or the name of Princeton University. 


 --- This software and any associated documentation is provided "as is" 

 PRINCETON UNIVERSITY MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS 
 OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR FITNESS FOR A 
 PARTICULAR PURPOSE, OR THAT  USE OF THE SOFTWARE, MODIFICATIONS, OR 
 ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, 
 TRADEMARKS OR OTHER INTELLECTUAL PROPERTY RIGHTS OF A THIRD PARTY.  

 Princeton University shall not be liable under any circumstances for 
 any direct, indirect, special, incidental, or consequential damages 
 with respect to any claim by USER or any third party on account of 
 or arising from the use, or inability to use, this software or its 
 associated documentation, even if Princeton University has been advised
 of the possibility of those damages.
*********************************************************************/

#include "duaffle.h"
#include "config.h"
#include <algorithm>
#include <ctime>
#include <sys/resource.h>

#define MAX_LINE_LENGTH 64000
#define MAX_WORD_LENGTH 1024
#define INIT_CLAUSES_SIZE 256

bool comp_qlevel (const int l1, const int l2) {
  if ( (l1 & QLEVEL_MASK) < (l2 & QLEVEL_MASK) ) return true;
  return false;
}

bool comp_var_priority (const CVariable * v1, const CVariable * v2) {
  assert (v1->value != UNKNOWN && v2->value != UNKNOWN);
  int pos1 = (v1->lit_clauses[1-(v1->value)]).size();
  int pos2 = (v2->lit_clauses[1-(v2->value)]).size();
  if (pos1 < pos2)
    return false;
  return true;
}       

long get_cpu_time(void) {
  struct rusage ru;
  getrusage(RUSAGE_SELF, &ru);
  return ( ru.ru_utime.tv_sec*1000 +
           ru.ru_utime.tv_usec/1000+
           ru.ru_stime.tv_sec*1000 +
           ru.ru_stime.tv_usec/1000 );
}

//===================================================================================
void CLitPoolElement::dump(ostream & os) {
  os << (var_sign()?" -":" +") << var_index();
  /* Ashish: watched literals not used
     if (is_watched())
     os << "*";
  */
}

//===================================================================================
CClause::CClause(void) {
  num_exist_lits = 0;
  num_univ_lits = 0;
  exist_lits = NULL;
  univ_lits = NULL;
  unit_lit = -1;
}

void CClause::init(int num_exist, int num_univ, int cl_idx) {
  num_exist_lits = num_exist;
  num_univ_lits = num_univ;
  if (exist_lits != NULL) 
    delete [] exist_lits;
  exist_lits = new CLitPoolElement[num_exist +2];
  if (univ_lits != NULL)
    delete [] univ_lits;
  univ_lits = new CLitPoolElement[num_univ + 2];

  exist_lits[0].set_index(cl_idx);
  exist_lits[num_exist + 1].set_index(cl_idx);
  univ_lits[0].val() = 0;
  univ_lits[num_univ + 1].val() = 0;
}

void CClause::dump(ostream & os ) { 
  if (status==DELETED_CL) 
    os << "\t\t\t======removed=====";
  os << "E: " ;
  for (CLitPoolElement * ptr = exist_lits + 1; ptr->is_literal(); ++ptr)
    os << (*ptr) ;
  os << "  U: " ;
  for (CLitPoolElement * ptr = univ_lits + 1; ptr->is_literal(); ++ptr)
    os << (*ptr) ;
  os << endl;
}

//===================================================================================
CCube::CCube(void) {
  num_exist_lits = 0;
  num_univ_lits = 0;
  exist_lits = NULL;
  univ_lits = NULL;
  unit_lit = -1;
}

void CCube::init(int num_exist, int num_univ, int cube_idx) {
  num_exist_lits = num_exist;
  num_univ_lits = num_univ;
  if (exist_lits != NULL) 
    delete [] exist_lits;
  exist_lits = new CLitPoolElement[num_exist +2];
  if (univ_lits != NULL)
    delete [] univ_lits;
  univ_lits = new CLitPoolElement[num_univ + 2];

  exist_lits[0].set_index(cube_idx);
  exist_lits[num_exist + 1].set_index(cube_idx);
  univ_lits[0].val() = 0;
  univ_lits[num_univ + 1].val() = 0;
}

void CCube::dump(ostream & os ) { 
  if (status==DELETED_CL) 
    os << "\t\t\t======removed=====";
  os << "E: " ;
  for (CLitPoolElement * ptr = exist_lits + 1; ptr->is_literal(); ++ptr)
    os << (*ptr) ;
  os << "  U: " ;
  for (CLitPoolElement * ptr = univ_lits + 1; ptr->is_literal(); ++ptr)
    os << (*ptr) ;
  os << endl;
}

//===================================================================================

void CVariable::dump(ostream & os) {
  os << (is_univ()?"U ": "E ")<< " QL: " << q_level 
     << "  V: " << value << "  DL: " << dlevel << "  AsgnPos: "<< asgn_stack_pos 
     << "  Ante: " << antecedence << endl;
  for (unsigned j=0; j< 2; ++j) {
    os << (j==0?"Pos ":"Neg ") <<  "(" ;
    for (int i=0; i< lit_clauses[j].size(); ++i )
      os << lit_clauses[j][i] << "  " ;
    os << ")" ;
    os << endl;
  }
  os << endl;
}

void CVariable::init(void) {
  antecedence = NULL_CLAUSE;
  value = UNKNOWN;
  is_universal = false;
  is_flipped = false;
  is_in_cube = false;
  q_level = -1;
  dlevel = -1;
  asgn_stack_pos = -1;
  score_pos = -1;
  scores[0] = scores[1] = 0;
}

//===================================================================================

CSolver::CSolver(void) {
  init();
}

CSolver::~CSolver() {
  for (int i=0; i< clauses.size(); ++i) {
    if (clauses[i].exist_lits != NULL)
      delete [] clauses[i].exist_lits;
    if (clauses[i].univ_lits != NULL)
      delete [] clauses[i].univ_lits;
  }
}

void CSolver::detail_dump_cube(int cube_idx, ostream & os) {
  os << "CUBE " << cube_idx <<" : ";
  CCube & cube = cubes[cube_idx];
  if (cube.status==DELETED_CUBE) 
    os << "\t\t\t======removed=====";
  os << "N0: " << (cube_count_0_n1[cube_idx] >> 16) 
     << " Nn1: " << (cube_count_0_n1[cube_idx]&0xffff);
  os << "  E: ";
  for (CLitPoolElement * ptr = cube.exist_lits + 1; ptr->is_literal(); ++ptr) {
    char value;
    if (literal_value(ptr)==0) value = '0';
    else if (literal_value(ptr)==1) value = '1';
    else value = 'X';
    os << (*ptr) << "(" << value << "@" << variables[ptr->var_index()].dlevel << ":" << 
      variables[ptr->var_index()].q_level << ")  ";
  }
  os << "  U: ";
  for (CLitPoolElement * ptr = cube.univ_lits + 1; ptr->is_literal(); ++ptr) {
    char value;
    if (literal_value(ptr)==0) value = '0';
    else if (literal_value(ptr)==1) value = '1';
    else value = 'X';
    os << (*ptr) << "(" << value << "@" << variables[ptr->var_index()].dlevel << ":" << 
      variables[ptr->var_index()].q_level << ")  ";
  }
  os << endl;
}

void CSolver::detail_dump_clause(int cl_idx, ostream & os) {
  os << "CL " << cl_idx <<" : ";
  CClause & cl = clauses[cl_idx];
  if (cl.status==DELETED_CL) 
    os << "\t\t\t======removed=====";
  os << "N1: " << (cls_count_1_n0[cl_idx] >> 16) 
     << " Nn0: " << (cls_count_1_n0[cl_idx]&0xffff);
  os << "  E: ";
  for (CLitPoolElement * ptr = cl.exist_lits + 1; ptr->is_literal(); ++ptr) {
    char value;
    if (literal_value(ptr)==0) value = '0';
    else if (literal_value(ptr)==1) value = '1';
    else value = 'X';
    os << (*ptr) << "(" << value << "@" << variables[ptr->var_index()].dlevel << ":" << 
      variables[ptr->var_index()].q_level << ")  ";
  }
  os << "  U: ";
  for (CLitPoolElement * ptr = cl.univ_lits + 1; ptr->is_literal(); ++ptr) {
    char value;
    if (literal_value(ptr)==0) value = '0';
    else if (literal_value(ptr)==1) value = '1';
    else value = 'X';
    os << (*ptr) << "(" << value << "@" << variables[ptr->var_index()].dlevel << ":" << 
      variables[ptr->var_index()].q_level << ")  ";
  }
  os << endl;
}

void CSolver::dump_assignment_stack(ostream & os ) {
  os << "Assignment Stack:  ";
  for (int i=0; i<= dlevel; ++i) {
    if (i > 0) {
      if ( variables[assignment_stack[i][0]>>1].is_flipped)
        os << "FL";
      else 
        os <<"NF";
    }
    os << "(" <<i << ":";
            
    for (int j=0; j<assignment_stack[i].size(); ++j )
      os << (assignment_stack[i][j]&0x1?"-":"+")
         << (assignment_stack[i][j] >> 1) << " ";
    os << ") " << endl;
  }
  os << endl;
}

void CSolver::dump_implication_queue(ostream & os) {
  queue<pair<int, int> > my_queue(implication_queue);
  while(!my_queue.empty()) {
    int lit = my_queue.front().first;
    int ante = my_queue.front().second;
    my_queue.pop();
    os << "(" << ((lit&1)?"-":"+") << (lit>>1) 
       << ":" << ante << ")  ";
  }
  os << endl;
}

void CSolver::dump(ostream & os) {
  for (int i=0;i< clauses.size(); ++i)
    detail_dump_clause(i);
  for (int i=0;i< cubes.size(); ++i)
    detail_dump_cube(i);
  for (int i=1; i<variables.size(); ++i) 
    os <<"VID: " << i << "  " <<  variables[i];
  dump_assignment_stack();
}

void CSolver::init(void) {
  stats.num_added_clauses= 0;
  stats.num_added_cubes= 0;   // Ashish
  stats.num_added_exist_literals= 0;
  stats.num_added_univ_literals= 0;
  stats.num_original_clauses= 0;
  stats.num_original_cubes= 0;   // Ashish
  stats.num_original_exist_literals= 0;
  stats.num_original_univ_literals= 0;
  stats.num_deleted_clauses= 0;
  stats.num_deleted_cubes= 0;
  stats.num_unsat_clauses = 0;
  stats.num_sat_cubes = 0;   // Ashish
    
  stats.num_implications= 0;
  stats.num_free_variables= 0;
  stats.num_decisions= 0;
  stats.start_cpu_time= 0;
  stats.num_backtracks= 0;
  stats.max_dlevel= 0;
  stats.num_conflicts = 0;
  stats.num_sat_leaves = 0;

  params.time_limit           = 24*3600;
  params.bubble_init_step     = 64;
  params.decay_period         = 0x200;
  params.deletion_period      = 0x5000;
  params.cl_min_del_length    = 30;
  params.cl_max_length        = 100;
  params.cl_max_unrelevance   = 20;
  params.cube_min_del_length  = 30;
  params.cube_max_length      = 100;
  params.cube_max_unrelevance = 20;

  clauses.reserve(INIT_CLAUSES_SIZE);
  assignment_stack.resize(1);
  unit_lit_stack.resize(1);
}

int CSolver::get_free_clause_idx(void) {
  int cl_idx;
  if (!unused_clause_idx.empty()) {   
    cl_idx = unused_clause_idx.back();
    unused_clause_idx.pop_back();
  }
  else {
    cl_idx = clauses.size();
    clauses.resize(cl_idx + 1);
    cls_count_1_n0.resize(cl_idx + 1);
  }
  return cl_idx;
}

int CSolver::get_free_cube_idx(void) {
  int cube_idx;
  if (!unused_cube_idx.empty()) {     
    cube_idx = unused_cube_idx.back();
    unused_cube_idx.pop_back();
  }
  else {
    cube_idx = cubes.size();
    cubes.resize(cube_idx + 1);
    cube_count_0_n1.resize(cube_idx + 1);
  }
  return cube_idx;
}

// Ashish: method not used at all
// void CSolver::mark_watched_lits(int cl_idx)
// {
//   CClause & cl = clauses[cl_idx];
//   if (cl.num_exist_lits > 1) {
//     int max_dl = -1;
//     CLitPoolElement * max_lit = NULL;
//     CLitPoolElement * ptr;
//     for (ptr = cl.exist_lits + 1; ptr->is_literal(); ++ptr){
//       int vid = ptr->var_index();
//       int sign = ptr->var_sign();
//       if (literal_value(ptr) != 0) {
//         variables[vid].watched[sign].push_back(ptr);
//         ptr->set_watch(1);
//         break;
//       }
//       else if (variables[vid].dlevel > max_dl) {
//         max_dl = variables[vid].dlevel;
//         max_lit = ptr;
//       }
//     }       
//     if (!ptr->is_literal()) {
//       int vid = max_lit->var_index();
//       int sign = max_lit->var_sign();
//       variables[vid].watched[sign].push_back(max_lit);
//       max_lit->set_watch(1);
//     }
//     max_dl = -1;
//     max_lit = NULL;
//     for (ptr = cl.exist_lits + cl.num_exist_lits; ptr->is_literal(); --ptr) {
//       int vid = ptr->var_index(); 
//       int sign = ptr->var_sign();
//       if (ptr->is_watched())
//         continue;
//       if (literal_value(ptr) != 0) {
//         variables[vid].watched[sign].push_back(ptr);
//         ptr->set_watch(-1);
//         break;
//       }
//       else if (variables[vid].dlevel > max_dl) {
//         max_dl = variables[vid].dlevel;
//         max_lit = ptr;
//       }
//     }
//     if (!ptr->is_literal()) {
//       int vid = max_lit->var_index();
//       int sign = max_lit->var_sign();
//       variables[vid].watched[sign].push_back(max_lit);
//       max_lit->set_watch(-1);
//     }
//   }
// }

void CSolver::verify_integrity_one_cl(int cl_idx) {
  int num_0 = 0;
  int num_1 = 0;
  for (CLitPoolElement * ptr = clauses[cl_idx].exist_lits + 1; ptr->is_literal(); ++ptr) {
    if (literal_value(ptr) == 0)
      ++num_0;
    else if (literal_value(ptr) == 1)
      ++num_1;
  }
  for (CLitPoolElement * ptr = clauses[cl_idx].univ_lits + 1; ptr->is_literal(); ++ptr) {
    if (literal_value(ptr) == 0)
      ++num_0;
    else if (literal_value(ptr) == 1)
      ++num_1;
  }
  assert (cls_count_1_n0[cl_idx] == ((num_1 << 16) + (clauses[cl_idx].num_exist_lits + clauses[cl_idx].num_univ_lits - num_0)));
}


void CSolver::verify_integrity(void) {
  static int entry = 0;
  ++ entry;
  for (int i=0; i< clauses.size(); ++i) {
    int cl_idx = i;
    int unit_lit = -1;
    verify_integrity_one_cl(cl_idx);
    if (cls_count_1_n0[cl_idx] == 1) {
      for (CLitPoolElement * ptr = clauses[cl_idx].exist_lits + 1; ptr->is_literal(); ++ptr) {
        int vid = ptr->var_index();
        if (variables[vid].value == UNKNOWN) {
          assert(unit_lit == -1);
          unit_lit = ptr->s_var();
        }
      }
      assert (unit_lit == clauses[cl_idx].unit_lit);
    }
  }
}

int CSolver::add_orig_clause(vector<int>& lits) {
  vector<int> exist;
  vector<int> univ;
  int cl_idx = get_free_clause_idx();
  CClause & cl = clauses[cl_idx];
  for (vector<int>::iterator itr = lits.begin(); itr != lits.end(); ++itr) {          
    int vid = (*itr)>>1;
    int sign = ((*itr) & 0x1);
    if (variables[vid].is_exist())
      exist.push_back(*itr);
    else 
      univ.push_back(((*itr) << QLEVEL_SHIFT) + variables[vid].q_level);
    variables[vid].lit_clauses[sign].push_back(cl_idx);
  }
  ++ stats.num_unsat_clauses;
  cl.init(exist.size(), univ.size(), cl_idx);
  cls_count_1_n0[cl_idx] = exist.size() + univ.size();
  //set up the existential array
  for (int i=0, sz = exist.size(); i< sz; ++i)
    cl.exist_lits[i+1].set(exist[i]);
  //set up the universal array
  ::sort(univ.begin(), univ.end(), comp_qlevel);
  for (int i=0, sz = univ.size(); i< sz; ++i) {
    assert (i==0 || variables[univ[i-1]>>(QLEVEL_SHIFT + 1)].q_level <= variables[univ[i]>>(QLEVEL_SHIFT +1)].q_level);
    cl.univ_lits[i+1].set(univ[i] >> QLEVEL_SHIFT);
  }
  //    mark_watched_lits(cl_idx);
  clauses[cl_idx].status = ORIGINAL_CL;
  stats.num_added_exist_literals += exist.size();
  stats.num_added_univ_literals += univ.size();
  ++ stats.num_added_clauses;
  return cl_idx;
}

// Ashish
int CSolver::add_orig_cube(vector<int>& lits) {
  vector<int> exist;
  vector<int> univ;
  int cube_idx = get_free_cube_idx();
  CCube & cube = cubes[cube_idx];
  for (vector<int>::iterator itr = lits.begin(); itr != lits.end(); ++itr) {          
    int vid = (*itr)>>1;
    int sign = ((*itr) & 0x1);
    if (variables[vid].is_univ())
      univ.push_back(*itr);
    else 
      exist.push_back(((*itr) << QLEVEL_SHIFT) + variables[vid].q_level);
    variables[vid].lit_cubes[sign].push_back(cube_idx);
  }
  ++ stats.num_sat_cubes;
  cube.init(exist.size(), univ.size(), cube_idx);
  cube_count_0_n1[cube_idx] = exist.size() + univ.size();
  //set up the universal array
  for (int i=0, sz = univ.size(); i< sz; ++i)
    cube.univ_lits[i+1].set(univ[i]);
  //set up the existential array
  ::sort(exist.begin(), exist.end(), comp_qlevel);
  for (int i=0, sz = exist.size(); i< sz; ++i) {
    assert (i==0 || variables[exist[i-1]>>(QLEVEL_SHIFT + 1)].q_level <= variables[exist[i]>>(QLEVEL_SHIFT +1)].q_level);
    cube.exist_lits[i+1].set(exist[i] >> QLEVEL_SHIFT);
  }
  //    mark_watched_lits(cube_idx);
  cubes[cube_idx].status = ORIGINAL_CUBE;
  stats.num_added_exist_literals += exist.size();
  stats.num_added_univ_literals += univ.size();
  ++ stats.num_added_cubes;
  return cube_idx;
}

void CSolver::back_track(int blevel) {
  DBG0(cout << "Back track to " << blevel <<" ,currently in "<< dlevel << " level" << endl;);
  DBG2(dump());
  assert(blevel <= dlevel);
  for (int i=dlevel; i>= blevel; --i) {
    vector<int> & assignments = assignment_stack[i];
    for (int j=assignments.size()-1 ; j>=0; --j) 
      unset_var_value(assignments[j]>>1);
    assignments.clear();

    vector<int> & unit_lits = unit_lit_stack[i];
    for (int j=unit_lits.size()-1 ; j>=0; --j) {
      if (unit_lits[j] < CUBE_MASK)
        clauses[unit_lits[j]].unit_lit = -1;
      else 
        cubes[unit_lits[j] - CUBE_MASK].unit_lit = -1;
    }
    unit_lits.clear();
  }
  dlevel = blevel - 1;
  if (dlevel < 0 ) 
    dlevel = 0;
  ++ stats.num_backtracks;
  DBG2 (dump());
}

void CSolver::flip_decision_at_dl(int dl) {
  assert (dl <= dlevel);
  int dlit = assignment_stack[dl][0];
  back_track(dl);
  ++dlevel;
  variables[dlit>>1].is_flipped = true;
  queue_implication(dlit^0x1, NULL_CLAUSE, false);
}

void CSolver::deduce(void) {
  //    verify_integrity();
  while (!implication_queue.empty() && conflicts.empty() && sat_cubes.empty()) {
    int lit = implication_queue.front().first;
    int idx = implication_queue.front().second;
    int vid = lit>>1;
    CVariable & var = variables[vid];
    implication_queue.pop();
    if (idx >= CUBE_MASK) {
      assert (var.is_univ());
      idx -= CUBE_MASK;
    }
    if ( var.value == UNKNOWN) {
      assert (vid != 0);
      set_var_value(vid, !(lit&0x1), idx);
      var.asgn_stack_pos = assignment_stack[dlevel].size();
      assignment_stack[dlevel].push_back(lit);
    }
    else if (var.value == (lit&0x1) ) { //a conflict
      assert (0 && "Can't happen ");
      if (var.is_exist())
        conflicts.push_back(idx);
      else 
        sat_cubes.push_back(idx);
    }
  }
  //if loop exited because a conflict, we need to clean implication queue
  while(!implication_queue.empty()) 
    implication_queue.pop();        
}

bool CSolver::time_out(void) {
  if (get_cpu_time() - stats.start_cpu_time > params.time_limit * 1000)
    return true;
  return false;
}

bool CSolver::can_branching(void) {
  if (implication_queue.empty() && stats.num_free_variables > 0) //some other routine added a implication
    return true;
  return false;
}

bool CSolver::is_conflicting(void) {
  if (!conflicts.empty())
    return true;
  // Ashish: add the following for comp_op == AND
  if (comb_op == AND && stats.num_sat_cubes == 0)
    return true;
  return false;
}

bool CSolver::is_satisfied(void) {
  DBG1(
       if (!sat_cubes.empty())
       cout << "found SAT: sat_cubes not empty!" << endl;
       if (stats.num_free_variables == 0)
       cout << "found SAT: no free variables!" << endl;
       if (comb_op == OR && stats.num_unsat_clauses == 0)
       cout << "found SAT: no unsat clauses and \"or\" as the combining op!" << endl;
       );

  if (!sat_cubes.empty())
    return true;
  if (stats.num_free_variables == 0)
    return true;
  // Ashish: be careful with the following!
  if (comb_op == OR && stats.num_unsat_clauses == 0)
    return true;
  return false;
}

int CSolver::solve(void) {
  init_solve();
  DBG1(dump());

  DEDUCTION_RESULT result = preprocess();
  if (result == SATISFIED) {
    stats.outcome = SATISFIABLE;
    return stats.outcome;
  }
  else if (result == CONFLICT) {
    DBG0(cout << "Preprocessing: conflict clause " << conflicts[0]
         << endl << clauses[conflicts[0]];);
    stats.outcome = UNSATISFIABLE;
    return stats.outcome;
  }

  while(1) {
    if (can_branching()) {
      run_periodic_functions();
      decide_next_branch();
    }
    while (1) {
      deduce();
      // If comb_op is OR, give priority to is_satisfied check,
      // otherwise give priority to is_conflicting check
      if (comb_op == OR) {
        if (is_satisfied()) {
          int blevel = analyze_satisfiability();
          if (blevel < 0) {
            stats.outcome = SATISFIABLE;
            return stats.outcome;
          }
        }
        else if (is_conflicting()) {
          int blevel = analyze_conflicts();
          if (blevel < 0) {
            stats.outcome = UNSATISFIABLE;
            return stats.outcome;
          }
        }
      }
      else {   // comb_op == AND
        if (is_conflicting()) {
          int blevel = analyze_conflicts();
          if (blevel < 0) {
            stats.outcome = UNSATISFIABLE;
            return stats.outcome;
          }
        }
        else if (is_satisfied()) {
          int blevel = analyze_satisfiability();
          if (blevel < 0) {
            stats.outcome = SATISFIABLE;
            return stats.outcome;
          }
        }
      }
      break;
    }
    if (time_out()) { 
      stats.outcome = TIME_OUT;
      return stats.outcome;
    }
  }
  return stats.outcome;
}

void CSolver::init_solve(void) {
  stats.start_cpu_time = get_cpu_time();
  stats.num_original_clauses = stats.num_added_clauses;
  stats.num_original_cubes = stats.num_added_cubes;   // Ashish
  stats.num_original_exist_literals = stats.num_added_exist_literals;
  stats.num_original_univ_literals = stats.num_added_univ_literals;
  stats.num_free_variables = variables.size() - 1;
  stats.outcome = UNDECIDED;
  q_unassigned_counts.resize(q_groups.size());
  for (unsigned i=0; i< q_unassigned_counts.size();++i) 
    q_unassigned_counts[i] = q_groups[i].size();

  for (unsigned i=0, sz = variables.size(); i< sz; ++i) {
    int q_level_compensation = 1024 * 1024 * (q_groups.size() - variables[i].q_level);
    variables[i].scores[0] = variables[i].lit_clauses[0].size() + q_level_compensation;
    variables[i].scores[1] = variables[i].lit_clauses[1].size() + q_level_compensation;
  }
  ordered_vars.resize( num_variables());
  update_var_score();
  dlevel = 0;
}

void CSolver::decay_variable_score(void) {
  unsigned int i, sz;
  for(i=1, sz = variables.size(); i<sz; ++i) {
    CVariable & var = variables[i];
    int q_level_compensation = 1024 * 1024 * (q_groups.size() - var.q_level);
    var.scores[0] = var.scores[0]/2 + q_level_compensation;
    var.scores[1] = var.scores[1]/2 + q_level_compensation;
  }
  for (i=0, sz = ordered_vars.size(); i<sz; ++i) {
    ordered_vars[i].second = ordered_vars[i].first->score();
    /*      assert (i==0 || _var_order[i].second <= _var_order[i-1].second );
            assert ( _var_order[i].first->value() != UNKNOWN || 
            _max_score_pos <= i); */
  }
}

inline bool cmp_var_stat(const pair<CVariable *,int> & v1, 
                            const pair<CVariable *,int> & v2) {
  if (v1.second >= v2.second) return true;
  return false;
};

void CSolver::update_var_score(void) {
  unsigned int i,sz;
  for (i=1, sz=variables.size(); i<sz; ++i) {
    ordered_vars[i-1].first = & variables[i];
    ordered_vars[i-1].second = variables[i].score();
  }
  stable_sort(ordered_vars.begin(), ordered_vars.end(), cmp_var_stat);
  for (i=0, sz= ordered_vars.size(); i<sz; ++i) 
    ordered_vars[i].first->score_pos = i;
  max_score_pos = 0;
}

void CSolver::adjust_one_lit(int lit) {
  int var_idx = (lit>>1);
  int var_sign = (lit&0x1);
  CVariable & var = variables[var_idx];
  int orig_score = var.score();
  ++ var.scores[var_sign];
  int new_score = var.score();
  if (orig_score == new_score) 
    return;
  int pos = var.score_pos;
  int orig_pos = pos;
  assert (ordered_vars[pos].first == & var);
  assert (ordered_vars[pos].second == orig_score);
  /* this is a binary search */
  int bubble_step = params.bubble_init_step;
  for (pos = orig_pos ; pos >= 0; pos -= bubble_step)
    if (ordered_vars[pos].second >= new_score) break;
  pos += bubble_step;
  for ( bubble_step = (bubble_step >> 1); bubble_step > 0; bubble_step = (bubble_step >> 1)) {
    if ( pos - bubble_step >= 0) {
      if (ordered_vars[pos - bubble_step].second < new_score)
        pos -= bubble_step;
    }
  }
  //now found the position, do a swap
  ordered_vars[orig_pos] = ordered_vars[pos];
  ordered_vars[orig_pos].first->score_pos = orig_pos;

  ordered_vars[pos].first = & var;
  ordered_vars[pos].second = new_score;
  ordered_vars[pos].first->score_pos = pos;
}

void CSolver::cls_adjust_variable_order(int clid) {
  for (CLitPoolElement * ptr = clauses[clid].exist_lits + 1; ptr->is_literal(); ++ptr)
    adjust_one_lit(ptr->s_var());
  for (CLitPoolElement * ptr = clauses[clid].univ_lits + 1; ptr->is_literal(); ++ptr)
    adjust_one_lit(ptr->s_var());
}

void CSolver::cube_adjust_variable_order(int cube_id) {
  for (CLitPoolElement * ptr = cubes[cube_id].exist_lits + 1; ptr->is_literal(); ++ptr)
    adjust_one_lit(ptr->s_var());
  for (CLitPoolElement * ptr = cubes[cube_id].univ_lits + 1; ptr->is_literal(); ++ptr)
    adjust_one_lit(ptr->s_var());
}

void CSolver::set_var_value(int v, int value, int ante) {
  DBG1 (cout << "\tSetting\t" << (value>0?"+":"-") << v 
        << " at " << dlevel << " because " << ante<< endl;);
  assert (variables[v].value == UNKNOWN);
  assert (value == 0 || value == 1);
  variables[v].value = value;
  variables[v].dlevel = dlevel;
  variables[v].antecedence = ante;
  if (variables[v].is_univ()) 
    set_univ_var_value(v, value);
  else 
    set_exist_var_value(v, value);
  ++ stats.num_implications;
  -- stats.num_free_variables;
  -- q_unassigned_counts[variables[v].q_level];
}


void CSolver::set_univ_var_value(int v, int value) {
  vector<int> & pos_cls = variables[v].lit_clauses[!value];
  for (vector<int>::iterator itr = pos_cls.begin(); itr != pos_cls.end(); ++itr) {
    int cl_idx = *itr;
    if (cls_count_1_n0[cl_idx] < (1<<16)) {
      -- stats.num_unsat_clauses;
    }
    cls_count_1_n0[cl_idx] += (1<<16);
  }

  vector<int> & neg_cls = variables[v].lit_clauses[value];
  for (vector<int>::iterator itr = neg_cls.begin(); itr != neg_cls.end(); ++itr) {
    int cl_idx = *itr;
    --cls_count_1_n0[cl_idx];
    if (cls_count_1_n0[cl_idx] == 1) {
      unit_lit_stack[dlevel].push_back(cl_idx);
      assert (clauses[cl_idx].unit_lit == -1);
      for (CLitPoolElement * ptr = clauses[cl_idx].exist_lits + 1; ptr->is_literal(); ++ptr) {
        int vid = ptr->var_index();
        if (variables[vid].value == UNKNOWN) {
          clauses[cl_idx].unit_lit = ptr->s_var();
          break;
        }
      }
      int unit_lit = clauses[cl_idx].unit_lit;
      if (unit_lit == -1)
        // -2 indicates the unit lit is a universal variable
        // and therefore no unit prop
        clauses[cl_idx].unit_lit = -2;
      else
        queue_implication(unit_lit, cl_idx, false);
    }
    else if (cls_count_1_n0[cl_idx] == 0) {
      DBG0(cout << "Clause " << cl_idx << " violated" << endl;);
      conflicts.push_back(cl_idx);
    }
  }

  vector<int> & pos_cubes = variables[v].lit_cubes[!value];
  for (vector<int>::iterator itr = pos_cubes.begin(); itr < pos_cubes.end(); ++itr) {
    int cube_idx = *itr;
    --cube_count_0_n1[cube_idx];
    if (cube_count_0_n1[cube_idx] == 1) {
      unit_lit_stack[dlevel].push_back(cube_idx + CUBE_MASK);
      assert (cubes[cube_idx].unit_lit == -1);
      for (CLitPoolElement * ptr = cubes[cube_idx].univ_lits + 1; ptr->is_literal(); ++ptr) {
        int vid = ptr->var_index();
        if (variables[vid].value == UNKNOWN) {
          cubes[cube_idx].unit_lit = ptr->s_var();
          break;
        }
      }
      int unit_lit = cubes[cube_idx].unit_lit;
      if (unit_lit == -1)
        // -2 indicates the unit lit is an existential variable
        // and therefore no unit prop
        cubes[cube_idx].unit_lit = -2;
      else
        queue_implication(unit_lit^0x1, cube_idx, true);
    }
    else if (cube_count_0_n1[cube_idx] == 0) {
      DBG0(cout << "Cube " << cube_idx << " satisfied" << endl;);
      sat_cubes.push_back(cube_idx);
    }
  }

  vector<int> & neg_cubes = variables[v].lit_cubes[value];
  for (vector<int>::iterator itr = neg_cubes.begin(); itr < neg_cubes.end(); ++itr) {
    int cube_idx = *itr;
    if (cube_count_0_n1[cube_idx] < (1<<16)) {
      -- stats.num_sat_cubes;
    }
    cube_count_0_n1[cube_idx] += (1<<16);
  }
}

void CSolver::set_exist_var_value(int v, int value)  {
  vector<int> & pos = variables[v].lit_clauses[!value];
  for (vector<int>::iterator itr = pos.begin(); itr < pos.end(); ++itr) {
    int cl_idx = *itr;
    if (cls_count_1_n0[cl_idx] < (1<<16)) {
      -- stats.num_unsat_clauses;
    }
    cls_count_1_n0[cl_idx] += (1<<16);
  }
    
  vector<int> & neg = variables[v].lit_clauses[value];
  for (vector<int>::iterator itr = neg.begin(); itr < neg.end(); ++itr) {
    int cl_idx = *itr;
    --cls_count_1_n0[cl_idx];
    if (cls_count_1_n0[cl_idx] == 1) {
      unit_lit_stack[dlevel].push_back(cl_idx);
      assert (clauses[cl_idx].unit_lit == -1);
      for (CLitPoolElement * ptr = clauses[cl_idx].exist_lits + 1; ptr->is_literal(); ++ptr) {
        int vid = ptr->var_index();
        if (variables[vid].value == UNKNOWN) {
          clauses[cl_idx].unit_lit = ptr->s_var();
          break;
        }
      }
      int unit_lit = clauses[cl_idx].unit_lit;
      if (unit_lit == -1)
        // -2 indicates the unit lit is a universal variable
        // and therefore no unit prop
        clauses[cl_idx].unit_lit = -2;
      else
        queue_implication(unit_lit, cl_idx, false);
    }
    else if (cls_count_1_n0[cl_idx] == 0) {
      DBG0(cout << "Clause " << cl_idx << " violated" << endl;);
      conflicts.push_back(cl_idx);
    }
  }

  vector<int> & pos_cube = variables[v].lit_cubes[!value];
  for (vector<int>::iterator itr = pos_cube.begin(); itr != pos_cube.end(); ++itr) {
    int cube_idx = *itr;
    --cube_count_0_n1[cube_idx];
    if (cube_count_0_n1[cube_idx] == 1) {
      unit_lit_stack[dlevel].push_back(cube_idx + CUBE_MASK);
      assert (cubes[cube_idx].unit_lit == -1);
      for (CLitPoolElement * ptr = cubes[cube_idx].univ_lits + 1; ptr->is_literal(); ++ptr) {
        int vid = ptr->var_index();
        if (variables[vid].value == UNKNOWN) {
          cubes[cube_idx].unit_lit = ptr->s_var();
          break;
        }
      }
      int unit_lit = cubes[cube_idx].unit_lit;
      if (unit_lit == -1)
        // -2 indicates the unit lit is an existential variable
        // and therefore no unit prop
        cubes[cube_idx].unit_lit = -2;
      else
        queue_implication(unit_lit^0x1, cube_idx, true);
    }
    else if (cube_count_0_n1[cube_idx] == 0) {
      DBG0(cout << "Cube " << cube_idx << " satisfied" << endl;);
      sat_cubes.push_back(cube_idx);
    }
  }

  vector<int> & neg_cube = variables[v].lit_cubes[value];
  for (vector<int>::iterator itr = neg_cube.begin(); itr != neg_cube.end(); ++itr) {
    int cube_idx = *itr;
    if (cube_count_0_n1[cube_idx] < (1<<16)) {
      -- stats.num_sat_cubes;
    }
    cube_count_0_n1[cube_idx] += (1<<16);
  }
}

void CSolver::unset_var_value(int vid) {
  static int entry = 0;
  ++ entry;
  DBG1(cout <<"Unset var " << (variables[vid].value?"+":"-") << vid << endl;);
  CVariable & var = variables[vid];
  assert (variables[vid].value != UNKNOWN);

  if (var.is_univ())
    unset_univ_var_value(vid);
  else 
    unset_exist_var_value(vid);

  var.value = UNKNOWN;
  var.antecedence = NULL_CLAUSE;
  var.dlevel = -1;
  var.asgn_stack_pos = -1;

  if (var.score_pos < max_score_pos)
    max_score_pos = var.score_pos;
    
  ++ stats.num_free_variables;
  ++ q_unassigned_counts[variables[vid].q_level];
}

void CSolver::unset_univ_var_value(int v) {
  int value = variables[v].value;
  vector<int> & pos_cls = variables[v].lit_clauses[!value];
  for (vector<int>::iterator itr = pos_cls.begin(); itr != pos_cls.end(); ++itr) {
    int cl_idx = *itr;
    cls_count_1_n0[cl_idx] -= (1<<16);
    if (cls_count_1_n0[cl_idx] < (1<<16)) 
      ++ stats.num_unsat_clauses;
  }
  vector<int> & neg_cls = variables[v].lit_clauses[value];
  for (vector<int>::iterator itr = neg_cls.begin(); itr != neg_cls.end(); ++itr)
    ++cls_count_1_n0[*itr];

  vector<int> & pos_cube = variables[v].lit_cubes[!value];
  for (vector<int>::iterator itr = pos_cube.begin(); itr < pos_cube.end(); ++itr)
    ++cube_count_0_n1[*itr];
  vector<int> & neg_cube = variables[v].lit_cubes[value];
  for (vector<int>::iterator itr = neg_cube.begin(); itr != neg_cube.end(); ++itr) {
    int cube_idx = *itr;
    cube_count_0_n1[cube_idx] -= (1<<16);
    if (cube_count_0_n1[cube_idx] < (1<<16)) {
      ++ stats.num_sat_cubes;
    }
  }
}

void CSolver::unset_exist_var_value(int v) {
  int value = variables[v].value;
  vector<int> & pos_cls = variables[v].lit_clauses[!value];
  for (vector<int>::iterator itr = pos_cls.begin(); itr < pos_cls.end(); ++itr) {
    int cl_idx = *itr;
    cls_count_1_n0[cl_idx] -= (1<<16);
    if (cls_count_1_n0[cl_idx] < (1<<16)) 
      ++ stats.num_unsat_clauses;
  }
  vector<int> & neg_cls = variables[v].lit_clauses[value];
  for (vector<int>::iterator itr = neg_cls.begin(); itr != neg_cls.end(); ++itr)
    ++cls_count_1_n0[*itr];

  vector<int> & pos_cube = variables[v].lit_cubes[!value];
  for (vector<int>::iterator itr = pos_cube.begin(); itr != pos_cube.end(); ++itr)
    ++cube_count_0_n1[*itr];
  vector<int> & neg_cube = variables[v].lit_cubes[value];
  for (vector<int>::iterator itr = neg_cube.begin(); itr != neg_cube.end(); ++itr) {
    int cube_idx = *itr;
    cube_count_0_n1[cube_idx] -= (1<<16);
    if (cube_count_0_n1[cube_idx] < (1<<16)) {
      ++ stats.num_sat_cubes;
    }
  }
}


void CSolver::make_decision(int s_var) {
  DBG0 (cout << endl << "**Decision " << stats.num_decisions << " at Level " << dlevel + 1;
        cout <<": " << s_var << "\ti.e. " << (s_var&0x1?"-":" ") ;
        cout <<(s_var>>1)  << endl; );
  queue_implication(s_var, NULL_CLAUSE, false);
  ++dlevel;
  if (dlevel > stats.max_dlevel) {
    assert (dlevel == stats.max_dlevel + 1);
    assignment_stack.resize(assignment_stack.size() + 1);
    unit_lit_stack.resize(assignment_stack.size());
    stats.max_dlevel = dlevel;
  }
}

int CSolver::branch_DLIS(void) {
  int s_var = 0;
  int best_score = -1;
  for (int k=0; k< q_groups.size(); ++k) {
    if (q_unassigned_counts[k] == 0) 
      continue;
    vector<int> & current = q_groups[k];
    for(int l=0, sz=current.size(); l< sz; ++l) {
      int vid = current[l];
      CVariable & v = variables[vid];
      if (v.value != UNKNOWN)
        continue;       
      int pos_score = 0;
      int neg_score = 0;
      vector<int> & pos_lit_cls = v.lit_clauses[0];
      for (vector<int>::iterator itr = pos_lit_cls.begin(); itr != pos_lit_cls.end(); ++itr) {
        int cl_id = *itr;
        if (is_sat_clause(cl_id)) continue;
        else ++pos_score;
      }
      vector<int> & neg_lit_cls = v.lit_clauses[0];
      for (vector<int>::iterator itr = neg_lit_cls.begin(); itr != neg_lit_cls.end(); ++itr) {
        int cl_id = *itr;
        if (is_sat_clause(cl_id)) continue;
        else ++neg_score;
      }
      if (pos_score + neg_score > best_score) {
        if ( s_var==0 || v.q_level <= variables[s_var>>1].q_level) {
          s_var = vid + vid + ((pos_score > neg_score)?0:1);
          best_score = pos_score + neg_score;
        }
      }
    }
    assert (s_var != 0);
  }
  return s_var;
}

int CSolver::branch_VSIDS_FAST(void) {
  //note: For VSIDS, the q_level have already been taken into account
  //by the score
  int s_var = 0;
  for (unsigned i=max_score_pos; i<ordered_vars.size(); ++i) {
    CVariable & var = *ordered_vars[i].first;
    if (var.value==UNKNOWN) {
      //move th max score position pointer
      max_score_pos = i;
      int sign = ((var.scores[0] > var.scores[1]) ? 0 : 1);
      int var_idx = ordered_vars[i].first - (&(*variables.begin()));
      s_var = var_idx + var_idx + sign;
      break;
    }
  }
  assert (s_var != 0);
  return s_var;
}

int CSolver::branch_VSIDS_SLOW(void) {
  //note: For VSIDS, the q_level have already been taken into account
  //by the score
  int s_var = 0;
  int best_score = 0;
  for(int i=1, sz=variables.size(); i< sz; ++i) {
    CVariable & v = variables[i];
    if (v.value != UNKNOWN)
      continue;   
    if (v.score() > best_score) {
      s_var = i + i + ((v.scores[0] > v.scores[1])?0:1);
      best_score = v.score();
    }
  }
  assert (s_var != 0);
  return s_var;
}

void CSolver::decide_next_branch(void) {
  int s_var = branch_VSIDS_FAST();

  assert (s_var >= 2); //there must be a free var somewhere
  assert (variables[s_var>>1].q_level == 0 ||
          q_unassigned_counts[variables[s_var>>1].q_level-1] == 0);
  DBG1(dump_assignment_stack(););

  variables[s_var>>1].is_flipped = false;
  ++ stats.num_decisions;     
  make_decision(s_var);
  return;
}


bool CSolver::is_exist_dlevel(int dl) {
  int vid = (assignment_stack[dl][0] >> 1);
  if (variables[vid].is_exist()) {
    return true;
  }
  assert (variables[vid].is_univ());
  return false;
}

bool CSolver::is_conflicting_clause(int cl_idx) {
  CLitPoolElement * ptr;
  for (ptr = clauses[cl_idx].exist_lits + 1; ptr->is_literal(); ++ptr)
    if (literal_value(ptr) != 0 )
      return false;
  for (ptr = clauses[cl_idx].univ_lits + 1; ptr->is_literal(); ++ptr)
    // if (literal_value(ptr) == 1)
    if (literal_value(ptr) != 0)
      return false;
  return true;
}

bool CSolver::is_sat_cube(int cube_idx) {
  CLitPoolElement * ptr;
  for (ptr = cubes[cube_idx].univ_lits + 1; ptr->is_literal(); ++ptr)
    if (literal_value(ptr) != 1)
      return false;
  for (ptr = cubes[cube_idx].exist_lits + 1; ptr->is_literal(); ++ptr)
    // if (literal_value(ptr) == 0 )
    if (literal_value(ptr) != 1 )
      return false;
  return true;
}

void CSolver::run_periodic_functions(void) {
  if (stats.num_backtracks == 0) return;
  if ((stats.num_decisions % params.decay_period)==0) 
    decay_variable_score();
  if ((stats.num_decisions % params.deletion_period)==0) 
    delete_unrelevant_clauses_and_cubes();
}

void CSolver::delete_unrelevant_clauses_and_cubes(void) {
  int orig_cls = stats.num_deleted_clauses;
  int orig_cbs = stats.num_deleted_cubes;
  vector<bool> is_ante;
  is_ante.resize(clauses.size());
  for (int i=0; i< is_ante.size(); ++i)
    is_ante[i] = false;
  for (int i=1; i< variables.size(); ++i) {
    if (variables[i].is_exist() && 
        variables[i].antecedence != NULL_CLAUSE) 
      is_ante[variables[i].antecedence] = true;
  }
  for (int i= stats.num_original_clauses; i< clauses.size(); ++i) {
    CClause & cl = clauses[i];
    if (cl.status == CONFLICT_CL && is_ante[i] == false) {
      if (cl.num_exist_lits + cl.num_univ_lits < params.cl_min_del_length)
        continue;
      int num_1 = (cls_count_1_n0[i] >> 16);
      int num_n0 = (cls_count_1_n0[i]&0xffff);
      int unrelevance = num_1 + num_n0; //weird relevance, but does it matter?
      int total_lits = cl.num_exist_lits + cl.num_univ_lits;
      if ( total_lits > params.cl_max_length || unrelevance > params.cl_max_unrelevance ) {
        clauses[i].status = DELETED_CL;
        delete [] clauses[i].univ_lits;
        delete [] clauses[i].exist_lits;
        clauses[i].univ_lits = NULL;
        clauses[i].exist_lits = NULL;
        unused_clause_idx.push_back(i);
        ++ stats.num_deleted_clauses;
      }
    }
  }
  is_ante.resize(cubes.size());
  for (int i=0; i< is_ante.size(); ++i)
    is_ante[i] = false;
  for (int i=1; i< variables.size(); ++i) {
    if (variables[i].is_univ() && 
        variables[i].antecedence != NULL_CUBE) 
      is_ante[variables[i].antecedence] = true;
  }
  for (int i=0; i< cubes.size(); ++i) {
    CCube & cb = cubes[i];
    if (cb.status == SAT_CUBE && is_ante[i] == false) {
      if (cb.num_exist_lits + cb.num_univ_lits < params.cube_min_del_length)
        continue;
      int num_0 = (cube_count_0_n1[i] >> 16);
      int num_n1 = (cube_count_0_n1[i]&0xffff);
      int unrelevance = num_0 + num_n1; //weird relevance, but does it matter?
      int total_lits = cb.num_exist_lits + cb.num_univ_lits;
      if ( total_lits > params.cube_max_length || unrelevance > params.cube_max_unrelevance ) {
        cubes[i].status = DELETED_CUBE;
        delete [] cubes[i].univ_lits;
        delete [] cubes[i].exist_lits;
        cubes[i].univ_lits = NULL;
        cubes[i].exist_lits = NULL;
        unused_cube_idx.push_back(i);
        ++ stats.num_deleted_cubes;
      }
    }
  }
  for (int i=1; i< variables.size(); ++i) {
    for (int j=0; j< 2; ++j) {
      vector<int> & lit_cls = variables[i].lit_clauses[j];
      for (vector<int>::iterator itr = lit_cls.begin(); itr < lit_cls.end(); ++itr) {
        if (clauses[*itr].status == DELETED_CL) {
          *itr = lit_cls.back();
          lit_cls.pop_back();
          -- itr;
        }
      }
      vector<int> & lit_cbs = variables[i].lit_cubes[j];
      for (vector<int>::iterator itr = lit_cbs.begin(); itr < lit_cbs.end(); ++itr) {
        if (cubes[*itr].status == DELETED_CUBE) {
          *itr = lit_cbs.back();
          lit_cbs.pop_back();
          -- itr;
        }
      }
    }
  }
  //      cout << "Deleted " << stats.num_deleted_clauses - orig_cls << " Clauses and "
  //       << stats.num_deleted_cubes - orig_cbs << " Cubes, has " 
  //       << clauses.size() - unused_clause_idx.size()<< " Clauses and "
  //       << cubes.size() - unused_cube_idx.size() << " Cubes " << endl;
}
        
    

DEDUCTION_RESULT CSolver::preprocess(void) {
  for (unsigned i=0; i< clauses.size(); ++i) {
    CClause & cl = clauses[i];
    if (cl.num_exist_lits == 1 && cl.num_univ_lits == 0) {
      cl.unit_lit = (cl.exist_lits + 1)->s_var();
      unit_lit_stack[dlevel].push_back(i);
      queue_implication(cl.unit_lit, i, false);
    }
    else if (cl.num_exist_lits == 0 && cl.num_univ_lits == 1) {
      cl.unit_lit = -2;
      unit_lit_stack[dlevel].push_back(i);
    }
  }

  for (unsigned i=0; i< cubes.size(); ++i) {
    CCube & cube = cubes[i];
    if (cube.num_univ_lits == 1 && cube.num_exist_lits == 0) {
      cube.unit_lit = (cube.univ_lits + 1)->s_var();
      unit_lit_stack[dlevel].push_back(i + CUBE_MASK);
      queue_implication(cube.unit_lit^0x1, i, true);
    }
    else if (cube.num_univ_lits == 0 && cube.num_exist_lits == 1) {
      cube.unit_lit = -2;
      unit_lit_stack[dlevel].push_back(i + CUBE_MASK);
    }
  }

  deduce();
  // If comb_op is OR, give priority to is_satisfied check,
  // otherwise give priority to is_conflicting check
  if (comb_op == OR) {
    if (is_satisfied())
      return SATISFIED;
    else if (is_conflicting())
      return CONFLICT;
  }
  else {   // comb_op == AND
    if (is_conflicting())
      return CONFLICT;
    else if (is_satisfied())
      return SATISFIED;
  }
  return NO_DEDUCTION;
}

// Ashish: the following method never used!
// int CSolver::first_level_invert_quantifier(void)
// {
//     if (q_groups.size() < 2) 
//      return NO_CONFLICT;
//     assert (q_groups.size() >= 3);
//     assert (dlevel == 0);
//     if (q_groups[1].size() <= 14) { // 2^14 == 8192
//      cout << "Inverting found " ;
//      vector<int> & univ_vars = q_groups[1];
//      int num_try = (1 << univ_vars.size());
//      for (int i=0; i< num_try ; ++i) {
//          assert (implication_queue.empty());
//          int svar = univ_vars[0] + univ_vars[0] + (i & 0x1); 
//          make_decision(svar);
//          assert (dlevel == 1);
//          for (int j=1; j< univ_vars.size(); ++j) {
//              int svar = univ_vars[j] + univ_vars[j] + ((i>>j)&0x1); 
//              queue_implication(svar, NULL_CLAUSE, false);
//          }   
//          deduce();
//          if (is_conflicting()) {
//              back_track(1);
//              return CONFLICT;
//          }
//          vector<int> & assigned = assignment_stack[1];
//          vector<int> forced;
//          for (int j=0; j< assigned.size(); ++j) {
//              int vid = (assigned[j]>>1);
//              if (variables[vid].q_level == 0)
//                  forced.push_back(assigned[j]);
//          }
//          back_track(1);
//          for (int j=0; j< forced.size(); ++j) {
//              cout << ((forced[j]&0x1)?" -":" +") << (forced[j] >> 1);
//              queue_implication(forced[j], NULL_CLAUSE, false);
//          }
//          deduce();
//          assert(!is_conflicting());
//      }
//      cout << endl;
//     }
//     else { //random probe with 5000 vectors
//     }
//     return NO_CONFLICT;
// }

//----------------------------------------------------------------------------------
int CSolver::analyze_satisfiability(void) {
  ++ stats.num_sat_leaves;
#ifdef SMART_SAT_ANALYSIS
  return analyze_satisfiability_smart();
#else
  return analyze_satisfiability_naive();
#endif
}

int CSolver::last_unflipped_univ_dl(void) {
  int i;
  for (i=dlevel; i > 0; --i) {
    int vid = (assignment_stack[i][0] >> 1);
    if (variables[vid].is_univ() && (variables[vid].is_flipped == false))
      break;
  }
  return i;
}

int CSolver::analyze_satisfiability_naive(void) {
  DBG1(cout << endl << "Found SAT, backtrack to the last unflipped universal lits " << endl;);
  int dl = last_unflipped_univ_dl();
  if (dl < 1) {
    stats.outcome = SATISFIABLE;
    return -1;
  }
  if (!sat_cubes.empty()) {
    DBG0(cout << "Clearing sat_cubes" << endl;);
    sat_cubes.clear();   // Ashish: should do this, right? (wasn't there originally)
  }
  flip_decision_at_dl(dl);
  return dlevel;
}


void CSolver::set_up_consensus(int cube_idx) {
  assert (sat_univ.empty());
  assert (sat_lits_exist.empty());
  CCube & cube = cubes[cube_idx];
  for (int i=1; i<= cube.num_univ_lits; ++i) {
    assert (literal_value(cube.univ_lits + i) == 1);
    int vid = cube.univ_lits[i].var_index();
    sat_univ.insert(&variables[0] + vid);
  }   
  for (int i=1; i<= cube.num_exist_lits; ++i) {
    int svar = cube.exist_lits[i].s_var();
    int q_level = variables[svar>>1].q_level;
    sat_lits_exist.insert((svar<< QLEVEL_SHIFT) + q_level);
  }
}

void CSolver::consensus_one_lit(void) {
  assert (sat_univ.size() >= 1);
  CVariable * v_ptr = *sat_univ.begin();
  sat_univ.erase (sat_univ.begin());
  int var_index = v_ptr - &variables[0];
  int dl = v_ptr->dlevel;
  int cube_idx = v_ptr->antecedence;
  assert (cube_idx != NULL_CUBE);
  CCube & cube = cubes[cube_idx];
  for (int i=1; i<= cube.num_univ_lits; ++i) {
    int vid = cube.univ_lits[i].var_index();
    if (vid == var_index) continue;
    assert (literal_value(cube.univ_lits + i) == 1);
    sat_univ.insert(&variables[0] + vid);
  }   
  for (int i=1; i<= cube.num_exist_lits; ++i) {
    int svar = cube.exist_lits[i].s_var();
    int q_level = variables[svar>>1].q_level;
    sat_lits_exist.insert((svar<< QLEVEL_SHIFT) + q_level);
  }
}

bool CSolver::is_asserting_cube(void) {
  AsgnVarSet::iterator itr1 = sat_univ.begin();
  CVariable * v1 = * itr1;
  assert (sat_univ.size()>=1);
  if (sat_univ.size() > 1) {
    ++itr1;
    CVariable * v2 = * itr1;
    assert (v1->dlevel >= v2->dlevel);
    if (v1->dlevel == v2->dlevel)
      return false;
  }
  for (QLevelLitsSet::iterator itr = sat_lits_exist.begin();
       itr != sat_lits_exist.end(); ++itr) {
    int lit = ((*itr) >> QLEVEL_SHIFT);
    int vid = (lit >>1);
    if (variables[vid].value == UNKNOWN ||
        variables[vid].dlevel >= v1->dlevel) {
      if (variables[vid].q_level < v1->q_level)
        return false;
    }
    else if (svar_value(lit) == 0)
      return false;
  }
  return true;
}

int CSolver::add_sat_cube(void) {
  int cube_idx = get_free_cube_idx();
  CCube & cube = cubes[cube_idx];
  int max_qlevel = 0;
  for (AsgnVarSet::iterator itr = sat_univ.begin(); 
       itr != sat_univ.end(); ++itr) {
    int vid = (*itr) - (& (*variables.begin()));
    if (variables[vid].q_level > max_qlevel)
      max_qlevel = variables[vid].q_level;
  }
  while (!sat_lits_exist.empty()) {
    QLevelLitsSet::iterator itr1 = sat_lits_exist.end();
    -- itr1;
    int lit = ((int)(*itr1) >> QLEVEL_SHIFT);
    int vid = (lit>>1);
    assert (variables[vid].q_level == ((int)(*itr1) & QLEVEL_MASK));
    if (variables[vid].q_level > max_qlevel)
      sat_lits_exist.erase(itr1);
    else 
      break;
  }

  cube.init( sat_lits_exist.size(), sat_univ.size(), cube_idx);

  int i = 1;
  for (AsgnVarSet::iterator itr = sat_univ.begin(); 
       itr != sat_univ.end(); ++itr) {
    int vid = (*itr) - (&(*variables.begin()));
    int lit = vid + vid + 1 - (*itr)->value;
    assert (svar_value(lit)==1);
    int sign = (lit&0x1);
    cube.univ_lits[i].set(lit);
    variables[vid].lit_cubes[sign].push_back(cube_idx);
    ++i;
  }
  i = 1;
  int num_0 = 0, num_1 = 0;
  for (QLevelLitsSet::iterator itr1 = sat_lits_exist.begin(); 
       itr1 != sat_lits_exist.end(); ++itr1) {
    int lit = ((int)(*itr1) >> QLEVEL_SHIFT);
    int vid = (lit>>1);
    int sign = (lit&0x1);
    if (svar_value(lit) == 0)
      ++ num_0;
    else if (svar_value(lit) == 1)
      ++ num_1;
    cube.exist_lits[i].set(lit);
    variables[vid].lit_cubes[sign].push_back(cube_idx);
    assert (i==1 || variables[vid].q_level >= variables[cube.exist_lits[i-1].var_index()].q_level);
    ++i;
  }
  cubes[cube_idx].status = SAT_CUBE;
  cube_count_0_n1[cube_idx] = (num_0 << 16) + (sat_lits_exist.size() - num_1) ;  //all universal lits must have value 1
  if (num_0 == 0) {
    ++ stats.num_sat_cubes;
  }
  ++ stats.num_added_cubes;   // Ashish
  return cube_idx;
}

void CSolver::gen_sat_induced_cube(void) {
  DBG1(cout << "Begin Generating Cube from Clauses : " << endl;);
  DBG1(dump());
  assert (stats.num_free_variables == 0 || stats.num_unsat_clauses==0);
  vector<int> count_1;
  count_1.resize(stats.num_original_clauses);
  vector<int> free_vars;
  for (int i=0; i< stats.num_original_clauses; ++i) {
    count_1[i] = (cls_count_1_n0[i] >> 16);
    assert(count_1[i] > 0);
  }
  //1. All essential variables must be included
  for (int i=0; i< count_1.size(); ++i) {
    if (count_1[i] == 1) { //essential
      CClause & cl = clauses[i];
      CLitPoolElement * ptr;
      for ( ptr = cl.exist_lits + 1; ptr->is_literal(); ++ptr)
        if (literal_value(ptr)==1) 
          break;
      if (!ptr->is_literal()) {
        for ( ptr = cl.univ_lits + 1; ptr->is_literal(); ++ptr) {
          if (literal_value(ptr)==1) 
            break;
        }
      }
      assert (ptr->is_literal() && literal_value(ptr) == 1);
      int vid = ptr->var_index();
      CVariable & var = variables[vid];   
      var.is_in_cube = true; //must be in the cube, because its essential
      vector<int> & pos_cls = var.lit_clauses[1-var.value];
      for (vector<int>::iterator itr = pos_cls.begin(); itr < pos_cls.end(); ++itr) {
        int cl_id = *itr;
        if (cl_id < stats.num_original_clauses) 
          count_1[cl_id] = 0; //clear that count, meaning: the clause is covered
        else break;
      }
    }
  }
  //2. Prioritize the variables
  vector<CVariable *> priority_var;
  for (int i=1; i< variables.size(); ++i) {
    if (variables[i].is_in_cube == false && variables[i].value != UNKNOWN)
      priority_var.push_back(& variables[i]);
  }
  ::stable_sort (priority_var.begin(), priority_var.end(), comp_var_priority);
  //3. Now cover the cubes according to proprity of the vars
  for (int i=0; i< priority_var.size(); ++i) {
    CVariable & var = * priority_var[i];
    if (var.is_in_cube == true)
      continue;
    vector<int> & pos_cls = var.lit_clauses[1-var.value];
    for (vector<int>::iterator itr = pos_cls.begin(); itr < pos_cls.end(); ++itr) {
      int cl_id = *itr;
      if (cl_id >= stats.num_original_clauses) //original clauses always occupy the earlier part of the array
        break;
      if (count_1[cl_id] > 0) {
        count_1[cl_id] = 0;
        var.is_in_cube = true;
      }           
    }
  }
  //4. verify all clauses got covered
  for (int i=0; i< count_1.size(); ++i) 
    assert (count_1[i] == 0);
  //6. get the cube
  for (int i=1; i< variables.size(); ++i) {
    CVariable & var = variables[i];
    if (var.is_in_cube == true) {
      var.is_in_cube = false;
      if (var.is_univ()) 
        sat_univ.insert(&var);
      else {
        assert (var.value != UNKNOWN);
        int svar = i + i + 1 - var.value; 
        int q_level = var.q_level;
        sat_lits_exist.insert((svar<< QLEVEL_SHIFT) + q_level);
      }
    }
  }
}

int CSolver::analyze_satisfiability_smart(void) {
  static int entry = 0;
  ++ entry;
  assert (implication_queue.empty());
  if (dlevel == 0){ //already at level 0. SAT means problem SAT
    sat_cubes.clear();
    back_track(0);
    return -1;
  }
  if (sat_cubes.empty()) {
    gen_sat_induced_cube();
    if (sat_univ.size() == 0) {
      sat_univ.clear();
      sat_lits_exist.clear();
      back_track(0);
      return -1;
    }
  }
  else 
    set_up_consensus(sat_cubes[0]);
  while (1) {
    if (is_asserting_cube()) {
      int dl = (*sat_univ.begin())->dlevel;
      int vid = (assignment_stack[dl][0] >> 1);
      if (variables[vid].is_univ()) {
        //the decision level should not be a pure literal dlevel
        assert (variables[vid].is_flipped == false); 
        break;
      }
    }
    consensus_one_lit();
    CVariable * v_ptr = *sat_univ.begin();
    if (sat_univ.empty() || v_ptr->dlevel == 0) { 
      //the highest lit is already at 0, so nothing we can do
      sat_cubes.clear();
      sat_univ.clear();
      sat_lits_exist.clear();
      back_track(0);
      return -1;
    }
  }
  int added_cube = add_sat_cube();
  if (added_cube < 0 ) { //memory out.
    stats.is_mem_out = true;
    sat_cubes.clear();
    sat_univ.clear();
    sat_lits_exist.clear();
    return -1; 
  }
  cube_adjust_variable_order(added_cube);
  DBG0( cout << "**Add Cube " <<added_cube << endl;
        detail_dump_cube(added_cube);
        cout << endl; );
  //find out the backtrack level
  //it should be the max of all the exitential variables
  //and all the universal variables that has dlevel < dl 
  int back_dl = 0;
  CVariable * unit_ptr = (*sat_univ.begin());
  if (sat_univ.size() > 1) {
    AsgnVarSet::iterator itr1 = sat_univ.begin();
    ++itr1;
    back_dl = (*itr1)->dlevel;
  }
  int unit_dl = back_dl;
  for (CLitPoolElement * ptr = cubes[added_cube].exist_lits + 1; 
       ptr->is_literal(); ++ptr) {
    int lit = ptr->s_var();
    int vid = (lit>>1);
    CVariable & v = variables[vid];
    if (v.q_level < unit_ptr->q_level) {
      assert (svar_value(lit) == 1);
      if (v.dlevel > back_dl)
        back_dl = v.dlevel;
    }
  }


  assert (back_dl < unit_ptr->dlevel);
  int unit_vid = unit_ptr - &variables[0];
  int unit_lit = unit_vid + unit_vid + 1 - unit_ptr->value;
  cubes[added_cube].unit_lit = unit_lit;
  unit_lit_stack[unit_dl].push_back(added_cube + CUBE_MASK);
  back_track(back_dl + 1);


  //    assert (cube_count_0_univ_n1[added_cube]==1);
  CVariable & unit_var = variables[unit_vid];
  bool imply = true;
  for (CLitPoolElement * ptr = cubes[added_cube].exist_lits + 1; ptr->is_literal(); ++ptr) {
    if (variables[ptr->var_index()].q_level > unit_var.q_level)
      break;
    else if (variables[ptr->var_index()].value == UNKNOWN) {
      imply = false;
      break;
    }
  }
  assert (imply);

  DBG0(cout << "Satisfiability Analasis: SAT at level: " << back_dl + 1;
       cout << "  Assertion Cube is " << added_cube<< endl; );

  queue_implication(unit_lit^0x1, added_cube, true);
            
  for (unsigned i=1; i< sat_cubes.size(); ++i) //after resolve the first SAT cube, others must also be resolved
    assert(!is_sat_cube(sat_cubes[i]));
  sat_cubes.clear();
  sat_univ.clear();
  sat_lits_exist.clear();
  return back_dl;
}

//----------------------------------------------------------------------------------
int CSolver::last_unflipped_exist_dl(void) {
  int i;
  for (i=dlevel; i > 0; --i) {
    int vid = (assignment_stack[i][0] >> 1);
    if (variables[vid].is_exist() && (variables[vid].is_flipped == false))
      break;
  }
  return i;
}

int CSolver::add_conf_clause(void) {
  int cl_idx = get_free_clause_idx();
  CClause & cl = clauses[cl_idx];
  int max_qlevel = 0;
  for (AsgnVarSet::iterator itr = conf_exist.begin(); 
       itr != conf_exist.end(); ++itr) {
    int vid = (*itr) - (&(*variables.begin()));
    if (variables[vid].q_level > max_qlevel)
      max_qlevel = variables[vid].q_level;
  }
  while (!conf_lits_univ.empty()) {
    QLevelLitsSet::iterator itr1 = conf_lits_univ.end();
    -- itr1;
    int lit = ((int)(*itr1) >> QLEVEL_SHIFT);
    int vid = (lit>>1);
    assert (variables[vid].q_level == ((int)(*itr1) & QLEVEL_MASK));
    if (variables[vid].q_level > max_qlevel)
      conf_lits_univ.erase(itr1);
    else 
      break;
  }
    
  cl.init(conf_exist.size(), conf_lits_univ.size(), cl_idx);
  stats.num_added_univ_literals += conf_lits_univ.size();
  stats.num_added_exist_literals += conf_exist.size();
    
  int i = 1;
  for (AsgnVarSet::iterator itr = conf_exist.begin(); 
       itr != conf_exist.end(); ++itr) {
    int vid = (*itr) - (&(*variables.begin()));
    int lit = vid + vid + (*itr)->value;
    assert (svar_value(lit)==0);
    int sign = (lit&0x1);
    assert (svar_value(lit) == 0);
    cl.exist_lits[i].set(lit);
    variables[vid].lit_clauses[sign].push_back(cl_idx);
    ++i;
  }
  i = 1;
  int num_1 = 0, num_0 = 0;
  for (QLevelLitsSet::iterator itr1 = conf_lits_univ.begin(); 
       itr1 != conf_lits_univ.end(); ++itr1) {
    int lit = ((int)(*itr1) >> QLEVEL_SHIFT);
    int vid = (lit>>1);
    int sign = (lit&0x1);
    if (svar_value(lit) == 1)
      ++ num_1;
    else if (svar_value(lit) == 0)
      ++ num_0;
    cl.univ_lits[i].set(lit);
    variables[vid].lit_clauses[sign].push_back(cl_idx);
    assert (i==1 || variables[vid].q_level >= variables[cl.univ_lits[i-1].var_index()].q_level);
    ++i;
  }
  clauses[cl_idx].status = CONFLICT_CL;
  cls_count_1_n0[cl_idx] = (num_1 << 16) + (conf_lits_univ.size() - num_0) ;  //all existential lits must have value 0
  if (num_1 == 0) {
    ++ stats.num_unsat_clauses;
  }
  //    assert (is_conflicting_clause(cl_idx));
  ++ stats.num_added_clauses;
  return cl_idx;
}

int CSolver::analyze_conflicts(void) {
#ifdef SMART_CONFLICT_ANALYSIS
  int l = analyze_conflicts_smart();
#else
  int l = analyze_conflicts_naive();
#endif
  ++ stats.num_conflicts;
  DBG1( dump_assignment_stack(););
  return l;
}

int CSolver::analyze_conflicts_naive(void) {
  int dl = last_unflipped_exist_dl();
  if (dl < 1) {
    return -1;
  }
  conflicts.clear();
  flip_decision_at_dl(dl);
  return dl;
}

void CSolver::dump_conf_cls(void) {
  AsgnVarSet exist = conf_exist;
  cout << "Exist: ";
  for (AsgnVarSet::iterator itr = exist.begin(); 
       itr != exist.end(); ++itr) {
    int vid = (*itr) - (&(*variables.begin()));
    int lit = vid + vid + (*itr)->value;
    assert (svar_value(lit)==0);
    int sign = (lit&0x1);
    cout << (sign?" -":"+") << vid;
  }
  cout << "    Univ: ";
  QLevelLitsSet univ = conf_lits_univ;
  for (QLevelLitsSet::iterator itr1 = univ.begin(); 
       itr1 != univ.end(); ++itr1) {
    int lit = ((int)(*itr1) >> QLEVEL_SHIFT);
    int vid = (lit>>1);
    int sign = (lit&0x1);
    cout << (sign?" -":" +") << vid;
  }
  cout << endl;
}

void CSolver::set_up_resolve(int cl_idx) {
  assert (conf_exist.empty());
  assert (conf_lits_univ.empty());
  CClause & cl = clauses[cl_idx];
  for (int i=1; i<= cl.num_exist_lits; ++i) {
    assert (literal_value (cl.exist_lits + i) == 0);
    int vid = cl.exist_lits[i].var_index();
    conf_exist.insert(&variables[0] + vid);
  }   
  for (int i=1; i<= cl.num_univ_lits; ++i) {
    int svar = cl.univ_lits[i].s_var();
    int q_level = variables[svar>>1].q_level;
    conf_lits_univ.insert((svar<< QLEVEL_SHIFT) + q_level);
  }
}

void CSolver::resolve_one_lit(void) {
  static int entry = 0;
  ++ entry;

  assert (!conf_exist.empty());
  CVariable * v_ptr = *conf_exist.begin();
  conf_exist.erase (conf_exist.begin());
  int var_index = v_ptr - &variables[0];
  int dl = v_ptr->dlevel;
  int cl_idx = v_ptr->antecedence;
  assert (cl_idx != NULL_CLAUSE);
  CClause & cl = clauses[cl_idx];
  for (int i=1; i<= cl.num_exist_lits; ++i) {
    int vid = cl.exist_lits[i].var_index();
    if (vid == var_index) continue;
    assert (literal_value(cl.exist_lits + i) == 0);
    conf_exist.insert(&variables[0] + vid);
  }
  for (int i=1; i<= cl.num_univ_lits; ++i) {
    int svar = cl.univ_lits[i].s_var();
    //      if (variables[svar].value != UNKNOWN && 
    //          variables[svar].dlevel <= dl) {
    //          assert (svar_value(svar) == 0);
    int q_level = variables[svar>>1].q_level;
    conf_lits_univ.insert((svar<< QLEVEL_SHIFT) + q_level);
    //      }
  }
}

bool CSolver::is_asserting_clause (void) {
  AsgnVarSet::iterator itr1 = conf_exist.begin();
  CVariable * v1 = * itr1;
  assert (conf_exist.size()>=1);
  if (conf_exist.size() > 1) {
    ++itr1;
    CVariable * v2 = * itr1;
    assert (v1->dlevel >= v2->dlevel);
    if (v1->dlevel == v2->dlevel)
      return false;
  }
  for (QLevelLitsSet::iterator itr = conf_lits_univ.begin();
       itr != conf_lits_univ.end(); ++itr) {
    int lit = ((*itr) >> QLEVEL_SHIFT);
    int vid = (lit >>1);
    if (variables[vid].value == UNKNOWN ||
        variables[vid].dlevel >= v1->dlevel) {
      if (variables[vid].q_level < v1->q_level)
        return false;
    }
    else if (svar_value(lit) == 1)
      return false;
  }
  return true;
}

int CSolver::analyze_conflicts_smart(void) {
  assert (conflicts.size() > 0);
  assert (implication_queue.empty());
  if (dlevel == 0){ //already at level 0. Conflict means unsat.
    conflicts.clear();
    back_track(0);
    return -1;
  }
  int cl_idx = conflicts[0];
  DBG0(cout <<endl << "Conflict clause: " << cl_idx << endl;
       detail_dump_clause(cl_idx); 
       cout << endl;);
  set_up_resolve(cl_idx);
  assert (is_asserting_clause()==false);
  while (1) {
    resolve_one_lit();
    CVariable * v_ptr = *conf_exist.begin();
    if (conf_exist.size()==0 || v_ptr->dlevel == 0) { 
      //the highest lit is already at 0, so nothing we can do
      conflicts.clear();
      conf_exist.clear();
      conf_lits_univ.clear();
      back_track(0);
      return -1;
    }
    if (is_asserting_clause()) {
      int dl = (*conf_exist.begin())->dlevel;
      int vid = (assignment_stack[dl][0] >> 1);
      if (variables[vid].is_exist()) {
        //the decision level should not be a pure literal dlevel
        assert (variables[vid].is_flipped == false); 
        break;
      }
    }
  }
  int added_cl = add_conf_clause();
  if (added_cl < 0 ) { //memory out.
    stats.is_mem_out = true;
    conflicts.clear();
    conf_exist.clear();
    conf_lits_univ.clear();
    assert (implication_queue.empty());
    return -1; 
  }
  cls_adjust_variable_order(added_cl);
  DBG0( cout << "**Add Clause " <<added_cl << endl;
        detail_dump_clause(added_cl);
        cout << endl; );
  //find out the backtrack level
  //it should be the max of all the exitential variables
  //and all the universal variables that has dlevel < dl 

  int back_dl = 0;
  CVariable * unit_ptr = (*conf_exist.begin());
  if (conf_exist.size() > 1) {
    AsgnVarSet::iterator itr1 = conf_exist.begin();
    ++itr1;
    back_dl = (*itr1)->dlevel;
  }
  int unit_dl = back_dl;
  for (CLitPoolElement * ptr = clauses[added_cl].univ_lits + 1; 
       ptr->is_literal(); ++ptr) {
    int lit = ptr->s_var();
    int vid = (lit>>1);
    CVariable & v = variables[vid];
    if (v.q_level < unit_ptr->q_level) {
      assert (svar_value(lit) == 0);
      if (v.dlevel > back_dl)
        back_dl = v.dlevel;
    }
  }

  assert (back_dl < unit_ptr->dlevel);
  int unit_vid = unit_ptr - &variables[0];
  int unit_lit = unit_vid + unit_vid + unit_ptr->value;
  clauses[added_cl].unit_lit = unit_lit;
  unit_lit_stack[unit_dl].push_back(added_cl);
  back_track(back_dl + 1);

  //    assert (cls_count_1_exist_n0[added_cl]==1);
  CVariable & unit_var = variables[unit_vid];
  bool imply = true;
  for (CLitPoolElement * ptr = clauses[added_cl].univ_lits + 1; ptr->is_literal(); ++ptr) {
    if (variables[ptr->var_index()].q_level > unit_var.q_level)
      break;
    else if (variables[ptr->var_index()].value == UNKNOWN) {
      imply = false;
      break;
    }
  }
  assert(imply);
    
  DBG0(cout << "Conflict Analasis: conflict at level: " << back_dl + 1;
       cout << "  Assertion Clause is " << added_cl<< endl; );

  queue_implication(unit_lit, added_cl, false);
            
  for (unsigned i=1; i< conflicts.size(); ++i) //after resolve the first conflict, others must also be resolved
    assert(!is_conflicting_clause(conflicts[i]));
  conflicts.clear();
  conf_exist.clear();
  conf_lits_univ.clear();
  return back_dl;
}
//----------------------------------------------------------------------------------

// Ashish
void CSolver::set_combining_op(char * opname) {
  if (!strcmp(opname, "and"))
    comb_op = AND;
  else
    comb_op = OR;   // the default
}



void CSolver::read_cnfdnf(char * filename) {
#ifdef INPUT_FORMAT_RINTANEN
  // Ashish: Rintanen format not supported yet
  cerr << "ERROR: Rintanen format not supported yet." << endl;
  exit(1);
  read_rintanen(filename);
#else
  read_qdimacs(filename);
#endif
}

void CSolver::read_rintanen(char * filename ) {
  char line_buffer[MAX_LINE_LENGTH];
  char word_buffer[MAX_WORD_LENGTH];
  set<int> clause_vars;
  set<int> clause_lits;
  int var_num;
  int cl_num;
  unsigned qlevel = 0;
  int quantified_vars = 0;
  int added_cls = 0;
  ifstream inp (filename, ios::in);
  if (!inp) {
    cerr << "Can't open input file" << endl;
    exit(1);
  }
  while (inp.getline(line_buffer, MAX_LINE_LENGTH)) {
    if (inp.fail()) {
      cerr << "Too large an input line. Unable to continue..." << endl;
      exit(2);
    }

    if (line_buffer[0] == 'c') { continue; }
    else if (line_buffer[0] == 'p') {
      int arg = sscanf (line_buffer, "p cnf %d %d", &var_num, &cl_num);
      if( arg < 2 ) {
        cerr << "Unable to read number of variables and clauses" << endl;
        exit(3);
      }
      set_variable_number(var_num); //first element not used.
    }
    else if (line_buffer[0] == 'a' || line_buffer[0] == 'e') {
      cerr << "Error: Are you sure input is in Rintanen's format instead of Q-DIMACS format? " << endl;
      exit(1);
    }
    else if (line_buffer[0] == 'q') { 
      if (q_groups.size() <= qlevel)
        q_groups.resize (qlevel + 1);
      char *lp = line_buffer + 1; 
      do {
        char *wp = word_buffer;
        while (*lp && ((*lp == ' ') || (*lp == '\t'))) {
          lp++;
        }
        while (*lp && (*lp != ' ') && (*lp != '\t') && (*lp != '\n')) {
          *(wp++) = *(lp++);
        }
        *wp = '\0';                                 // terminate string
        if (strlen(word_buffer) != 0) {     // check if number is there
          int var_idx = atoi (word_buffer);
          assert (var_idx >= 0);
          if( var_idx > 0) {
            assert (variables[var_idx].q_level == -1);
            variables[var_idx].q_level = qlevel;
            if ((qlevel & 0x1) == 0) 
              variables[var_idx].set_exist();
            else 
              variables[var_idx].set_univ();
            ++ quantified_vars;
            q_groups[qlevel].push_back(var_idx);
          }   
        }
      } while(*lp);
      ++ qlevel;
      assert (qlevel < (QLEVEL_MASK/2));
    }
    else {                             // Clause definition or continuation
      char *lp = line_buffer;
      do {
        char *wp = word_buffer;
        while (*lp && ((*lp == ' ') || (*lp == '\t'))) {
          lp++;
        }
        while (*lp && (*lp != ' ') && (*lp != '\t') && (*lp != '\n')) {
          *(wp++) = *(lp++);
        }
        *wp = '\0';                                 // terminate string

        if (strlen(word_buffer) != 0) {     // check if number is there
          int var_idx = atoi (word_buffer);
          int sign = 0;

          if( var_idx != 0) {
            if( var_idx < 0)  { var_idx = -var_idx; sign = 1; }
            clause_vars.insert(var_idx);
            clause_lits.insert( (var_idx << 1) + sign);
          }   
          else {
            //add this clause
            ++ added_cls;
            if (clause_vars.size() != 0 && (clause_vars.size() == clause_lits.size())) { //yeah, can add this clause
              vector <int> temp;
              for (set<int>::iterator itr = clause_lits.begin();
                   itr != clause_lits.end(); ++itr)
                temp.push_back (*itr);
              add_orig_clause(temp);
            }
            else {} //it contain var of both polarity, so is automatically satisfied, just skip it
            clause_lits.clear();
            clause_vars.clear();
          }
        }
      } while (*lp);
    }
  }
  assert (cl_num == added_cls);
  assert (quantified_vars == num_variables() && "Not all variables are quantified");
  // Ashish: with DNF parts, the innermost quantifier may not be
  // existential
  //    assert (qlevel & 0x1 == 1 && "Innermost and outermost quantifier must be existential");
  assert (clause_vars.size() == 0);   //some benchmark has no 0 in the last clause
}

void CSolver::read_qdimacs(char * filename ) {
  // Ashish: modifed to read dnf cubes as well
  char combining_op[32];
  int file_section = 0;
  char line_buffer[MAX_LINE_LENGTH];
  char word_buffer[MAX_WORD_LENGTH];
  set<int> constraint_vars;
  set<int> constraint_lits;
  int var_num;
  int cl_num;
  int cube_num;
  unsigned qlevel = 0;
  int quantified_vars = 0;
  int added_cls = 0, added_cubes = 0;
  ifstream inp (filename, ios::in);
  if (!inp) {
    cerr << "Can't open input file" << endl;
    exit(1);
  }
  while (inp.getline(line_buffer, MAX_LINE_LENGTH)) {
    if (inp.fail()) {
      cerr << "Too large an input line. Unable to continue..." << endl;
      exit(2);
    }

    if (line_buffer[0] == 'c') { continue; }
    else if (line_buffer[0] == 'p') {
      int arg = sscanf (line_buffer, "p cnfdnf %s %d %d %d", 
                        combining_op, &var_num, &cl_num, &cube_num);
      if( arg < 4 ) {
        if (strcmp(combining_op, "or") && strcmp(combining_op, "and")) {
          cerr << "ERROR: Operator type in input file must be \"or\" or \"and\"" << endl;
          exit(3);
        }
        
        cerr << "ERROR: Unable to read number of variables, clauses, and cubes" << endl;
        exit(3);
      }
      set_combining_op(combining_op); // Ashish: set the combining operator
      set_variable_number(var_num); //first element not used.
    }
    else if (line_buffer[0] == 'q' ) {
      cerr << "Error: Are you sure input is in Q-DIMACS format instead of Rintanen's format? " << endl;
      exit(1);
    }
    else if (line_buffer[0] == 'a' || line_buffer[0] == 'e') { 
      if (line_buffer[0] == 'a' && qlevel == 0)
        ++ qlevel;

      if (q_groups.size() <= qlevel)
        q_groups.resize (qlevel + 1);

      if (line_buffer[0] == 'e')
        assert ((qlevel & 0x1) == 0);

      char *lp = line_buffer + 1;       
      do {
        char *wp = word_buffer;
        while (*lp && ((*lp == ' ') || (*lp == '\t'))) {
          lp++;
        }
        while (*lp && (*lp != ' ') && (*lp != '\t') && (*lp != '\n')) {
          *(wp++) = *(lp++);
        }
        *wp = '\0';                                 // terminate string
        if (strlen(word_buffer) != 0) {     // check if number is there
          int var_idx = atoi (word_buffer);
          assert (var_idx >= 0);
          if( var_idx > 0) {
            assert (variables[var_idx].q_level == -1);
            variables[var_idx].q_level = qlevel;
            if ((qlevel & 0x1) == 0) 
              variables[var_idx].set_exist();
            else 
              variables[var_idx].set_univ();
            ++ quantified_vars;
            q_groups[qlevel].push_back(var_idx);
          }     
        }
      } while(*lp);
      ++ qlevel;
      assert (qlevel < (QLEVEL_MASK/2));
    }
    else {                             // Clause/cube definition or continuation
      char *lp = line_buffer;
      do {
        char *wp = word_buffer;
        while (*lp && ((*lp == ' ') || (*lp == '\t'))) {
          lp++;
        }
        while (*lp && (*lp != ' ') && (*lp != '\t') && (*lp != '\n')) {
          *(wp++) = *(lp++);
        }
        *wp = '\0';                                 // terminate string

        if (strlen(word_buffer) != 0) {     // check if number is there
          int var_idx = atoi (word_buffer);
          int sign = 0;

          if( var_idx != 0) {
            if( var_idx < 0)  { var_idx = -var_idx; sign = 1; }
            constraint_vars.insert(var_idx);
            constraint_lits.insert( (var_idx << 1) + sign);
          }     
          else {
            if (constraint_vars.size() == 0) {
              ++ file_section;
              continue;
            }

            //add this constraint
            if (file_section == 0)
              ++ added_cls;
            else
              ++ added_cubes;
            if (constraint_vars.size() == constraint_lits.size()) { //yeah, can add this constraint
              vector <int> temp;
              for (set<int>::iterator itr = constraint_lits.begin();
                   itr != constraint_lits.end(); ++itr)
                temp.push_back (*itr);
              if (file_section == 0)
                add_orig_clause(temp);
              else
                add_orig_cube(temp);
            }
            else { //it contain var of both polarity
              // if clause, then automatically satisfied - nothing to do
              // if cube, then automatically unsatisfiable!
              if (file_section == 1) {
                cerr << "Cube number " << added_cubes << " of input is trivially unsat "
                     << endl
                     << "RESULT: UNSAT" << endl;
                exit(0);
              }
            }
            constraint_lits.clear();
            constraint_vars.clear();
          }
        }
      } while (*lp);
    }
  }
  // Ashish: the following appears to be unnecessary
  // assert (cl_num == added_cls);
  // assert (cube_num == added_cubes);

  assert (quantified_vars == num_variables() && "Not all variables are quantified");
  // Ashish: with DNF parts, the innermost quantifier may not be existential
  // assert (qlevel & 0x1 == 1 && "Innermost and outermost quantifier must be existential");
  assert (constraint_vars.size() == 0);         //some benchmark has no 0 in the last constraint
}

// Ashish: print out values of variables in the first (existential) layer
void CSolver::print_solution_e0(void) {
  cout << "Solution (first existential layer) : " << endl << "  ";
  for (int i=0; i<num_variables(); i++)
    if (variables[i].q_level == 0)
      cout << i << '(' << variables[i].value << "), ";
  cout << endl;
}
