/***********************************************************************/
/*                                                                     */
/*   svm_struct_api.c                                                  */
/*                                                                     */
/*   Definition of API for attaching implementing SVM learning of      */
/*   structures (e.g. parsing, multi-label classification, HMM)        */ 
/*                                                                     */
/*   Author: Thorsten Joachims                                         */
/*   Date: 03.07.04                                                    */
/*                                                                     */
/*   Copyright (c) 2004  Thorsten Joachims - All rights reserved       */
/*                                                                     */
/*   This software is available for non-commercial use only. It must   */
/*   not be modified and distributed without prior permission of the   */
/*   author. The author is not responsible for implications from the   */
/*   use of this software.                                             */
/*                                                                     */
/***********************************************************************/
/*                                                                     */
/*   Large-Margin Training of Submodular Summarization Methods         */
/*   Last modified: 28 jul 2011                                        */
/*   Copyright (c) 2011 Ruben Sipos - All rights reserved              */
/*                                                                     */
/***********************************************************************/

#define _CPPCODE_ //hack for C/C++ friendship in svm_struct_api_types.h


#define OPT_BINARIZE
//#define OPT_SKIPCLS1
//#define OPT_NSCO4
#define OPT_SCO
#define OPT_NGS
#define OPT_NGS4
#define OPT_NGS4B
#define OPT_FEATSv11
#define OPT_FEATSv2
//#define OPT_DCS
//#define OPT_NORSCALE
//#define OPT_PRFv2

//----------------------------


#include <stdio.h>
#include <string.h>

#include <string>
#include <vector>
#include <map>
#include <set>
using namespace std;

typedef struct cpplabel {
	int docid;
	vector<int> sentences;
#ifdef OPT_NGS4
	double lossMN;
	struct cpplabel* Y;
#endif
} cppLABEL;

typedef struct sentence_s {
	int cls;
	int slen;
	int art;
	int sent;
	map<int,int> doc_word_counts;
} sentence_t;

typedef struct cpppattern {
	int docid;
	int N;
	sentence_t* sentences;
} cppPATTERN;

typedef cppPATTERN* PATTERN;
typedef cppLABEL* LABEL;

double loss1(LABEL y, LABEL ybar);
double loss2(LABEL y, LABEL ybar, int sel);


extern "C" {
#include "svm_struct/svm_struct_common.h"
#include "svm_struct_api.h"
} //extern "C"


#ifndef OPT_FEATSv11
const double th[] = {0.0001, 0.0005, 0.001, 0.002, 0.005, 0.01, 0.02, 0.05, 0.1, 0.15, 0.2, 0.3, 0.5, 0.8};
#else
const double th[] = {0.001, 0.005, 0.01, 0.05, 0.1, 0.15, 0.2, 0.3, 0.5, 0.65, 0.8, 0.9, 0.95, 0.99};
#endif
const int thL = sizeof(th)/sizeof(double);
const int psiLen = 2*410+6+100+2000+2000; //600+500*6;
const double r = 0.3;
const int B = 665;



#define myassert(exp) \
	if (exp) ; \
	else myassert_report( (char*)#exp, (char*)__FILE__, (char*)__BASE_FILE__, __LINE__ )
void myassert_report(char *exp, char *file, char *baseFile, int line)
{
	printf("myassert(%s) FAIL @ %s (%s) %i\n", exp, file, baseFile, line);
	exit(-1);
}


void denewline(char* s)
{
	while (*s)
	{
		if (*s == '\r' || *s == '\n')
		{
			*s = 0;
			return;
		}
		s++;
	}
}

char* list2str(vector<int>& l, char* sep)
{
	static char s[1024];
	s[0] = 0;
	char tmp[1024];
	for (int i=0; i<(int)l.size(); i++)
	{
		sprintf(tmp, "%i%s", l[i], (i+1==(int)l.size())?(""):(sep));
		strcat(s, tmp);
	}
	return s;
}


struct gcache_s {
	SAMPLE sample;
    bool clx;
    bool rec;

	int dotsN;
	double**** dots;

	map<string,int> wmap;
	map<int,string> invwmap;
	set<int> caps;
	set<int> stops;
	map<int,int>* hdrs;

	map<int,int>* DFs;
	map<int,int> CFs;
	int* DWcnt;
	map<int,int>* SOcnt;
	map<int,set<int> >* AOcnt;
	map<int,int>* fstsnb;

	double* minpsi2;
	double* maxpsi2;
	bool mm;
	double* hi;
	double* lo;
} gcache;


#define showdot() \
	printf("."); fflush(stdout)

#define ffcn_wgt(c,x,d,k,v) \
	switch (c) { \
	case ffcn_wgt_1: (x) = (v); break; \
	case ffcn_wgt_tfiyCF: (x) = (double)(v) / gcache.CFs[k]; break; \
	case ffcn_wgt_tfiyDF: (x) = (double)(v) / gcache.DFs[d][k]; break; \
	case ffcn_wgt_DTFICF: (x) = (double)gcache.DFs[d][k] / gcache.CFs[k]; break; \
	default: printf("ERR: ffcn_wgt default case\n"); \
	}

#define ffcn_ky(c,x,d,k,o) \
	switch (c) { \
	case ffcn_ky_all: (x) = true; break; \
	case ffcn_ky_caps: (x) = gcache.caps.count(k); break; \
	case ffcn_ky_stops2: (x) = (gcache.stops.count(k)==0)?(true):(false); break; \
	\
	case ffcn_ky_min5DF: (x) = gcache.DFs[d][k] >= (o); break; \
	case ffcn_ky_max5DF: (x) = gcache.DFs[d][k] <= (o); break; \
	case ffcn_ky_min5pct: (x) =  (float)gcache.DFs[d][k] / gcache.DWcnt[d] >= (o); break; \
	case ffcn_ky_max5pct: (x) =  (float)gcache.DFs[d][k] / gcache.DWcnt[d] <= (o); break; \
	case ffcn_ky_WDR: tmpx = 0; for (int wdr=0; wdr<gcache.sample.n; wdr++) \
		if (gcache.DFs[d].count(k)) tmpx++; (x) = (float)tmpx / gcache.sample.n <= (o); break; \
	case ffcn_ky_WSR: (x) = (float)gcache.SOcnt[d][k] / N <= (o); break; \
	case ffcn_ky_len5: (x) = gcache.invwmap[k].length() >= (o); break; \
	case ffcn_ky_infstsent: (x) = gcache.fstsnb[d][k] <= (o); break; \
	case ffcn_ky_inarts: (x) = gcache.AOcnt[d][k].size() >= (o); break; \
        case ffcn_ky_inhdrs: (x) = gcache.hdrs[d][k] >= (o); break; \
	default: printf("ERR: ffcn_ky default case\n"); \
	} 
	//if ((c)!=ffcn_ky_stops2) (x) = (x) & (gcache.stops.count(k)==0); //remove stops

struct dotz_s {

	static double*** cdots(int mywgt, int myky, double o)
	{
		double*** mydots = new double**[gcache.sample.n];
		for (int d=0; d<gcache.sample.n; d++)
		{
			PATTERN x = gcache.sample.examples[d].x;
			int N = x->N;
			mydots[d] = new double*[N];
			for (int i=0; i<N; i++)
			{
				mydots[d][i] = new double[N];
				for (int j=0; j<N; j++)
				{
					if (i == j)
						continue;
					
					bool b = false;
					double tmp1 = 0, tmp2 = 0;
					int tmpx;
					double dsum = 0;
					for (map<int,int>::iterator it = x->sentences[i].doc_word_counts.begin(); it != x->sentences[i].doc_word_counts.end(); it++)
						if (x->sentences[j].doc_word_counts.count(it->first))
						{
							ffcn_ky(myky, b, d, it->first, o);
							if (b)
							{
								ffcn_wgt(mywgt, tmp1, d, it->first, it->second);
								ffcn_wgt(mywgt, tmp2, d, it->first, x->sentences[j].doc_word_counts[it->first]);
								dsum += tmp1 * tmp2;
							}
						}
					double nsum1 = 0;
					for (map<int,int>::iterator it = x->sentences[i].doc_word_counts.begin(); it != x->sentences[i].doc_word_counts.end(); it++)
					{
#ifdef OPT_FEATSv11
						ffcn_wgt(mywgt, tmp1, d, it->first, it->second);
						nsum1 += tmp1*tmp1;
#else
						nsum1 += it->second*it->second;
#endif
					}
					double nsum2 = 0;
					for (map<int,int>::iterator it = x->sentences[j].doc_word_counts.begin(); it != x->sentences[j].doc_word_counts.end(); it++)
					{
#ifdef OPT_FEATSv11
						ffcn_wgt(mywgt, tmp1, d, it->first, it->second);
						nsum2 += tmp1*tmp1;
#else
						nsum2 += it->second*it->second;
#endif
					}
					if (nsum1*nsum2 < 1e-5)
						dsum /= sqrt(1e-5);
					else
						dsum /= sqrt(nsum1*nsum2);

					mydots[d][i][j] = dsum;
				}
			}
		}
		showdot();
		return mydots;
	}

	enum {
	ffcn_nil,
	ffcn_wgt_1,
	ffcn_wgt_tfiyCF,
	ffcn_wgt_tfiyDF,
	ffcn_wgt_DTFICF,
	ffcn_ky_all,
	ffcn_ky_caps,
	ffcn_ky_stops2,
	ffcn_ky_min5DF,
	ffcn_ky_max5DF,
	ffcn_ky_min5pct,
	ffcn_ky_max5pct,
	ffcn_ky_WDR,
	ffcn_ky_WSR,
	ffcn_ky_len5,
	ffcn_ky_infstsent,
	ffcn_ky_inarts,
	ffcn_ky_inhdrs,
	ffcn_fin
	};

} dotz;




void fillcache(SAMPLE sample, char* file)
{
	printf("Populating EX cache...\n");

	gcache.sample = sample;

	//load wmap
    {
        FILE* f = fopen("../data/wmap.str", "r");
	if (!f) printf("ERR: missing wmap.str\n");
        while (!feof(f))
        {
            int w;
            char s[1024];
            fscanf(f, "%i %s\n", &w, s); //FIXME ?? was &s
            gcache.wmap.insert(pair<string,int>(s, w));
        }
        fclose(f);
    }

	//inv wmap
	for (map<string,int>::iterator it = gcache.wmap.begin(); it != gcache.wmap.end(); it++)
		gcache.invwmap.insert(pair<int,string>(it->second, it->first));

	//caps
	for (map<string,int>::iterator it = gcache.wmap.begin(); it != gcache.wmap.end(); it++)
		if (isupper(it->first[0]))
			gcache.caps.insert(it->second);

	//stops
    {
        FILE* f = fopen("../data/stops", "r");
	if (!f) printf("ERR: missing stops\n");
        while (!feof(f))
        {
            char line[1024];
            fgets(line, 1000, f);
            denewline(line);
            if (strlen(line)<1)
                continue;
            gcache.stops.insert(gcache.wmap[line]);
        }
        fclose(f);
    }

	//precompute DFs
	gcache.DFs = new map<int,int>[sample.n];
	gcache.SOcnt = new map<int,int>[sample.n];
	gcache.AOcnt = new map<int,set<int> >[sample.n];
	gcache.fstsnb = new map<int,int>[sample.n];
	gcache.DWcnt = new int[sample.n];
	for (int d=0; d<sample.n; d++)
	{
		PATTERN x = sample.examples[d].x;
		int N = x->N;
		for (int i=0; i<N; i++)
		{
#ifndef OPT_NSCO4
			if (x->sentences[i].cls > 1)
				continue;
#endif
			for (map<int,int>::iterator it = x->sentences[i].doc_word_counts.begin(); it != x->sentences[i].doc_word_counts.end(); it++)
			{
				if (!gcache.DFs[d].count(it->first))
					gcache.DFs[d][it->first] = 0;
				if (!gcache.SOcnt[d].count(it->first))
					gcache.SOcnt[d][it->first] = 0;
				if (!gcache.AOcnt[d].count(it->first))
					gcache.AOcnt[d][it->first] = set<int>(&x->sentences[i].art, (&x->sentences[i].art)+1);
				if (!gcache.fstsnb[d].count(it->first))
					gcache.fstsnb[d][it->first] = x->sentences[i].sent;
				gcache.DFs[d][it->first] += it->second;
				gcache.SOcnt[d][it->first] += 1;
				gcache.AOcnt[d][it->first].insert(x->sentences[i].art);
				if (gcache.fstsnb[d][it->first] > x->sentences[i].sent)
					gcache.fstsnb[d][it->first] = x->sentences[i].sent;
				gcache.DWcnt[d] += 1;
			}
		}
	}

	//load CFs
    {
        FILE* f = fopen("../data/CFs.str", "r");
	if (!f) printf("ERR missing CFs.str\n");
        while (!feof(f))
        {
            int w;
            int c;
            fscanf(f, "%i %i\n", &w, &c);
            gcache.CFs.insert(pair<int,int>(w,c));
        }
        fclose(f);
    }

#ifdef OPT_FEATSv2
    {
	    //load HDR
	    int hc = 0;
	    gcache.hdrs = new map<int,int>[sample.n];
	    int d = 0;
	    FILE* f = fopen(file, "r");
	    if (!f) printf("ERR missing '%s'\n", file);
	    while (!feof(f))
	    {
		    char fn[1024];
		    fn[0] = 0;
		    fgets(fn, 1000, f);
		    for (int i=0; i<(int)strlen(fn); i++)
			    if (fn[i]=='\n'||fn[i]=='\r')
				    fn[i] = 0;
		    if (strlen(fn)==0)
			    break;
		    if (strncmp(fn, "../data/svm", 11)) printf("ERR unsupported data path\n");
		    char fn2[1024];
		    strcpy(fn2, "../data/hdr_d");
		    strcat(fn2, fn+11);
		    strcat(fn2, "t");
		    FILE* ff = fopen(fn2, "r");
		    if (!ff) printf("ERR missing '%s'\n", fn2);
		    //printf("ldHDR %s\n", fn2);
		    while (!feof(ff))
		    {
			    char hd[1024];
			    hd[0] = 0;
			    fgets(hd, 1000, ff);
			    char* p = hd;
			    while (*p)
			    {
				    *p = tolower(*p);
				    p++;
			    }
			    if (strlen(hd)==0)
				    break;
			    p = strtok(hd, " ,.'\"!;:");
			    while (p)
			    {
				    if (gcache.wmap.count(p))
				    {
					    int k = gcache.wmap[p];
					    if (!gcache.hdrs[d].count(k))
						    gcache.hdrs[d][k] = 0;
					    gcache.hdrs[d][k]++;
					    //printf("ldHDR %i %s\n", k, p);
					    hc++;
				    }
				    p = strtok(NULL, " ,.'\"!;:");
			    }
		    }
		    fclose(ff);
		    d++;
	    }
	    fclose(f);
	    printf("ldHDR %i\n", hc);
    }
#endif //OPT_FEATSv2

	//try to load psiminmax
    {
        FILE* f = fopen("psiminmax", "r");
        if (f)
        {
            gcache.mm = true;
            gcache.minpsi2 = new double[psiLen];
            gcache.maxpsi2 = new double[psiLen];
            for (int i=0; i<psiLen; i++)
                fscanf(f, "%lf %lf\n", &gcache.minpsi2[i], &gcache.maxpsi2[i]);
            printf("Using PMM.\n");
            fclose(f);
        } else {
            gcache.mm = false;
            printf("Not using PMM.\n");
        }
    }

	gcache.lo = new double[psiLen];
	gcache.hi = new double[psiLen];
    for (int i=0; i<psiLen; i++)
    {
        gcache.lo[i] = 1e9;
        gcache.hi[i] = -1e9;
    }


    //ngs6 f(full) a(allonly) ax(noall) m(nominmax) t(nov2) w(noWxR) c(nocapsstopslen)
    printf("  DOTz(6f) "); fflush(stdout);
    gcache.dotsN = 27+9; //27+9
	double**** dots = new double***[gcache.dotsN];
	gcache.dots = dots;
	int q=0;
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_1, dotz.ffcn_ky_all, 0);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyCF, dotz.ffcn_ky_all, 0);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyCF, dotz.ffcn_ky_caps, 0);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyCF, dotz.ffcn_ky_stops2, 0);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_all, 0);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_DTFICF, dotz.ffcn_ky_all, 0);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_min5DF, 5);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_min5DF, 10);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_min5DF, 20);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_max5DF, 5);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_max5DF, 10);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_max5DF, 20);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_min5pct, 0.05);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_min5pct, 0.1);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_min5pct, 0.3);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_max5pct, 0.05);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_max5pct, 0.1);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_max5pct, 0.3);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_WDR, 0.2);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_WDR, 0.4);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_WDR, 0.7);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_WSR, 0.05);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_WSR, 0.1);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_WSR, 0.3);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_len5, 5);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_len5, 7);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_len5, 9);
#ifdef OPT_FEATSv2
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_infstsent, 1);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_infstsent, 2);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_infstsent, 3);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_inarts, 2);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_inarts, 3);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_inarts, 5);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_inhdrs, 1);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_inhdrs, 2);
	dots[q++] = dotz.cdots(dotz.ffcn_wgt_tfiyDF, dotz.ffcn_ky_inhdrs, 3);
#endif //OPT_FEATSv2
	//myassert(q==gcache.dotsN);
	gcache.dotsN = q;
    printf(" (%i) done\n", gcache.dotsN);
    printf("Cache is hot.\n");
    return;
}


void        svm_struct_learn_api_init(int argc, char* argv[])
{
  /* Called in learning part before anything else is done to allow
     any initializations that might be necessary. */
    gcache.clx = false;
    gcache.rec = false;
}

void        svm_struct_learn_api_exit()
{
  /* Called in learning part at the very end to allow any clean-up
     that might be necessary. */
}

void        svm_struct_classify_api_init(int argc, char* argv[])
{
  /* Called in prediction part before anything else is done to allow
     any initializations that might be necessary. */
    gcache.clx = true;
    gcache.rec = false;
}

void        svm_struct_classify_api_exit()
{
  /* Called in prediction part at the very end to allow any clean-up
     that might be necessary. */
}

LABEL mknewgs(int docid, cppPATTERN* x, cppLABEL* y, STRUCT_LEARN_PARM *sparm)
{
  LABEL ybar = new cppLABEL;
  ybar->docid = y->docid;
#ifdef OPT_NGS4
  //create 4 manual labels
  ybar->Y = new cppLABEL[4];
  for (int i=0; i<4; i++)
  {
	  ybar->Y[i].docid = y->docid;
	  for (int j=0; j<x->N; j++)
		  if (x->sentences[j].cls == i+1)
			  ybar->Y[i].sentences.push_back(j);
  }
#endif //OPT_NGS4
  int N = x->N;
  char* opt = new char[N];
  for (int i=0; i<N; i++)
	  opt[i] = 0;
  while (true)
  {
	  int best = -1;
	  double best_score = 1e10;
	  ybar->sentences.clear();
	  for (int i=0; i<N; i++)
		  if (opt[i])
			  ybar->sentences.push_back(i);
	  for (int did=0; did<N; did++)
	  {
		  if (opt[did])
			  continue;
#ifndef OPT_NSCO2
//#ifdef OPT_SKIPCLS1
          //if (gcache.clx && x->sentences[did].cls == 1)
          if (x->sentences[did].cls > 0)
              continue;
//#endif //OPT_SKIPCLS1
#endif //OPT_NSCO2
		  ybar->sentences.push_back(did);
		  
		  int sumc = 0;
		  for (int i=0; i<(int)ybar->sentences.size(); i++)
			  sumc += x->sentences[ybar->sentences[i]].slen;
#ifdef OPT_NGS4
		  //AVG LOSS 4
		  double score = 0;
		  for (int i=0; i<4; i++)
			  score += loss1(&ybar->Y[i], ybar);
			  //score = max(score,loss(&Y[i], ybar, sparm));
		  score /= 4;
#else //OPT_NGS4
		  double score = loss(y, ybar, sparm);
#endif //OPT_NGS4
		  ybar->sentences.pop_back();
		  if (sumc > B)
			  continue;
		  if (score < best_score)
		  {
			  best_score = score;
			  best = did;
			  //printf("%i ", did); fflush(stdout);
		  }
	  }
	  if (best_score > 1e9)
		  break;
	  //printf("%f >>>\n", best_score);
	  opt[best] = 1;
  }

  ybar->docid = x->docid;
  ybar->sentences.clear();
  for (int did=0; did<N; did++)
	  if (opt[did])
		  ybar->sentences.push_back(did);
#ifdef OPT_NGS4
  //AVG LOSS 4
  double LL = 0;
  if (struct_verbosity>1) printf("    avg ");
  for (int i=0; i<4; i++)
  {
	  LL += loss1(&ybar->Y[i], ybar);
	  if (struct_verbosity>1) printf("%f ", loss1(&ybar->Y[i], ybar));
  }
  if (struct_verbosity>1) printf("\n");
  LL /= 4;
  ybar->lossMN = LL;

#else //OPT_NGS4
  double LL = loss(y, ybar, sparm);
#endif //OPT_NGS4
  int cost = 0;
  for (int i=0; i<(int)ybar->sentences.size(); i++)
	  cost += x->sentences[ybar->sentences[i]].slen;
#ifdef OPT_NGS4
  if (struct_verbosity>1) printf("  NGS-4avg: %i %i [%s] %f\n", ybar->docid, cost, list2str(ybar->sentences, (char*)", "), LL);
#else //OPT_NGS4
  if (struct_verbosity>1) printf("  NGS-1st: %i %i [%s] %f\n", ybar->docid, cost, list2str(ybar->sentences, (char*)", "), LL);
#endif //OPT_NGS4

  return ybar;
}




SAMPLE      read_struct_examples(char *file, STRUCT_LEARN_PARM *sparm)
{
  /* Reads struct examples and returns them in sample. The number of
     examples must be written into sample.n */
  SAMPLE   sample;  /* sample */
  EXAMPLE  *examples;
  long     n;       /* number of examples */

#ifdef OPT_NGS
  printf("\n");
#endif
  
  n = 0;
  FILE* f = fopen(file, "r");
  myassert(f);
  char input_file[1024];
  while (fgets(input_file, 1000, f))
	  n++;
  examples=(EXAMPLE *)my_malloc(sizeof(EXAMPLE)*n);
  sample.examples=examples;
  sample.n = n;
  gcache.sample = sample; //needed by NGS

  fseek(f, 0, SEEK_SET);
  int docid = 0;
  while (fgets(input_file, 1000, f))
  {
	  denewline(input_file);
	  FILE* fi = fopen(input_file, "r");
	  if (!fi)
		  printf("read_struct_examples()\n  File '%s' not found!\n", input_file);
	  myassert(fi);
	  char line[4*1024];
	  int N = 0;
	  while (fgets(line, 4*1000, fi))
		  N++;
	  fseek(fi, 0, SEEK_SET);

	  cppPATTERN* x = new cppPATTERN;
	  cppLABEL* y = new cppLABEL;
	  x->docid = docid;
	  x->N = N;
	  x->sentences = new sentence_t[N];
	  y->docid = docid;
	  int snb = 0;
	  while (fgets(line, 4*1000, fi))
	  {
		  char* tok;
		  tok = strtok(line, " \t\n");
		  x->sentences[snb].cls = atoi(tok);
          //if (x->sentences[snb].cls > 1)
          //    x->sentences[snb].cls = 1;

	  //skip ex# >20
//	  if (x->sentences[snb].cls == 0 && snb >= 20)
//		  continue;

          if (false && x->sentences[snb].cls > 1)
          {
              x->N = snb;
              break;
              snb++;
              continue;
          }
		  if (x->sentences[snb].cls == 1)
			  y->sentences.push_back(snb);
		  while ((tok = strtok(NULL, " \t\n")))
		  {
			  char tokcpy[1024];
			  strcpy(tokcpy, tok);
			  tok = tokcpy;
			  while (*tok && *tok != ':')
				  tok++;
			  *tok = 0;
			  tok++;
			  int k = atoi(tokcpy);
			  int v = atoi(tok);
			  if (k == 1)
				  x->sentences[snb].slen = v;
			  if (k == 2)
				  x->sentences[snb].art = v;
			  if (k == 3)
				  x->sentences[snb].sent = v;
			  if (k < 10) //FEATSv1.2
				  continue;
			  x->sentences[snb].doc_word_counts.insert(pair<int,int>(k,v));
		  }
		  snb++;
	  }
	  if (x->N != snb) printf("JOOOOOOOOOOOOOOOOOOOOOOOOOOJ\n");
	  x->N = snb; //skip ex# >20 FIX/HACK

      examples[docid].x = x;
	  examples[docid].y = y;

      //------------------------------------------------
      // NEW GS COMPUTATION
#ifdef OPT_NGS
	  examples[docid].y = mknewgs(docid, x, y, sparm);
#endif //OPT_NGS
      //------------------------------------------------
	  
	  docid++;
      fclose(fi);
  }
  fclose(f);

  printf("  %i examples read\n", (int)n);

  //print compile opts
  printf("Compiled with ");
#ifdef OPT_BINARIZE
  printf("OPT_BINARIZE ");
#endif
#ifdef OPT_SCO
  printf("OPT_SCO ");
#endif
#ifdef OPT_NSCO4
  printf("OPT_NSCO4 ");
#endif
#ifdef OPT_SKIPCLS1
  printf("OPT_SKIPCLS1 ");
#endif
#ifdef OPT_NGS
  #ifdef OPT_NGS4
    #ifdef OPT_NGS4B
      printf("OPT_NGS4B ");
    #else //OPT_NGS4B
      printf("OPT_NGS4A ");
    #endif //OPT_NGS4B
  #else //OPT_NGS4
    printf("OPT_NGS1 ");
  #endif //OPT_NGS4
#endif //OPT_NGS
#ifdef OPT_DCS
  printf("OPT_DCS ");
#endif
#ifdef OPT_FEATSv11
  printf("OPT_FEATSv11b ");
#endif
#ifdef OPT_FEATSv2
  printf("OPT_FEATSv2 ");
#endif
#ifdef OPT_NORSCALE
  printf("OPT_NORSCALE ");
#endif
#ifdef OPT_PRFv2
    printf("OPT_PRFv2 ");
#else
    printf("OPT_PRFv3 ");
#endif
  printf("\n");

  fillcache(sample, file);

  return(sample);
}

void        init_struct_model(SAMPLE sample, STRUCTMODEL *sm, 
			      STRUCT_LEARN_PARM *sparm, LEARN_PARM *lparm, 
			      KERNEL_PARM *kparm)
{
  /* Initialize structmodel sm. The weight vector w does not need to be
     initialized, but you need to provide the maximum size of the
     feature space in sizePsi. This is the maximum number of different
     weights that can be learned. Later, the weight vector w will
     contain the learned weights for the model. */

  sm->sizePsi=psiLen; /* replace by appropriate number of features */
}

CONSTSET    init_struct_constraints(SAMPLE sample, STRUCTMODEL *sm, 
				    STRUCT_LEARN_PARM *sparm)
{
  /* Initializes the optimization problem. Typically, you do not need
     to change this function, since you want to start with an empty
     set of constraints. However, if for example you have constraints
     that certain weights need to be positive, you might put that in
     here. The constraints are represented as lhs[i]*w >= rhs[i]. lhs
     is an array of feature vectors, rhs is an array of doubles. m is
     the number of constraints. The function returns the initial
     set of constraints. */
  CONSTSET c;
  long     sizePsi=sm->sizePsi;
  long     i;
  WORD     words[2];

  if(1) { /* normal case: start with empty set of constraints */
    c.lhs=NULL;
    c.rhs=NULL;
    c.m=0;
  }
  else { /* add constraints so that all learned weights are
            positive. WARNING: Currently, they are positive only up to
            precision epsilon set by -e. */
    c.lhs=(DOC**)my_malloc(sizeof(DOC *)*sizePsi);
    c.rhs=(double*)my_malloc(sizeof(double)*sizePsi);
    for(i=0; i<sizePsi; i++) {
      words[0].wnum=i+1;
      words[0].weight=1.0;
      words[1].wnum=0;
      /* the following slackid is a hack. we will run into problems,
         if we have move than 1000000 slack sets (ie examples) */
      c.lhs[i]=create_example(i,0,1000000+i,1,create_svector(words,(char*)"",1.0));
      c.rhs[i]=0.0;
    }
  }
  return(c);
}

LABEL       classify_struct_example(PATTERN x, STRUCTMODEL *sm, 
				    STRUCT_LEARN_PARM *sparm)
{
  /* Finds the label yhat for pattern x that scores the highest
     according to the linear evaluation function in sm, especially the
     weights sm.w. The returned label is taken as the prediction of sm
     for the pattern x. The weights correspond to the features defined
     by psi() and range from index 1 to index sm->sizePsi. If the
     function cannot find a label, it shall return an empty label as
     recognized by the function empty_label(y). */
  LABEL y;

  /* insert your code for computing the predicted label y here */

  LABEL ybar = NULL;
  y = find_most_violated_constraint_marginrescaling(x, ybar, sm, sparm);

  return(y);
}


LABEL       find_most_violated_constraint_slackrescaling(PATTERN x, LABEL y, 
						     STRUCTMODEL *sm, 
						     STRUCT_LEARN_PARM *sparm)
{
  /* Finds the label ybar for pattern x that that is responsible for
     the most violated constraint for the slack rescaling
     formulation. For linear slack variables, this is that label ybar
     that maximizes

            argmax_{ybar} loss(y,ybar)*(1-psi(x,y)+psi(x,ybar)) 

     Note that ybar may be equal to y (i.e. the max is 0), which is
     different from the algorithms described in
     [Tschantaridis/05]. Note that this argmax has to take into
     account the scoring function in sm, especially the weights sm.w,
     as well as the loss function, and whether linear or quadratic
     slacks are used. The weights in sm.w correspond to the features
     defined by psi() and range from index 1 to index
     sm->sizePsi. Most simple is the case of the zero/one loss
     function. For the zero/one loss, this function should return the
     highest scoring label ybar (which may be equal to the correct
     label y), or the second highest scoring label ybar, if
     Psi(x,ybar)>Psi(x,y)-1. If the function cannot find a label, it
     shall return an empty label as recognized by the function
     empty_label(y). */
  LABEL ybar = NULL;

  /* insert your code for computing the label ybar here */

  return(ybar);
}

/*
vector<cppLABEL*>* oldcs = NULL;
int alloo = 0;
bool lbleq(LABEL a, LABEL b) {
	if (a->docid != b->docid) return false;
	if (a->sentences.size() != b->sentences.size()) return false;
	for (int i=0; i<(int)a->sentences.size(); i++)
		if (a->sentences[i] != b->sentences[i])
			return false;
	return true;
}
double viol(PATTERN x, double yscore, LABEL y, LABEL ybar, STRUCTMODEL *sm, STRUCT_LEARN_PARM *sparm) {
	//w*(y-ybar)-loss
	SVECTOR* f = psi(x, ybar, sm, sparm);
	WORD* wd = f->words;
	double score = 0;
	while (wd->wnum) { score += sm->w[wd->wnum] * wd->weight; wd++; }
	double LL = loss(y, ybar, sparm);
	return -yscore + score + LL;
}
*/


LABEL       find_most_violated_constraint_marginrescaling(PATTERN x, LABEL y, 
						     STRUCTMODEL *sm, 
						     STRUCT_LEARN_PARM *sparm)
{
  /* Finds the label ybar for pattern x that that is responsible for
     the most violated constraint for the margin rescaling
     formulation. For linear slack variables, this is that label ybar
     that maximizes

            argmax_{ybar} loss(y,ybar)+psi(x,ybar)

     Note that ybar may be equal to y (i.e. the max is 0), which is
     different from the algorithms described in
     [Tschantaridis/05]. Note that this argmax has to take into
     account the scoring function in sm, especially the weights sm.w,
     as well as the loss function, and whether linear or quadratic
     slacks are used. The weights in sm.w correspond to the features
     defined by psi() and range from index 1 to index
     sm->sizePsi. Most simple is the case of the zero/one loss
     function. For the zero/one loss, this function should return the
     highest scoring label ybar (which may be equal to the correct
     label y), or the second highest scoring label ybar, if
     Psi(x,ybar)>Psi(x,y)-1. If the function cannot find a label, it
     shall return an empty label as recognized by the function
     empty_label(y). */
  LABEL ybar = new cppLABEL;
  ybar->docid = x->docid;
  if (struct_verbosity>1) printf("\n-------------- find_most_violated_constraint -------------------\n");
  if (struct_verbosity<=1 && gcache.clx) printf(".");
  int N = x->N;
  char* opt = new char[N];
  for (int i=0; i<N; i++)
	  opt[i] = 0;
  double prevbest = 0;
  int ylen = 0;
  if (gcache.rec || !gcache.clx)
    for (int i=0; i<(int)y->sentences.size(); i++)
	  ylen += x->sentences[y->sentences[i]].slen;
  while (true)
  {
	  int best = -1;
	  double best_score = -1e10;
	  double best_zscore = -1;
	  double best_LL = -1;
      
	  ybar->sentences.clear();
	  for (int i=0; i<N; i++)
		  if (opt[i])
			  ybar->sentences.push_back(i);
      
	  for (int did=0; did<N; did++)
	  {
		  if (opt[did])
			  continue;
#ifdef OPT_SKIPCLS1
          //if (gcache.clx && x->sentences[did].cls == 1)
          if (x->sentences[did].cls > 0)
              continue;
#endif //OPT_SKIPCLS1
#ifdef OPT_SCO
          if (!gcache.rec)
          if (gcache.clx && x->sentences[did].cls > 0)
              continue;
#endif
#ifndef OPT_NSCO2
          if (x->sentences[did].cls > 1)
              continue;
#endif
		  double score = 0;
          ybar->sentences.push_back(did);
		  SVECTOR* f = psi(x, ybar, sm, sparm);
		  WORD* wd = f->words;
		  while (wd->wnum)
		  {
			  score += sm->w[wd->wnum] * wd->weight;
			  wd++;
		  }
		  free_svector(f);
		  
		  int sumc = 0;
		  for (int i=0; i<(int)ybar->sentences.size(); i++)
			  sumc += x->sentences[ybar->sentences[i]].slen;
		  double LL = 0;
		  if (!gcache.clx)
		  {
			  LL = loss(y, ybar, sparm);
			  score += LL;
		  }
		  ybar->sentences.pop_back();
		  double zscore = score;
#ifdef OPT_NORSCALE
		  //score = (score - prevbest) / x->sentences[did].slen;
		  score = score / sumc;
#else
		  score = (score - prevbest) / pow(x->sentences[did].slen, r);
#endif
		  if (score < 0 || (sumc > ylen && !gcache.clx) || (sumc > B && gcache.clx))
			  continue;
		  if (score > best_score)
		  {
			  best_score = score;
			  best_zscore = zscore;
			  best = did;
			  best_LL = LL;
			  if (struct_verbosity>1) printf("%i ", did); fflush(stdout);
		  }
	  }
	  if (best_score < -1e9)
		  break;
	  if (struct_verbosity>1) printf("%f %f %f >>>\n", best_score, best_zscore, best_LL);
	  prevbest = best_zscore;
	  opt[best] = 1;
  }

  ybar->docid = x->docid;
  ybar->sentences.clear();
  for (int did=0; did<N; did++)
	  if (opt[did])
		  ybar->sentences.push_back(did);
  
  double LL = -1.0;
  if (gcache.rec || !gcache.clx)
      LL = loss(y, ybar, sparm);
  if (gcache.clx) //LL@clx hax (unk side-effects)
  {
      y = gcache.sample.examples[x->docid].y;
      LL = loss2(y, ybar, 2);
  }
  int cost = 0;
  for (int i=0; i<(int)ybar->sentences.size(); i++)
	  cost += x->sentences[ybar->sentences[i]].slen;
  //if (gcache.rec || !gcache.clx)
  if (true)
      if (struct_verbosity>1) printf("True: %i [%s]\n", ybar->docid, list2str(y->sentences, (char*)", "));
  if (struct_verbosity>1) printf("Most violated: [%s]\n", list2str(ybar->sentences, (char*)", "));
  if (struct_verbosity>1) printf("Loss: %f\n", LL);
  SVECTOR* f = psi(x, ybar, sm, sparm);
  WORD* wd = f->words;
  if (struct_verbosity>1) printf("Psi:  ");
  /*
  for (int i=0; i<5 && wd[i].wnum; i++)
  //for (int i=0; wd[i].wnum; i++)
  //    if (fabs(wd[i].weight) > 1e-100)
      printf("%i:%.5f ", wd[i].wnum-3, wd[i].weight);
  */
  for (int i=0; i<psiLen && wd[i].wnum; i++)
	  if (wd[i].wnum>=0+3+ 2*(gcache.dotsN * (1+thL)) && wd[i].wnum<5+3+ 2*(gcache.dotsN * (1+thL)))
		  if (struct_verbosity>1) printf("%i:%.5f:%.5f ", (int)wd[i].wnum-3, wd[i].weight, sm->w[wd[i].wnum]);
  if (struct_verbosity>1) printf("\n");
  wd = f->words;
  if (struct_verbosity>1) printf("Wgts: ");
  for (int i=0; i<5 && wd[i].wnum; i++)
  //for (int i=0; wd[i].wnum; i++)
      if (struct_verbosity>1) printf("%i:%.5f ", (int)wd[i].wnum-3, sm->w[wd[i].wnum]);
  free_svector(f);
  if (struct_verbosity>1) printf("\n");
  //printf("Cost & Score: %i %f (%f)\n", cost, -1.0, -1.0);
  if (struct_verbosity>1) printf("Cost & Score: %i _ (_)\n", cost);
  delete opt;
   
  double score1, score2;
    {
        score1 = 0.0;
        if (gcache.clx)
            y = gcache.sample.examples[x->docid].y;
        SVECTOR* f = psi(x, y, sm, sparm);
		  WORD* wd = f->words;
		  while (wd->wnum)
		  {
			  score1 += sm->w[wd->wnum] * wd->weight;
			  wd++;
		  }
		  free_svector(f);
        score2 = 0.0;
        f = psi(x, ybar, sm, sparm);
		  wd = f->words;
		  while (wd->wnum)
		  {
			  score2 += sm->w[wd->wnum] * wd->weight;
			  wd++;
		  }
		  free_svector(f);
          if (struct_verbosity>1) printf("I && II: y=%f yb=%f %s\n", score1, score2, (score2 < score1)?(">MISS<"):(""));
          if (score2 < score1)
          {
              //printf("YBYBYB\n\n\nYBYBYB\n\n\n\n");
          }
    }  

#ifdef OPT_DCS
    if (!gcache.rec && !gcache.clx)
    {
        gcache.rec = true;
        gcache.clx = true;
        LABEL ybar2 = find_most_violated_constraint_marginrescaling(x, y, sm, sparm);
        gcache.clx = false;
        double score3 = 0.0;
        SVECTOR* f = psi(x, ybar2, sm, sparm);
        WORD* wd = f->words;
        while (wd->wnum)
        {
            score3 += sm->w[wd->wnum] * wd->weight;
            wd++;
        }
        free_svector(f);
        printf("1.. 2.. 3.. -> %s\n", (score3>score2)?("IMPR"):("1ST"));
        gcache.rec = false;
        //if (score3>score2)
        if (score2 < score1)
            return ybar2;
    }
#endif

  return(ybar);
}



int         empty_label(LABEL y)
{
  /* Returns true, if y is an empty label. An empty label might be
     returned by find_most_violated_constraint_???(x, y, sm) if there
     is no incorrect label that can be found for x, or if it is unable
     to label x at all */
	//printf("CALL empty_label [%s]\n", list2str(y->sentences, ", "));
    if (y->sentences.size() == 0)
        return true;
  return(0);
}

#ifndef OPT_BINARIZE
#define psiedge(dummyx, dummypsi, dummyoff, a, b, pl) \
	for (int n=0; n<gcache.dotsN; n++) { \
		double t = gcache.dots[n][x->docid][a][b]; \
		thePsi[offset+n*(1+thL)+0] += (pl)?(t):(-t); \
	}
#else //OPT_BINARIZE
#define psiedge(dummyx, dummypsi, dummyoff, a, b, pl) \
	for (int n=0; n<gcache.dotsN; n++) { \
		double t = gcache.dots[n][x->docid][a][b]; \
		thePsi[offset+n*(1+thL)+0] += (pl)?(t):(-t); \
		for (int k=0; k<thL; k++) { \
			if (t > th[k]) \
				thePsi[offset+n*(1+thL)+1+k] += (pl)?(0.01):(-0.01); \
            else \
                break; \
	}}
#endif //OPT_BINARIZE

void psiedge__(PATTERN x, double* thePsi, int offset, int a, int b, bool pl)
{
	for (int n=0; n<gcache.dotsN; n++)
	{
		double t = gcache.dots[n][x->docid][a][b];
		thePsi[offset+n*(1+thL)+0] += (pl)?(t):(-t);
#ifdef OPT_BINARIZE
		for (int k=0; k<thL; k++)
		{
			if (t > th[k])
				thePsi[offset+n*(1+thL)+1+k] += (pl)?(0.01):(-0.01);
		}
#endif //OPT_BINARIZE
	}
}		



SVECTOR     *psi(PATTERN x, LABEL y, STRUCTMODEL *sm,
		 STRUCT_LEARN_PARM *sparm)
{
  /* Returns a feature vector describing the match between pattern x
     and label y. The feature vector is returned as a list of
     SVECTOR's. Each SVECTOR is in a sparse representation of pairs
     <featurenumber:featurevalue>, where the last pair has
     featurenumber 0 as a terminator. Featurenumbers start with 1 and
     end with sizePsi. Featuresnumbers that are not specified default
     to value 0. As mentioned before, psi() actually returns a list of
     SVECTOR's. Each SVECTOR has a field 'factor' and 'next'. 'next'
     specifies the next element in the list, terminated by a NULL
     pointer. The list can be though of as a linear combination of
     vectors, where each vector is weighted by its 'factor'. This
     linear combination of feature vectors is multiplied with the
     learned (kernelized) weight vector to score label y for pattern
     x. Without kernels, there will be one weight in sm.w for each
     feature. Note that psi has to match
     find_most_violated_constraint_???(x, y, sm) and vice versa. In
     particular, find_most_violated_constraint_???(x, y, sm) finds
     that ybar!=y that maximizes psi(x,ybar,sm)*sm.w (where * is the
     inner vector product) and the appropriate function of the
     loss + margin/slack rescaling method. See that paper for details. */
  SVECTOR *fvec=NULL;

  int N = x->N;
  double thePsi[psiLen+10];
  for (int i=0; i<psiLen+10; i++)
      thePsi[i] = 0.0;


  int offset = 3+ 0;
  for (int yi=0; yi<(int)y->sentences.size(); yi++)
  {
	  for (int i=0; i<N; i++)
	  {
#ifndef OPT_NSCO2
		  if (x->sentences[i].cls > 1)
			  continue;
#endif
		  bool cont = false;
		  for (int yj=0; yj<(int)y->sentences.size(); yj++)
			  if (i == y->sentences[yj])
				  cont = true;
		  if (cont)
			  continue;
		  psiedge(x, thePsi, offset, i, y->sentences[yi], true);
	  }
  }

  offset = 3+ gcache.dotsN * (1+thL);
  for (int yi=0; yi<(int)y->sentences.size(); yi++)
	  for (int yj=0; yj<(int)y->sentences.size(); yj++)
	  {
		  if (y->sentences[yi] == y->sentences[yj])
				  continue;
		  psiedge(x, thePsi, offset, y->sentences[yi], y->sentences[yj], false);
	  }
/*
  offset = 3+ 2*(gcache.dotsN * (1+thL));
  for (int yi=0; yi<y->sentences.size(); yi++)
  {
	  thePsi[offset+1] += (x->sentences[y->sentences[yi]].sent==1);
	  thePsi[offset+2] += (x->sentences[y->sentences[yi]].sent<=2);
	  thePsi[offset+3] += (x->sentences[y->sentences[yi]].sent<=3);
  }

  //SQ&RT
  offset = 3+ 2*(gcache.dotsN * (1+thL)) + 100;
  for (int i=0; i<offset; i++)
  {
	  thePsi[offset+i] = thePsi[i]*thePsi[i];
	  if (thePsi[i] < 0)
		  thePsi[2*offset+i] = -sqrt(-thePsi[i]);
	  else
		  thePsi[2*offset+i] = sqrt(thePsi[i]);
  }
*/
  //OVERFIT
  //for (int i=0; i<psiLen; i++)
  //    thePsi[i] = 0.0;
  //for (int yi=0; yi<y->sentences.size(); yi++)
  //    thePsi[x->docid*0+y->sentences[yi]] = 1.0;

  for (int i=0; i<psiLen; i++)
  {
	  if (thePsi[i] < gcache.lo[i])
		  gcache.lo[i] = thePsi[i];
	  if (thePsi[i] > gcache.hi[i])
		  gcache.hi[i] = thePsi[i];
  }

  if (gcache.mm)
	  for (int i=0; i<psiLen; i++)
		  thePsi[i] /= 0.01 + fabs(gcache.maxpsi2[i]-gcache.minpsi2[i]);
	  
  fvec = create_svector_n(thePsi, psiLen, NULL, 1.0);

  /*
  printf("S: [%s]\n", list2str(y->sentences, ", "));
  SVECTOR* f = fvec; //psi(x, ybar, sm, sparm);
  WORD* wd = f->words;
  printf("Psi:  ");
  //for (int i=0; i<5 && wd[i].wnum; i++)
  for (int i=0; wd[i].wnum; i++)
      printf("%i:%.5f ", wd[i].wnum-3, wd[i].weight);
  printf("\n");
  exit(-4);
  */

  return(fvec);
}


double loss1(LABEL y, LABEL ybar)
{

#ifndef OPT_PRFv2
        //PRFv3-loss
        set<int> yset;
	int ysz = 0;
        set<int> ybset;
	int ybsz = 0;
        PATTERN x = gcache.sample.examples[y->docid].x;
        for (int yi=0; yi<(int)y->sentences.size(); yi++)
                for (map<int,int>::iterator it=x->sentences[y->sentences[yi]].doc_word_counts.begin(); it!=x->sentences[y->sentences[yi]].doc_word_counts.end(); it++)
		{ 
			yset.insert(it->first);
			ysz += it->second;
		}
        for (int yi=0; yi<(int)ybar->sentences.size(); yi++)
                for (map<int,int>::iterator it=x->sentences[ybar->sentences[yi]].doc_word_counts.begin(); it!=x->sentences[ybar->sentences[yi]].doc_word_counts.end(); it++)
		{
                        ybset.insert(it->first);
			ybsz += it->second;
		}
        int q = 0;
        for (set<int>::iterator it=yset.begin(); it!=yset.end(); it++)
                if (ybset.count(*it))
		{
			if (ybset.count(*it) > yset.count(*it))
                        	q += yset.count(*it);
			else
				q += ybset.count(*it);
		}
        if (ybset.size()==0 || yset.size()==0 || q==0)
                return 1;
        double p = (double)q / ybsz;
        double r = (double)q / ysz;
        double f = 2*p*r/(p+r);
        return max(0.0,1-f);

#else

	//PRFv2-loss
        set<int> yset;
        set<int> ybset;
        PATTERN x = gcache.sample.examples[y->docid].x;
        for (int yi=0; yi<y->sentences.size(); yi++)
                for (map<int,int>::iterator it=x->sentences[y->sentences[yi]].doc_word_counts.begin(); it!=x->sentences[y->sentences[yi]].doc_word_counts.end(); it++)
                        yset.insert(it->first);
        for (int yi=0; yi<ybar->sentences.size(); yi++)
                for (map<int,int>::iterator it=x->sentences[ybar->sentences[yi]].doc_word_counts.begin(); it!=x->sentences[ybar->sentences[yi]].doc_word_counts.end(); it++)
                        ybset.insert(it->first);
        int q = 0;
        for (set<int>::iterator it=yset.begin(); it!=yset.end(); it++)
                if (ybset.count(*it))
                        q++;
        if (ybset.size()==0 || yset.size()==0 || q==0)
                return 1;
        double p = (double)q / ybset.size();
        double r = (double)q / yset.size();
        double f = 2*p*r/(p+r);
        return max(0.0,1-f);

#endif

}

double      loss(LABEL y, LABEL ybar, STRUCT_LEARN_PARM *sparm)
{
#ifndef OPT_NGS
	//no ngs
	//TODO untested
	return loss1(y, ybar);
#endif

#ifndef OPT_NGS4
	//single ngs
	//TODO untested
	return loss1(y->Y[0], ybar);
#endif

#ifndef OPT_NGS4B
	//A-ver 4-ngs
	double lossMP = loss1(y, ybar);
	//return lossMP;
	return max(0.0, lossMP - y->lossMN);

#else
	//B-ver 4-ngs
        double score = 0;
        for (int i=0; i<4; i++)
        	score += loss1(&y->Y[i], ybar);
        	//score = max(score,loss(&Y[i], ybar, sparm));
        score /= 4;
	//return score;
	return max(0.0, score - y->lossMN);
#endif //OPT_NGS4B
}

double      loss2(LABEL y, LABEL ybar, int sel)
	//no min clamp loss
{
double lossMP, score;
switch (sel) {
	case 1:
	lossMP = loss1(y, ybar);
	return lossMP;
	//return max(0.0, lossMP - y->lossMN);
	
	case 2:
        score = 0;
        for (int i=0; i<4; i++)
        	score += loss1(&y->Y[i], ybar);
        	//score = max(score,loss(&Y[i], ybar, sparm));
        score /= 4;
	return score;
	//return max(0.0, score - y->lossMN);
	
	default:
	myassert(0);
	return 0.0;
	}
}


int         finalize_iteration(double ceps, int cached_constraint,
			       SAMPLE sample, STRUCTMODEL *sm,
			       CONSTSET cset, double *alpha, 
			       STRUCT_LEARN_PARM *sparm)
{
  /* This function is called just before the end of each cutting plane iteration. ceps is the amount by which the most violated constraint found in the current iteration was violated. cached_constraint is true if the added constraint was constructed from the cache. If the return value is FALSE, then the algorithm is allowed to terminate. If it is TRUE, the algorithm will keep iterating even if the desired precision sparm->epsilon is already reached. */
  return(0);
}

void        print_struct_learning_stats(SAMPLE sample, STRUCTMODEL *sm,
					CONSTSET cset, double *alpha, 
					STRUCT_LEARN_PARM *sparm)
{
  /* This function is called after training and allows final touches to
     the model sm. But primarly it allows computing and printing any
     kind of statistic (e.g. training error) you might want. */
}

void        print_struct_testing_stats(SAMPLE sample, STRUCTMODEL *sm,
				       STRUCT_LEARN_PARM *sparm, 
				       STRUCT_TEST_STATS *teststats)
{
  /* This function is called after making all test predictions in
     svm_struct_classify and allows computing and printing any kind of
     evaluation (e.g. precision/recall) you might want. You can use
     the function eval_prediction to accumulate the necessary
     statistics for each prediction. */
}

void        eval_prediction(long exnum, EXAMPLE ex, LABEL ypred, 
			    STRUCTMODEL *sm, STRUCT_LEARN_PARM *sparm, 
			    STRUCT_TEST_STATS *teststats)
{
  /* This function allows you to accumlate statistic for how well the
     predicition matches the labeled example. It is called from
     svm_struct_classify. See also the function
     print_struct_testing_stats. */
  if(exnum == 0) { /* this is the first time the function is
		      called. So initialize the teststats */
  }
}

void        write_struct_model(char *file, STRUCTMODEL *sm, 
			       STRUCT_LEARN_PARM *sparm)
{
  /* Writes structural model sm to file file. */
    write_model(file, sm->svm_model);
    //save psiminmax
    FILE* f = fopen("psiminmax2", "w");
    for (int i=0; i<psiLen; i++)
        fprintf(f, "%f %f\n", gcache.lo[i], gcache.hi[i]);
    fclose(f);
    //save wgts
    char wfn[1024];
    strcpy(wfn, file);
    strcat(wfn, ".wgts");
    f = fopen(wfn, "w");
    for (int i=0; i<psiLen; i++)
        fprintf(f, "%i %f\n", i, sm->w[i]);
    fclose(f);
}

STRUCTMODEL read_struct_model(char *file, STRUCT_LEARN_PARM *sparm)
{
  /* Reads structural model sm from file file. This function is used
     only in the prediction module, not in the learning module. */
	STRUCTMODEL s;
	s.svm_model = read_model(file);
	return s;
}

void        write_label(FILE *fp, LABEL y)
{
  /* Writes label y to file handle fp. */
    fprintf(fp, "%s\n", list2str(y->sentences, (char*)" "));
} 

void        free_pattern(PATTERN x) {
  /* Frees the memory of x. */
}

void        free_label(LABEL y) {
  /* Frees the memory of y. */
}

void        free_struct_model(STRUCTMODEL sm) 
{
  /* Frees the memory of model. */
  /* if(sm.w) free(sm.w); */ /* this is free'd in free_model */
  if(sm.svm_model) free_model(sm.svm_model,1);
  /* add free calls for user defined data here */
}

void        free_struct_sample(SAMPLE s)
{
  /* Frees the memory of sample s. */
  int i;
  for(i=0;i<s.n;i++) { 
    free_pattern(s.examples[i].x);
    free_label(s.examples[i].y);
  }
  free(s.examples);
}

void        print_struct_help()
{
  /* Prints a help text that is appended to the common help text of
     svm_struct_learn. */
  printf("         --* string  -> custom parameters that can be adapted for struct\n");
  printf("                        learning. The * can be replaced by any character\n");
  printf("                        and there can be multiple options starting with --.\n");
}

void         parse_struct_parameters(STRUCT_LEARN_PARM *sparm)
{
  /* Parses the command line parameters that start with -- */
  int i;

  for(i=0;(i<sparm->custom_argc) && ((sparm->custom_argv[i])[0] == '-');i++) {
    switch ((sparm->custom_argv[i])[2]) 
      { 
      case 'a': i++; /* strcpy(learn_parm->alphafile,argv[i]); */ break;
      case 'e': i++; /* sparm->epsilon=atof(sparm->custom_argv[i]); */ break;
      case 'k': i++; /* sparm->newconstretrain=atol(sparm->custom_argv[i]); */ break;
      default: printf("\nUnrecognized option %s!\n\n",sparm->custom_argv[i]);
	       exit(0);
      }
  }
}

void        print_struct_help_classify()
{
  /* Prints a help text that is appended to the common help text of
     svm_struct_classify. */
  printf("         --* string -> custom parameters that can be adapted for struct\n");
  printf("                       learning. The * can be replaced by any character\n");
  printf("                       and there can be multiple options starting with --.\n");
}

void         parse_struct_parameters_classify(STRUCT_LEARN_PARM *sparm)
{
  /* Parses the command line parameters that start with -- for the
     classification module */
  int i;

  for(i=0;(i<sparm->custom_argc) && ((sparm->custom_argv[i])[0] == '-');i++) {
    switch ((sparm->custom_argv[i])[2]) 
      { 
      /* case 'x': i++; strcpy(xvalue,sparm->custom_argv[i]); break; */
      default: printf("\nUnrecognized option %s!\n\n",sparm->custom_argv[i]);
	       exit(0);
      }
  }
}



