/***

femat_sparse -- sparse matrix assembly for MATLAB
Copyright (c) 2008
David Bindel

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

****/

/* @T
 * The [[Reassembler]] class accumulates element contributions into an
 * existing compressed sparse column matrix data structure.  Elements
 * that don't fit into the existing index structure are silently ignored.
 *
 * @q */

#include <vector>
#include <algorithm>


class Reassembler {
public:
    template<class intT>
    Reassembler(intT* jc1, intT* ir1, int m, int n) :
        jc(n+1), ir(jc1[n]), pr(jc1[n]), m(m), n(n) {
        std::copy(jc1, jc1+n+1,   jc.begin());
        std::copy(ir1, ir1+jc[n], ir.begin());
        clear();
    }

    void clear() {
        std::fill(pr.begin(), pr.end(), 0);
    }

    int get_m()   { return m;     }
    int get_n()   { return n;     }
    int get_nnz() { return jc[n]; }

    int find_row(int klo, int khi, int i);
    void add_entries(int* i, int* j, double* A, int m, int n,
                     int index_base = 0);

    template<class intT>
    void fill_csc(intT* jc, intT* ir, double* pr);

private:
    std::vector<int> jc;
    std::vector<int> ir;
    std::vector<double> pr;
    int m, n;
};


int Reassembler::find_row(int klo, int khi, int i)
{
    while (klo < khi) {
        int kmid = klo + (khi-klo)/2;
        int imid = ir[kmid];
        if (imid < i)
            klo = kmid+1;
        else if (imid > i)
            khi = kmid;
        else
            return kmid;
    }
    return -1;
}


void Reassembler::add_entries(int* iAe, int* jAe, double* Ae, int mAe, int nAe,
                              int index_base)
{
    for (int jj = 0; jj < nAe; ++jj, Ae += mAe) {
        int j = jAe[jj]-index_base;
        int col_start = jc[j];
        int col_end   = jc[j+1];
        if (j >= 0 && j < n) {
            for (int ii = 0; ii < mAe; ++ii) {
                int i = iAe[ii]-index_base;
                int k = find_row(col_start, col_end, i);
                if (k >= 0)
                    pr[k] += Ae[ii];
                else 
                    mexPrintf("Failed element update (%d, %d)\n", i, j);
            }
        }
    }
}


template<class intT>
void Reassembler::fill_csc(intT* jc_out, intT* ir_out, double* pr_out)
{
    std::copy(jc.begin(), jc.end(), jc_out);
    std::copy(ir.begin(), ir.end(), ir_out);
    std::copy(pr.begin(), pr.end(), pr_out);
}
