#!/bin/sh
# \
exec wish "$0" ${1+"$@"}

if {$tcl_version < 7.6} {
    puts stderr "requires Tcl 7.6 or later"
    exit
}

set AppName "TED"

# ted.tcl --
#
#   The Tcl Extension Kit/Installer.  This file creates "tek" (teki) files for Tcl/Tk packages


# Notes about the organization of this file:
#   Major sections are delimited by a string of "-" characters -- e.g. "-------"
#   At the beginning of each major section,the purpose and routines in that
#   section are described.
#   ted-specific packages reside in the "tekilib" directory below teki.tcl

set home [string trimright [file dirname [info script]] ./]
set home [file join [pwd] $home tekilib]
lappend auto_path $home
set TekiInfo(library) $home
unset home

package require Debug
package require Undo

# -----------------------------------------------------------------------
# Help Messages

set HelpMessages {\
\
"The base directory specifies the relative position of all the selected files. The base directory \
is excluded from the files added to the bundle, i.e, the directory \
structure is maintained exclusive of the Base Directory.\
\n\nFor Example: If tcl files are stored in /tcl-dp/foo and the base directory is /tcl-dp\
then the files added from the tcl directory will be added as foo/???? and the directory\
 structure after foo will be maintained." \
 \
"\nPackage Name: A human readable name for the package.  \
For example, \"Tcl-DP\" or \"Bob's FTP Package\" \
\n\nVersion: Package Version (e.g., 4.0) \
\nDescription: A one line description for the package. \
\n\nRequires: It is a list of {package version} pairs indicating which other packages are required. It is not\
 used currently, but it could be used later to auto-load the required packages. \
\n\nUpdate URL: It is a URL where package updates can be found.  It is used by the Teki \"update\" command\
 to automatically install new versions of a package. \
\n\nRegister URL: It is a URL where users can go to fill out a registration form. It's currently not used. \
\n\nSource URL: \"Source URL\" and \"Source Directory\" are the base directories for finding the files to\
 copy for an over the web install.  The Source URL is the base URL where all the documentation,\
 tcl, binary, data, and example files can be found -- it is prepended to every file in the package\
 for web based installs.\
\n\nBrowse URL: The Browse URL gives the URL address for .tkc, or the Table of Contents File, \
used for Web based installs."\
\
"Select Following Files and Directories: \
\n\nSource Directory: Location of package files with respect to the base directory. \
If you have only one package in the bundle file this directory is usually \".\" \
\n\nCopyright File: File containing copyright information (e.g., license.txt) \
\n\nInformation File: File containing other package related information (e.g., readme.txt) \
\n\nDocument Directory: The directory where the document files are located (e.g., doc)" \
 \
"You can associate object files with system names. \
\nObject Files are a list of architecture dependent files" \
\
"System Name: Name of the operating system ,eg.,HPUX,WIN95/NT \
\nOperating Sys.: A short id. which is usually associated with the system,eg.,HP*,Win* \
\nOS Version No.: All supported Versions,eg.,9*,* respectively\
\n\nThe OS and OS version are determined at runtime by matching (glob-style) the OS and OS\
 version in the system definition to the tcl_platform(os) and tcl_platform(osVersion)\
 variables." \
 \
"These are the default architectures for which the install takes place if the user selects\
 the \"typical\" install option (as opposed to \"custom\" installation)." \
 \
"Specify if the Document/Example/Data Files are installed if the user selects\
 the \"typical\" install option (as opposed to \"custom\" installation)."\
\
"The package files are the Tcl, Document, Example and Data files associated with a package.\
\nDouble clicking on the file browser adds the file while double clicking on a selected file\
 removes it.\
\n\nBase Directory: The base directory specifies the relative location of the respective package\
 files. This path is removed from the file path before adding it to the .tek file. For example\
 if you are creating the .tek file for Tcl-DP Version 4.0 then you should have copied all the\
 files related to the package to a directory dp4.0, now the base directory becomes the path to\
 dp4.0, inclusive of dp4.0\
 \n\nNOTE: You can always change the base directory. The base directory is just to inform Ted the\
 path to exclude. But do remember that because the directory structure is preserved at the time\
 of installation TEKI will look for the files in the specifed file path relative to the specified\
 source directory."\
\
"System Names is a list of system definitions.  These names are used to identify\
 architectural dependencies at runtime. Each definition is a three-tuple\
 consisting of the system name, the operating system (OS), and the OS version number.\
 The OS and OS version are determined at runtime by matching (glob-style) the OS and\
 OS version in the system definition to the tcl_platform(os) and tcl_platform(osVersion)\
 variables."\
\
"These are the default packages which will be installed if the user selects\
 the \"typical\" install option (as opposed to \"custom\" installation)."\
\
"A bundle group of packages that can be installed with one .tek file."\
\
"An application may require or may consist of many different individual packages,\
 like libraries etc. to be installed before it can run. The .tek file can specify\
 settings for multiple packages which are bundled togather installed at runtime."\
\
"\"Source URL\" and \"Source Directory\" are the base directories for finding the\
 files to copy for an over the web install.  The Source URL is the base URL where\
 all the documentation, tcl, binary, data, and example files can be found -- it is\
 prepended to every file in the package for web based installs."}
# -----------------------------------------------------------------------

# Define some useful fonts

if {[info exists tk_version] == 0} {
	puts "Error: Unable to run $AppName, TK not found !"
	return
} else {
	set TedInfo(gui) 1
    package require Wise
    set majorversion [lindex [split $tk_version .] 0]
    if {$majorversion >= 8} {
        set TedInfo(varfont) [font create -family Times -weight bold -size 18]
        set TedInfo(fixedfont) [font create -family Courier -size 10]
    } else {
        set TedInfo(varfont) *-Times-Bold-R-Normal--*-240-*-*-*-*-*-*
        set TedInfo(fixedfont) -*-Courier-Medium-R-Normal--*-120-*-*-*-*-*-*
    }

	set TekiInfo(varfont) $TedInfo(varfont)
	set TekiInfo(fixedfont) $TedInfo(fixedfont)
    unset majorversion

	proc TedError {msg} {
	bgerror "$msg"
	}
	proc TedWarning {msg default options} {
		set rv [Wise_Message "Warning" $msg $options 0]
		return $rv
	}

	proc TedHelp {help {fromfile 0}} {
	global HelpMessages

		if {![catch {set st [wm state .wisemsg]}]} {
			if {$st == "normal"} {
				return
			}
		}

		if $fromfile {
		    if {[string length $help] && [file exists $help] && [file readable $help]} {
				set f [open $help r]
				set helptext ""

				while {[gets $f line] >= 0} {
					set helptext "$helptext\n$line"
				}

				close $f
			} else {
				set helptext "Unable to open help file <$help>"
			}
		} else {
			set helptext [format "[lindex $HelpMessages $help]"]
		}

		Wise_Message "Ted Help" $helptext OK 1
	}
}

# -----------------------------------------------------------------------
#
# User Interface
#
# This section of code creates and manages the user interface.

#
# TedCreateMenus
#    Create the menu bar across the top of the window
#
# Arguments:
#    None
#
proc TedCreateMenus {} {
	global tclPkgInfo TedInfo
	Debug_Enter TedCreateMenus

	set wmenu .menu
	menu $wmenu -tearoff 1

	# File Menu
	set m .menu.file
	menu $m -tearoff 0
	$wmenu add cascade -label "File" -menu $m -underline 0
	$m add command -label "New" -underline 0 -command TedDoFileNew
	$m add command -label "Open" -underline 0 -command TedDoFileOpen
	$m add command -label "Save" -underline 0 -command TedDoFileSave
	$m add command -label "Save As" -underline 5 -command TedDoFileSaveAs
	$m add command -label "Close" -underline 0 -command TedDoFileClose
	$m add separator
	$m add command -label "Exit" -underline 1 -command TedDoFileExit

	# Bundle Menu
	set m .menu.bundle
	menu $m -tearoff 0
	$wmenu add cascade -label "Bundle" -menu $m -underline 0
	$m add command -label "Add Package" -underline 0 -command "TedMainWinEventHandler addpkg"
	$m add command -label "Remove Package" -underline 0 -command "TedMainWinEventHandler rempkg"
	$m add separator
	$m add command -label "System Names" -underline 0 -command "TedMainWinEventHandler sysnames"
	$m add command -label "Default Installs" -underline 8 -command "TedMainWinEventHandler definstalls"
	$m add command -label "Default Arch." -underline 8 -command "TedMainWinEventHandler defarchs"
	$m add command -label "Default Packages" -underline 8 -command "TedMainWinEventHandler defpkgs"
	$m add separator
	$m add command -label "Defaults Wizard" -underline 9 -command "TedMainWinEventHandler defwiz"

	# Package Menu
	set m .menu.package
	menu $m -tearoff 0
	$wmenu add cascade -label "Package" -menu $m -underline 0
	$m add command -label "Base Directory" -underline 0 -command "TedMainWinEventHandler basedir"
	$m add command -label "Settings" -underline 0 -command "TedMainWinEventHandler pkgsettings"
	$m add command -label "Directories & Misc. Files" -underline 0 -command "TedMainWinEventHandler pkgdirs"
	$m add command -label "Files" -underline 0 -command "TedMainWinEventHandler pkgfiles"
	$m add command -label "Object Files" -underline 0 -command "TedMainWinEventHandler pkgobjfiles"
	$m add separator
	$m add command -label "Wizard" -underline 0 -command "TedMainWinEventHandler pkgwiz"

	# Help Menu
	set m .menu.help
	menu $m -tearoff 0
	$wmenu add cascade -label "Help" -menu $m -underline 0
	$m add command -label "What is Base Directory?" -underline 0 -command "TedHelp 0"
	$m add command -label "What is a Bundle?" -underline 10 -command "TedHelp 10"
	$m add command -label "What is a Package?" -underline 10 -command "TedHelp 11"
	$m add command -label "What is the Source URL?" -underline 12 -command "TedHelp 12"
	$m add separator
	$m add command -label "TEKI File Format" -underline 12 -command "TedHelp tek-file-format.txt 1"
	$m add command -label "Help" -underline 0 -command "TedDisplayInitHelp ted-readme 1"

	.menu.bundle entryconfigure "Remove Package" -state disabled
	.menu entryconfigure Package -state disabled

	. configure -menu $wmenu

	Debug_Leave
}

#
# TedCreateMainWindow
#    Create the main window -- this window allows a user to add packages to the current project
#
# Arguments:
#    none
#

proc TedCreateMainWindow {} {
	global tclPkgInfo TedInfo TedPkgInfo
	Debug_Enter TedCreateMainWindow

	set w .packages
	set TedInfo(pkgWin) $w
	frame $w
	pack $w -side left -fill both -expand yes

	# Define the left section of the main window
	set w1 $w.leftFrame
	frame $w1
	pack $w1 -side left -fill both -expand no

	# Define the Bundle information section
	set w11 $w1.bundleInfoWin
	frame $w11
	pack $w11 -side top -fill both -expand yes
	
	label $w11.lbl -text "Bundle Information" -anchor w
	pack $w11.lbl -side top -fill x -expand no

    scrollbar $w11.scroll -command "$w11.text yview"
    text $w11.text -relief sunken -bd 2 -yscrollcommand "$w11.scroll set" \
						-setgrid 1 -height 7 -width 30
	pack $w11.scroll -side right -fill y -expand no
	pack $w11.text -side left -fill both -expand yes

	# Define Packages Listbox Frame Window
	set w12 $w1.pkgListFrame
	frame $w12
	pack $w12 -side bottom -fill both -expand yes

	label $w12.lbl -text "Packages" -anchor w
	pack $w12.lbl -side top -fill x -expand no

	set w121 $w12.pkgBtnWin
	frame $w121
	pack $w121 -side bottom -fill x -expand no
	set TedInfo(pkgIdEntry) ""
	button $w121.addPkg -width 10 -padx 1 -pady 1 -text "Add Package" -command {TedMainWinEventHandler addpkg}
	button $w121.remPkg -width 10 -padx 1 -pady 1 -text "Remove Package" -command {TedMainWinEventHandler rempkg}
	pack $w121.addPkg $w121.remPkg -side left -fill x -expand yes

	scrollbar $w12.scrollPkg -command "$w12.listPkg yview"
	listbox $w12.listPkg -font $TedInfo(fixedfont) -relief sunken -width 30 \
				 -bd 2 -yscrollcommand "$w12.scrollPkg set" -setgrid 1 -height 8 \
				 -selectmode single
	pack $w12.scrollPkg -side right -fill y -expand no
	pack $w12.listPkg -side left -fill both -expand yes

	# Create the Package's basic information section
	set w2 $w.pkgInfoWin
	frame $w2
	pack $w2 -side right -fill both -expand yes

	label $w2.lbl -text "Package Information" -anchor w
	pack $w2.lbl -side top -fill x -expand no

    scrollbar $w2.scroll -command "$w2.text yview"
    text $w2.text -relief sunken -bd 2 -yscrollcommand "$w2.scroll set" \
						-setgrid 1 -height 15 -width 40
	pack $w2.scroll -side right -fill y -expand no
	pack $w2.text -side left -fill both -expand yes

    bind $w12.listPkg <ButtonRelease-1> {TedUpdatePkgInfo}
	TedUpdateBundleInfo
	TedUpdatePkgInfo
	TedDisplayInitHelp "ted-readme"

	Debug_Leave
}

# Function to display the initial help screen

proc TedDisplayInitHelp {filename {fflag 0}} {
	global dontshowhelp

	Debug_Enter TedDisplayInitHelp
	set inifile "ted.ini"
	set dontshowhelp 0

	if {[file exists $inifile] && [file readable $inifile]} {
		set f [open $inifile r]
		set line [split [gets $f]]
		close $f

		if {[lindex $line 0] == "ShowHelp" && [lindex $line 1] == "=" && [lindex $line 2] == 0} {
			set dontshowhelp 1

			if {$fflag == 0} {
				Debug_Leave
				return
			}
		}
	}


	# Wait for the splash screen to be destroyed
	catch {tkwait window .about-Ted}

    toplevel .inihelp
    grid rowconfigure .inihelp 0 -weight 1
    WiseCreateLogo
    label .inihelp.logo -image logoImage
    scrollbar .inihelp.scroll -command ".inihelp.text yview"
    text .inihelp.text -relief sunken -bd 2 -yscrollcommand ".inihelp.scroll set" -setgrid 1 -height 10 -width 60
	button .inihelp.bok -text OK -command "set doneMsg 1" -width 8
	checkbutton .inihelp.shelp -variable dontshowhelp -text "Do not show this screen on startup" -anchor w
    grid .inihelp.text -column 1 -row 0 -columnspan 2 -sticky nesw
    grid .inihelp.scroll -column 3 -row 0 -sticky ns
	grid .inihelp.shelp -row 1 -column 1 -sticky nesw
	grid .inihelp.bok -row 2 -column 1
    grid .inihelp.logo -row 0 -column 0 -sticky n
	wm resizable .inihelp 0 0
    wm title .inihelp "TED - Help"

	if {[file exists $filename] && [file readable $filename]} {
		set f [open $filename r]
		set helptext ""

		while {[gets $f line] >= 0} {
			set helptext "$helptext\n$line"
		}

		close $f
	} else {
		set helptext "Unable to open help file <$filename>"
	}

    .inihelp.text insert end $helptext
	Wise_CenterWindow .inihelp

    set doneMsg 0
    vwait doneMsg
	set f [open $inifile w]

	if {$f != ""} {
		puts $f "ShowHelp = [expr $dontshowhelp == 0]"
		close $f
	}

	destroy .inihelp
	unset dontshowhelp
	Debug_Leave
}

proc TedUpdateBundleInfo {} {
	global TedInfo TedPkgInfo

	Debug_Enter TedUpdateBundleInfo
	set w $TedInfo(pkgWin)
	set w1 $w.leftFrame.bundleInfoWin
	set w2 $w.leftFrame.pkgListFrame.pkgBtnWin
    $w1.text delete 1.0 end

	if {!$TedInfo(fileOpen)} {
		$w2.addPkg configure -state disabled
		$w2.remPkg configure -state disabled
		.menu entryconfigure "Bundle" -state disabled
		.menu entryconfigure "Package" -state disabled
		Debug_Leave
		return
	}

	expr {$TedPkgInfo(defaultInstallDoc) ? [set defdocs YES]:[set defdocs  NO]}
	expr {$TedPkgInfo(defaultInstallExamples) ? [set defexamples YES]:[set defexamples NO]}
	expr {$TedPkgInfo(defaultInstallData) ? [set defdata YES]:[set defdata NO]}

	set bundleInfo "Bundle teki File = $TedPkgInfo(tekiFile)\
	\nSystem Names = "

	set sysnames "\n"

	foreach pkg $TedPkgInfo(systemNames) {
		set sysnames "$sysnames\n\t\t$pkg"
	}

	set bundleInfo "$bundleInfo $sysnames\
	\nPackages Available = $TedPkgInfo(available)\
	\nDefault Packages = $TedPkgInfo(defaultPackages)\
	\nDefault Architecture = $TedPkgInfo(defaultArch)\
	\nDefault Install Documentation = $defdocs\
	\nDefault Install Examples = $defexamples\
	\nDefault Install Data = $defdata"
    $w1.text insert end $bundleInfo

	$w2.addPkg configure -state normal
	$w2.remPkg configure -state normal
	.menu entryconfigure Bundle -state normal
	.menu entryconfigure Package -state normal
	Debug_Leave
}

proc TedUpdatePkgInfo {} {
	global TedInfo TedPkgInfo

	Debug_Enter TedUpdatePkgInfo
	set w $TedInfo(pkgWin)
	set w1 $w.leftFrame.pkgListFrame.listPkg
	set w2 $w.leftFrame.pkgListFrame.pkgBtnWin
    $w.pkgInfoWin.text delete 1.0 end

	if {!$TedInfo(fileOpen)} {
		return
	}

	set selid [$w1 curselection]

	if {$selid == ""} {
		$w2.remPkg configure -state disabled
		.menu.bundle entryconfigure "Remove Package" -state disabled
		.menu entryconfigure "Package" -state disabled
		Debug_Leave
		return
	}

	set selPkg [$w1 get $selid]

	if {$TedPkgInfo($selPkg,BaseDir) == ""} {
		.menu.package entryconfigure "Directories & Misc. Files" -state disabled
		.menu.package entryconfigure "Files" -state disabled
		.menu.package entryconfigure "Object Files" -state disabled
		.menu.package entryconfigure "Wizard" -state disabled
	} else {
		.menu.package entryconfigure "Directories & Misc. Files" -state normal
		.menu.package entryconfigure "Files" -state normal
		.menu.package entryconfigure "Object Files" -state normal
		.menu.package entryconfigure "Wizard" -state normal
	}

	set tclFiles ""
	
	foreach file $TedPkgInfo($selPkg,tclFiles) {
		set tclFiles "$tclFiles\n\t\t$file"
	}

	set exampleFiles ""
	
	foreach file $TedPkgInfo($selPkg,exampleFiles) {
		set exampleFiles "$exampleFiles\n\t\t$file"
	}

	set dataFiles ""
	
	foreach file $TedPkgInfo($selPkg,dataFiles) {
		set dataFiles "$dataFiles\n\t\t$file"
	}

	set docFiles ""
	
	foreach file $TedPkgInfo($selPkg,docFiles) {
		set docFiles "$docFiles\n\t\t$file"
	}

	set objFiles ""
	
	foreach file $TedPkgInfo($selPkg,objFiles) {
		set objFiles "$objFiles\n\t\t$file"
	}

	set pkgInfo "Package Id. = $selPkg\
	\nPackage Name = $TedPkgInfo($selPkg,name)\
	\nVersion =  $TedPkgInfo($selPkg,version)\
	\nDescription =  $TedPkgInfo($selPkg,description)\
	\nRequires =  $TedPkgInfo($selPkg,requires)\
	\nTeki File =  $TedPkgInfo($selPkg,tekiFile)\
	\n\nUpdate URL =  $TedPkgInfo($selPkg,updateURL)\
	\n\nRegister URL =  $TedPkgInfo($selPkg,registerURL)\
	\n\nSource URL =  $TedPkgInfo($selPkg,srcURL)\
	\n\nTcl Files = $tclFiles\
	\n\nExample Files = $exampleFiles\
	\n\nData Files = $dataFiles\
	\n\nDocument Dir. = $TedPkgInfo($selPkg,docDir)\
	\nDocument Files = $docFiles\
	\n\nObject Files = $objFiles"

    $w.pkgInfoWin.text insert end $pkgInfo

	$w2.remPkg configure -state normal
	.menu.bundle entryconfigure "Remove Package" -state normal
	.menu entryconfigure Package -state normal
	Debug_Leave
}

proc TedUpdatePkgList {} {
	global TedInfo TedPkgInfo
	
	Debug_Enter TedUpdatePkgList
	set w $TedInfo(pkgWin)
	set w1 $w.leftFrame.pkgListFrame.listPkg
	set w2 $w.leftFrame.pkgListFrame.pkgBtnWin

	$w1 delete 0 end

	if {!$TedInfo(fileOpen)} {
		TedUpdateBundleInfo
		TedUpdatePkgInfo
		Debug_Leave
		return
	}


	if {$TedPkgInfo(available) != ""} {
		foreach pkg $TedPkgInfo(available) {
			$w1 insert end $pkg
		}

		$w1 selection set 0 0
	} else {
	}

	TedUpdateBundleInfo
	TedUpdatePkgInfo
	Debug_Leave
}

proc TedMainWinEventHandler {event} {
	global TedInfo TedPkgInfo tmpPkgInfo

	Debug_Enter TedMainWinEventHandler
	set w $TedInfo(pkgWin)
	set w1 $w.leftFrame.pkgListFrame.listPkg

	# Get the current selected package

	set selId [$w1 curselection]

	if {$selId != ""} {
		set selPkg [$w1 get $selId]
	}

	switch $event {
		addpkg {
			set TedInfo(pkgIdEntry) [TedGetPkgId]

			if {$TedInfo(pkgIdEntry) != ""} {
				set TedInfo(fileSaved) 0

				foreach pkg $TedPkgInfo(available) {
					if {$TedInfo(pkgIdEntry) == $pkg} {
						TedWarning "Package Id. should be unique !\
						\nPackage Id.: $TedInfo(pkgIdEntry)\ already exists." 0 OK
						Debug_Leave
						return
					}
				}

				lappend TedPkgInfo(available) $TedInfo(pkgIdEntry)
				set TedPkgInfo($TedInfo(pkgIdEntry),destDir) $TedInfo(pkgIdEntry)
				TedInitPkgInfo $TedInfo(pkgIdEntry)
				set $TedInfo(pkgIdEntry) ""
				TedUpdatePkgList
			}			
		}

		rempkg {
			if {[TedWarning "This will delete all information related to this package !" 0 {Cancel OK}]} {
				set TedInfo(fileSaved) 0
				set newlist ""

				foreach pkg $TedPkgInfo(available) {
					if {$pkg != $selPkg} {
						lappend newlist $pkg
					}
				}

				set TedPkgInfo(available) $newlist
				TedDeletePkgInfo $selPkg
				TedUpdatePkgList
			}
		}

		pkgwiz {
			set rv [TedPackageWizard $selPkg]
			if {$rv == "finish" || $rv == "next"} {
				set TedInfo(fileSaved) 0
			}
		}

		defwiz {
			TedDefaultWizard
		}

		sysnames {
			TedInitTmpInfo

			# Create copies of information for all packages
			foreach pkg $tmpPkgInfo(available) {
				TedInitTmpPkgInfo $pkg
			}

			.wise.next configure -state disabled
			.wise.back configure -state disabled

			set rv [TedGetBundleSystemNames]

			if {$rv == "finish"} {
				set TedInfo(fileSaved) 0
				TedUpdateTedPkgInfo

				# Update information for all packages
				foreach pkg $tmpPkgInfo(available) {
					TedCopyTmpToTedPkgInfo $pkg
				}
			}

			wm withdraw .wise
			.wise.next configure -state normal
			.wise.back configure -state normal
		}

		defarchs {
			TedInitTmpInfo
			.wise.next configure -state disabled
			.wise.back configure -state disabled

			set rv [TedGetBundleDefaultArch]

			if {$rv == "finish"} {
				set TedInfo(fileSaved) 0
				TedUpdateTedPkgInfo
			}

			wm withdraw .wise
			.wise.next configure -state normal
			.wise.back configure -state normal
		}

		definstalls {
			TedInitTmpInfo
			.wise.next configure -state disabled
			.wise.back configure -state disabled

			set rv [TedGetBundleDefInstallDocExDat]

			if {$rv == "finish"} {
				set TedInfo(fileSaved) 0
				TedUpdateTedPkgInfo
			}

			wm withdraw .wise
			.wise.next configure -state normal
			.wise.back configure -state normal
		}

		defpkgs {
			TedInitTmpInfo
			.wise.next configure -state disabled
			.wise.back configure -state disabled

			set rv [TedGetBundleDefaultPackages]

			if {$rv == "finish"} {
				set TedInfo(fileSaved) 0
				TedUpdateTedPkgInfo
			}

			wm withdraw .wise
			.wise.next configure -state normal
			.wise.back configure -state normal
		}

		basedir {
			set tmpdir [Wise_GetDirName $TedPkgInfo($selPkg,BaseDir) "Select Base Directory" "TedHelp 0"]

			if {$tmpdir != ""} {
				set TedPkgInfo($selPkg,BaseDir) $tmpdir
			}

			TedUpdatePkgInfo
		}

		pkgsettings {
			TedInitTmpInfo
			TedInitTmpPkgInfo $selPkg
			.wise.next configure -state disabled
			.wise.back configure -state disabled

			set rv [TedGetPkgSettings $selPkg]

			if {$rv == "finish"} {
				set TedInfo(fileSaved) 0
				TedUpdateTedPkgInfo
				TedCopyTmpToTedPkgInfo $selPkg
			}

			wm withdraw .wise
			.wise.next configure -state normal
			.wise.back configure -state normal
		}

		pkgfiles {
			TedInitTmpInfo
			TedInitTmpPkgInfo $selPkg
			.wise.next configure -state disabled
			.wise.back configure -state disabled

			set rv [TedGetPkgFiles $selPkg]

			if {$rv == "finish"} {
				set TedInfo(fileSaved) 0
				TedUpdateTedPkgInfo
				TedCopyTmpToTedPkgInfo $selPkg
			}

			wm withdraw .wise
			.wise.next configure -state normal
			.wise.back configure -state normal
		}

		pkgobjfiles {
			TedInitTmpInfo
			TedInitTmpPkgInfo $selPkg
			.wise.next configure -state disabled
			.wise.back configure -state disabled

			set rv [TedGetPkgObjFiles $selPkg]

			if {$rv == "finish"} {
				set TedInfo(fileSaved) 0
				TedUpdateTedPkgInfo
				TedCopyTmpToTedPkgInfo $selPkg
			}

			wm withdraw .wise
			.wise.next configure -state normal
			.wise.back configure -state normal
		}

		pkgdirs {
			TedInitTmpInfo
			TedInitTmpPkgInfo $selPkg
			.wise.next configure -state disabled
			.wise.back configure -state disabled

			set rv [TedGetPkgDirOptions $selPkg]

			if {$rv == "finish"} {
				set TedInfo(fileSaved) 0
				TedUpdateTedPkgInfo
				TedCopyTmpToTedPkgInfo $selPkg
			}

			wm withdraw .wise
			.wise.next configure -state normal
			.wise.back configure -state normal
		}

		help {
			TedHelp "readme" 1
		}
	}

	TedUpdateBundleInfo
	TedUpdatePkgInfo
	Debug_Leave
}

# Dialog box for getting a new package id.

# Function to Create Dialog box

proc TedCreatePkgDialog {} {
	global PkgDialog tmpPkgInfo

	set PkgDialog ".pkgDlg"
	set w $PkgDialog
	toplevel $w
	wm title $w "Package Id."
	Wise_CenterWindow $PkgDialog
    grid rowconfigure $w 1 -weight 1

	label $w.l_pkgId -text "Package Id." -anchor w -width 15
	entry $w.e_pkgId -width 22 -relief sunken -textvariable tmpPkgInfo(pkgId)
	label $w.help -text "Package Id. is an unique identifier, eg. dp4.0" -anchor w -width 35
	button $w.ok -text "   OK   " -command "PkgDlgEventHandler ok"
	button $w.cancel -text " Cancel " -command "PkgDlgEventHandler cancel"

	grid $w.l_pkgId -row 0 -column 0
	grid $w.e_pkgId  -row 0 -column 1 -columnspan 2
	grid $w.help -row 1 -column 0 -columnspan 3 -sticky w
	grid $w.ok -row 2 -column 0
	grid $w.cancel -row 2 -column 2
}

proc PkgDlgEventHandler {event} {
	global PkgDialog tmpPkgInfo

	switch $event {
		ok {
			destroy $PkgDialog
			unset PkgDialog
		}

		cancel {
			set tmpPkgInfo(pkgId) ""
			destroy $PkgDialog
			unset PkgDialog
		}
	}
}

proc TedGetPkgId {} {
	global PkgDialog tmpPkgInfo

	set tmpPkgInfo(pkgId) ""
	TedCreatePkgDialog
	grab set $PkgDialog
	tkwait window $PkgDialog
	return $tmpPkgInfo(pkgId)
}

#------------------------------------------------------
# File Menu Option Handlers

# New File menu option handler

proc TedDoFileNew {} {
	global TedInfo TedPkgInfo

	Debug_Enter TedDoFileNew

	# Close previously open file

	if {$TedInfo(fileOpen)} {
		if {![TedDoFileClose]} {
			Debug_Leave
			return
		}
	}

	if [info exists TedPkgInfo] {
		unset TedPkgInfo
	}

	if [info exists tmpPkgInfo] {
		unset tmpPkgInfo
	}

	TedInitPkgIndependentInfo
	set TedInfo(fileOpen) 1

    set currPkg $TedPkgInfo(tekiFile)
    wm title . "TED -- $currPkg"
    wm iconname . "TED\[$currPkg\]"
	TedUpdatePkgList

	# Enable menu options
	.menu.file entryconfigure "Save" -state normal
	.menu.file entryconfigure "Save As" -state normal
	.menu.file entryconfigure "Close" -state normal

	DebugLeave
}

# File Open menu option handler

proc TedDoFileOpen {} {
	global TedInfo TedPkgInfo tcl_platform

	Debug_Enter TedDoFileOpen

#	Get the teki install file
    set filename [tk_getOpenFile -title "Open TEKI file" -filetypes {{{Tek Files} .tek} {{All Files} *}}]

	if {$filename == ""} {
		Debug_Leave
        return;
    }

	if {![TedDoFileClose]} {
		Debug_Leave
		return
	}

	if {$tcl_platform(platform) == "windows"} {
		set filename [string tolower $filename]
	}

	set TedPkgInfo(pkgFilename) $filename
	set TedPkgInfo(tekiFile) [file tail $TedPkgInfo(pkgFilename)]
	if {![TedReadVersion1.0 $TedPkgInfo(pkgFilename)]} {
		TedWarning "Unable to open TEKI File $filename !" 0 OK
		Debug_Leave
		return
	}

	catch {cd [file dirname $TedPkgInfo(pkgFilename)]}
	TedInitBaseDir [file dirname $TedPkgInfo(pkgFilename)]

    set currPkg $TedPkgInfo(tekiFile)
    wm title . "TED -- $currPkg"
    wm iconname . "TED\[$currPkg\]"
	set TedInfo(fileOpen) 1

	# Enable menu options
	.menu.file entryconfigure "Save" -state normal
	.menu.file entryconfigure "Save As" -state normal
	.menu.file entryconfigure "Close" -state normal

	TedUpdatePkgList
	Debug_Leave
}

# File Save menu option handler

proc TedDoFileSave {} {
	global TedPkgInfo TedInfo

	Debug_Enter TedDoFileSave
	if {$TedPkgInfo(pkgFilename) == ""} {
		TedDoFileSaveAs
		Debug_Leave
		return
	} else {
		TedWriteVersion1.0 $TedPkgInfo(pkgFilename)
	}

	set TedInfo(fileSaved) 1
	Debug_Leave
}

# File Save As menu option handler

proc TedDoFileSaveAs {} {
	global TedPkgInfo TedInfo

	Debug_Enter TedDoFileSaveAs
#	Get the teki install file
    set filename [tk_getSaveFile -title "Save As TEKI file" -defaultextension tek -initialfile $TedPkgInfo(pkgFilename) -filetypes {{{Tek Files} .tek} {{All Files} *}}]

	if {$filename != ""} {
		set TedPkgInfo(pkgFilename) $filename
		set TedPkgInfo(tekiFile) [file tail $TedPkgInfo(pkgFilename)]
		TedWriteVersion1.0 $TedPkgInfo(pkgFilename)

		set currPkg $TedPkgInfo(tekiFile)
		wm title . "TED -- $currPkg"
		wm iconname . "TED\[$currPkg\]"
		set TedInfo(fileSaved) 1
		TedUpdatePkgInfo
		TedUpdateBundleInfo
		Debug_Leave
		return $TedPkgInfo(pkgFilename)
	} else {
		Debug_Leave
		return ""
	}
}

# Function to close the currently open file
# Returns 1 if closed successfully
# Returns 0 if file close cancelled

proc TedDoFileClose {} {
	global TedInfo TedPkgInfo tmpPkgInfo

	Debug_Enter TedDoFileClose
	
	if {!$TedInfo(fileOpen)} {
		Debug_Leave
		return 1
	}

	if {!$TedInfo(fileSaved)} {
		set rv [TedWarning "File not saved !\nModifications made will be lost." 0 {"Save" "OK" "Cancel"}]

		if {$rv == 2} {
			Debug_Leave
			return 0
		} elseif {$rv == 0} {
			if {[TedDoFileSaveAs] == ""} {
				Debug_Leave
				return 0
			}			
		}
	}

	set TedInfo(fileOpen) 0
	TedUpdatePkgList

	# Disable Save, Save As menu options
	.menu.file entryconfigure "Save" -state disabled
	.menu.file entryconfigure "Save As" -state disabled
	.menu.file entryconfigure "Close" -state disabled


	if {[info exists TedPkgInfo]} {
		unset TedPkgInfo
	}

	if {[info exists tmpPkgInfo]} {
		unset tmpPkgInfo
	}

	wm title . "TED"
	wm iconname . "TED"
	Debug_Leave
	return 1
}

proc TedDoFileExit {} {
	global StartDir TedInfo TedPkgInfo

	Debug_Enter TedDoFileExit

	if {![TedDoFileClose]} {
		Debug_Leave
		return
	}

	cd $StartDir
	unset TedInfo

	if {[info exists TedPkgInfo]} {
		unset TedPkgInfo
	}

	if {[info exists tmpPkgInfo]} {
		unset tmpPkgInfo
	}
	exit
}

#-----------------------------------------------------------------------
# Function:	TedSplashScreen
#			Creates a Splash Window which appears initially
#

proc TedSplashScreen {} {
    global .menu TekiInfo

    Debug_Enter TedSplashScreen
	toplevel .about-Ted
	set w .about-Ted.splash
	Wise_CenterWindow .about-Ted

	if {[catch {image create photo sImage -file "$TekiInfo(library)/ted.gif"}]} {
		frame $w -width 100m -height 50m -relief groove -bd 4
		pack $w -fill both -padx 1m -pady 2m
		message $w.msg1 -font *-Times-BOLD-I-Normal--*-220-*-*-*-*-*-* -text "TED" -width 100m
		message $w.msg2 -font *-Times-BOLD-R-Normal--*-200-*-*-*-*-*-* -text "TEKI File Editor" -width 100m
		message $w.msg3 -font *-Times-BOLD-R-Normal--*-200-*-*-*-*-*-* -text "(c) Cornell University 1998" -width 100m
		message $w.msg4 -font *-Times-Light-R-Normal--*-200-*-*-*-*-*-* -text "Brian Smith" -width 50m
		message $w.msg5 -font *-Times-Light-I-Normal--*-200-*-*-*-*-*-* -text "bsmith@cs.cornell.edu" -width 50m
		message $w.msg6 -font *-Times-Light-R-Normal--*-200-*-*-*-*-*-* -text "Sumit Kapoor" -width 50m
		message $w.msg7 -font *-Times-Light-I-Normal--*-200-*-*-*-*-*-* -text "sumit@cs.cornell.edu" -width 50m
		pack $w.msg1 $w.msg2 $w.msg3 \
		$w.msg4 $w.msg5 $w.msg6 \
		$w.msg7 -side top -fill both
	} else {
		canvas $w -width 400 -height 250 -relief groove -bd 4
		pack $w -fill both -padx 4 -pady 4
		$w create image 0 0 -image sImage -anchor nw
	}

	raise .about-Ted .
    bind .about-Ted <Button-1> {destroy .about-Ted}
	focus .about-Ted
	after 5000 {destroy .about-Ted}
    Debug_Leave
}

#-----------------------------------------------------------------------

#
# TedCreateUI
#    Create the TED User Interface in the main window (".")
#        1. Create the menus
#        2. Create & show the browse windows
#
proc TedCreateUI {} {
    global tclPkgInfo TedInfo TedPkgInfo

    Debug_Enter TedCreateUI
	TedSplashScreen
    TedCreateMenus
    TedCreateMainWindow
	
	# Create wise wizard
	WiseMakeWizard 1
	wm withdraw .wise

    set currPkg $TedPkgInfo(tekiFile)
    wm title . "TED -- $currPkg"
    wm iconname . "TED\[$currPkg\]"
    Debug_Leave
}

#---------------------------------------------------------------------
# The following section contains the various package information wizard
# screens and their event handlers

######################################################################
# Package File Selections

proc TedCreateFileSelectionWindow {pkgId} {
	global FilesWin tclPkgInfo TedInfo
	Debug_Enter TedCreateFileSelectionWindow

	WiseMakeWizard 1
	wm title .wise "Package Files--$pkgId"
	.wise.title configure -text "$pkgId Package Files"
	set FilesWin ".wise.filesWin"
    set w $FilesWin
	frame $w -width 100m -height 200m -bd 1
	grid $w -row 1 -column 1 -columnspan 5 -sticky nesw
	set eId 0

	set w1 $w.browser
	frame $w1
	pack $w1 -side left -fill both -expand yes

	# Define Directory Listbox Frame Window
	set w11 $w1.dirListWin
	frame $w11
	pack $w11 -side top -fill both -expand no
	scrollbar $w11.scroll_d -command "$w11.list_d yview"
	listbox $w11.list_d -font $TedInfo(fixedfont) -relief sunken -width 25 \
				 -bd 2 -yscrollcommand "$w11.scroll_d set" -setgrid 1 -height 8 \
				 -selectmode single
	pack $w11.scroll_d -side right -fill y
	pack $w11.list_d -expand yes -fill both
	set TedInfo(DBrowser) $w11.list_d

	# Define Files Listbox Frame Window
	set w12 $w1.dirFileWin
	frame $w12
	pack $w12 -fill both -expand yes -side top
	scrollbar $w12.scroll_f -command "$w12.list_f yview"
	listbox $w12.list_f -font $TedInfo(fixedfont) -relief sunken -width 25 \
				 -bd 2 -yscrollcommand "$w12.scroll_f set" -setgrid 1 -height 12 \
				 -selectmode extended
	pack $w12.scroll_f -side right -fill y
	pack $w12.list_f -expand yes -fill both
	set TedInfo(FBrowser) $w12.list_f

	# Create the file selection button frame
	set w2 $w.projOptions
	frame $w2
	pack $w2 -side left -fill none -expand no
	button $w2.btnAdd -width 5 -padx 1 -pady 1 -text "->" -command "TedFileSelOptionHandler $pkgId add"
	button $w2.btnRemove -width 5 -padx 1 -pady 1 -text "<-" -command "TedFileSelOptionHandler $pkgId remove"
	pack $w2.btnAdd $w2.btnRemove -expand no -side top -fill none -pady 4m

	# Create the Project's Selected Files section of the frame
	set w3 $w.selectedFiles
	frame $w3
	pack $w3 -side right -fill both -expand yes

	set w31 $w3.btnbar
	frame $w31
	pack $w31 -side top -fill x -expand no
    grid rowconfigure $w31 1 -weight 1

	# Set File Selection Buttons Window,variable
	set TedInfo(FileSelBtnWindow) $w31
	label $w31.l_basedir -width 8 -padx 1 -text "Base Dir."
	label $w31.basedir -width 16 -padx 1 -textvariable tmpPkgInfo($pkgId,BaseDir) \
			-relief sunken -background WHITE -anchor w
	button $w31.btnbdir -width 8 -padx 1 -pady 1 -text "Browse.." \
			-command "TedFileSelOptionHandler $pkgId basedir"

	button $w31.btntcl -width 8 -padx 1 -pady 1 -text TCL -relief sunken \
			-command "TedUpdateSelList $pkgId tcl"
	button $w31.btndoc -width 8 -padx 1 -pady 1 -text Document \
			-command "TedUpdateSelList $pkgId doc"
	button $w31.btnexample -width 8 -padx 1 -pady 1 -text Example \
			-command "TedUpdateSelList $pkgId example"
	button $w31.btndata -width 8 -padx 1 -pady 1 -text Data \
			-command "TedUpdateSelList $pkgId data"
	grid $w31.l_basedir  -row 0 -col 0 -sticky ew -padx 2 -pady 2
	grid $w31.basedir  -row 0 -col 1 -columnspan 2 -sticky ew -padx 2 -pady 2
	grid $w31.btnbdir -row 0 -col 3 -sticky ew -padx 2 -pady 2
	grid $w31.btntcl $w31.btndoc $w31.btnexample $w31.btndata -row 1 -padx 0

	scrollbar $w3.scrolly -command "$w3.list yview"
	scrollbar $w3.scrollx -orient horizontal -command "$w3.list xview"
	listbox $w3.list -font $TedInfo(fixedfont) -relief sunken -width 25 \
				 -bd 2 -yscrollcommand "$w3.scrolly set" -xscrollcommand "$w3.scrollx set" \
				 -setgrid 1 -height 10 \
				 -selectmode extended
	set TedInfo(SelFilesListWindow) $w3.list
	pack $w3.scrollx -side bottom -fill x
	pack $w3.scrolly -side right -fill y
	pack $w3.list -expand yes -fill both

	bind $TedInfo(DBrowser) <Double-1> TedDBrowserEventHandler
	bind $TedInfo(FBrowser) <Double-1> "TedFileSelOptionHandler $pkgId add"
	bind $TedInfo(SelFilesListWindow) <Double-1> "TedFileSelOptionHandler $pkgId remove"

	# Initialize directory list box
	TedDBrowserEventHandler update

	# Initialize file list box
	TedUpdateFBrowser

	# Update selected project files listbox
	set TedInfo(FileSelectBtn) tcl
	TedUpdateSelList $pkgId tcl

	Debug_Leave
}

#
# TedUpdateSelList
# Function to update the selected files section of the window
#

proc TedUpdateSelList {pkgId btnName} {
	global TedInfo tmpPkgInfo
	Debug_Enter TedUpdateSelList

	# Raise previously selected button
	set w $TedInfo(FileSelBtnWindow).btn$TedInfo(FileSelectBtn)
	$w configure -relief raised

	# Depress selected button
	set TedInfo(FileSelectBtn) $btnName
	set w $TedInfo(FileSelBtnWindow).btn$btnName
	$w configure -relief sunken

	set w $TedInfo(SelFilesListWindow)
	set varname "[set TedInfo(FileSelectBtn)]Files"

	$w delete 0 end

	if {[string compare $tmpPkgInfo($pkgId,$varname) ""] != 0} {
		foreach file $tmpPkgInfo($pkgId,$varname) {
			$w insert end $file
		}
	}

	Debug_Leave
}

#
# TedDBrowserEventHandler --
#    This procedure is called when the user clicks on a directory in the directory list box
#    It is also called to initialize the directory listbox
#

proc TedDBrowserEventHandler {{event "dblclk"}} {
	global DriveList PresentDir TedInfo tcl_platform
    Debug_Enter "TedDBrowserEventHandler"

	set w $TedInfo(DBrowser)

	if {$event == "dblclk"} {
		set selid [$w curselection]

		if {[string compare $selid ""] != 0} {
			set c [$w get $selid]

			if [regexp {^\[-} $c] {
				set PresentDir "[string index $c 2]:/"
			} else {
				set PresentDir $c
			}

			catch {cd $PresentDir}
		}
	}

	$w delete 0 end
	$w insert end ".."

	set directories [glob -nocomplain */]

	if {[string compare $directories ""] != 0} {
		set directories [lsort -dictionary $directories]
	}

	if {[string compare $DriveList ""] != 0} {
		set directories [concat $directories $DriveList]
	}
	
	if {[string compare $directories ""] != 0} {
		foreach file $directories {
			$w insert end $file
		}
	}

	TedUpdateFBrowser
	set PresentDir [pwd]

	if {$tcl_platform(platform) == "windows"} {
		set PresentDir [string tolower $PresentDir]
	}

    Debug_Leave
}

#
# TedUpdateFBrowser --
#    It is also called to initialize the files listbox
#

proc TedUpdateFBrowser {} {
	global DriveList PresentDir TedInfo
    Debug_Enter "TedUpdateFBrowser"

	set w $TedInfo(FBrowser)

	# Delete all the elements of the listbox
	$w delete 0 end

	set directories [glob -nocomplain */]
	set files [glob -nocomplain *]

	foreach dir $directories {
		regsub / $dir "" dir
		set ind [lsearch $files $dir]

		if {$ind != -1} {
			set files [lreplace $files $ind $ind]
		}
	}

	if {[string compare $files ""] != 0} {
		set files [lsort -dictionary $files]
	}

	if {[string compare $files ""] != 0} {
		foreach file $files {
			$w insert end $file
		}
	}

    Debug_Leave
}

#
#	TedFileSelOptionHandler
#
#

proc TedFileSelOptionHandler {pkgId option} {
	global TedInfo tmpPkgInfo PresentDir
    Debug_Enter TedFileSelOptionHandler

	if {$option == "basedir"} {
		if {$tmpPkgInfo($pkgId,BaseDir) != ""} {
			set tmpdir [Wise_GetDirName $tmpPkgInfo($pkgId,BaseDir) "Select Base Directory" "TedHelp 0"]
		} else {
			set tmpdir [Wise_GetDirName [pwd] "Select Base Directory" "TedHelp 0"]
		}

		if {$tmpdir == "" && $tmpPkgInfo($pkgId,BaseDir) == ""} {
			grab release .wise
			TedWarning "You need to select a base directory before you can add files to the package !" 0 OK
			grab set .wise
		} else {
			set tmpPkgInfo($pkgId,BaseDir) $tmpdir
		}

		if {$tmpPkgInfo($pkgId,BaseDir) != ""} {
			cd $tmpPkgInfo($pkgId,BaseDir)
		}

		# Update file browser
		TedUpdateFBrowser
		TedDBrowserEventHandler
	} elseif {$tmpPkgInfo($pkgId,BaseDir) != ""} {
		switch $option {
			add {
				set w $TedInfo(FBrowser)
				set varname "[set TedInfo(FileSelectBtn)]Files"

				set selid [$w curselection]

				if {[string compare $selid ""] != 0} {
					foreach c $selid {
						set file [$w get $c]
						set file "$PresentDir/$file"

						if {$TedInfo(FileSelectBtn) == "doc" && $tmpPkgInfo($pkgId,docDir) != ""} {
							set bdir "$tmpPkgInfo($pkgId,BaseDir)/$tmpPkgInfo($pkgId,docDir)"
						} else {
							set bdir "$tmpPkgInfo($pkgId,BaseDir)"
						}

						set bDirInd [string first $bdir $file]

						if {$bDirInd != -1} {
							set ind [expr {$bDirInd + [string length $bdir] + 1}]
						} else {
							set ind 0
						}

						set file [string range $file $ind end]
						set ind [lsearch $tmpPkgInfo($pkgId,$varname) $file]

						if {$ind == -1} {
							regsub :// $file ":/" file
							set tmpPkgInfo($pkgId,$varname) [linsert $tmpPkgInfo($pkgId,$varname) 0 $file]
						}
					}

					set tmpPkgInfo($pkgId,$varname) [lsort -dictionary $tmpPkgInfo($pkgId,$varname)]
				}
			}

			remove {
				set w $TedInfo(SelFilesListWindow)
				set varname "[set TedInfo(FileSelectBtn)]Files"

				set selid [$w curselection]

				if {[string compare $selid ""] != 0} {
					foreach c $selid {
						set file [$w get $c]
						set f_id [lsearch $tmpPkgInfo($pkgId,$varname) $file]
						set tmpPkgInfo($pkgId,$varname) [lreplace $tmpPkgInfo($pkgId,$varname) $f_id $f_id]
					}
				}
			}

			wizard {
				TedWizard
			}
		}

		TedUpdateSelList $pkgId $TedInfo(FileSelectBtn)
	} else {
		# If option is wizard then call wizard proc
		
		if {$option == "wizard"} {
			TedWizard
		} else {
			grab release .wise
			TedWarning "Please select a Base Directory First before proceeding !" 0 OK
			grab set .wise
		}
	}

    Debug_Leave
}

proc TedGetPkgFiles {pkgId} {
	global FilesWin tmpPkgInfo WiseInfo

	set WiseInfo(done) 0
	TedCreateFileSelectionWindow $pkgId

	while {$WiseInfo(done) == 0} {
		grab set .wise
		vwait WiseInfo(done)
		grab release .wise

		if {$WiseInfo(done) == "help"} {
			TedHelp 7

			# Reset WiseInfo(done)
			set WiseInfo(done) 0
		}
	}

	destroy $FilesWin
	unset FilesWin
	return $WiseInfo(done)
}

######################################################################
# Package Settings Information

# This window updates the following information related to a package
# name version description requires updateURL registerURL srcURL browseURL

# Create Package Settings Window

proc TedCreateSettingsWindow {pkgId} {
	global SettingsWin tmpPkgInfo TedInfo

	WiseMakeWizard 1
	wm title .wise "Package Settings--$pkgId"
	.wise.title configure -text "$pkgId Package Settings"
	set SettingsWin ".wise.setWin"
    set w $SettingsWin
	frame $w -width 100m -height 200m -bd 1
	grid $w -row 1 -column 1 -columnspan 5 -sticky nesw
	set eId 0

	foreach element $TedInfo(settingsName) {
		frame $w.f_$element
		pack $w.f_$element -side top
		label $w.f_$element.lbl_$element -text [lindex $TedInfo(settingsText) $eId] -width 15 -anchor w
		entry $w.f_$element.ent_$element -width 40 -relief sunken -bd 2 -textvariable tmpPkgInfo($pkgId,$element)
		pack $w.f_$element.lbl_$element $w.f_$element.ent_$element -side left -padx 1m -pady 1m
		incr eId 1
	}
}

proc TedGetPkgSettings {pkgId} {
	global SettingsWin tmpPkgInfo WiseInfo

	Debug_Enter TedGetPkgSettings
	set WiseInfo(done) 0
	TedCreateSettingsWindow $pkgId

	while {$WiseInfo(done) == 0} {
		grab set .wise
		vwait WiseInfo(done)
		grab release .wise

		if {$WiseInfo(done) == "help"} {
			TedHelp 1

			# Reset WiseInfo(done)
			set WiseInfo(done) 0
		}
	}

	destroy $SettingsWin
	unset SettingsWin
	Debug_Leave
	return $WiseInfo(done)
}

######################################################################
# Package Directories and File Settings Window

# This window gets the foolwing package related information
# srcDir destDir docDir copyright infoFile initFile

# Create Window

proc TedCreateDirOptionsWindow {pkgId} {
	global DirOptionsWin tmpPkgInfo TedInfo

	WiseMakeWizard 1
	wm title .wise "Package Directories and Files--$pkgId"
	.wise.title configure -text "$pkgId Package Directories and Files"
	set DirOptionsWin .wise.dirOptionsWin

    set w $DirOptionsWin
	frame $w -width 100m -height 200m -bd 1
	grid $w -row 1 -column 1 -columnspan 5 -sticky nesw
	set eId 0

	foreach element $TedInfo(dirOptionsName) {
		frame $w.f_$element
		pack $w.f_$element -side top
		set label_txt [lindex $TedInfo(dirOptionsText) $eId]
		label $w.f_$element.lbl_$element -text $label_txt -width 18 -anchor w
		label $w.f_$element.ent_$element -width 20 -relief sunken -anchor w -bd 2 -textvariable tmpPkgInfo($pkgId,$element)
		button $w.f_$element.btn_$element -command "DirOptionsEventHandler $pkgId button $element \"$label_txt\""  -text Change
		pack $w.f_$element.lbl_$element $w.f_$element.ent_$element $w.f_$element.btn_$element -side left -padx 1m -pady 1m
		incr eId 1
	}
}

proc DirOptionsEventHandler {pkgId {event ""} {element ""} {txtMsg ""}} {
	global tcl_platform DirOptionsWin tmpPkgInfo WiseInfo

	switch $event {
		button {
# Search in string for patterns "Directory" or "File",
# if found then use appropriate dialog box for selection

			if {[string first "Directory" $txtMsg] != -1} {
				set tmpdir $tmpPkgInfo($pkgId,BaseDir)/$tmpPkgInfo($pkgId,$element)

				# Release grab
				grab release .wise

				# Get directory name
				set tmpdir [Wise_GetDirName $tmpdir "Select $txtMsg"]

				# Set grab
				grab set .wise

				if {$tmpdir != ""} {
				# Remove the base directory from the directory name
				# Check if windows platform then convert everything to lower case and then compare
					if {$tcl_platform(platform) == "windows"} {
						set bDirInd [string first [string tolower $tmpPkgInfo($pkgId,BaseDir)] [string tolower $tmpdir]]
					} else {
						set bDirInd [string first $tmpPkgInfo($pkgId,BaseDir) $tmpdir]
					}

					if {$bDirInd != -1} {
						set ind [expr {$bDirInd + [string length $tmpPkgInfo($pkgId,BaseDir)] + 1}]
					} else {
						set ind 0
					}

					# Check if the selected directory is same as Base Directory,if so put "."

					set tmpdir  [string range $tmpdir $ind end]

					if {$tmpdir == ""} {
						set tmpdir "."
					}
				}

				set tmpPkgInfo($pkgId,$element) $tmpdir
			} elseif {[string first "File" $txtMsg] != -1} {
				# Release grab
				grab release .wise

				# Get the teki install file
				set tmpfile [tk_getOpenFile -title "Select $txtMsg" -filetypes {{{Text Files} .txt} {{All Files} *}}]

				# Set grab
				grab set .wise

				if {$tmpfile != ""} {
				# Remove the base directory from the file name
				# Check if windows platform then convert everything to lower case and then compare
					if {$tcl_platform(platform) == "windows"} {
						set bFileInd [string first [string tolower $tmpPkgInfo($pkgId,BaseDir)] [string tolower $tmpfile]]
					} else {
						set bFileInd [string first $tmpPkgInfo($pkgId,BaseDir) $tmpfile]
					}


					if {$bFileInd != -1} {
						set ind [expr {$bFileInd + [string length $tmpPkgInfo($pkgId,BaseDir)] + 1}]
					} else {
						set ind 0
					}

					set tmpPkgInfo($pkgId,$element) [string range $tmpfile $ind end]
				} else {
					set tmpPkgInfo($pkgId,$element) $tmpfile
				}
			}
		}
	}
}

proc TedGetPkgDirOptions {pkgId} {
global DirOptionsWin tmpPkgInfo WiseInfo

	set WiseInfo(done) 0
	TedCreateDirOptionsWindow $pkgId

	while {$WiseInfo(done) == 0} {
		grab set .wise
		vwait WiseInfo(done)
		grab release .wise

		if {$WiseInfo(done) == "help"} {
			TedHelp 2

			# Reset WiseInfo(done)
			set WiseInfo(done) 0
		}

		if {$tmpPkgInfo($pkgId,srcDir) == ""} {
			TedWarning "Please select a source directory before proceeding." 0 OK

			 set WiseInfo(done) 0
		}
	}

	destroy $DirOptionsWin
	unset DirOptionsWin
	return $WiseInfo(done)
}

######################################################################
# Functions to get systems related object files

# Create Window

proc TedCreateObjFilesWindow {pkgId} {
	global ObjFilesWin

	WiseMakeWizard 1
	wm title .wise "Object Files--$pkgId"
	.wise.title configure -text "$pkgId Object Files"
	set ObjFilesWin ".wise.objFilesWin"
	set w $ObjFilesWin
	frame $w -width 100m -height 200m -bd 1
	grid $w -row 1 -column 1 -columnspan 5 -sticky nesw

	# Create the bottom button frame
	set w3 $ObjFilesWin.buttonFrame
	frame $w3 -width 50m -height 5m -bd 1
	pack $w3 -side bottom
	button $w3.btnModifySysNames -text "Modify System Names" -command "ObjFilesWinEventHandler $pkgId button modSysNames"
	button $w3.btnAddObjFile -text "Add Object File" -command "ObjFilesWinEventHandler $pkgId button addObj"
	button $w3.btnDeleteObj -text "Delete Object File" -command "ObjFilesWinEventHandler $pkgId button deleteObj"
	pack $w3.btnModifySysNames -side left -padx 8 -pady 4
	pack $w3.btnAddObjFile $w3.btnDeleteObj -side left -padx 0 -pady 4

	set w1 $ObjFilesWin.sysFrame
	frame $w1 -width 10m -height 20m -bd 1
	pack $w1 -side left
    scrollbar $w1.scroll -command "$w1.list yview"
	listbox $w1.list  -relief sunken -height 15 -width 25 -selectmode single \
						 -yscrollcommand "$w1.scroll set"
    pack $w1.scroll -side right -fill y
	pack $w1.list -side left -fill both

	set w2 $ObjFilesWin.objFrame
	frame $w2 -width 20m -height 20m -bd 1
	pack $w2 -side left
    scrollbar $w2.scrolly -command "$w2.list yview"
	scrollbar $w2.scrollx -orient horizontal -command "$w2.list xview"
	listbox $w2.list  -relief sunken -height 14 -width 35 -selectmode multiple \
						 -yscrollcommand "$w2.scrolly set"  -xscrollcommand "$w2.scrollx set"
	pack $w2.scrollx -side bottom -fill x
    pack $w2.scrolly -side right -fill y
	pack $w2.list -side left -fill both

	ObjFilesWinEventHandler $pkgId update
    bind $w1.list <ButtonRelease-1> "ObjFilesWinEventHandler $pkgId lbtnclk"
    bind $w1.list <Double-1> "ObjFilesWinEventHandler $pkgId button addObj"
    bind $w2.list <Double-1> "ObjFilesWinEventHandler $pkgId button deleteObj"
}

proc ObjFilesWinEventHandler {pkgId {event ""} {btnType ""}} {
	global ObjFilesWin selectedSys tmpPkgInfo tcl_platform WiseInfo

	Debug_Enter ObjFilesWinEventHandler
	set w1 $ObjFilesWin.sysFrame.list
	set w2 $ObjFilesWin.objFrame.list
	set noSys [llength $tmpPkgInfo(systemNames)]
	set noSysObj [llength $tmpPkgInfo($pkgId,objFiles)]

	switch $event {
		button {
			switch $btnType {
				modSysNames {
					grid forget $ObjFilesWin 
					set old_nstate [lindex [.wise.next configure -state] 4]
					set old_bstate [lindex [.wise.back configure -state] 4]
					.wise.next configure -state disabled
					.wise.back configure -state disabled

					foreach pkg $tmpPkgInfo(available) {
						# Skip current package because modifications may have been made

						if {$pkg != $pkgId} {
							TedInitTmpPkgInfo $pkg
						}
					}

					set rv [TedGetBundleSystemNames]
					set WiseInfo(done) 0

					if {$rv == "finish"} {
						TedUpdateTedPkgInfo

						# Update information for all packages
						foreach pkg $tmpPkgInfo(available) {
							TedCopyTmpToTedPkgInfo $pkg
						}
					} elseif {$rv == "cancel"} {
						# If cancel is presses then restore old values to tmpPkgInfo
						# TedGetBundleSystemNames updates tmPkgInfo

						TedInitTmpInfo
					}

					.wise.next configure -state $old_nstate
					.wise.back configure -state $old_bstate
					grid $ObjFilesWin -row 1 -column 1 -columnspan 5 -sticky nesw
					ObjFilesWinEventHandler $pkgId update
				}

				deleteObj {
					# First check if anything in the object files listbox is selected
					set selid [$w2 curselection]

					if {$selid != -1} {
						# First find the list of object files corresponding to 
						# the selected system name

						for {set i 0} {$i < $noSysObj && [lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] 0] \
						!= [lindex $selectedSys 0]} {incr i 1} {}

						set tmpDList [list ""]

						if {$i < $noSysObj} {
							foreach j $selid {
								set k [expr $j+1]

								# Add selected file names to a temporary list
								set tmpDList [lappend $tmpDList [lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] $k]]
							}

							# Now copy all the undeleted names to a temporary list
							# First copy the system name
							set tmpList [lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] 0]

							# Copy all the object files
							set tmpObjFiles [lrange [lindex $tmpPkgInfo($pkgId,objFiles) $i] 1 end]

							foreach file $tmpObjFiles {
								if {[lsearch $tmpDList $file] == -1} {
									lappend tmpList $file
								}
							}

							set tmpPkgInfo($pkgId,objFiles) [lreplace $tmpPkgInfo($pkgId,objFiles) $i $i $tmpList]

							# Update listboxes
							ObjFilesWinEventHandler $pkgId update
						}
					}
				}

				addObj {
					# Release grab
					grab release .wise

					# Get the teki install file
					# First check for the platform

					if {$tcl_platform(platform) == "windows"} {
						set tmpfile [tk_getOpenFile -title "Select Object File" -filetypes {{{Windows dll} .dll} {{All Files} *}}]
					} else {
						set tmpfile [tk_getOpenFile -title "Select Object File" -filetypes {{{Unix obj. files} .so} {{All Files} *}}]
					}

					# Set grab
					grab set .wise

					if {$tmpfile != ""} {
					# Remove the base directory from the file name
					# Check if windows platform then convert everything to lower case and then compare
						if {$tcl_platform(platform) == "windows"} {
							set bFileInd [string first [string tolower $tmpPkgInfo($pkgId,BaseDir)] [string tolower $tmpfile]]
						} else {
							set bFileInd [string first $tmpPkgInfo($pkgId,BaseDir) $tmpfile]
						}

						if {$bFileInd != -1} {
							set ind [expr {$bFileInd+[string length $tmpPkgInfo($pkgId,BaseDir)]+1}]
						} else {
							set ind 0
						}

						set tmpfile [string range $tmpfile $ind end]

						# First find the list of object files corresponding to 
						# the selected system name

						for {set i 0} {$i < $noSysObj && [lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] 0] \
						!= [lindex $selectedSys 0]} {incr i 1} {}

						set tmpList [lindex $tmpPkgInfo($pkgId,objFiles) $i]
						lappend tmpList $tmpfile
						set tmpPkgInfo($pkgId,objFiles) [lreplace $tmpPkgInfo($pkgId,objFiles) $i $i $tmpList]

						# Update listboxes
						ObjFilesWinEventHandler $pkgId update
					}
				}
			}
		}

		update {
			$w1 delete 0 end
			$w2 delete 0 end

			for {set i 0} {$i < $noSys} {incr i 1} {
				$w1 insert end [lindex $tmpPkgInfo(systemNames) $i]
			}


			if {![info exists selectedSys]} {
				set selectedSys [lindex $tmpPkgInfo(systemNames) 0]
			}

			for {set i 0} {[lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] 0] != [lindex $selectedSys 0] \
			&& $i < $noSysObj} {incr i 1} {}

			if {$i < $noSysObj} {
				set j 1
				set file [lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] $j]

				while {$file != ""} {
					$w2 insert end $file

					incr j 1
					set file [lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] $j]
				}
			}
		}

		lbtnclk {
			set selid [$w1 curselection]

			if {$selid != ""} {
				set selectedSys [lindex $tmpPkgInfo(systemNames) $selid]

				# Delete all the elements of the object files listbox
				$w2 delete 0 end

				for {set i 0} {[lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] 0] != [lindex $selectedSys 0] \
				&& $i < $noSysObj} {incr i 1} {}

				if {$i < $noSysObj} {
					set j 1
					set file [lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] $j]

					while {$file != ""} {
						$w2 insert end $file

						incr j 1
						set file [lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] $j]
					}
				}
			}
		}
	}

	Debug_Leave
}

proc TedGetPkgObjFiles {pkgId} {
	global selectedSys ObjFilesWin tmpPkgInfo WiseInfo SystemWin

	if {$tmpPkgInfo($pkgId,BaseDir) == ""} {
		if {[TedWarning "Please select a base directory before proceeding !" 0 {OK Cancel}]} {
			return cancel
		}

		set tmpPkgInfo($pkgId,BaseDir) [Wise_GetDirName [pwd] "Select Base Directory" "TedHelp 0"]
	}

	set WiseInfo(done) 0
	TedCreateObjFilesWindow $pkgId

	while {$WiseInfo(done) == 0} {
		grab set .wise
		vwait WiseInfo(done)
		grab release .wise

		# Find out if the systems window exists

		if {[info exists SystemWin]} {
			unset SystemWin
			set WiseInfo(done) 0
		} elseif {$WiseInfo(done) == "help"} {
			TedHelp 3

			# Reset WiseInfo(done)
			set WiseInfo(done) 0
		}
	}

	destroy $ObjFilesWin
	unset ObjFilesWin
	unset selectedSys
	return $WiseInfo(done)
}

######################################################################
# Ted Package Wizard

# This wizard basically gets the information related to the different
# packages in the project

proc TedPackageWizard {pkgId {mainWiz 0}} {
	global TedInfo tmpPkgInfo

	set oldDir [pwd]

	# Main Wizard takes care of creating temporary copies
	# There is no main wizard now

	if {!$mainWiz} {
		# Initialize tmpPkgInfo
		TedInitTmpInfo
		TedInitTmpPkgInfo $pkgId
	}

	if {$tmpPkgInfo($pkgId,BaseDir) != ""} {
		if {[file executable $tmpPkgInfo($pkgId,BaseDir)]} {
			cd $tmpPkgInfo($pkgId,BaseDir)
		} else {
			TedWarning "Unable to change to base directory $tmpPkgInfo($pkgId,BaseDir), please select another base directory !" 0 OK
			set tmpPkgInfo($pkgId,BaseDir) ""
		}
	}

	set wizSeq {{TedGetPkgSettings} {TedGetPkgDirOptions} {TedGetPkgFiles} {TedGetPkgObjFiles}}

	set wizId 0
	set rv [[lindex $wizSeq $wizId] $pkgId]
	set seqLen [expr [llength $wizSeq]-1]

	while {$rv != "finish" && $rv != "cancel"} {
		if {$rv == "next" && $wizId != $seqLen} {
			incr wizId 1
			set rv [[lindex $wizSeq $wizId] $pkgId]
		} elseif {$rv == "next" && $wizId == $seqLen} {
			break
		} elseif {$rv == "back" && $wizId != 0} {
			incr wizId -1
			set rv [[lindex $wizSeq $wizId] $pkgId]
		} elseif {$rv == "back" && $wizId == 0} {
			break
		}
	}

	wm withdraw .wise

	# Main Wizard takes care of updating TedPkgInfo

	if {!$mainWiz} {
		if {$rv == "finish"} {
			# Update TedPkgInfo
			TedUpdateTedPkgInfo
			TedCopyTmpToTedPkgInfo $pkgId
		}
	}

	cd $oldDir
	return $rv
}

# Function to create a temporary copy of the package settings for the wizard

proc TedInitTmpPkgInfo {pkgId} {
	global TedPkgInfo tmpPkgInfo TedInfo

	Debug_Enter TedInitTmpPkgInfo
	# Get the Base Directory first

	if {[info exists TedPkgInfo($pkgId,BaseDir)]} {
		set tmpPkgInfo($pkgId,BaseDir) $TedPkgInfo($pkgId,BaseDir)
	} else {
		set tmpPkgInfo($pkgId,BaseDir) ""
	}

	foreach element $TedInfo(pkgFileTypes) {
		if [info exists TedPkgInfo($pkgId,$element)] {
			set tmpPkgInfo($pkgId,$element) $TedPkgInfo($pkgId,$element)
		} else {
			set tmpPkgInfo($pkgId,$element) ""
		}
	}

	foreach element $TedInfo(settingsName) {
		if [info exists TedPkgInfo($pkgId,$element)] {
			set tmpPkgInfo($pkgId,$element) $TedPkgInfo($pkgId,$element)
		} else {
			set tmpPkgInfo($pkgId,$element) ""
		}
	}

	foreach element $TedInfo(dirOptionsName) {
		if [info exists TedPkgInfo($pkgId,$element)] {
			set tmpPkgInfo($pkgId,$element) $TedPkgInfo($pkgId,$element)
		} else {
			set tmpPkgInfo($pkgId,$element) ""
		}
	}

	set tmpPkgInfo($pkgId,objFiles) $TedPkgInfo($pkgId,objFiles)
	Debug_Leave
}

# Function to update TedPkgInfo

proc TedCopyTmpToTedPkgInfo {pkgId} {
	global TedPkgInfo tmpPkgInfo TedInfo

	set TedPkgInfo($pkgId,BaseDir) $tmpPkgInfo($pkgId,BaseDir)

	foreach element $TedInfo(pkgFileTypes) {
		set TedPkgInfo($pkgId,$element) $tmpPkgInfo($pkgId,$element)
	}

	foreach element $TedInfo(settingsName) {
		set TedPkgInfo($pkgId,$element) $tmpPkgInfo($pkgId,$element)
	}

	foreach element $TedInfo(dirOptionsName) {
		set TedPkgInfo($pkgId,$element) $tmpPkgInfo($pkgId,$element)
	}

	set TedPkgInfo($pkgId,objFiles) $tmpPkgInfo($pkgId,objFiles)
}

# ---------------------------------------------------------------------
# System Settings Window and Default Wizard Windows

######################################################################
# System/Platform Information

# Create Window

proc TedCreateSystemsWindow {} {
	global SystemWin tmpPkgInfo

	WiseMakeWizard 1
	wm title .wise "System Names--$tmpPkgInfo(tekiFile)"
	.wise.title configure -text "$tmpPkgInfo(tekiFile) System Names"
	set SystemWin ".wise.sysWin"
	set w $SystemWin
	frame $w -width 100m -height 200m -bd 1
	grid $w -row 1 -column 1 -columnspan 5 -sticky nesw

	# Create the bottom button frame
	set w2 $SystemWin.buttonFrame
	frame $w2 -width 50m -height 5m -bd 1
	pack $w2 -side bottom
	button $w2.btnNewSys -text "New System" -command "SystemsWinEventHandler button new"
	button $w2.btnModifySys -text "Modify System" -command "SystemsWinEventHandler button modify"
	button $w2.btnDeleteSys -text "Delete System" -command "SystemsWinEventHandler button delete"
	pack $w2.btnNewSys $w2.btnModifySys $w2.btnDeleteSys -side left -padx 0m -pady 4

	set w1 $SystemWin.sysFrame
	frame $w1 -width 10m -height 20m -bd 1
	pack $w1 -side left
    scrollbar $w1.scroll -command "$w1.list yview"
	listbox $w1.list  -relief sunken -height 12 -width 45 -selectmode single \
						 -yscrollcommand "$w1.scroll set"
    pack $w1.scroll -side right -fill y
	pack $w1.list -side left -fill both

	SystemsWinEventHandler "update"
    bind $w1.list <ButtonRelease-1> "SystemsWinEventHandler lbtnclk"
    bind $w1.list <Double-1> "SystemsWinEventHandler button modify"
}

proc SystemsWinEventHandler {{event ""} {btnType ""}} {
	global SystemWin currentSys tmpPkgInfo tcl_platform

	Debug_Enter SystemsWinEventHandler
	set w1 $SystemWin.sysFrame.list
	set noSys [llength $tmpPkgInfo(systemNames)]

	switch $event {
		button {
			switch $btnType {
				new {
					# Release grab
					grab release .wise

					# Get the new system name
					set tmpSys [TedEditSysName]
					lappend tmpPkgInfo(systemNames) $tmpSys

					# Set grab
					grab set .wise

					# Add System name to the Object Files List of each of the packages also

					foreach pkgId $tmpPkgInfo(available) {
						lappend tmpPkgInfo($pkgId,objFiles) [lindex $tmpSys 0]
					}

					SystemsWinEventHandler update
				}

				modify {
					# Release grab
					grab release .wise

					# Edit selected system name
					set tmpSys [TedEditSysName $currentSys]

					# Set grab
					grab set .wise

					if {$tmpSys != ""} {
						for {set i 0} {$i < $noSys && ![string match [lindex $tmpPkgInfo(systemNames) $i] $currentSys]} {incr i 1} {}

						set tmpPkgInfo(systemNames) [lreplace $tmpPkgInfo(systemNames) $i $i $tmpSys]
						
						# If the name has been modified then also update the object files list of all the packages

						set tmpSysName [lindex $currentSys 0]

						foreach pkgId $tmpPkgInfo(available) {
							if {$tmpSysName != [lindex $tmpSys 0]} {
								for {set i 0} {$i < $noSys && $tmpSysName != [lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] 0]} \
								{incr i 1} {}

								set tmpPkgInfo($pkgId,objFiles) [lreplace $tmpPkgInfo($pkgId,objFiles) $i $i [list [lindex $tmpSys 0] \
								[lrange [lindex $tmpPkgInfo($pkgId,objFiles) $i] 1 end]]]
							}
						}

						set currentSys $tmpSys
						SystemsWinEventHandler update
					}
				}

				delete {
					if {$noSys == 1} {
						# Release grab
						grab release .wise

						TedWarning "You need to specify atleast one architecture.\nIf you want to delete this architecture, specify a new architecture first !" 0 OK

						# Set grab
						grab set .wise

						Debug_Leave
						return
					}

					set selid [$w1 curselection]

					# First check if anything in the system names listbox is selected
					if {$selid != -1} {
						grab release .wise
						set rv [TedWarning "This will delete all object files related to this system from all the packages !" 0 {Cancel OK}]
						grab set .wise

						if {$rv} {
							set $currentSys [lindex tmpPkgInfo(systemNames) $selid]
							# Delete the list of associated object files from all the packages

							foreach pkgId $tmpPkgInfo(available) {
								# First find the list of object files corressponding to 
								# the selected system name

								for {set i 0} {$i < $noSys && [lindex [lindex $tmpPkgInfo($pkgId,objFiles) $i] 0] \
								!= [lindex $currentSys 0]} {incr i 1} {}

								set tmplist ""
								set j 0

								foreach obj $tmpPkgInfo($pkgId,objFiles) {
									if {$j != $i} {
										lappend tmplist [list $obj]
									}

									incr j 1
								}

								set tmpPkgInfo($pkgId,objFiles) $tmplist
							}

							# Delete selected system
							# Get index of selected item
							set tmpPkgInfo(systemNames) [lreplace $tmpPkgInfo(systemNames) $selid $selid]

							# Unset currentSys so that update initializes it
							unset currentSys

							# Update listboxes
							SystemsWinEventHandler update
						}
					}
				}
			}
		}

		update {
			$w1 delete 0 end

			for {set i 0} {$i < $noSys} {incr i 1} {
				$w1 insert end [lindex $tmpPkgInfo(systemNames) $i]
			}

			if {![info exists currentSys] &&$tmpPkgInfo(systemNames) != ""} {
				set currentSys [lindex $tmpPkgInfo(systemNames) 0]
				$w1 selection set 0 0
			}
		}

		lbtnclk {
			set selid [$w1 curselection]

			if {$selid != ""} {
				set currentSys [lindex $tmpPkgInfo(systemNames) $selid]
			}
		}
	}

	Debug_Leave
}

proc TedGetBundleSystemNames {} {
	global currentSys SystemWin tmpPkgInfo WiseInfo ObjFilesWin

	set WiseInfo(done) 0
	TedCreateSystemsWindow

	while {$WiseInfo(done) == 0} {
		grab set .wise
		vwait WiseInfo(done)
		grab release .wise

		if {$WiseInfo(done) == "help"} {
			TedHelp 8

			# Reset WiseInfo(done)
			set WiseInfo(done) 0
		}
	}

	destroy $SystemWin
	unset currentSys

	# Inform TedGetPkgObjFiles that the systems window exists
	
	if {[info exists ObjFilesWin] == 0} {
		unset SystemWin
	}

	return $WiseInfo(done)
}

# Edit/Get platform/system names

# Function to Create Dialog box

proc TedCreateSysDialog {} {
	global SysDialog tmpPkgInfo

	set SysDialog ".sysDlg"
	set w $SysDialog
	toplevel $w
	wm title $w "System Name"
	Wise_CenterWindow $SysDialog
    grid rowconfigure $w 1 -weight 1

	label $w.l_sysname -text "System Name" -anchor w -width 14
	entry $w.e_sysname -width 8 -relief sunken -textvariable tmpPkgInfo(sysName)
	label $w.l_sysid -text "Operating Sys." -anchor w -width 14
	entry $w.e_sysid -width 8 -relief sunken -textvariable tmpPkgInfo(sysOS)
	label $w.l_sysver -text "OS Version No." -anchor w -width 14
	entry $w.e_sysver -width 8 -relief sunken -textvariable tmpPkgInfo(sysOSVer)
	button $w.ok -text "   OK   " -command "SysDlgEventHandler ok"
	button $w.cancel -text " Cancel " -command "SysDlgEventHandler cancel"
	button $w.help -text " Help " -command "grab release $SysDialog;TedHelp 8;catch {grab set $SysDialog};"

	grid $w.l_sysname -row 0 -column 0
	grid $w.e_sysname  -row 0 -column 1
	grid $w.l_sysid -row 1 -column 0
	grid $w.e_sysid -row 1 -column 1
	grid $w.l_sysver -row 2 -column 0
	grid $w.e_sysver -row 2 -column 1
	grid $w.ok -row 0 -column 2
	grid $w.cancel -row 1 -column 2
	grid $w.help -row 2 -column 2
}

proc SysDlgEventHandler {event} {
	global SysDialog tmpPkgInfo

	switch $event {
		ok {
			set tmpPkgInfo(retval) [concat $tmpPkgInfo(sysName) $tmpPkgInfo(sysOS) $tmpPkgInfo(sysOSVer)]
			destroy $SysDialog
			unset SysDialog
		}

		cancel {
			set tmpPkgInfo(retval) ""
			destroy $SysDialog
			unset SysDialog
		}
	}
}

proc TedEditSysName {{sys ""}} {
	global SysDialog tmpPkgInfo

	if {$sys == ""} {
		set tmpPkgInfo(sysName) ""
		set tmpPkgInfo(sysOS) ""
		set tmpPkgInfo(sysOSVer) ""
	} else {
		set tmpPkgInfo(sysName) [lindex $sys 0]
		set tmpPkgInfo(sysOS) [lindex $sys 1]
		set tmpPkgInfo(sysOSVer) [lindex $sys 2]
	}

	TedCreateSysDialog
	grab set $SysDialog
	tkwait window $SysDialog
	return $tmpPkgInfo(retval)
}

######################################################################
# Default Examples/Document/Data Options

proc TedGetBundleDefInstallDocExDat {} {
	global tmpPkgInfo WiseInfo

	# Create list to be sent to the Wise_Radiolist
	set tmpCBoxList [list [list "Documents" 0 \
	$tmpPkgInfo(defaultInstallDoc)] [list "Examples" 1 \
	$tmpPkgInfo(defaultInstallExamples)] [list "Data Files" 2 \
	$tmpPkgInfo(defaultInstallData)]]

	Wise_CreateWizList

	# Initialize retval so that the condition for while loop is true
	set rv "help"

	while {$rv == "help"} {
		set retval [Wise_Checklist "$tmpPkgInfo(tekiFile) Default Doc./Examples/Data Options" $tmpCBoxList]
		set rv [lindex $retval 0]
		set rlist [lrange $retval 1 end]


		if {[lindex $retval 0] == "help"} {
				TedHelp 6
		} elseif {$rlist != ""} {
			# Initialize default parameters to 0
			set tmpPkgInfo(defaultInstallDoc) 0
			set tmpPkgInfo(defaultInstallExamples) 0
			set tmpPkgInfo(defaultInstallData) 0

			foreach i $rlist {
				switch $i {
					0 {
						set tmpPkgInfo(defaultInstallDoc) 1
					}

					1 {
						set tmpPkgInfo(defaultInstallExamples) 1
					}

					2 {
						set tmpPkgInfo(defaultInstallData) 1
					}
				}
			}
		}
	}

	Wise_DestroyWizList
	return $rv
}

######################################################################
# Default Architecture Options

proc TedGetBundleDefaultArch {} {
	global tmpPkgInfo WiseInfo

	Debug_Enter TedGetBundleDefaultArch
	# Create list to be sent to the Wise_Checklist

	set noSys [llength $tmpPkgInfo(systemNames)]
	set tmpCBoxList ""

	for {set i 0} {$i < $noSys} {incr i 1} {
		set tmpSysName [lindex [lindex $tmpPkgInfo(systemNames) $i] 0]

		if {[lsearch $tmpPkgInfo(defaultArch) $tmpSysName] != -1} {
			set val 1
		} else {
			set val 0
		}

		lappend tmpCBoxList [list $tmpSysName $i $val]
	}

	Wise_CreateWizList

	# Initialize retval so that the condition for while loop is true
	set rv "help"

	while {$rv == "help"} {
		set retval [Wise_Checklist "$tmpPkgInfo(tekiFile) Default Architectures" $tmpCBoxList]
		set rv [lindex $retval 0]
		set rlist [lrange $retval 1 end]

		if {[lindex $retval 0] == "help"} {
				TedHelp 5
		} elseif {$rlist != ""} {
			set defArch ""

			foreach i $rlist {
				lappend defArch [lindex [lindex $tmpPkgInfo(systemNames) $i] 0]
			}

			set tmpPkgInfo(defaultArch) $defArch
		} else {
			TedWarning "Please select atleast one default platform before proceeding !" 0 OK
			set rv "help"
		}
	}

	Wise_DestroyWizList
	Debug_Leave
	return $rv
}


######################################################################
# Default Packages

proc TedGetBundleDefaultPackages {} {
	global tmpPkgInfo WiseInfo

	if {$tmpPkgInfo(available) == ""} {
		TedWarning "No packages available !" 0 OK
		return cancel
	}

	# Create list to be sent to the Wise_Radiolist
	set tmpCBoxList ""

	foreach pkg $tmpPkgInfo(available) {
		if {[lsearch -exact $tmpPkgInfo(defaultPackages) $pkg] != -1} {
			lappend tmpCBoxList [list $pkg $pkg 1]
		} else {
			lappend tmpCBoxList [list $pkg $pkg 0]
		}
	}

	Wise_CreateWizList

	# Initialize retval so that the condition for while loop is true
	set rv "help"

	while {$rv == "help"} {
		set retval [Wise_Checklist "$tmpPkgInfo(tekiFile) Default Install Packages" $tmpCBoxList]
		set rv [lindex $retval 0]
		set rlist [lrange $retval 1 end]

		if {$rv == "help"} {
			TedHelp 9
		} elseif {$rv == "next" || $rv == "finish"} {
			if {$rlist == ""} {
				TedWarning "Please select atleast one default package before proceeding !" 0 OK
				set rv "help"
			} else {
				set tmpPkgInfo(defaultPackages) $rlist
			}
		}
	}

	Wise_DestroyWizList
	return $rv
}

######################################################################
# Default Wizard

# This is the default wizard which updates default bundle settings

proc TedDefaultWizard {} {
	global TedInfo tmpPkgInfo

	set oldDir [pwd]
	# Initialize tmpPkgInfo
	TedInitTmpInfo

	# Create temporary copies of all package related info

	foreach pkg $tmpPkgInfo(available) {
		TedInitTmpPkgInfo $pkg
	}

	set wizSeq {{TedGetBundleDefInstallDocExDat} {TedGetBundleDefaultArch} {TedGetBundleDefaultPackages}}

	set wizId 0
	set rv [[lindex $wizSeq $wizId]]
	set seqLen [expr [llength $wizSeq]-1]

	while {$rv != "finish" && $rv != "cancel"} {
		if {$rv == "next" && $wizId != $seqLen} {
			incr wizId 1
			set rv [[lindex $wizSeq $wizId]]
		} elseif {$rv == "next" && $wizId == $seqLen} {
			set rv "finish"
		} elseif {$rv == "back" && $wizId != 0} {
			incr wizId -1
			set rv [[lindex $wizSeq $wizId]]
		} elseif {$rv == "back" && $wizId == 0} {
			set rv "cancel"
		}
	}

	if {$rv == "finish" || $rv == "next"} {
		set TedInfo(fileSaved) 0
		# Initialize tmpPkgInfo
		TedUpdateTedPkgInfo

		# Create temporary copies of all package related info

		foreach pkg $tmpPkgInfo(available) {
			TedCopyTmpToTedPkgInfo $pkg
		}

		TedUpdateBundleInfo
	}

	wm withdraw .wise
	cd $oldDir
	return $rv
}

proc TedGetPackagesInfo {} {
	global tmpPkgInfo

	set noPkgs [expr [llength $tmpPkgInfo(available)]-1]
	set cPkgNo 0
	set rv [TedPackageWizard [lindex $tmpPkgInfo(available) $cPkgNo] 1]

	while {($rv != "back" && $cPkgNo != 0) && ($rv != "next" && $cPkgNo != $noPkgs) && ($rv != "finish") && ($rv != "cancel")} {
		if {$rv == "back"} {
			incr cPkgNo 1
		} elseif {$rv == "next"} {
			incr cPkgNo -1
		}

		set rv [TedPackageWizard [lindex $tmpPkgInfo(available) $cPkgNo] 1]
	}

	return $rv
}

proc TedInitTmpInfo {} {
	global TedPkgInfo tmpPkgInfo

	Debug_Enter TedInitTmpInfo
	set tmpPkgInfo(systemNames)	$TedPkgInfo(systemNames)
	set tmpPkgInfo(available) $TedPkgInfo(available)
	set tmpPkgInfo(defaultPackages) $TedPkgInfo(defaultPackages)
	set tmpPkgInfo(defaultArch) $TedPkgInfo(defaultArch)
	set tmpPkgInfo(defaultInstallDoc) $TedPkgInfo(defaultInstallDoc)
	set tmpPkgInfo(defaultInstallExamples) $TedPkgInfo(defaultInstallExamples)
	set tmpPkgInfo(defaultInstallData) $TedPkgInfo(defaultInstallData)
	set tmpPkgInfo(tekiFile) $TedPkgInfo(tekiFile)
	Debug_Leave
}

proc TedUpdateTedPkgInfo {} {
	global TedPkgInfo tmpPkgInfo

	Debug_Enter TedUpdateTedPkgInfo
	set TedPkgInfo(systemNames) $tmpPkgInfo(systemNames)
	set TedPkgInfo(available) $tmpPkgInfo(available)
	set TedPkgInfo(defaultPackages) $tmpPkgInfo(defaultPackages)
	set TedPkgInfo(defaultArch) $tmpPkgInfo(defaultArch)
	set TedPkgInfo(defaultInstallDoc) $tmpPkgInfo(defaultInstallDoc)
	set TedPkgInfo(defaultInstallExamples) $tmpPkgInfo(defaultInstallExamples)
	set TedPkgInfo(defaultInstallData) $tmpPkgInfo(defaultInstallData)
	set TedPkgInfo(tekiFile) $tmpPkgInfo(tekiFile)
	Debug_Leave
}

#**********************************************************************

#
# TedGetDrives
# If the platform is windows,the function sets the
#
#

proc TedGetDrives {} {
	global tcl_platform DriveList

	set DriveList ""
	set pdir [pwd]

	switch $tcl_platform(platform) {
		windows {
			foreach c {c d e f g h i j k l m n o p q r s t u v w x y z} {
				set drive "$c:/"

				if {[catch {cd $drive}] == 0} {
					set DriveList [concat $DriveList "\[-$c-\]"]
				}
			}
		}
	}

	cd $pdir
}

#
#	TedInitPkgInfo
#	Function to initialize project parameters
#
#

proc TedInitPkgInfo {pkgId} {
	global TedPkgInfo TedInfo

	foreach filetype $TedInfo(pkgFileTypes) {
		if {[info exists TedPkgInfo($pkgId,$filetype)]} {
			unset TedPkgInfo($pkgId,$filetype)
		}

		set TedPkgInfo($pkgId,$filetype) ""
	}

	# Initialize object files

	set $TedPkgInfo(systemNames) ""

	foreach sys $TedPkgInfo(systemNames) {
		lappend TedPkgInfo($pkgId,objFiles) [list [list [lindex $sys 0]]]
	}

	foreach element $TedInfo(settingsName) {
		set TedPkgInfo($pkgId,$element) ""
	}

	foreach element $TedInfo(dirOptionsName) {
		if {[info exists TedPkgInfo($pkgId,$element)]} {
			unset TedPkgInfo($pkgId,$element)
		}

		switch $element {
			srcDir {
				set TedPkgInfo($pkgId,$element) "."
			}
			
			docDir {
				set TedPkgInfo($pkgId,$element) ""
			}

# Destination directory is same as the package id.			
#			destDir {
#				set TedPkgInfo($pkgId,$element) [file tail [pwd]]
#			}
			
			copyright {
				set TedPkgInfo($pkgId,$element) ""
			}
			
			infoFile {
				set TedPkgInfo($pkgId,$element) ""
			}
			
			initFile {
				set TedPkgInfo($pkgId,$element) ""
			}
		}
	}

	set TedPkgInfo($pkgId,pkgId) "$pkgId"
	set TedPkgInfo($pkgId,name) "NewBundle"
	set TedPkgInfo($pkgId,version) "1.0"
	set TedPkgInfo($pkgId,description) "New Bundle"
	set TedPkgInfo($pkgId,requires) "Tcl 7.6"
	set TedPkgInfo($pkgId,tekiFile) $TedPkgInfo(tekiFile)
	set TedPkgInfo($pkgId,srcDir) "."
	set TedPkgInfo($pkgId,BaseDir) ""
}

proc TedInitPkgIndependentInfo {} {
	global TedInfo TedPkgInfo

	# Initialize Package Independant Information
	set TedPkgInfo(systemNames) $TedInfo(systemNames)
	set TedPkgInfo(available) ""
	set TedPkgInfo(defaultPackages) ""
	set TedPkgInfo(defaultArch) [lindex [lindex $TedInfo(systemNames) 0] 0]
	set TedPkgInfo(defaultInstallDoc) 1
	set TedPkgInfo(defaultInstallExamples) 1
	set TedPkgInfo(defaultInstallData) 1
	set TedPkgInfo(tekiFile) "NewProject.tek"
	set TedPkgInfo(pkgFilename) ""
}

# Initially I thought that the destination directory can be added to the 
# location of the teki file to get the base directory
# but it is a non trivial problem and many issues to be resolved

# dirname contains the base location to which the destination dir
# of each package is added

proc TedInitBaseDir2 {dirname} {
	global TedPkgInfo

	foreach pkg $TedPkgInfo(available) {
		if {$TedPkgInfo($pkg,destDir) == "" || [llength $TedPkgInfo(available)] == 1} {
			set TedPkgInfo($pkg,BaseDir) $dirname
		} else {
			set TedPkgInfo($pkg,BaseDir) "$dirname/$TedPkgInfo($pkg,destDir)"
		}
	}
}

# Just initialize the base directory to the location of the teki file
# dirname specifies the base directory

proc TedInitBaseDir {dirname} {
	global TedPkgInfo

	foreach pkg $TedPkgInfo(available) {
		set TedPkgInfo($pkg,BaseDir) $dirname
	}
}

# Unsets all information related to a package given by pkgId

proc TedDeletePkgInfo {pkgId} {
	global TedPkgInfo

	# destDir has been removed

	set patterns {",name)" ",version)" ",description)" ",requires)" ",tekiFile)" ",updateURL)"\
			  ",registerURL)" ",srcURL)" ",srcDir)" ",copyright)" ",infoFile)"\
			  ",tclFiles)" ",dataFiles)" ",docDir)" ",docFiles)" ",exampleFiles)" ",objFiles)"\
			  ",initFile)" ",browseURL"}

	foreach pat $patterns {
		if {[info exists TedPkgInfo($pkgId$pat]} {
			unset TedPkgInfo($pkgId$pat
		}
	}
}

#
# TedInit
# Initialize Ted parameters
#

proc TedInit {} {
	global TedInfo StartDir

	# Set File Saved flag
	set TedInfo(fileSaved) 1
	set TedInfo(fileOpen) 1

	#
	# Predefined system names
	#
	set TedInfo(systemNames) {
			{Win95/NT Win* *}
			{Solaris SunOS 5*}
			{SunOS SunOS 4*}
			{HPUX HP* 9*}
			{Linux Linux* 2*}
	}

	TedInitPkgIndependentInfo

	# set the global startup directory variable
	set StartDir [pwd]

	# Sets the DriveList global variable to the drives if any on windows platform
	TedGetDrives

	# List of package setting elements
    set TedInfo(pkgFileTypes) "tclFiles dataFiles docFiles exampleFiles"

	# List of package setting elements
    set TedInfo(settingsName) "name version description requires updateURL registerURL srcURL browseURL"

	# Corressponding Text for each element
	set TedInfo(settingsText) {{Package Name} {Version} {Description} {Package Requires} \
					 {Update URL} {Register URL} {Source URL} {Browse URL}}

	# List of package directory settings elements
    set TedInfo(dirOptionsName) "srcDir docDir copyright infoFile initFile"

	# Corressponding Text for each element
	set TedInfo(dirOptionsText) {{Source Directory} {Document Directory} {Copyright File} \
									{Information File} {Initialization File}}
}

# ---------------------------------------------------------------------
# Functions for reading and saving Teki file

# Function to extract string delimited by braces

proc GetString {strbuf stindex} {
	set t [string index $strbuf $stindex]
	set b1 "{"
	set b2 "}"
#Brace count
	set flag 0
	set flag2 0
# Flag to check if first iteration
	set flag3 0
	set tbuf ""

# Check if there are enclosing brackets

	if {$t != $b1} {
		set flag2 1
	}

	while {1} {
		if {$flag2 == 0 && $t == $b1} {
			set flag [expr $flag+1]
		}

		if {$flag2 == 0 && $t == $b2} {
			set flag [expr $flag-1]
		}

# Filter opening set of braces
		if {($flag > 0 && $flag3 == 1) || $flag2 == 1} {
			set tbuf $tbuf$t
		}

		set flag3 1

		set stindex [expr $stindex+1]
		set t [string index $strbuf $stindex]

		if {$flag2 == 0 && $flag == 0} {
			break
		} elseif {$flag2 == 1 && $t == "\n"} {
			break
		} elseif {$flag2 == 1 && $t == ""} break

# Return error if closing brace not found and end of string reached
		if {$t == ""} {
			return -1
		}

	}

	return $tbuf
}

#
# Function:	TedReadVersion1.0
#			Read .tek file and load the various parameters
# Parameters:	tek_fname	- ".tek" file name
# Returns:		1 On Success
#				-1 If unable to open file
#

proc TedReadVersion1.0 {tek_fname} {
global TedInfo TedPkgInfo

Debug_Enter TedReadVersion1.0 

# In the patterns list the first element is the regular expression
# and the second is the variable name. The third element specifies
# wether to take the variable name as is or not (0/1)

# Important: Do not move the identifiers and variables,position matters

set patterns {{" *System *Names *= *" "TedPkgInfo(systemNames)" 0}
			  {" *Package *Id. *= *" "pkgid" 0}
			  {" *Package *Name *= *" ",name)" 1}
			  {" *Version *= *" ",version)" 1}
			  {" *Description *= *" ",description)" 1}
			  {" *Requires *= *" ",requires)" 1}
			  {" *Teki *File *= *" ",tekiFile)" 1}
			  {" *Update *URL *= *" ",updateURL)" 1}
			  {" *Register *URL *= *" ",registerURL)" 1}
			  {" *Source *URL *= *" ",srcURL)" 1}
			  {" *Source *Directory *= *" ",srcDir)" 1}
			  {" *Destination *Directory *= *" ",destDir)" 1}
			  {" *Copyright *= *" ",copyright)" 1}
			  {" *Information *File *= *" ",infoFile)" 1}
			  {" *Tcl *Files *= *" ",tclFiles)" 1}
			  {" *Data *Files *= *" ",dataFiles)" 1}
			  {" *Document *Directory *= *" ",docDir)" 1}
			  {" *Document *Files *= *" ",docFiles)" 1}
			  {" *Example *Files *= *" ",exampleFiles)" 1}
			  {" *Object *Files *= *" ",objFiles)" 1}
			  {" *Initialization *File *= *" ",initFile)" 1}
			  {" *Browse *URL *= *" ",browseURL)" 1}
			  {" *Default *Packages *= *" "TedPkgInfo(defaultPackages)" 0}
			  {" *Default *Architecture *= *" "TedPkgInfo(defaultArch)" 0}
			  {" *Default *Install *Documentation *= *" "TedPkgInfo(defaultInstallDoc)" 0}
			  {" *Default *Install *Examples *= *" "TedPkgInfo(defaultInstallExamples)" 0}
			  {" *Default *Install *Data *= *" "TedPkgInfo(defaultInstallData)" 0}}

    if {![string length $tek_fname] || ![file exists $tek_fname] || ![file readable $tek_fname]} {
			error "Unable to read version 1.0 TEKI file $tek_fname."
			return
    }

	file stat $tek_fname flstat
    set f [open $tek_fname r]
	set fl_buf ""
	set chars_read 1
	set nextPackage ""
	set pkgno 0
	set pat [lindex $patterns 1]

	while {$chars_read >= 0} {
		set fl_buf $nextPackage

		while 1 {
			set chars_read [gets $f line]

			if {$chars_read < 0} {
				break
			}

			# Check for the start of the next package parameters
			# Indicated by the line containing "Package Id. = ???"

			if {[regexp -indices [lindex $pat 0] $line x] != 0} {
				if {$pkgno} {
					set nextPackage $line
					incr pkgno 1
					break
				}

				incr pkgno 1
			}


	# Remove all square brackets
	#			regsub -all \[|\] $line \# line
			regsub -all -nocase {\#+[^#]*} $line "" line

			if {[string length $line] != 0} {
				set fl_buf "$fl_buf\n$line"
			}
		}

		set j 0
			 
		foreach i $patterns {
			if {[regexp -indices [lindex $i 0] $fl_buf x] != 0} {
				set tstr [GetString $fl_buf [expr [lindex $x 1]+1]]

				if {$tstr == -1} {
					global errorCode errorInfo
					error "Error reading version 1.0 TEKI file $filename. Missing brace !" \
					$errorInfo $errorCode
				}

	# Check if variable name is substituted as is

				if {[lindex $i 2] == 0} {
					set varname [lindex $i 1]
				} else {
					set varname "TedPkgInfo($pkgid[lindex $i 1]"
				}

				if {($j == 14 || $j == 15 || $j == 17 || $j == 18) && ([string first "*" $tstr] != -1 || [string first "?" $tstr] != -1)} {
					set $varname ""

					foreach filename $tstr {
						if {[string first "\{" $filename] == -1 && [string first "\}" $filename] == -1} {
							lappend $varname [glob $filename]
						}
					}
				} else {
					set $varname $tstr
				}


	#			set x "puts \$$varname"
	#			eval $x
			}

			set j [expr $j+1]
		}

		if {$pkgno == 1} {
			set TedPkgInfo(available) $pkgid
		} else {
			lappend TedPkgInfo(available) $pkgid
		}
	}

	close $f

	Debug_Leave
}

#
# Function:	TedWriteVersion1.0
#			Write .tek file
# Parameters:	tek_fname	- ".tek" file name
# Returns:		1 On Success
#				-1 If unable to open file
#

proc TedWriteVersion1.0 {tek_fname} {
	global TedInfo TedPkgInfo

	Debug_Enter TedWriteVersion1.0

	# In the patterns1 list the first element is the descriptor
	# and the second is the variable name.

	set patterns1 {{"Package Name = " "name"}
			  {"Version = " "version"}
			  {"Description = " "description"}
			  {"Requires = " "requires"}
			  {"Teki File = " "tekiFile"}
			  {"Update URL = " "updateURL"}
			  {"Register URL = " "registerURL"}
			  {"Source URL = " "srcURL"}
			  {"Source Directory = " "srcDir"}
			  {"Destination Directory = " "destDir"}
			  {"Copyright = " "copyright"}
			  {"Information File = " "infoFile"}
			  {"Tcl Files = " "tclFiles"}
			  {"Data Files = " "dataFiles"}
			  {"Document Directory = " "docDir"}
			  {"Document Files = " "docFiles"}
			  {"Example Files = " "exampleFiles"}
			  {"Object Files = " "objFiles"}
			  {"Initialization File = " "initFile"}
			  {"Browse URL = " "browseURL"}}

	set patterns2 {{"Default Packages = " "defaultPackages"}
			  {"Default Architecture = " "defaultArch"}
			  {"Default Install Documentation = " "defaultInstallDoc"}
			  {"Default Install Examples = " "defaultInstallExamples"}
			  {"Default Install Data = " "defaultInstallData"}}

	if {[catch {set f [open $tek_fname w]}]} {
		TedError "Unable to write version 1.0 TEKI f $tek_fname."
		return
	} else {
		puts $f "# TekiFile 1.0\n#\n# TEKI install file\n#"

		puts $f "System Names = \{$TedPkgInfo(systemNames)\}"
		
		foreach pkg $TedPkgInfo(available) {
			puts $f "\n###############################################################"
			puts $f "# Information for Package: $pkg\n"
			puts $f "Package Id. = $pkg"
			set TedPkgInfo($pkg,tekiFile) $TedPkgInfo(tekiFile)

			foreach varname $patterns1 {
				if {[lindex $varname 1] == "destDir"} {
					if {[info exists TedPkgInfo($pkg,destDir)]} {
						 if {$TedPkgInfo($pkg,destDir) != ""} {
							puts $f "[lindex $varname 0]\{$TedPkgInfo($pkg,destDir)\}"
						 } else {
							puts $f "[lindex $varname 0]\{$pkg\}"
						 }
					} else {
							puts $f "[lindex $varname 0]\{$pkg\}"
					}
				} else {
					if {[info exists TedPkgInfo($pkg,[lindex $varname 1])]} {
						puts $f "[lindex $varname 0]\{$TedPkgInfo($pkg,[lindex $varname 1])\}"
					} else {
						puts $f "[lindex $varname 0]\{\}"
					}
				}
			}
		}

		puts $f "\n###############################################################"
		puts $f "\n# Default Settings\n"
		foreach varname $patterns2 {
			if {[info exists TedPkgInfo([lindex $varname 1])]} {
				puts $f "[lindex $varname 0]\{$TedPkgInfo([lindex $varname 1])\}"
			} else {
				puts $f "[lindex $varname 0]\{\}"
			}
		}

		close $f
	}

	Debug_Leave
}


# ---------------------------------------------------------------------
#
# Initialization code
#

if $TedInfo(gui) {
	TedInit
    TedCreateUI
    wm deiconify .
}

if {$tk_version < 8.0} {
   proc fcopy args {
        eval unsupported0 $args
    }
}
