/*
 * Nest simulation library - process stack memory management routines
 * 
 * $Header: swap.c,v 2.5 88/03/31 14:50:42 dupuy Rel $
 */

#include "nest.h"

#include "defs.h"
#include "swap.h"

/*
 * This routine preallocates memory space for the heap, putting it on the
 * freelist, used to keep track of all the free slots in the heap, and also
 * initializes the swaptable, an array indexed by node id of allocated slots.
 * swaptable[0] is a header node, whose forward pointer is the swaplist and
 * backward pointer is freelist, such that swaplist is a forward linked ring,
 * freelist is a forward linked ring, and together they form a backward linked
 * ring.
 */

_swap_init (blocks, size)
unsigned        blocks,
                size;
{
    /* array of allocated swap_elems for nodes */

    if ((swaptable = newarray (blocks + 1, swap_elem)) is nil)
        return (_error ("too many nodes: %s"));

    /* create first node of freelist */

    freelist = new (swap_elem);

    /* allocate space on the heap for the swap area */

    if ((baseptr = (pointer) malloc (size)) is nil)
        return (_error ("initial stack storage too large: %s"));

    stacksize = size;

    freelist->blockhead = baseptr;
    freelist->size = size;

    /* connect freelist, swaplist into a ring */

    swaplist = swaptable;               /* initially, node zero on swaplist */
    freelist->next = freelist;          /* initially, freelist has one node */
    freelist->next->back = swaptable;   /* swaplist,freelist */

    return (None);                      /* no errors */
}



/*
 * This routine allocates space for the node specified as blockid, taking
 * space from a freelist entry, and updating the lists as appropriate.  It
 * returns a pointer to the (uninitialized) space, nil if an error occurred.
 */


pointer         _allocate (blockid, size)
ident           blockid;                /* node to allocate space for */
unsigned        size;                   /* size of space requested */
{
    extern pointer  getspace ();
    swap_ptr        parent;             /* passed from getspace to insert */

    if ((swaptable[blockid].blockhead   /* successful allocation */
         = getspace (size, &parent))
            isnt nil)
    {
        swaptable[blockid].size = size; /* copy size */

        insert (blockid, parent);       /* insert swap_elem on swaplist */
    }

    return (swaptable[blockid].blockhead);
}


/*
 * This routine deallocates space for the node specified as blockid, putting
 * the freed space back on the freelist, and deleting the swap_elem from the
 * swaplist (it is still on the swaptable).  It returns a pointer to the space
 * being deallocated, without destroying its contents.  It sets the size
 * argument to the size of the space deallocated.
 */

pointer         _deallocate (blockid, size)
ident           blockid;
unsigned       *size;
{
    extern pointer  putspace ();
    pointer         blockptr;

    *size = swaptable[blockid].size;    /* return size of memory being freed */

    blockptr = putspace (blockid);      /* put back on freelist, return ptr */

    if (blockptr isnt nil)              /* delete from swaplist, still on
                                         * swaptable */
        delete (blockid);

    return (blockptr);                  /* return pointer to freed space */
}



/*
 * This routine looks through freelist to see if a suitable memory slot is
 * available.  If one is found, it returns the address of the memory slot, if
 * not, it calls trushcollect() to do storage compaction and checks again. If
 * there is still not enough space, returns nil.  It sets the parent arg to
 * point at the freelist parent of the allocated block.
 */

static pointer  getspace (size, parent)
unsigned        size;                   /* size requested */
swap_ptr       *parent;                 /* pointer to parent freenode */
{
    swap_ptr        freeptr;            /* local ptr to freelist swap_elem */
    pointer         blockptr;           /* return pointer */

    freeptr = freelist;                 /* initialize freeptr */

    do                                  /* for each possible free block */
    {
        if (freeptr->size >= size)      /* block is large enough; use it */
        {
            blockptr = freeptr->blockhead;

            freeptr->blockhead += size; /* reduce free block space by size */
            freeptr->size -= size;

            *parent = freeptr;          /* save freeptr */
            return (blockptr);          /* return ptr to space allocated */
        }

        nextn (freeptr);                /* work towards higher blocks */
    }                                   /* (reduces need for compaction) */
    while (freeptr isnt freelist);

    if (trushcollect (size))            /* if compaction found enough space */
    {
        blockptr = freeptr->blockhead;

        freeptr->blockhead += size;     /* reduce free block space by size */
        freeptr->size -= size;

        *parent = freeptr;              /* save freeptr */
        return (blockptr);              /* return ptr to space allocated */
    }

    return (nil);                       /* still not enough space */
}



/*
 * This routine puts the freed space back on a freelist node and adjusts the
 * block sizes.  It returns a pointer to the freed space without destroying
 * it.
 */

static pointer  putspace (blockid)
ident           blockid;
{
    swap_ptr        swapptr;


    swapptr = &swaptable[blockid];

    /* add space back to lower slot on freelist */

    swapptr->back->size += swapptr->size;

    /* add space (may be 0) in the upper slot to the lower slot */

    swapptr->back->size +=
        swapptr->back->next->size;

    /* return pointer to space returned */

    return (swapptr->blockhead);
}



/*
 * This routine inserts swaptable[blockid] onto swaplist, and creates an empty
 * freenode to match it, which is inserted onto the freelist.
 */

static          insert (blockid, parent)
ident           blockid;                /* node to be inserted */
swap_ptr        parent;                 /* where the free space was obtained */
{
    swap_ptr        freeptr,            /* local ptr to sblock */
                    swapptr;

    swapptr = &swaptable[blockid];      /* insert swaplist node */

    swapptr->next = parent->back->next;
    parent->back->next = swapptr;

    freeptr = new (swap_elem);          /* insert new empty freelist node */
    freeptr->next = parent;
    parent->back->back->next = freeptr;

    /* empty freelist node points to the lower boundary of parent slot */

    freeptr->blockhead = swapptr->blockhead;
    freeptr->size = 0;

    freeptr->back = parent->back;       /* link the two nodes into back ring */
    swapptr->back = freeptr;
    parent->back = swapptr;
}

/*
 * This routine deletes swaptable[blockid] from the swaplist, along with the
 * corresponding freelist node.  It is still on swaptable, however, and the
 * information is not damaged.
 */

static          delete (blockid)
ident           blockid;
{
    swap_ptr        swapptr,            /* ptr to (soon) deleted swapnode */
                    freeptr,            /* ptr to (soon) deleted freenode */
                    parent,             /* ptr to combined freenode */
                    next;


    swapptr = &swaptable[blockid];      /* set four pointers */
    parent = swapptr->back;
    freeptr = parent->next;
    next = swapptr->next;

    parent->next = freeptr->next;       /* skip freeptr going forwards */
    parent->back->next = next;          /* skip swapptr going forwards */
    next->back = parent;                /* skip both going backwards */

    swapptr->blockhead = nil;           /* reset swapptr node */
    swapptr->size = 0;
    parent = nil;
    next = nil;

    dispose (freeptr);                  /* trash freeptr node */
}


/*
 * This routine does garbage compaction. It moves the allocated memory slots
 * toward the lower address end of the heap, and combines the free space into
 * the freelist entry at the high end of the list.
 */

static          trushcollect (size)
unsigned        size;
{
    swap_ptr        swapptr,            /* ptrs moving along the lists */
                    freeptr;
    pointer         storage;
    int             offset;
    extern          copybytes ();


    freeptr = freelist->next;           /* start from the lowest slot */
    swapptr = swaplist;

    while (swapptr isnt swaptable)
    {
        if (freeptr->size isnt 0)       /* there is room to collect */
        {
            bcopy (swapptr->blockhead, freeptr->blockhead, (int) swapptr->size);

            swapptr->blockhead = freeptr->blockhead;

            freeptr->next->blockhead -= freeptr->size;
            freeptr->next->size += freeptr->size;
            freeptr->size = 0;          /* bubble down the space */
        }

        nextn (freeptr);
        nextn (swapptr);
    }

    if (freeptr->size < size)
    {
        offset = grab (size, freeptr->size);

/*
        (void) fprintf (stderr,
              "%d bytes stack storage insufficient: getting %d more bytes\n",
                        stacksize, offset);
*/

        if ((storage = realloc (swaplist->blockhead, stacksize + offset))
                is nil)
            return (false);             /* not enough memory available */

        StackSize = stacksize += offset;
        freeptr->size = size;

        offset = storage - baseptr;     /* calculate relocation of storage */

        if (offset)                     /* storage area was moved */
        {
            offsetblocks (freelist, offset);
            offsetblocks (swaplist, offset);
        }
    }

    return (true);                      /* return true if space was found */
}


static          grab (size, freesize)
unsigned        size,
                freesize;
{

    /*
     * Grab additional space to bring stack area up to multiple of 512, less
     * 16 bytes overhead.  This could be more intelligent, but it's not bad.
     */

    return (((size + 15) / 512 + 1) * 512 - freesize - 16);
}

static          offsetblocks (blocklist, offset)
swap_ptr        blocklist;              /* list of blocks to be offset */
int             offset;                 /* offset to be added to blockhead */
{
    swap_ptr        blockptr = blocklist;

    do                                  /* always do first entry on list */
    {                                   /* (either baseptr or first free) */
        blockptr->blockhead += offset;
        nextn (blockptr);
    }
    while (blockptr isnt blocklist);
}

#ifdef debug
static          dumptable ()
{
    swap_ptr        blockptr = swaptable;
    bool            free = true;


    (void) printf ("\nswap storage: 0x%lx-0x%lx\t[%ld bytes]\n",
                   blockptr->blockhead, blockptr->blockhead + stacksize - 1,
                   stacksize);

    nextf (blockptr, back);

    while (blockptr isnt swaptable)
    {
        if (blockptr->size)
            (void) printf ("%s block: 0x%lx-0x%lx\t[%ld bytes]\n",
                           free ? "free" : "used",
                           blockptr->blockhead,
                           blockptr->blockhead + blockptr->size - 1,
                           blockptr->size);
        else
            (void) printf ("%s block: 0x%lx\t\t[%ld bytes]\n",
                           free ? "free" : "used",
                           blockptr->blockhead, blockptr->size);

        free = not free;
        nextf (blockptr, back);
    }
}

#endif
