/*
 * Nest simulation library - main simulation loop routines
 * 
 * Completely rewritten 7/97... S. Keshav, Cornell University
 */

 /* What is the main idea of the simulator?
 *
 *	The key idea in the simulator is to call a function, have it
 * execute, then block, so that other functions can run. On blocking,
 * a node can either ask to wake up at a specific time, or block until
 * it receives a message.
 *
 * How do you call a function?
 * 	The function pointer is stored in a "process table", and
 * called by the simulate() function.
 * 
 * Which function do you call next?
 *	The one that has the earliest unblock time. In the beginning
 *  all node functions are put on a list and started one by one. When
 *  a node function executes, it is allowed to send messages, which 
 *  cause other nodes to unblock in the future. I put this node in the
 *  event queue, and when it's time comes, it is executed. Note that the
 *  event queue has only one entry per node. If a node could be unblocked
 *  multiple times (because of multiple messages waiting for it), I
 *  only add it to the event queue for the earliest message.
 * 
 *  Does that mean that a node may need to be taken off the event queue
 *  and added to an earlier position when a mesage arrives?
 * 	Yes. See ipc.c to see how this is done.
 * 
 * What happens when a function blocks?
 * 	I store its stack state and register contents, then 
 * jump to the start of the simulator main loop, where we call 
 * the next function to be run.
 * 
 * How do you restore a function?
 * 	I copy back the registers and stack state, then call
 * the function normally, as I would for a procedure call.
 * 
 * Ah, so that means that to make my simulation run faster, I should
 * have as little state as possible on a node function's stack.
 * 	Yes.
 * 
 * Great, now I understand the idea. Let me read the code.
	Have fun!
 */

#include "nest.h"
#include "graph.h"

#define XSize 8192                      /* size of temporary stack */
#include <errno.h>

#include "defs.h"
#include "process.h"
#include "schedule.h"
#include "simulate.h"

/* To turn on debugging,#define D(x) x */
#define D(x) 

/*
 * Define internal global variables which are used throughout the simulator.
 */

jmp_buf  _scheduler;             /* jump buffer for top of "loop" */
char    *_stackbase;             /* base of the swapped stack */
timev    _passtime;              /* length of simulation passes */
timev    _sync_time;             /* synchronization time of this pass */

int      errno;

int             simulate (nodes, stacksize, initgraph, port)
unsigned        nodes,                  /* number of nodes in graph */
                stacksize;              /* total size of all nodes' stacks */
graph          *initgraph;              /* initial network graph */
int             port;                   /* port number for simulation */
{
    char            tmpstack[XSize];
    int             state ;
    ident       next_node;
    process    * proc;


    /* Initializations */

    Nodes = nodes;                      /* [re]initialize globals */
    StackSize = stacksize;

    _sched_init (nodes);                /* all on done_list, n_id = node */
    _network_init (nodes);              /* no connections anywhere */
    _message_init (nodes);              /* no messages for anybody   */
    					
    if (_serv_init (nodes, initgraph, port))
        return (_complain (""));	/* initial network configuration */

    if (_swap_init (nodes, stacksize))  
        return (_complain (""));	/* set up area for storing stacks */

    /* start the monitor for the first time */
    Graph->header->timenow = _sync_time;
    (*(Graph->header->monitor)) (Graph);

    /* top of "loop" created by setjmp() and longjmp() */
    _stackbase = nest_stack ();
    state = setjmp (_scheduler);        

    /* here is the call to the scheduler to find the node that has
     * the first unblock time. 
     */
    _current_node = _get_next_node ();

    D(printf("new current node is %d\n", _current_node);)

    if(_current_node is -1) 	/* end of simulation */
	return;

    if(_current_node is 0) /* special case for the monitor */
			   /* reinsert into event queue after pass time */
    {
	Graph->header->timenow = _sync_time;
	(*Graph->header->monitor) (Graph);
        time_add (_sync_time, _passtime);

	/* insert this node for the next pass */
	_proc_table[0].unblocked = _sync_time;
	_sched_table[0].unblocked = _sync_time;
	_insert_sched(0);
	longjmp(_scheduler, Normal);
    }
    else
    {
	/* proc has everything we want to know about the function */
        proc = &(_proc_table[_current_node]);
	proc->runtime = proc->unblocked;

        if (not proc->started)  /* start function on node 
				 * for the first time */
        {
	    proc->started = true;
	    if(proc->function) {
	        D(printf("starting function on node %d\n", _current_node);)
		(*proc->function) (_current_node);   /* start the process */
	    }
	    /* we should never get here, since a node function is typically
	     * an infinite loop, but who knows ! 
	     */
	    proc->started = false;
	    longjmp(_scheduler, Normal);
        }
        else		/* restore the node to previous state */
        {
            /* copy its stack back and dealloc */
	    D(printf("restoring function on node %d\n", _current_node);)
#if STACKDIR < 1
            nest_xstack (tmpstack + sizeof (tmpstack) - 256);
#else                                   /* -256 above in case we postdecrement */
            nest_xstack (tmpstack);
#endif
            if (nest_restore (_current_node))
                fatal ("simulate: restore: ");

	    /* now jump back to the point where the node blocked (see below) */
#ifdef __alpha
	    longjmp(_proc_table[_current_node].environment, 1); /* start running it again */
#else
            nest_cook (&(_proc_table[_current_node].environment)); /* start running it again */
#endif
        }
    }
}

/*
 * Blocks a node. Sets the unblocked time of this
 * node to the `unblocktime', and the reason for blocking to `reason'. Will
 * return only after process runtime is at least unblocktime (if unblocktime
 * value is zero, it may never return).
 */

_block (unblocktime, reason)
timev           unblocktime;
int             reason;
{
    process *proc = &(_proc_table[_current_node]);
    extern int      errno;
    int             olderrno;
    pointer         stack;


    D(printf("blocking on node %d, time %d, %d\n", 
	_current_node, unblocktime.tv_sec, unblocktime.tv_usec);)

    /* for efficiency, the unblock time is stored both in the proc
     * table and the sched table 
     */
    _proc_table[_current_node].unblocked = unblocktime;
    _sched_table[_current_node].unblocked = unblocktime;

    /* insert it in the right place, depending on the unblock time */
    _insert_sched(_current_node);

    stack = nest_stack ();		/* find the  stack pointer of this frame*/
    if (nest_store (stack, 0))	 	/* save stack in heap */	
        fatal ("_block: store: ");

#ifdef __alpha
    if(not setjmp(proc->environment)) /* store registers */
#else
    if(not nest_freeze (&proc->environment))	/* store registers */
#endif
        longjmp (_scheduler, Normal);       /* stop running this node */

    /* return here when restarting this thread */
    D(printf("restored node %d\n", _current_node);)
    proc->whyblocked = UNBLOCKED;

    return (None);
}
