/*
 * Nest simulation library - scheduler routines
 * Merged scheds.c sched2s.c procs.c timers.c 6/16/97.. keshav
 * Completely cleaned up 7/7/97
 */

#include "nest.h"
#include "graph.h"
#include "defs.h"

#include "simulate.h"
#include "schedule.h"
#include "process.h"

/* to turn on debugging, #define D(x) x */
#define D(x) 

/*
 * The `_sched_table', which holds the four distinct scheduling queues that
 * make up the scheduler database, is an array (indexed by node id's) of
 * "node_queue" structures, with embedded node queues known as the execution,
 * and blocked, queues, and is defined like this:
 */

nodeptr  _sched_table;           /* size of `nodes' arg to simulate() */

nodeptr  _blocked_queue;         /* nodes which have blocked on recv */
nodeptr  _wait_queue;            /* #1 waiting node on exec. queue */

process *_proc_table;            /* list of processes waiting to run */
char	*_scheduled;		 /* vector of scheduled processes */

/*
 * Initialization and maintenance of scheduler table
 */

_sched_init (nodes)
ident           nodes;
{
    reg ident       id;


    _sched_table = newarray (nodes + 1, node);
    _blocked_queue = nil;
    _wait_queue = nil;         

    _proc_table = newarray (nodes + 1, process);
    _proc_table[0].function = (funcptr) Graph->header->monitor;	
				/* store monitor in node 0 */

    _scheduled = newarray (nodes + 1, char);

    /* put all the nodes in the wait queue */

    id = 0;
    while (id <= nodes)
    {
        _wait_queue = _push_node (id, _wait_queue);
        _wait_queue->n_id = id++;
    }

    /* first time to call the monitor */
    _sync_time = time_zero;     
}

/* add a node to the head of a queue */

node           *_push_node (nodeid, squeue)
ident           nodeid;
register node  *squeue;
{
    node  *firstnode = &(_sched_table[nodeid]);

    _scheduled[nodeid] = 1;
    _proc_table[nodeid].unblocked = time_zero;
    _sched_table[nodeid].unblocked = time_zero;

    firstnode->next = squeue;
    if (squeue isnt nil)
        squeue->prev = firstnode;
    return (firstnode);
}


/* Remove a node from a wait queue. This is O(1) */

_remove_sched (nodeid, state)
ident           nodeid;
int             state;
{
    node  *dropnode = &(_sched_table[nodeid]),
          *nextnode = dropnode->next,
          *prevnode = dropnode->prev,
          **queuep;



    if (nextnode isnt nil)
        nextnode->prev = prevnode;
    if (prevnode isnt nil)
        prevnode->next = nextnode;

    switch (state)
    {

      case BlockedQueue:
        queuep = &_blocked_queue;
        break;

      case WaitQueue:
        queuep = &_wait_queue;
        break;

    }

    if (*queuep is dropnode)
        *queuep = nextnode;

    dropnode->next = dropnode->prev = nil;
    _scheduled[nodeid] = 0;
}

/* unlink and hand over the next node to simulate */

ident           _get_next_node ()
{
    node  *curnode;


    curnode = _wait_queue;

    if(_wait_queue isnt nil)
    {
    	_wait_queue = _wait_queue->next;
	if(_wait_queue isnt nil)
            _wait_queue->prev = nil;
    }

    if(curnode isnt nil)
    {
        /* unlink current node from everything else */
        curnode->next = curnode->prev = nil;
        _scheduled[curnode->n_id] = 0;
	return curnode->n_id;
    }
    else
	return -1;
}

/* Insert a blocked node onto the appropriate queue. */

_insert_sched (nodeid)
ident           nodeid;
{
    node  	   *newnode = &(_sched_table[nodeid]);
    timev          unblocktime;
    node * 	tmp;


    /* can't schedule a node twice */
    if(_scheduled[nodeid])
	return;
    else
	_scheduled[nodeid] = 1;

    unblocktime = _proc_table[nodeid].unblocked;

    D(printf("Inserting node %d, unblocktime %d,%d\nBefore:\n", 
	nodeid, (int)unblocktime.tv_sec, (int)unblocktime.tv_usec);)
    D(print_wait();)
    D(printf("After:\n");)

    newnode->prev = newnode->next = nil;

    if (time_iszero (unblocktime)) /* put in blocked queue */
    {
        _blocked_queue = _push_node (nodeid, _blocked_queue);
	D(printf("inserted in blocked queue\n");)
	return;
    }

    if(_wait_queue is nil)
    {
        _wait_queue = newnode;
        _wait_queue->n_id = nodeid;
     	D(print_wait();)
	return;
    }

    /* the next line is the heart of the simulator: find the sorted
     * position to insert a node. Should be replaced by a heap (but
     * this will make the deletion time O(log N). This list will
     * have at most N elements, where N is the number of nodes in
     * this simulation, so a simple linked list is fastest for N < 50.
     */

    for(tmp = _wait_queue; 
	(tmp->next isnt nil) and (time_after (unblocktime, tmp->unblocked)); 
	tmp = tmp->next);

    if(tmp->next is nil and time_after(unblocktime, tmp->unblocked)) 
	/* really broke out because of end of list */
    {
	tmp->next = newnode;
	newnode->prev = tmp;
    }
    else
    {
	if(tmp->prev)
	    tmp->prev->next = newnode;
        else
	    _wait_queue = newnode;

       	newnode->prev = tmp->prev;
       	newnode->next = tmp;
       	tmp->prev = newnode;
    }
    D(print_wait();)
}

/* print out the wait queue */
print_wait()
{
    nodeptr tmp;

    for(tmp = _wait_queue; tmp isnt nil; tmp = tmp->next)
	printf("node %d, unblocktime %d, %d\n", tmp->n_id, 
	    tmp->unblocked.tv_sec, tmp->unblocked.tv_usec);

}

