/*
 * Nest simulation library - process definitions
 */

/* 
 * A 'process' is a function running on a node. A process can be in
 * one of two states: blocked or ready. If it is blocked, it is put
 * in a blocked queue. Otherwise, it is in the wait queue, with its
 * unblocked time set to the earliest time it can awake. A blocked
 * node has an unblock time of zero.
 *
 * The key data strcutures in teh simulator are the process table
 * and the message table. The process table stores the node id
 * its state, the function it runs, and the earliest time it can
 * run (its unblocked time). Nodes in teh process table are linked
 * into a linked list. We always run the function at the head of
 * of the 'ready' list.
 *
 * On sending a message, we move a node from the blocked state to
 * the unblocked state, if necessary. If the message arrives earlier
 * than an existing message, the node is rescheduled. Otherwise, it
 * is left as is. On a recvm(), we _always_ block until the earliest
 * available message.
 * 
 * A process state is a compound structure as defined below:
 */

/* for DEC Alpha, we'll use setjmp/longjmp instead of cook/freeze */

#ifdef __alpha
#include <setjmp.h>
#define istate jmp_buf
#else
#include "state.h"                      /* to get local "istate" definition */
#endif

typedef struct process_state
{
    funcptr function;                   /* pointer to function for this node */

    flag            started;            /* true if started, false otherwise */
    int             whyblocked;         /* cause of blocking (see below) */

    timev	    runtime;		/* current time at this node */
				
    timev           unblocked;          /* time when this node could restart 
					 * time_zero => waiting on recv */
					/* could change while node is running */
    istate          environment;        /* buffer for internal machine state */
} process;

/*
 * The function field is the function which is running as this process.
 * 
 * The started field indicates the state of the process's function.  It is set to
 * true when the function is called, and set to false when the function
 * returns.
 * 
 * The whyblocked field indicates the cause of blocking, if any, and also whether
 * a node is blocked at all.  It is set by the _block() function and cleared
 * (set to 0 = UNBLOCKED) each time the node is started or restarted.
 * 
 * The unblocked field is the time at which a node may be scheduled for activity.
 * This may be the arrival time of the first message which is available to
 * unblock the process if it is waiting. 
 * The scheduler tables are sorted by this field;
 * The runtime is the current time of the node. 
 * They are all "timeval" structures as defined in the file
 * <sys/time.h>, with "long" fields for both microseconds and seconds.
 * The environment field is a buffer for the machine (dependent) state, and
 * should only be used by the assembler routines that do the switching of
 * processes.  It is similar to a "jmp_buf" structure, as defined in the file
 * <setjmp.h>, but may vary from machine to machine.
 */

/*
 * The following are the codes defined for the whyblocked field.
 */

#define UNBLOCKED    0                  /* not blocked at all */
#define RECEIVE     -1                  /* blocked on receive from anyone */
#define ADVANCE     -4                  /* blocked to use up fake cpu time */
/*
 * The following are integer codes for process states which are actually
 * represented by being in one of the various scheduler queues.  They are used
 * as arguments by functions which manipulate the scheduler table, so that a
 * function which deals with the process table can ignore scheduler types.
 */

#ifndef DoneQueue
#define DoneQueue            0          /* = started is false */
#define BlockedQueue        -1          /* = not time_nonzero(unblocked) */
#define WaitQueue            1          /* = time_after(sync_time,unblocked) */
#endif

/*
 * The `_proc_table', which keeps track of the state of the function
 * associated with each node, is simply an array (indexed by node id's) of
 * "process_state" structures, and is defined like this:
 */

extern process *_proc_table;            /* size of `nodes' arg to simulate() */

/* scheduled is 1 if the node is in the wait queue, and 0 otherwise.
 * Used to prevent double scheduling of a node */

extern char *_scheduled;            	/* size of `nodes' arg to simulate() */
