/*
 * Nest simulation library - graphlanguage definitions
 * 
 * $Header: graph.h,v 2.5 88/03/31 14:49:19 dupuy Rel $
 */

/*
 * These structures may be changed in future versions of Nest, and won't exist
 * in Net-MATE.  So try to keep any code which uses anything other than
 * position, real_point, or node_list structures in its own module.
 */

#define word  unsigned long             /* a 4 byte unsigned */
#define half  unsigned short            /* a 2 byte unsigned */
#define byte  unsigned char             /* a 1 byte unsigned */

/*
 * A graph is defined as having a header, a list of nodes in the graph, and a
 * list of edges which each connect two nodes.  While in byte-stream form, the
 * endpoints of an edge are referred to by their order in the node list; in
 * the graphlanguage data structures, pointers are used.
 */

#define nil_node    0                   /* undefined node number */
#define Node_Base   1                   /* first node number */

#define Graph_Magic ((word) 0xAD86DFB2) /* all tools initials are in hex */
#define Header_Size 32                  /* size of header (in bytes) */

/*
 * A node_list is a single-linked list of struct node_list sorted by ident.
 * Inherited by some IPLS types, and used by Nest and some other programs
 * using Graph Language.  Not properly part of Graph Language, but a useful
 * structure which doesn't have any better place to be.
 */

typedef struct
{
    long            x;
    long            y;
    long            z;
} position;

typedef struct
{
    double          x;
    double          y;
    double          z;
} real_point;

#define grnodedat struct nest_node

typedef struct graph_node *grnodeptr;
typedef struct graph_node               /* a node */
{
    grnodeptr       next;
    grnodedat      *nodedata;           /* user-defined data structure */
    word            number;             /* node number; not always valid */
} grnode;

#define gredgedat struct nest_edge

typedef struct graph_edge *gredgeptr;
typedef struct graph_edge               /* an edge connecting two nodes  */
{
    gredgeptr       next;
    gredgedat      *edgedata;           /* user-defined data structure */
    grnodeptr       node1,              /* edge endpoints */
                    node2;              /* if directed graph; from 1 to 2 */
} gredge;

#define grhead struct nest_head

typedef struct graph
{
    grnodeptr       nodes;              /* linked list of nodes  */
    gredgeptr       edges;              /* linked list of edges  */
    half            flags;              /* topological and data flags */
    half            protocol;           /* system/version of user-defined */
    grhead         *header;             /* user-defined data */
} graph;

#define Directed    0x0001              /* edges imply 1=>2 connection only */
#define Divided     0x0002              /* graph not fully connected */
#define Loopbacks   0x0004              /* edges may connect node to itself */
#define Bare_Nodes  0x0008              /* not all nodes in valid edges */
#define Hang_Edges  0x0010              /* not all edges have good endpoints */
#define Topo_Flags  0x001F

#define Node_IDs    0x0100              /* ID number present */
#define Node_Names  0x0200              /* string name in next bytes  */
#define Positions   0x0C00              /* no, 1-, 2-, or 3-D positions */
#define Pos_Shift   10                  /* bit offset of Positions, above */
#define Pos_Real    0x1000              /* floating point/integer Positions */
#define Edge_Weight 0x2000              /* integer edge weights */
#define Real_Weight 0x4000              /* floating point edge weights */
#define Data_Flags  0x7F00
#define Nest_Graph  0xEAD0              /* Nest simulation networks */

/*
 * Graph operations
 */

extern graph   *readgraph ();
extern grhead  *readheader ();
extern grnodedat *readnode ();
extern gredgedat *readedge ();
extern byte    *writegraph ();
extern word     writeheader (),
                writenode (),
                writeedge ();
extern int	freeheader (),
                freeedge ();

/*
 * Types provided for in Graph Language flags
 */

typedef struct node_list *node_ptr;
typedef struct node_list **node_list;
typedef struct node_list
{
    node_ptr        next;               /* rest of sorted list */
    ident           id;                 /* identifier for the node */
} node_item;

#define Nest_Flags (Node_IDs | (3 << Pos_Shift) | Edge_Weight)

/*
 * Some macros useful for reading and writing graphlanguage structures to and
 * from bytestreams.  We need different macros for different machines because
 * of the pointer type punning and byteorder differences.  The last versions
 * (for the ISI 68000 workstation) are fairly generic, and should even work
 * for machines with braindamaged compilers and weird byte orders. See
 * graphs.c and nestgraphs.c for examples of use.
 */

/* Since I've hacked out _readf, _write_f, I'll get rid of the sun and
vax stuff */ 
/*
#if defined(sun) || defined (hp9000s200)
#define readw(bptr) ((word) ntohl (*(((long *) (bptr))++)))
#define readh(bptr) ((half) ntohs (*(((short *) (bptr))++)))
#define readb(bptr) (*(((byte *) (bptr))++))

#define writew(bptr,var) (*((word *) (bptr))++ = htonl ((long) var))
#define writeh(bptr,var) (*((half *) (bptr))++ = htons ((short) var))
#define writeb(bptr,var) (*(bptr)=(var), bptr++)
#endif
*/

#ifdef vax
extern float 	_writef();
extern float 	_readf();

#define readf(bptr) (_readf (&bptr))
#define readw(bptr) ((bptr) += 4, ntohl (*(long *) ((bptr) - 4)))
#define readh(bptr) ((bptr) += 2, ntohs (*(short *) ((bptr) - 2)))
#define readb(bptr) (*((bptr)++))

#define writef(bptr,var) (_writef ((byte **) &bptr, (float) (var)))
#define writew(bptr,var) (*(word *)(bptr) = htonl ((long) var), (bptr) += 4)
#define writeh(bptr,var) (*(half *)(bptr) = htons ((short) var), (bptr) += 2)
#define writeb(bptr,var) (*((bptr)++) = (var))

#endif
#ifndef readw

#define nohackmacros

extern word     _readw ();
extern half     _readh ();
extern word     _writew ();
extern half     _writeh ();
extern float 	_writef();
extern float 	_readf();

#define readf(bptr) (_readf (&bptr))
#define readw(bptr) (_readw (&bptr))
#define readh(bptr) (_readh (&bptr))
#define readb(bptr) (*((bptr)++))

#define writef(bptr,var) (void) (_writef ((byte **) &bptr, (float) var))
#define writew(bptr,var) (void) (_writew ((byte **) &bptr, (word) var))
#define writeh(bptr,var) (void) (_writeh ((byte **) &bptr, var))
#define writeb(bptr,var) (*((bptr)++) = (var))
#endif

#define align(x) (sizeof (word) - (x % sizeof (word)))

/*
 * Nest specific extensions to graphlanguage
 */

typedef struct xfunc *extptr;
typedef struct xfunc **extlist;
typedef struct xfunc
{
    struct xfunc   *next;               /* pointer to next entry in list */
    function        addr;               /* address of function */
    char           *name;               /* function description string */
} extfunc;

typedef struct cfunc *chanptr;
typedef struct cfunc **chanlist;
typedef struct cfunc
{
    struct cfunc   *next;               /* pointer to next func on stack */
    function        addr;               /* address of function */
} chanfunc;

#undef grnodedat
typedef struct nest_node
{
    ident           nodeid;             /* node id number */
    function function;                  /* function to be run on node */
    int             status;             /* current node status */
    timev           cputime;            /* cputime used by node */
    boolean         unused,
                    repeat;
    boolean         halt,
                    start;
    /* added by keshav ... */
    ident 	    sink;		 /* sink to send message to */	
	char plot ;
					/* whether to plot this node
or not */
} grnodedat;

#undef gredgedat
typedef struct nest_edge
{
    long            weight;             /* weight associated with edge */
    chanptr         chanstack;          /* pointer to channel stack */
    boolean         unused;
    boolean         delete;
    float	    line_speed;
    float	    loss_prob;		/* prob. of a burst */
    float	    corruption_prob;	/* prob. of bit corruption */
    float	    loss_burst_size;    /* mean size of an error burst */
} gredgedat;

#undef grhead
typedef struct nest_head
{
    timev           passtime;           /* time length of a simulation pass  */
    timev           wakeups;            /* time before any real-time wakeups */
    long            delay;              /* default delay between nodes:usecs */
    boolean         logging;            /* log passes and node ids for each  */
    boolean         unused;             /* log reports of simulation actions */
    boolean         broadcast;          /* broadcast everything to neighbors */
    boolean         point2point;        /* don't allow broadcasts to be sent */
    timev           timenow;            /* current runtime of simulation     */
    unsigned        passes;             /* number of pass being simulated    */
    unsigned        nodes;              /* maximum number of nodes allowed   */
    graph          *(*monitor) ();      /* selected monitor function address */
    extptr          nodefns;            /* list of defined node functions    */
    extptr          chanfns;            /* list of defined channel functions */
    extptr          monfns;             /* list of defined monitor functions */
} grhead;

/*
 * Possible status values for graph node data struct.
 */

#define BLOCKED -3                      /* indicates indefinite recvm block */
#define STOPPED -2                      /* indicates indefinite slumber */

#define DIED    -1                      /* indicates software fault on node */
#define DONE    0                       /* indicates stopped node */

#define RUNNING 1                       /* indicates running node */
#define PAUSED  2                       /* indicates slumber (SLEEP or WAIT) */
#define WAITING 3                       /* indicates blocked on recvm */

/*
 * Graph related functions.
 */

extern graph   *nest_read_graph ();
extern byte    *nest_write_graph ();
extern graph   *nest_restore_graph ();

#ifndef CLIENT

extern graph   *nest_monitor ();

extern node_ptr get_neighbors ();

#endif
