/* HYPE:include(/usr/keshav/hype/labels.h) */
/* HYPE:define(file, ~/real/sim/sim/sim/sources/jk_tahoe.c) */

/* 
 * Jacobson-Karels FTP source. Originally written by S. Keshav.
 * Revised by S. McCanne@LBL. This is based on 4.3BSD Tahoe 
 */

#include "../kernel/real.h"

extern int             JK_DEBUG;

#define ALPHA_RTT 0.1
#define TCP_MAX_RT_SHIFT 13

static int             tcp_backoff[TCP_MAX_RT_SHIFT] =
{1, 2, 4, 8, 16, 32, 64, 64, 64, 64, 64, 64, 64};

static void     ftp_send ();
static void     ftp_timeout ();
static void     ftp_ack ();

/* HYPE:LABEL(frame) */

struct frame
{
    PKT_PTR         pkt;		/* pkt being handled */
    int             nn;			/* node number */
    int             back_off;		/* determines multiplier for timeout */
    int             line_busy;		/* set if output line is busy */
    int             num_dup_acks; 	/* number of consecutive duplicate 
				 	 * acks seen so far */
    char            retxed[MAX_WINDOW_SIZE];	/* this seq. # retxed */
    int             fr_thresh;		/* fast retx. thereshold */
    int             last_sent;		/* seq of last pkt. sent */
    int             max_sent;		/* max. seq of pkt. sent */
    float           rtt; 		/* actually, the estimate */
    int             last_ack; 		/* last ack recvd */
    int             cur_window; 	/* current window size */
    int             ssthresh; 		/* slow start threshold */
    int             count; 		/* count of # of increments made 
					 * to cur_window so far */
    float           mean_deviation;	/* of rtt estimate from mean */
    int             timer_id;		/* used to simulate single timer */
    int             max_window;		/* max window size allowed */
};

/* HYPE:LABEL(jk_tahoe) */

jk_tahoe()
{
    /* Dummy variables for nest that we don't use. */

    ident           sender, destn;
    long            key;
    struct frame    f;

    bzero ((char *) &f, sizeof f);

    f.nn = get_node_id ();
    abs_advance (node_start_time[f.nn]);
    f.max_window = ftp_window;
    f.timer_id = 1;

    printf ("JK tahoe source: %d ", f.nn);
    printf ("sending %d packets to sink %d\n", num_pkts[f.nn], assigned_sink[f.nn]);

    source_node_type[f.nn] = JK_TAHOE_SOURCE;

   /* from /bsd/netinet/tcp_subr.c line 251 */

    f.rtt = 0.0;
    f.mean_deviation = 1.5 * scale_factor;

    f.cur_window = 1;
    f.last_sent = f.last_ack = -1;
    f.ssthresh = f.max_window / 2;

    /* send the first packet */
    ftp_send (&f);

    while (1)
    {
	sender = recvm (&destn, &key, &f.pkt);
	switch (f.pkt->type)
	{

	case ACK:
	    ftp_ack (&f);
	    ftp_send (&f);
	    break;

	case INT:
	    f.line_busy = 0;
	    free (f.pkt);
	    ftp_send (&f);
	    break;

	case TIMEOUT:
	    ftp_timeout (&f);
	    break;

	default:
	    free (f.pkt);
	    pr_error ("Node %d: source recvd. an pkt of unkonwn type",  f.nn);
	    break;
	}
    }
}
/*
 * adjust the timeout to correspond to 0.5 sec coarseness 
 */


/* HYPE:LABEL(vj_timeout) */

static float
vj_timeout (fp)
    struct frame   *fp;
{
    float           timeout;
    float	    beta_rtt = 2.0;

    timeout = fp->rtt;
    timeout += beta_rtt * fp->mean_deviation;
    timeout += fp->rtt * RANDOM;
    timeout += 0.5;
    timeout = coarser (timeout);
    timeout *= tcp_backoff[fp->back_off];
    return timeout;
}

/* HYPE:LABEL(ftp_send) */

/*
 * see if a packet can be sent, and if so, send it. First check if 
 * there is any data in the transmission queue to send, if not,
 * make a packet and send it 
 */


static void
ftp_send (fp)
    struct frame   *fp;
{
    PKT_PTR         pkt;
    int             nn = fp->nn;
    int             nqueued;
    int             win;

    /* sanity */
    fp->pkt = 0;

    win = min (fp->cur_window, fp->max_window);
    nqueued = num_in_q (fp->nn);

    while (--nqueued >= 0)
    {
	/* first try to send timeout or fast rexmt packet; but only if an 
	 * ack has not been received in the interim. */

	pkt = deq (nn);
	if (pkt->seq_no > fp->last_ack)
	{
	    /* ok to send , even if later */
	    if (pkt->seq_no <= fp->last_ack + win)
	    {
		/* ok to send now */
		fp->retxed[pkt->seq_no % MAX_WINDOW_SIZE] = 1;
		pkt->gen_time = runtime ();
		fp->line_busy = 1;
		pkt->id = ++fp->timer_id;

		if(pkt->resent)
		    ++num_retransmissions[fp->nn];
		if(pkt->seq_no > fp->max_sent)
		    fp->max_sent = pkt->seq_no;
		fp->last_sent = pkt->seq_no;
		safe_send (pkt, vj_timeout (fp));
		return;
	    } else
		/* save for later */
		enq (nn, pkt);
	} else
	    free (pkt);
	/* drop pkt on the floor */
    }
    if (fp->last_sent < fp->last_ack + win && !fp->line_busy &&
		fp->last_ack < num_pkts[fp->nn])
    {
	int node = fp->nn;

	make_pkt(pkt);
	pkt->gen_time = runtime ();
	pkt->seq_no = max (fp->last_sent, fp->last_ack) + 1;
	pkt->resent = 0;
	pkt->id = ++fp->timer_id;

	fp->line_busy = 1;
	fp->last_sent = pkt->seq_no;

	if(pkt->seq_no > fp->max_sent)
	    fp->max_sent = pkt->seq_no;
	else 
	{
	   ++num_retransmissions[fp->nn];
	    pkt->resent = 1;
	}
	safe_send (pkt, vj_timeout (fp));
    }
}

/* HYPE:LABEL(ftp_timeout) */

static void
ftp_timeout (fp)
    struct frame   *fp;
{
    PKT_PTR         pkt = fp->pkt;
    int             seqno = pkt->seq_no;

    if (fp->timer_id is pkt->id and pkt->seq_no > fp->last_ack)
    {
	if(JK_DEBUG)
		printf("%f: node %d timeout of seq %d\n", make_float(runtime()),
			fp->nn, seqno);
	if (fp->back_off < TCP_MAX_RT_SHIFT -1)
	    ++fp->back_off;
	fp->fr_thresh = seqno;
	if (fp->back_off > TCP_MAX_RT_SHIFT / 4)
	{
	    fp->mean_deviation += fp->rtt;
	    fp->rtt = 0;
	}
	fp->retxed[seqno % MAX_WINDOW_SIZE] = 1;
	/* retxed used as a dirty bit to allow us to ignore corrupted tao
	 * values */
	seqno = fp->last_ack + 1;
	pkt->type = DATA;
	pkt->seq_no = seqno;
	fp->ssthresh = fp->cur_window / 2;
	fp->cur_window = 1;
	make_plot("win", fp->cur_window);
	fp->count = 0;
	pkt->resent = 1;

	if (seqno <= fp->last_ack + fp->cur_window && !fp->line_busy)
	{
	    pkt->gen_time = runtime ();
	    fp->line_busy = 1;
	    pkt->id = ++fp->timer_id;
	    num_retransmissions[fp->nn] ++;
	    fp->last_sent = pkt->seq_no;
	    if(pkt->seq_no > fp->max_sent)
	        fp->max_sent = pkt->seq_no;
	    safe_send (pkt, vj_timeout (fp)); 
	} else
	    enq (fp->nn, pkt);
    } else
	free (pkt);
}

/* HYPE:LABEL(ftp_ack) */


static void
ftp_ack (fp)
    struct frame   *fp;
{
    PKT_PTR         pkt;
    float           tao;
    float           err;
    timev           now, diff, old;
    int             i, last;

    now = runtime ();

    pkt = fp->pkt;
    last = fp->last_ack;
    /* VAN-J */

    make_plot("ack", pkt->seq_no);
    if (pkt->seq_no > last)
    {
	/* ack not seen before */
	fp->num_dup_acks = 0;


	if (fp->cur_window < fp->ssthresh)
	     ++fp->cur_window; 	/* exponential */
	else
	    if (++fp->count == fp->cur_window)
	    {
		++fp->cur_window; 	/* linear */
		fp->count = 0;
	    }
	make_plot("win", fp->cur_window);

	if (JK_DEBUG)
	    printf ("(%d,%d)Source %d :current window size is % d and ssthresh is % d\n ",
		    now.tv_sec, now.tv_usec, fp->nn,
		    fp->cur_window,
		    fp->ssthresh);

	for (i = 1; i <= pkt->seq_no - last; ++i)
	{
	    /* KARN */
	    if (!fp->retxed[(last + i) % MAX_WINDOW_SIZE])
	    {
		/* if pkt has been retxed - ignore the tao */
		fp->back_off = 0;
		/* yet another kludge - if an ack acknowledges more than
		 * one packet then ignore all but last tao */
		if (pkt->seq_no is last + 1)
		{
		    diff = time_minus (now, pkt->gen_time);
		    tao = make_float (diff); 	/* real round trip time */
		    if (fp->rtt is 0.0)
			fp->rtt = tao;

		    make_entry (tao, &rt_time[fp->nn]);

		    /* VAN-J */
		    err = tao - fp->rtt;
		    if (err < 0)
			err = -err;
		    fp->rtt += ALPHA_RTT * (tao - fp->rtt);
		    /* 0.25 from Jacobson's 88 SIGCOMM paper */

		    fp->mean_deviation += 0.25 * (err - fp->mean_deviation);
		    if (JK_DEBUG)
		    {
			printf ("Source %d : (%d,%d)Seq # %d :tao was %f\n",
				get_node_id (), now.tv_sec, now.tv_usec, pkt->seq_no, tao);
			printf ("round trip time re-estimated as %f and deviation was %f\n\n",
				fp->rtt, fp->mean_deviation);
		    }
		}
	    }
	    /* reset the bit anyhow */
	    fp->retxed[(last + i) % MAX_WINDOW_SIZE] = 0;
	}
	fp->last_ack += pkt->seq_no - last;
    } else if (pkt->seq_no is last)
    {
	int             win = min (fp->cur_window, fp->max_window);

	++fp->num_dup_acks;

	if (fp->num_dup_acks is 3 and (last + 1) > fp->fr_thresh)
	{
	     /* probably the packet got dropped */
	    PKT_PTR         np;
	    int 	    node = fp->nn;

	    fp->fr_thresh = last + 1 + win;
	    fp->num_dup_acks = 0;

	    make_pkt(np);
	    np->seq_no = last + 1;
	    np->resent = 1;
	    np->gs = 0;

	    fp->retxed[np->seq_no % MAX_WINDOW_SIZE] = 1;

	    /* reset ssthresh and cwnd - as if it really were a dropped
	     * packet */

	    fp->ssthresh = fp->cur_window / 2;
	    fp->cur_window = 1;
	    make_plot("win", fp->cur_window);
	    fp->count = 0;
	    if (fp->back_off < TCP_MAX_RT_SHIFT - 1)
		++fp->back_off;
	    if (fp->back_off > TCP_MAX_RT_SHIFT / 4)
	    {
		/* If backed off this far, clobber the srtt value, storing
		 * it in the mean deviation instead. */

		fp->mean_deviation += fp->rtt;
		fp->rtt = 0;
	    }
	    if (fp->line_busy)
		enq (fp->nn, np);
	    else
	    {
		fp->line_busy = 1;
		np->id = ++fp->timer_id;
		num_retransmissions[fp->nn] ++;
		fp->last_sent = np->seq_no;
	        if(pkt->seq_no > fp->max_sent)
	            fp->max_sent = pkt->seq_no;
		safe_send (np, vj_timeout (fp));
	    }
	}
    }
    free (pkt);
}



