/*
 * Implementation exercise 2: CSMA/CD: Station
 * 
 * An Engineering Approach to Computer Networking
 * 
 * S. Keshav
 * 
 */

/*
 * ARGS USED: average rate NOT USED : peak_rate, on_time, off_time,
 * interval
 * 
 * The source sends data at an average rate chosen by the average_rate
 * parameter. It also implements CSMA/CD in conjunction with the
 * ecn_master node.
 */

#include "../kernel/real.h"
#define RETRY_INTERVAL 	1   /* seconds */
extern int CSMA_DEBUG;

/* state diagram 
 * 
 *                BUSY/NACK
 *    NORMAL   ------------> WAIT_FOR_IDLE <----------------
 *      ^|     <-----------		                   |
 *      ||        IDLE					   |
 * IDLE ||  						   |
 *      ||  buffer non empty => send packet		   |
 *      |V 						   | 
 *     BUSY						   ^
 *      ^| 						   |
 * T'OUT||  NACK					   |
 *      ||  						   |
 *      |V  						   | TIMEOUT
 *    WAIT_FOR_TIMEOUT (stay in this state on IDLE)	   |
 *      ^|  					 	   |
 * IDLE ||  BUSY/NACK					   |
 *      ||  						   |
 *      |V   			                           |
 *    WAIT_FOR_IDLE_AND_TIMEOUT --------------------------->
 */

struct station_state {
    int		    seq_no;
    int             pkts_generated;
    int             retry_interval;
    PKT_PTR         last_pkt_sent;
};


ecn_slave()
{
    PKT_PTR         pkt, tick_pkt = (PKT_PTR) 0;
    int             num, node;
    ident           destn, sender, sink;
    long            key, delay;
    struct station_state s;

    node = get_node_id();
    source_node_type[node] = TEMPLATE;
    sink = assigned_sink[node];
    abs_advance(node_start_time[node]);

    printf("CSMA Station : %d ", get_node_id());
    printf("--> %d, start (%d, %d) average rate %.3f\n",
       sink, node_start_time[node].tv_sec, node_start_time[node].tv_usec,
	   ave_bandwidth[node]);

    /* intialize state */
    s.seq_no = 0;
    s.pkts_generated = 0;
    s.retry_interval = RETRY_INTERVAL;
    s.last_pkt_sent = 0;

    /* place the first packet in the queue */
    create_pkt(&s);

normal:
    state_print("NORMAL");

    /* if the last transmission was unsuccessful, send it */
    if (s.last_pkt_sent isnt 0) {
	sendit(&s, s.last_pkt_sent);
	goto busy;
    }

    /* if queued packets await, send one */
    if (num_in_q(node)) {
	pkt = deq(node);
	sendit(&s, pkt);
	goto busy;
    }

    /* wait for a packet */
    sender = recvm(&destn, &key, &pkt);
    if(CSMA_DEBUG)
	pkt_print(pkt);

    switch (pkt->type) {

    case BUSY:		     /* line no longer available, change state */
    case NACK:
	free(pkt);
	goto wait_for_idle;

    case TICK:		    /* enqueue the next packet from upper layer */
	create_pkt(&s);

    default:
	free(pkt);
	goto normal;
    }

wait_for_idle:
    state_print("WAIT_FOR_IDLE");
    sender = recvm(&destn, &key, &pkt);
    if(CSMA_DEBUG)
	pkt_print(pkt);

    switch (pkt->type) {

    case IDLE:
	free(pkt);
	goto normal;

    case TICK:
	create_pkt(&s);
	 
    default:
	free(pkt);
	goto wait_for_idle;
    }

busy:
    state_print("BUSY");
    sender = recvm(&destn, &key, &pkt);
    if(CSMA_DEBUG)
	pkt_print(pkt);

    switch (pkt->type) {

    case IDLE:
	free(pkt);
	free(s.last_pkt_sent);
	s.last_pkt_sent = 0;
	s.retry_interval = RETRY_INTERVAL; 	/* reset */
	goto normal;

    case NACK:
	pkt->type = TIMEOUT;
	set_timer(RANDOM * s.retry_interval, pkt);
	goto wait_for_timeout;
	 
    case TICK:
	create_pkt(&s);
	 
    default:
	free(pkt);
	goto busy;
    }

wait_for_timeout:
    state_print("WAIT_FOR_TIMEOUT");
    sender = recvm(&destn, &key, &pkt);
    if(CSMA_DEBUG)
	pkt_print(pkt);

    switch (pkt->type) {

    case TIMEOUT:
	free(pkt);
	sendit(&s, s.last_pkt_sent);
	s.retry_interval *= 2;		/* exponential backoff */
	goto busy;
	 
    case BUSY:
    case NACK:
	free(pkt);
	goto wait_for_idle_and_timeout;

    case TICK:
	create_pkt(&s);
	 
    default:
	free(pkt);
	goto wait_for_timeout;
    }

wait_for_idle_and_timeout:
    state_print("WAIT_FOR_IDLE_AND_TIMEOUT");
    sender = recvm(&destn, &key, &pkt);
    if(CSMA_DEBUG)
	pkt_print(pkt);

    switch (pkt->type) {

    case TIMEOUT:
	free(pkt);
	s.retry_interval *= 2;		/* exponential backoff */
	goto wait_for_idle;
	 
    case IDLE:
	free(pkt);
	goto wait_for_timeout;

    case TICK:
	create_pkt(&s);
	 
    default:
	free(pkt);
	goto wait_for_idle_and_timeout;
    }
}
 
static int 
sendit(s, pkt)
    struct station_state *s;
    PKT_PTR         pkt;
{
    int             node = get_node_id();

    pkt->gen_time = runtime();
    s->last_pkt_sent = (PKT_PTR) malloc((unsigned) sizeof(PKT));
    *(s->last_pkt_sent) = *pkt;
    send_pkt(pkt);
}

static int 
create_pkt(s)
struct station_state *s;
{
    PKT_PTR         pkt, tick_pkt;
    float           delay;
    int		    node = get_node_id();

    /* create a packet only if the source hasn't exceeded its limit */
    if (s->pkts_generated++ < num_pkts[node]) {
	make_pkt(pkt);
	pkt->size = 0; /* so master knows about it right away */
	pkt->proto_flag = ftp_size; /* save size here */
	pkt->type = DATA;
	pkt->seq_no = s->seq_no;
	s->seq_no = s->seq_no + 1;
	enq(node, pkt);

	/* wait for exponentially distributed time, so that the 
	 * modulating workload is Poisson */
	delay = expntl((ftp_size * 8) / ave_bandwidth[node]);
	make_pkt(tick_pkt);
	tick_pkt->type = TICK;
	set_timer(delay, tick_pkt);
    }
}

static int
state_print (s)
char * s;
{
  timev		    now;

  if(CSMA_DEBUG){
      now = runtime();
      printf("(%f) Node %d entered state %s\n", make_float(now), 
						get_node_id(), s);
  }
}

