/*
 * Implementation exercise 2: CSMA/CD: Transmission medium
 * 
 * An Engineering Approach to Computer Networking
 * 
 * S. Keshav
 * 
 */

/*
 * ARGS USED: diameter
 * 
 * Simuates a shared medium like Ethernet. The diameter is the maximum end to
 * end propagation delay.
 * 
 * I do not know of a clean way to _exactly_ simulate a shared medium with REAL.
 * Instead, this node simulates the medium with a 'master' node that has a
 * point to point link to every other station. At start up, the master
 * comes up with a random set of distances between it and each node,
 * constraining any pairwise sum to be no larger than the stated diameter.
 * Subsequent messages placed on the "medium" are delivered to the
 * stations (called slave) after the corresponding delay.
 * Actually, only the edge from the master to a slave node has a delay
 * associated with it, the slave to master direction has zero delay, so
 * that the master knows about a packet instantaneously. To do this 
 * cleanly, the packet size is carried in the pkt->proto_flag field,
 * instead of in the size field. 
 * 
 * In this code, We ignore the fact that a station could get a BUSY signal
 * from a packet passing by before the master forwards it. Thus, the
 * performance numbers are conservative.
 */

#include "../kernel/real.h"

extern int CSMA_DEBUG;
extern struct graph *Graph;  /* simulation input */
extern float    diameter;    /* from input file */
#define JAM_TIME 0.05	     /* 50 ms */

/* stores all the state associated with the medium */
struct medium_state {
    int             medium_busy;
    int             num_nbrs;
    struct {
	int             id;  /* id of the node */
    } nbr[MAX_NODES + 1];
} ;

ecn_master()
{

    PKT_PTR         pkt, cpkt;
    int             i, node;
    ident           destn, sender, sink;
    long            key, delay;
    timev           now;
    gredgeptr       edge;
    struct medium_state m;

    int             num_nbrs;
    int             pkt_size; 	/* pkt_size of packet being transmitted */
    int             pkt_source;	/* source of the packet transmitted */
    int             pkt_dest; 	/* destination of packet transmitted */
    float 	    fdelay;

    node = get_node_id();
    printf("Node %d: CSMA medium master\n", node);
    source_node_type[node] = ROUTER;
    sink = assigned_sink[node];
    m.medium_busy = 0;

    /*
     * Discover all the stations on the segment, and assign them a random
     * distance from the master. The distance between two nodes is the sum
     * of their distances from the master.
     */
    num_nbrs = 0;
    edge = Graph->edges;
    while (edge isnt 0) {
	int n1, n2;

	/* n1 and n2 are the ids of the nodes at each end */
	n1 = edge->node1->nodedata->nodeid;
	n2 = edge->node2->nodedata->nodeid;
	
	/* we only add latency to the edge from the master to slave */
	/* so master instantaneously knows of packet placed on wire */
	if((n1 is node) or (n2 is node)) {
	    edge->edgedata->weight = (long) (RANDOM * 0.5 * diameter * MILLION);
	    printf("Master selected delay %d microseconds on edge %d --> %d\n", 
		    edge->edgedata->weight, n1, n2);
	    if (n1 is node){
	        m.nbr[num_nbrs].id = n2;
	        _neighbor_update(n1, n2, edge->edgedata);
	    } else {
	        m.nbr[num_nbrs].id = n1;
	        _neighbor_update( n2, n1, edge->edgedata);
	    }
	    num_nbrs++;
	}
	edge = edge->next;
    }
    m.num_nbrs = num_nbrs;

    /* sit in an infinite loop, waiting for packets */
    for (ever) {
	sender = recvm(&destn, &key, &pkt);
	if(CSMA_DEBUG)
	    pkt_print(pkt);

	now = runtime();

	switch (pkt->type) {

	case DATA:
	    if (not m.medium_busy) {
		/* medium was idle. make it busy */
		m.medium_busy = 1;

		/* save packet size and source, then destroy original pkt */
		pkt_size = pkt->proto_flag; 	/* source saved it here */
		pkt_source = pkt->source;
		pkt_dest = pkt->dest;
		free(pkt);

		/* send a BUSY message to all neighbors */
		tell_all(m, BUSY);

		/* set a timer to send idle after packet transmission time */
		make_pkt(pkt);
		pkt->type = TIMER;
		set_timer((float)(pkt_size * 8 / line_speeds[node][pkt_source]), pkt);

	    } else {	     /* collision: jam the network by sending
			      * NACKs to all nodes */

		free(pkt);
		tell_all(m, NACK);
		
		/* set a timer to send idle after jam time */
		make_pkt(pkt);
		pkt->type = TIMER;
		set_timer(JAM_TIME, pkt);
	    }
	    make_plot("busy", 0);
	    make_plot("busy", 10);
	    break;

	case TIMER:
	    /* send an IDLE packet to all nodes */
	    tell_all(m, IDLE);
	    make_plot("busy", 10);
 	    make_plot("busy", 0);
	    m.medium_busy = 0;
	    break;

	case INT:
	    free(pkt);
	    break;

	default:
	    printf("pkt->type %d, pkt->so %d, pkt->dest  %d\n",
		   pkt->type, pkt->source, pkt->dest);
	    pr_error("source received a pkt of unknown type", node);
	}
    }
}

static int
tell_all(m, type)
    struct medium_state m;
    int             type;
{
    int             i, node = get_node_id();
    PKT_PTR         pkt;
    timev	    now;

    now = runtime();

    for (i = 0; i < m.num_nbrs; i++) {
	make_pkt(pkt);
	pkt->type = type;
	pkt->size = 0;
	pkt->dest = m.nbr[i].id;
	if(CSMA_DEBUG)
	    printf("(%f) Node %d sending %s pkt to node %d\n", 
		make_float(now), node, pkt_types[pkt->type], m.nbr[i].id);
	send_pkt(pkt);
    }
}
