/*
 * This is the router (switch). The code is pretty complicated because of
 * the multiple scheduling disciplines supported. If you wanted something
 * easier to understand, simply delete the scheduling disciplines that you
 * don't want to support (or write your own).
 */

#include "router.h"
extern int      START_R_DEBUG;	/* start and end of router debugging */
extern int      END_R_DEBUG;	/* set in ../kernel/switches.c */
extern int 	SCH_DEBUG;


router()
{
    int             count = 0, i, j, dest, final_dest, so;
    int             conv_id; /* conversation identifier  == VCI */
    int             qnum;    /* queue number */
    PKT_PTR         pkt, util_pkt, hrr_timer_pkt, xunet_timer_pkt;
    timev           now, diff;
    ident           node, destn, sendr;
    long            key;
    float 	    bw, delay;

    node = get_node_id();
    abs_advance(node_start_time[node]);
    source_node_type[node] = ROUTER;
    printf("Gateway : %d\n", node);

    /* set queue numbers */

    for (i = 0; i < MAX_NODES + 1; i++)
	if (route[node][i] is i)	/* i is on a trunk line */
	{
	    q_number[node][i] = count++;	/* assign q number */

 	    /* for HRR/XUNET : compute bandwidth levels  and start off 
	     * initial set of HRR_TIMER packets */

	    if (i and (policy[node] is HRR or policy[node] is XUNET))
	    {
		make_pkt(hrr_timer_pkt);
		hrr_timer_pkt->type = HRR_TIMER;
		bw = line_speeds[node][i]/ 8.0; /* bps -> bytes/sec */
		delay = ((float) hrr_timer_pkt->size / (float) bw);

		set_timer(delay, hrr_timer_pkt);
		compute_bw_levels(q_number[node][i], i);	/* i = dest */
	    }
           if (i and policy[node] is XUNET)
	    {
              /* for XUNET need to initialize phase timer and bandwidths */
                 make_pkt(xunet_timer_pkt);
                 xunet_timer_pkt->type = TIMER;
                 xunet_timer_pkt->jitter = q_number[node][i];  
			/* use the unused jitter field to hold line number */
                 phase[node][q_number[node][i]] = 0;
                 set_timer((float)(XU_PHASE_SIZE* scale_factor), xunet_timer_pkt);
                 xu_init_bw_levels(q_number[node][i], i); /* i = dest */
             }
	}
    for (j = 0; j < MAX_FAN_OUT; j++)
    {
	prev_cycle_begin_time[node][j] = time_zero;
	cycle_begin_time[node][j] = time_zero;
	q_change_time[node][j] = time_zero;
	area[node][j] = 0.0;
	prev_area[node][j] = 0.0;
	average_q_len[node][j] = 0.0;
	q_length[node][j] = 0;
	c_knee[node][j] = 0.0;
	prev_c_knee[node][j] = 0.0;
    }
    for (j = 0; j < MAX_CONVERSATIONS; j++)
    {
	pkts_sent[node][j] = 0.0;
	prev_pkts_sent[node][j] = 0.0;
	source_bit[node][j] = 0;
	source_bit_old[node][j] = 0;
	fq_weight[node][j] = 0;
	bandwidth_share[node][j] = 0;
	buffer_share[node][j] = 0;
    }

    for (j = 0; j < MAX_NODES + 1; j++)
	num_bits[node][j] = 0;

    /* set up a timer to compute utilizations */

    if (plot_option[node]) 
    {
        make_pkt(util_pkt);
        util_pkt->type = UTIL;
        set_timer(((float) util_time * scale_factor), util_pkt);
    }
    for (ever)
    {
	/* get a packet */
	sendr = recvm(&destn, &key, &pkt);
	now = runtime();

	so = pkt->source;
	final_dest = pkt->dest;
	dest = route[node][final_dest];

	/* dest is the actual next node - not the eventual destn */

	conv_id = hash(node, so, final_dest);

	/*
	 * if this is a new conversation, conv_id will be assigned a new
	 * number, else it is an old value. Since so and final_dest are
	 * fixed, conv_id serves as a VCI, though computed from datagram header.
	 */

	qnum = q_number[node][dest];	/* qnum is output interface number */
	q_num_of_conv[node][conv_id] = qnum; /* reverse mapping */

	pkt->arr_time = now;

	if (make_float(now) > START_R_DEBUG){
	    R_DEBUG = 1;
	    SCH_DEBUG = 1;
	}
	if (make_float(now) > END_R_DEBUG){
	    R_DEBUG = 0;
	    SCH_DEBUG = 0;
	}

	if (R_DEBUG)
	{
	    if (pkt->type > NUM_VIRTUAL_TYPES)
	    {
		printf("%f Router %d: Received %d ---(%d %s)---> %d\n",
			make_float(now), node, so, pkt->seq_no, 
			type_to_str(pkt->type), dest); 

		dump_q(so, conv_id);
	    }
	}

	/* Deal with the incoming packet depending on its type */

	switch (pkt->type)
	{
	case DATA:
	case ACK:
	    /* if a new conversation has started on some qnum
	     * add it to the list of conversations */

	    if (!check_conv(conv_id, head_conv_list[node][qnum]))
	    {
		add_conv(conv_id, &head_conv_list[node][qnum]);
		net_conv[node][qnum] ++;
	    }

	    /* need extra processing for some of the disciplines */

	    switch (policy[node])
	    {
	    case FQ:
	    case FQBIT: /* compute new finish number and remove old conversations */

		new_arrival(conv_id, qnum, pkt, dest);
		break;

	    case DECBIT:    /* update queue length */

		diff = time_minus(now, q_change_time[node][qnum]);
		area[node][qnum] += q_length[node][qnum] * (make_float(diff));
		q_length[node][qnum]++;
		q_change_time[node][qnum] = now;
		source_bit[node][conv_id] = 1;
		if (q_length[node][qnum] is 1)
		    do_transition(qnum, conv_id);
		break;

            case XUNET: /* if the VC is not active put it onto a service  list */

                if (num_buffers_in_q[node][conv_id] is 0)
			xu_new_arrival(conv_id, qnum);
                break;

	    case HRR:

		/* if the current list empty, put in the opposite parity list */

		if (num_buffers_in_q[node][conv_id] is 0)
		{
		    int             level;

		    level = conv_level[node][conv_id];
		    put_in_tail(1 - parity[node][qnum][level], level, conv_id, qnum);
		}
		break;
	    }		     /* end of special processing case */

	    if (busy[node][qnum]) /* line to dest is busy, so buffer */
	    {
		switch (policy[node])	/* buffering depends on policy */
		{
		case FQ:
		case FQBIT:
		case SFQ:
		case HRR:
		case XUNET:

		    /* if the packet is from a GS source, then check if it
		     * has exceeded quota, tossing it if it has */

		    if (pkt->gs)
		    {
			if (!check_buffers(pkt, node, conv_id))
			    put_in_op_q(conv_id, pkt);
			else
			{
			    free(pkt);
			    make_entry(1.0, &so_tossed[so]);
			    finish_num[node][conv_id] = save_finish[node][conv_id];
			    	/* for FQ reset finish number */
			}
		    } else
			/* BE packet  */
		    {
			if (op_q_full(qnum, pkt->size)) /* queue full,toss packet */
			    switch (decongest_mechanism)
			    {
			    case 0:
				decongest_first(qnum);
				break;
			    case 1:
				decongest_last(qnum, conv_id, pkt);
				break;
			    case 2:
				decongest_random(qnum);
				break;
			    case 3:
				decongest_all(qnum);
				break;
			    }
			/* place into buffer */
			if (put_in_op_q(conv_id, pkt) == ERROR)
			    pr_error("Bug in decongestion code \n ");
		    }	     /* end of BE packet case */
		    break;

		case FCFS:
		case DECBIT:
		    /* assume that GS and FCFS dont go together */
		    /* for fcfs and decbit, no need for a fancy decongest
		     * mechanism, (unless it is random) */

		    if (op_q_full(qnum, pkt->size))
		    {
			q_length[node][qnum]--;

			if (decongest_mechanism is 2 or decongest_mechanism is 3)	/* random  or all */
			{
			    if(decongest_mechanism is 2) 
				decongest_random(qnum);
			    else
				decongest_all(qnum);

			    if (put_in_op_q(conv_id, pkt) == ERROR)
				pr_error("Output buffer full - CONGESTION\n ");
			} else /* first or last, doesn't matter  - toss incoming packet */
			{
			    free(pkt);
			    make_entry(1.0, &so_tossed[so]);
			    source_bit[node][conv_id] = 1;
			}
		    } else   /* some place still left in the buffer */
			if (put_in_op_q(conv_id, pkt) == ERROR)
			    pr_error("Output buffer full - CONGESTION\n ");
		    break;
		}
	    } else	     /* end of if(busy), so line not busy */
		do_departure_processing(pkt, conv_id, qnum, dest);
	    break;

        case TIMER: /* time to change phase for xunet FRR level */
                if (policy[node] is XUNET)
                   { qnum = pkt->jitter;
                     phase[node][qnum] = 1 - phase[node][qnum];
                     set_timer((float)(XU_PHASE_SIZE* scale_factor), pkt);
                     if (!busy[node][qnum])
                        goto _wakeup_tick;
                    }
                else
                     pr_error("TIMER packet received and policy not XUNET\n");
                 break;

	case HRR_TIMER : /* place holder for empty transmission  slot */
	case INT:

	    /*
	     * line has freed up - if there is a packet, send it, else
	     * just unset the busy flag, and for HRR, set timer.
	     */

	    free(pkt);
_wakeup_tick: /* xunet phase has changed, see if something can be sent now */
	    pkt = select_from_op_q(qnum, dest);

	    if (R_DEBUG and pkt)
		printf("%f Router %d: Selected %d ---(%d %s)---> %d\n",
		make_float(now), node, pkt->source, pkt->seq_no, 
		type_to_str(pkt->type), dest);

	    if (pkt is (PKT_PTR) (-1))
		pr_error("problems with select_from_op_q: returns ERROR");

	    if (pkt != NULL)
	    {
		/* reset useful values */
		so = pkt->source;
		final_dest = pkt->dest;
		dest = route[node][final_dest];
		conv_id = hash(node, so, final_dest);
		qnum = q_number[node][dest];
		do_departure_processing(pkt, conv_id, qnum, dest);

	    } else /* idle: nothing in the queue to be sent */
	    {
		busy[node][qnum] = 0;

		/* for HRR/XUNET, set a timer for next time slot. Set busy to 1,
		 * forcing incoming pkts to be queued, and served with correct
		 * slippage  
		 */

		if (policy[node] is HRR or policy[node] is XUNET) 
		{
		    busy[node][qnum] = 1;
    		    make_pkt(hrr_timer_pkt);
    		    hrr_timer_pkt->type = HRR_TIMER;

    		    bw = line_speeds[node][dest]/ 8.0; /* bps -> bytes/sec */
    		    delay =  ((float) hrr_timer_pkt->size / bw);
    		    set_timer(delay, hrr_timer_pkt);
		}
	    }
    
	    break;

	case SETUP:	     /* admission control and bandwidth allocation */
	    /*
	     * pkt -> bandwidth is set to 0, if admission denied SETUP_ACK
	     * will carry this value back to source
	     */

	    if (!allocate_bw(pkt, conv_id, dest, qnum))
		pkt->ave_bandwidth = 0.0;
	    if (!allocate_buffers(pkt, conv_id, qnum))
	   	pkt->peak_bandwidth = 0.0;
	    if (pkt->ave_bandwidth isnt 0.0 and pkt->peak_bandwidth isnt 0.0) 
						/* accepted here */
	    {
	        if (policy[node] is FQ)
	    	{
		    add_conv(conv_id, &head_perm_conv_list[node][qnum]);
		    compute_fq_weights(conv_id, qnum);
		}
	        queue_or_send (pkt, conv_id, node, qnum, dest); 
	    }
	    else /* send back the bad news to the source */
	    {
		tear_down_connection(pkt, conv_id, qnum);
		pkt->dest = pkt->source;
		pkt->source = node;
		pkt->type = SETUP_ACK;
	 	dest = route[node][pkt->dest];
		qnum = q_number[node][dest];
		conv_id = hash(node, node, pkt->dest);
		queue_or_send (pkt, conv_id, node, qnum, dest);
	    }
	    break;

	case SETUP_ACK:
	    if (pkt->peak_bandwidth is 0.0 or pkt->ave_bandwidth is 0.0)
		/* rejected upstream */
	    {
		int             destn, qnumber;

		destn = route[node][pkt->source];
			/* find out dest in forward direction */
		qnumber = q_number[node][destn];
			/* and the queue it was in */
		tear_down_connection(pkt, conv_id, qnumber);
			/* and tear it down */
	    }
	    queue_or_send (pkt, conv_id, node, qnum, dest); 
	    break;

	case TEARDOWN:
	    tear_down_connection(pkt, conv_id, qnum);
	    queue_or_send (pkt, conv_id, node, qnum, dest); 

	case UTIL:	     /* compute utilizations for all o/p trunks */
	    {
		char            fname[20];

		for (i = 0; i < MAX_NODES + 1; i++)
		{
		    if (num_bits[node][i] > 0)
		    {
			sprintf(fname, "util.%d.", i);
			make_flt_plot(fname, ((float)num_bits[node][i]
			 /(float)(util_time*scale_factor*line_speeds[node][i])));
			num_bits[node][i] = 0;
		    }
		}
		set_timer(((float) util_time * (long)scale_factor), pkt);

		break;
	    }

	default:
	    printf("Spurious packet received at the router \n");
	    printf("%d: (%d, %d):  pkt: source %d, type %d seq %d\n",
		   node, now.tv_sec, now.tv_usec, pkt->source, pkt->type,
		   pkt->seq_no);
	    free(pkt);
	    break;
	} /* end of switch on packet type */
    }	/* end of infinite loop */
}

do_departure_processing(pkt, conv_id, qnum, dest)
    PKT_PTR         pkt;
    int             conv_id, qnum, dest;
{
    int             node, so;
    timev           now, diff;

    node = get_node_id();
    now = runtime();
    so = pkt->source;

    busy[node][qnum] = 1;

    /* collect stats */

    if (pkt->type is DATA)
    {
	diff = time_minus(now, pkt->arr_time);
	make_entry(make_float(diff), &qing_delay[node][so]);
	if (not pkt->resent)
	    make_entry(0.0, &nxmit[node][so]);
    }

    if (policy[node] is DECBIT)
    {
	/* since we removed a packet, need to update q lengths */

	diff = time_minus(now, q_change_time[node][qnum]);
	area[node][qnum] += q_length[node][qnum] * (make_float(diff));
	q_length[node][qnum]--;
	q_change_time[node][qnum] = now;
	diff = time_minus(now, prev_cycle_begin_time[node][qnum]);
	average_q_len[node][qnum] = (area[node][qnum] + prev_area[node][qnum])
	    / make_float(diff);
	pkts_sent[node][conv_id]++;
	c_knee[node][qnum]++;
	if ((pkt->type isnt ACK) and(!pkt->decbit) and(set_bit(conv_id, pkt)))
	    pkt->decbit = 1;
    }
    /* may need to remove this conversation */
    if (pkt->type > NUM_VIRTUAL_TYPES)
	remove_conv(conv_id, qnum, pkt);

    if (policy[node] is FQBIT)
	/*
	 * the packet that just went out doesnt count if there was no one
	 * else in the queue
	 */
    {
	int             pkts_in_q = 0;

	pkts_in_q = get_num_buffers_in_q(conv_id);
	if (num_active[node][qnum] != 0)
	    if (pkts_in_q >= 1)
		if (pkts_in_q + 1 > (int) ((float) op_qsize /
			   ((float) num_active[node][qnum] * 3.0) + 0.5))
		    pkt->decbit = 1;
    }

#ifndef OLD_FQ_CODE
    /* heybey/davin hack */

    if(policy[node] is FQ)
	round_number[node][qnum] = pkt->bid;
#endif
    num_bits[node][dest] += pkt->size * 8;

    if (!sendm(dest, 0, pkt))
	pr_error("sendm failed in router");

    if ((R_DEBUG) and pkt->type > NUM_VIRTUAL_TYPES)
        printf("%f Router %d: Forwarded %d ---(%d %s)---> %d\n",
	        make_float(now), node, so, pkt->seq_no, 
		type_to_str(pkt->type), dest); 

}

do_transition(qnum, new_conv_id)
    int             qnum;
    int             new_conv_id;
	/* qnum is the q number on which to do a transition */

{
    timev           now, diff;
    ident           node;
    CONV_PTR        conv;

    /* reset variables that track regeneration points */
    /* we need to identify the conversations on this line. */

    now = runtime();
    node = get_node_id();

    prev_cycle_begin_time[node][qnum] = cycle_begin_time[node][qnum];
    cycle_begin_time[node][qnum] = now;

    prev_area[node][qnum] = area[node][qnum];
    area[node][qnum] = 0.0;

    prev_c_knee[node][qnum] = c_knee[node][qnum];
    c_knee[node][qnum] = 0.0;

    conv = head_conv_list[node][qnum];
    while (conv != 0)
    {
	prev_pkts_sent[node][conv->conv_id] = pkts_sent[node][conv->conv_id];
	pkts_sent[node][conv->conv_id] = 0.0;
	source_bit_old[node][conv->conv_id] = source_bit[node][conv->conv_id];
	source_bit[node][conv->conv_id] = 0;
	conv = conv->next;
    }
}

set_bit(conv_id, pkt)
    PKT_PTR         pkt;
    int             conv_id;
{
    ident           node;
    timev           now, diff;
    float           capacity, fair_share, old_fair_share;
    int             qnum, num_unalloc, cid;
    float           sum_allocation, old_sum_allocation, demand;
    char            changed = 1;
    CONV_PTR        conv;


    now = runtime();
    node = get_node_id();
    qnum = q_num_of_conv[node][conv_id];

    if (average_q_len[node][qnum] <= 1.0)
	return 0;
    /*
     * 2 is the value from the DEC tech report. The router is overcrowded
     * and so we set bits on every packet
     */

    if (average_q_len[node][qnum] >= 2.0)
	return 1;

    if (make_float(prev_cycle_begin_time[node][qnum]) is 0.0)
	return 1;
    /*
     * when this condn is true, we dont really have a capacity estimate.
     * So, to be conservative, we drive the line towards idleness, so that
     * we can then force a transition to occur
     */

    num_unalloc = 0;
    fair_share = 0.0;
    old_fair_share = -1.0;
    sum_allocation = 0.0;
    old_sum_allocation = -1.0;

    /* first determine how many people need to be allocated */

    conv = head_conv_list[node][qnum];
    while (conv != 0)
    {
	cid = conv->conv_id;
	if (source_bit[node][cid] || source_bit_old[node][cid])
	    num_unalloc++;
	conv = conv->next;
    }
    capacity = ((prev_c_knee[node][qnum] + c_knee[node][qnum])
		* CAPACITY_FACTOR);
    while ((sum_allocation > old_sum_allocation) and(num_unalloc > 0))
    {
	old_fair_share = fair_share;
	old_sum_allocation = sum_allocation;
	fair_share = (capacity - sum_allocation) / num_unalloc;

	conv = head_conv_list[node][qnum];
	while (conv != 0)
	{
	    cid = conv->conv_id;
	    if (source_bit[node][cid] || source_bit_old[node][cid])
	    {
		demand = pkts_sent[node][cid] + prev_pkts_sent[node][cid];
		if (demand <= fair_share and demand > old_fair_share)
		{
		    num_unalloc--;
		    sum_allocation += demand;
		}
	    }
	    conv = conv->next;
	}
    }

    demand = pkts_sent[node][conv_id] + prev_pkts_sent[node][conv_id];
    if (demand > fair_share)
	return 1;
    else
	return 0;
}

remove_conv(conv_id, qnum, pkt)
    int             conv_id, qnum;
    PKT_PTR         pkt;

{
    int             node;

    node = get_node_id();
    /*
     * if this pkt has made its queue empty, remove the conversation from
     * the list of conversations
     */
    /* this conversation may still be active, (in FQ sense) though! */

    if (!get_num_buffers_in_q(conv_id))
    {
        if (delete_conv(conv_id, &head_conv_list[node][qnum]) is ERROR)
	    pr_error("error in deleting conversation ");
	net_conv[node][qnum] --;
    }
}

allocate_bw(pkt, conv_id, dest, qnum)
    PKT_PTR         pkt;
    ident           dest;
    int             conv_id, qnum;

{
    ident           node;

    node = get_node_id();
    switch (policy[node])
    {
    case FQ:
	if (pkt->ave_bandwidth < (BANDWIDTH_THRESHOLD * (line_speeds[node][dest] -
				       bandwidth_allocated[node][qnum])))
	    /* can accept */
	{
	    bandwidth_allocated[node][qnum] += pkt->ave_bandwidth;
	    bandwidth_share[node][conv_id] = pkt->ave_bandwidth;
	    return 1;
	} else
	    return 0;
	break;
    case HRR:
	return compute_level(pkt, conv_id, qnum);
	break;
    case XUNET:
        return xu_compute_class(pkt, conv_id, qnum);
        break;

    }
    return 1;
}

allocate_buffers(pkt, conv_id, qnum)
    PKT_PTR         pkt;
    int             conv_id, qnum;

{
    ident           node;
    int             buffers_needed;

    node = get_node_id();

    switch (policy[node])
    {
    case FQ:
	buffers_needed = (((pkt->ave_bandwidth * pkt->interval) -
	     (pkt->ave_bandwidth / pkt->peak_bandwidth) * pkt->interval *
			   pkt->ave_bandwidth)) / MILLION;
	break;
    case HRR:
	if (conv_level[node][conv_id] isnt 0)
	    buffers_needed = 3 * alloc_aj[node][conv_id];
	break;
    default:
	return 1;
    }
    if (op_qsize - buffers_allocated[node][qnum] >=
	(buffers_needed / BUFFER_THRESHOLD))
    {
	buffers_allocated[node][qnum] += buffers_needed;
	buffer_share[node][conv_id] = buffers_needed;
	return 1;
    } else
	return 0;
}

tear_down_connection(pkt, conv_id, qnum)
    PKT_PTR         pkt;
    int             qnum;
    int             conv_id;

{
    ident           node;

    node = get_node_id();
    switch (policy[node])
    {
    case FQ:
	bandwidth_allocated[node][qnum] -= bandwidth_share[node][conv_id];
	bandwidth_share[node][conv_id] = 0;
	if (delete_conv(conv_id, &head_perm_conv_list[node][qnum]) is -1)
		pr_error("trying to delete a conv that does not exist!");;
	break;
    case HRR:
	num_cells_allocated[node][qnum][conv_level[node][conv_id]]
	    -= alloc_aj[node][conv_id];
	break;
    case XUNET:
          xu_teardown( conv_id, qnum);
          break;
    }
    /* buffers have to be deallocated */
    buffers_allocated[node][qnum] -= buffer_share[node][conv_id];
}

compute_fq_weights(conv_id, qnum)
    int             conv_id, qnum;

{
    float           min_bw;
    ident           node;
    CONV_PTR        conv;

    min_bw = INFINITY;
    node = get_node_id();
    conv = head_perm_conv_list[node][qnum];
    while (conv != 0)
    {
	if (bandwidth_share[node][conv->conv_id] < min_bw)
	{
	    min_bw = bandwidth_share[node][conv->conv_id];
	}
	conv = conv->next;
    }
    conv = head_perm_conv_list[node][qnum];
    while (conv != 0)
    {
	fq_weight[node][conv->conv_id] = bandwidth_share[node][conv->conv_id] / min_bw;
	conv = conv->next;
    }
}


check_buffers(pkt, node, conv_id)
    PKT_PTR         pkt;
    ident           node;
    int             conv_id;

{
    /* turned off for the moement. Used by admission control scheme */
    /*
     * return (num_bytes_in_q[node][conv_id] >=
     * buffer_share[node][conv_id]);
     */
    return op_q_full(q_num_of_conv[node][conv_id]);

}

/* helper function used by SETUP, SETUP_ACK, TEARDOWN */

queue_or_send(pkt, conv_id, node, qnum, dest)
    PKT_PTR 	    pkt;
    int 	    conv_id, node, qnum, dest;
{
    if (busy[node][qnum]) /* can't forward the setup now, queue it */
    {
	if (policy[node] is XUNET)
	    xu_new_arrival(conv_id,qnum);

	if (policy[node] is HRR)
	{
	    int             level;
    
	    level = conv_level[node][conv_id];
	    put_in_tail(1 - parity[node][qnum][level], level, conv_id, qnum);
	}

	if (!check_conv(conv_id, head_conv_list[node][qnum]))
	{
	    add_conv(conv_id, &head_conv_list[node][qnum]);
				/* so that it is picked up later */
	    net_conv[node][qnum]++;
	}

	put_in_op_q(conv_id, pkt);
    } 
    else
	sendm(dest, 0, pkt);
}
