/*
 * hash.c 
 * generic hash function
 * S. Keshav Cornell University 7/11/97
 */

#include "real.h"
#include "hash.h"

struct hash_template * 
hash_create (size, compare, hashfn)
int size;
funcptr compare, hashfn; 
{
    struct hash_template * h;

    h = (struct hash_template*) malloc((unsigned)sizeof(struct hash_template));
    h->size = size;
    h->hashtab = (struct bucket **)malloc((unsigned)(size * sizeof(struct bucket)));
    bzero(h->hashtab, size*sizeof(struct bucket));
    h->compare = compare;
    h->hashfn = hashfn;
    return h;
}

/* locate the value associated with this input */

void *
hash_locate (h, input)
struct hash_template * h;
void *input;
{
    int index;
    struct bucket * b;

    index = (h->hashfn)(input, h->size);

    for(b = h->hashtab[index]; b isnt 0; b = b->next)
	 if((h->compare)(b->input, input))
	      return b->value;
    
   return (void*) -1;
}

/* insert a value into the hash table */
void
hash_insert(h, input, value)
struct hash_template * h;
void  *input, *value;
{
   int index;
   struct bucket * b;

   index = (h->hashfn)(input, h->size);

   b = (struct bucket *) malloc((unsigned) sizeof (struct bucket));
   b->input = input;
   b->value = value;
   b->next = h->hashtab[index];
   h->hashtab[index] = b;
}

/*-------------------------------------------------------------------*
 * the functions below use the generic functions for looking up 
 * "conversation ids", which are used by a router to keep track
 * of flow state. A second use is to map from a print name to an index.
 */

/* this is a source-dest tuple */
struct sode{
    int so;
    int de;
};

/* comparing tuples */
int
a_compare(a, b)
void *a, *b;

{ 
  struct sode* v1, *v2;

  v1 = a;
  v2 = b;
  return (v1->so is v2->so and v1->de is v2->de);
}

/* the hash */
int
a_hashfn(input, size)
void* input;
int size;
{
 struct sode *s;
 
 s = input;
 return (s->so * 7 + s->de * 11) % (size);
}

/* the inverse comparision (on conv ids) */
int b_compare(a,b)
void *a, *b;
{
int * v1, *v2;

v1 = a;
v2 = b;
return (*v1 == *v2);
}

/* hash fn for inverted table */
int b_hashfn(input, size)
int *input;
int size;
{
return (11 * (*input) % size);
}

/* globals needed for per-node hash tables */

struct hash_template * hashptr[MAX_NODES + 1];
struct hash_template * hinvptr[MAX_NODES + 1];
struct hash_template * hnameptr[MAX_NODES + 1];

int
hash_init()
{
  int i;

  for(i = 1; i <= MAX_NODES; i++) 
  {
      hashptr[i] = 0;
      hinvptr[i] = 0;
      hnameptr[i] = 0;
  }
}

/* hash from a source and destination: create entry if needed */
int
hash (node, so, de)
    ident           node;
    int             so, de;
{
    struct hash_template * h, *hinv;
    struct sode *s, *t;
    int *c, *d;
    static int conv_id = 0;

     h = hashptr[node];
     if(h is 0){
	h = hash_create (MAX_CONVERSATIONS * MAX_NODES, a_compare, a_hashfn);
	hashptr[node] = h;
    }

    s = (struct sode*) malloc((unsigned)sizeof(struct sode));
    s->so = so;
    s->de = de;
    if((c = hash_locate (h, s)) is (int*) -1) {
	c = (int *) malloc ((unsigned)sizeof(int));
	*c = conv_id;
	hash_insert(h, s, c);

        hinv = hinvptr[node];
        if(hinv is 0){
	    hinv = hash_create (MAX_CONVERSATIONS * MAX_NODES, b_compare, b_hashfn);
	    hinvptr[node] = hinv;
    	}
	hash_insert(hinv, c, s);
	conv_id ++;
    }

    return *c;
}

invert (node, conv_id, so, de)
    ident           node;
    int             conv_id, *so, *de;
{
 struct hash_template *hinv;
 struct sode * s;

 hinv = hinvptr[node];
 if(hinv is 0)
     return -1;
 else
    s = hash_locate(hinv, &conv_id);

if( s is (struct sode*) -1)
	return -1;

*so = s->so;
*de = s->de;
}

/*
testing ...

main()
{
int conv;
int so, de;

conv = hash (10, 110, 20);
printf ("%d \n", conv);
invert (10, conv, &so, &de);
printf ("%d, %d \n",so , de);
conv = hash (10, 4,14);
printf ("%d \n", conv);
invert (10, conv, &so, &de);
printf ("%d, %d \n",so , de);
invert (10, 1, &so, &de);
printf ("%d, %d \n",so , de);
conv = hash (10, 0, 90);
printf ("%d \n", conv);
invert (10, conv, &so, &de);
printf ("%d, %d \n",so , de);
}

*/
