/* *********************************************************************** */
/*   File:  distrib.cc                                                     */
/*                                                                         */
/*       Contents: This file contains the stochastic distribution          */
/*                functions. The file was taken from Edward Vopata's       */
/*                simulator code (Thesis) and modified appropriately.      */
/*                                                                         */
/* *********************************************************************** */

#include <math.h>
#include "distrib.h"

/********************************************************************
 * distrib.cc -- by Edward Vopata
 ********************************************************************
 * Stochastic distribution functions.
 *   the get_time() function returns a service or arrival time,
 *   based on the specified stochastic distribution function and
 *   parameters.
 ********************************************************************
 */

/* Forward Declaration of the stochastic distribution functions */

double uniform(double lower, double upper);
double normal(double mean,double stdev,double min,double max);
double expntl(double mean,double min,double max);
double erlang(double mean,double k,double min,double max);
double my_gamma(double mean,double k,double min,double max);
double beta( double k1, double k2);
double lognormal(double mean,double stdev);
double weibull(double shape,double scale);
double binomial(double num,double prob);
double poisson(double mean);

/********************************************************************
 * Function:
 *   get_time()
 * Parameter:
 *   dis - specifies the stochastic distribution function to use,
 *         and the parameter(s) to use with the function.
 * Summary:
 *   Use the specified stochastic distribution function to generate
 *   a random variable. The get_time function is used to generate
 *   service and arrival times.
 * Returns:
 *   a floating point value greater than zero (0). Note: the time
 *   return may be zero (0).
 ********************************************************************
 */
double get_time(dis)
distrib_struc dis;
{
  double time;  /* arrival or service time */

  if (dis.min < 0.0)  dis.min = 0.0;
  if (dis.max < 0.0)  dis.max = HUGE_VAL;  /* same as INF_TIME in 'glob.h' */


  switch(dis.type) {

      case FIXED:     time = dis.param1;
                      break;
  
      case UNIFORM:   time = uniform(dis.min,dis.max);
                      break;

      case NORMAL:    time = normal(dis.param1,dis.param2,dis.min,dis.max);
                      break;

      case EXPNTL:    time = expntl(dis.param1,dis.min,dis.max);
                      break;

      case ERLANG:    time = erlang(dis.param1,dis.param2,dis.min,dis.max);
                      break;

      case GAMMA:     time = my_gamma(dis.param1,dis.param2,dis.min,dis.max);
                      break;

/* not yet fixed for min and max */
      case POISSON:   time = poisson(dis.param1);
                      break;

/* not yet fixed for min and max */
      case BINOMIAL:  time = binomial(dis.param1,dis.param2);
                      break;

/* not yet fixed for min and max */
      case BETA:      time = beta(dis.param1,dis.param2);
                      break;

/* not yet fixed for min and max */
      case LOGNORMAL: time = lognormal(dis.param1,dis.param2);
                      break;

/* not yet fixed for min and max */
      case WEIBULL:   time = weibull(dis.param1,dis.param2);
                      break;
  }

  return time; 

} /* end get_time */


/***********************************************************
 *  Stochastic Distribution Functions.                     *
 *  These functions are from Monte Hall's Thesis [HALL88]  *
 ***********************************************************
 */

/**********************************************************/
/*           Discrete Statistical Distributions           */
/**********************************************************/

/*---- INTEGER UNIFORM [a,b] RANDOM VARIATE GENERATOR ----*/
/*                                                        */
/*  This function requires two integer bounds as input    */
/*  parameters which represent the range in which the     */
/*  integer random variates are generated.                */
/*                                                        */
/*--------------------------------------------------------*/

double uniform(double lower, double upper)
{
    double c;

    c =  lower + (upper - lower) * drand01();
    return (c);
}


/*---------- POISSON RANDOM VARIATE GENERATOR ------------*/
/*                                                        */
/*  This poisson distribution is usually used to model    */
/*  the number of arrivals in a given amount of time.     */
/*  It is related to the exponential function.  The mean  */
/*  is required as an input parameter, and an integer     */
/*  random variate is generated.                          */
/*                                                        */
/*--------------------------------------------------------*/

/* >>>>>>>>>>  not yet fixed for min and max <<<<<<<<< */

double poisson(double mean)
{
    double n;
    double x,y;

    n = 0.0;

    if (mean > 6.0) return ((double)normal(mean,sqrt(mean), 0.0, HUGE_VAL));
    else {
        y = exp(-1 * mean);
        x = drand01();

        while (x >= y) {
            n += 1.0;
            x = x * drand01();
        }
        return (n);
    }
}

/*---------- BINOMIAL RANDOM VARIATE GENERATOR -----------*/
/*                                                        */
/*  According to the SIMSCRIPT book description from      */
/*  which these functions were borrowed, the binomial     */
/*  distribution represents the integer number of         */
/*  successes in n independent trials, each having prob-  */
/*  ability of success p.                                 */
/*                                                        */
/*--------------------------------------------------------*/

/* >>>>>>>>>>  not yet fixed for min and max <<<<<<<<< */

double binomial(double num,double prob)
{
    register i;
    double sum = 0.0;

    for (i = 0; i < num; i++)
        if (drand01() <= prob) sum += 1.0;
    return (sum);
}


/**********************************************************/
/*          Continuous Statistical Distributions          */
/**********************************************************/

/*------------ BETA RANDOM VARIATE GENERATOR -------------*/
/*                                                        */
/*  The input parameters to beta are two variables, which */
/*  when put together in the formulas below determine the */
/*  mean (mu) and standard deviation (sd) of the distri-  */
/*  bution:                                               */
/*                                                        */
/*   mu = k1 / (k1 + k2)                                  */
/*   sd = sqrt((k1 * k2) / (sqr(k1 + k2) * (k1 + k2 + 1)  */
/*                                                        */
/*--------------------------------------------------------*/

/* >>>>>>>>>>  not yet fixed for min and max <<<<<<<<< */

double beta( double k1, double k2)
{
    double x, y;

    x = my_gamma(k1,k1, 0.0, HUGE_VAL);
    return (x / (x + my_gamma(k2,k2, 0.0, HUGE_VAL)));


}

/*----------- ERLANG RANDOM VARIATE GENERATOR ------------*/
/*                                                        */
/*  An erlang function is a special case of a gamma       */
/*  function when k is an integer.  If k = 1, then the    */
/*  erlang function is the same as the exponential        */
/*  function.  The mean (x) and a constant (k) are the    */
/*  input parameters to the function.  An extra test was  */
/*  added to this code to assure that the value of the    */
/*  variable e was not equal to zero, primarily so the    */
/*  logarithm function would not be passed a parameter    */
/*  equal to zero.                                        */
/*                                                        */
/*--------------------------------------------------------*/

/* HANDLES min AS A SHIFT FROM THE Y AXIS, AND max AS A CUT-OFF */

double erlang(double mean,double k,double min,double max)
{
    register i;
    double e,time;

 mean -= min;
 do{

    do {
        e = 1.0;
        for (i=0; i < k; i++) e *= drand01();
    } while (e == 0.0);
    time = -(mean/k) * log(e);

 } while (min + time > max);

 return ( min + time );
}

/*--------- EXPONENTIAL RANDOM VARIATE GENERATOR ---------*/
/*                                                        */
/*  The input parameter for an exponential distribution   */
/*  is the mean (x).  The variance for an exponential     */
/*  distribution is simply the square of the mean.        */
/*                                                        */
/*--------------------------------------------------------*/

/* HANDLES min AS A SHIFT FROM THE Y AXIS, AND max AS A CUT-OFF */

double expntl(double mean,double min,double max)
{
    double y,time;

 mean -= min;
 do{

    while ((y = drand01()) == 0.0);
    time = (-mean) * log(y);

 } while (min + time > max);

 return (min + time);
}

/*------------ GAMMA RANDOM VARIATE GENERATOR ------------*/
/*                                                        */
/*  The gamma function requires a mean (x) and a constant */
/*  (k) as input parameters.  If k is an integer, then    */
/*  this function is the same as the erlang function.  If */
/*  k is equal to one, this function is the same as the   */
/*  exponential function.  If k is equal to one-half,     */
/*  this function is the same as the chi-square distri-   */
/*  bution.  The density function for this distribution   */
/*  is given below:                                       */
/*                                                        */
/*    f(x) = ( (1 / (k-1)! * pow(b,k))  *                 */
/*             pow(x,(k-1)) * exp(-x/b) )                 */
/*                                                        */
/*      where the following holds:                        */
/*        k > 0,  b > 0, and x >= 0                       */
/*      and the mean is:  x = k * b                       */
/*      and the variance is:  var = sqr(b) * k            */
/*                                                        */
/*  The gamma function has smaller variance and more      */
/*  control in parameter selection, and therefore more    */
/*  realistically represents observed data, such as       */
/*  service times.  It is often used in preference to the */
/*  exponential function, and is closely related to the   */
/*  beta and erlang functions, according to the SIMSCRIPT */
/*  book from which these functions where borrowed.       */
/*                                                        */
/*--------------------------------------------------------*/

/* HANDLES min AS A SHIFT FROM THE Y AXIS, AND max AS A CUT-OFF */

double 
my_gamma(double mean,double k,double min,double max)
{
 double z,a,b,d,e,x,y,w,v, time;
 long kk;
 register i;

 mean -= min;
 do{

    z = 0.0;
    kk = (long) k;  /* truncation of k */
    d = k - kk;    /* fractional of k */

    if (kk != 0) {
        do {
            e = 1.0;
            for (i=0; i < kk; i++) e *= drand01();
        } while (e == 0.0);
        z = -(log(e));
        if (d == 0.0)  time = (mean / k) * z;
    }

    if (d != 0.0) {
        a = 1.0 / d;
        b = 1.0 / (1.0 - d);
        y = 2.0;

        while (y > 1.0) {
          x = pow(drand01(),a);
          y = (pow(drand01(),b)) + x;
        }

        w = x / y;
        while ((v = drand01()) == 0.0);
        y = -(log(v));
        time = ((w * y + z) * (mean / k));
    }

 } while (min + time > max);

 return (min + time);
}


/*--------- LOG NORMAL RANDOM VARIATE GENERATOR ----------*/
/*                                                        */
/*  This function requires a mean and standard deviation  */
/*  (sigma) as input parameters.  The log normal function */
/*  is often used to characterize skewed data.  The mean  */
/*  and variance of this distribution function are given  */
/*  below:                                                */
/*                                                        */
/*    mu = exp(mean + (sqr(sigma) / 2))                   */
/*    sig = exp( (mean * 2) + (sqr(sigma)) )  *           */
/*           ( (exp (sqr(sigma))) - 1)                    */
/*                                                        */
/*--------------------------------------------------------*/

/* >>>>>>>>>>  not yet fixed for min and max <<<<<<<<< */

double lognormal(double mean,double stdev)
{
    double s,u;

    s = log((stdev * stdev) / (mean * mean) + 1);
    u = log(mean) - (0.5 * s);
    return (exp(normal(u,sqrt(s), 0.0, HUGE_VAL)));
}

/*------------ NORMAL RANDOM VARIATE GENERATOR -----------*/
/*                                                        */
/*  The normal distribution function provides a "bell-    */
/*  shaped curve".  It requires the mean (mu) and stan-   */
/*  dard deviation (sigma) as input parameters.  If in-   */
/*  appropriate relative values of mean and standard      */
/*  deviation are entered, it is possible that the "tail" */
/*  of the function can extend into the negative region   */
/*  of the graph (x-axis).  This could cause some         */
/*  complications in regard to generating service times,  */
/*  which have no meaning if negative.  An extra test was */
/*  added to this code to recalculate a new random        */
/*  variate if a variate of less than zero is generated.  */
/*                                                        */
/*--------------------------------------------------------*/

/* HANDLES BOTH min AND max AS CUT-OFFS */

double normal(double mean,double stdev,double min,double max)
{
    double q,r,s,x,xx,y,yy;

    do {
        s = 2.0;
        while (s > 1.0) {
            x = drand01();
            y = (2.0 * drand01()) - 1;
            xx = x * x;
            yy = y * y;
            s = xx + yy;
        }
        while ((x = drand01()) == 0.0);
        r = sqrt((-2.0) * log(x)) / s;
        q = r * stdev * (xx - yy) + mean;
    } while (q < min  || q > max);

    return (q);
}

/*----------- WEIBULL RANDOM VARIATE GENERATOR -----------*/
/*                                                        */
/*  This function can represent several families of       */
/*  distribution functions depending on the values of the */
/*  input parameters.  If the shape parameter is equal to */
/*  one, then this function is the same as the exponen-   */
/*  tial function with a mean equal to the scale para-    */
/*  meter.  There is also a similarity between this       */
/*  function and the gamma distribution when the shape    */
/*  parameter is set equal to two.                        */
/*                                                        */
/*--------------------------------------------------------*/

/* >>>>>>>>>>  not yet fixed for min and max <<<<<<<<< */

double weibull(double shape,double scale)
{
    double x;

    while ((x = drand01()) == 0.0);
    return (scale * pow((-log(x)),(1.0 / shape)));
}

