function [ feature_locs ] = sd_symi(img, keypoints_fname, varargin)
% Intensity based symmetry detector.

sd_init;

params = sd_defaults();

use_small_radius_nms = 0;

for k = 1:2:length(varargin)
    
    switch(lower(varargin{k}))
        case 'min_img_size'
            params.min_img_size = varargin{k + 1};
        case 'max_img_size'
            params.max_img_size = varargin{k + 1};
        case 'n_iters'
            n_iters = varargin{k + 1};
        case 'level_scale_factor'
            params.level_scale_factor = varargin{k + 1};
        case 'k_width'
            params.k_width = varargin{k + 1};
        case 'geom_blur'
            params.geom_blur = varargin{k + 1};
        case 'uniform_blur'
            params.uniform_blur = varargin{k + 1};
        case 'n_ang_buckets'
            n_ang_buckets = varargin{k + 1};
        case 'n_rad_buckets'
            n_rad_buckets = varargin{k + 1};
        case 'r0'
            r0 = varargin{k + 1};
        case 'desc_support'
            desc_support = varargin{k + 1};
        case 'use_log_mapping'
            use_log_mapping = varargin{k + 1};
        case 'dist_func'
            params.dist_func = varargin{k + 1};
        case 'alpha'
            params.alpha = varargin{k + 1};
        case 'nms_radius'
            params.nms_radius = varargin{k + 1};
        case 'nms_thresh'
            params.nms_thresh = varargin{k + 1};
        case 'baseline_descriptor'
            baseline_descriptor = varargin{k + 1};
        case 'overlap_thresh'
            params.overlap_thresh = varargin{k + 1};
        case 'nms_feats_on'
            params.nms_feats_on = varargin{k + 1};
        case 'randomize_seed'
            params.randomize_seed = varargin{k + 1};
        case 'input_keypoints'
            params.input_keypoints = varargin{k + 1};
        case 'magnif'
            params.magnif = varargin{k + 1};            
        case 'use_small_radius_nms'
            use_small_radius_nms = varargin{k + 1};
                        
        case 'channels'
            params.channels = [];
            if sum(varargin{k + 1} == 'r') ~= 0
                params.channels = [params.channels 1];
            end
            if sum(varargin{k + 1} == 'h') ~= 0
                params.channels = [params.channels 2];
            end
            if sum(varargin{k + 1} == 'v') ~= 0
                params.channels = [params.channels 3];
            end
        otherwise
            error(sprintf('Unrecognized option: %s', lower(varargin{k})));
    end
end

% Print values
sd_log(1, sprintf('min_img_size = %d', params.min_img_size));
sd_log(1, sprintf('max_img_size = %d', params.max_img_size));
sd_log(1, sprintf('level_scale_factor = %f', params.level_scale_factor));
sd_log(1, sprintf('k_width = %d', params.k_width));
sd_log(1, sprintf('geom_blur = %d', params.geom_blur));
sd_log(1, sprintf('uniform_blur = %d', params.uniform_blur));
sd_log(1, sprintf('dist_func = %d', params.dist_func));
sd_log(1, sprintf('symscore_width = %d', params.symscore_width));
sd_log(1, sprintf('alpha = %f', params.alpha));
sd_log(1, sprintf('nms_radius = %d', params.nms_radius));
sd_log(1, sprintf('nms_thresh = %d', params.nms_thresh));
sd_log(1, sprintf('nms_feats_on = %d', params.nms_feats_on));
sd_log(1, sprintf('overlap_thresh = %d', params.overlap_thresh));
sd_log(1, sprintf('randomize_seed = %d', params.randomize_seed));
sd_log(1, sprintf('use_small_radius_nms = %d', use_small_radius_nms));
if ~isempty(params.input_keypoints)
    sd_log(1, sprintf('input_keypoints = %s', params.input_keypoints));
end


% Load image
img_fname = '';
if isa(img, 'char')
    img_fname = img;
    img = sd_imread(img_fname);
end
img_orig = img;

% if ~isa(img, 'double')
%     img = im2double(img);
% end
% 
% if size(img,3) >= 3
%     img = rgb2gray(img);
% elseif size(img,3) == 2
%     % second channel is alpha
%     img = img(:,:,1);
% end

%img = histeq(img);

% Build image pyramid
img_pyr = sd_build_image_pyramid2(img, params.level_scale_factor, params.min_img_size, params.max_img_size, params.uniform_blur, params.magnif);
n_levels = length(img_pyr.levels);

% Compute sym dist independently for each level
sd_log(1, 'Computing sym distance for each level');
symdist_pyr = struct('levels', {{}}, 'level_scale_factor', img_pyr.level_scale_factor);

% LoG kernel
log_filt = fspecial('log', params.log_w, params.log_std);
log_filt = log_filt / sum(max(log_filt(:),0)) ;

% Scale factor that brings feature size to nms radius
if use_small_radius_nms
    featsize2radius = (4 / params.k_width) * params.uncertainty_radius;
else
    featsize2radius = 2;
end

for i = 1:n_levels
    sd_log(2, sprintf('Processing level %d', i));
    symscale = sd_symscale_donut(img_pyr.levels{i}, params.k_width);
    symscale = conv2(symscale, log_filt, 'same');
    
    symdist_pyr.levels{end + 1} = symscale;    
end

% Locate the features on the pyramid
sd_log(1, 'Locating features on pyramid');
[ feature_locs ] = sd_locate_features(symdist_pyr, params.nms_thresh, params.nms_radius, params.channels, params.k_width);

% Suppress repeated detections across scales
if params.nms_feats_on
    sd_log(1, 'Suppressing repeated detections across scales')
    [ feature_locs ] = sd_nms_features2(feature_locs, params.overlap_thresh, featsize2radius);
    sd_log(1, sprintf('Found a total of %d features\n', size(feature_locs,1)));
end

sd_write_keypoints(feature_locs, keypoints_fname, size(img));

end

