function sd_symg_generate_symmetry_features(filename, base_sym_scale, scale_factor, num_scales, ...
                                            grad_scale, hist_scale, num_bins, nms_thresh, ...
                                            use_rotational_symmetry, mag_factor, generate_regions, ...
                                            max_image_size, output_prefix, output_fname, debug)
% Generate a set of symmetry feature detections across scale-space 
% for the given image.  Symmetry features are computed by computing 
% oriented edge correlations.
% 
% Inputs:
%     filename       : name of the file to process (e.g., 'foo.jpg')
%     base_sym_scale : at each image scale, how big should the
%                      symmetry filter be?
%     scale_factor   : scale factor between each level (e.g., 2^(-1/8))
%     num_scales     : how many scales to generate
%     grad_scale     : how much to smooth images before computing gradients?
%     hist_scale     : the size of the Gaussian used when smoothing the edge orientation histograms
%     num_bins       : how many orientation bins to use?
%     nms_thresh     : threshold for non-maxima suppression
%     use_rotational_symmetry : use rotational symmetry, rather than
%                      bilateral symmetry
%     mag_factor     : amount by which to resize the image prior to
%                      computing pyramid (good values: 1 or 2)
%     generate_regions : generate (non-circular) regions instead of points
%     max_image_size : maximum image size, images larger than this will be downsampled
%     output_prefix  : prefix to use for output files
%     output_fname   : real filename to use
%
% Outputs: written to files
%
% EXAMPLE: sd_symg_generate_symmetry_features('foo.jpg', 12.0, 1.0, 1.0, 8, 0.4, 2000, 'foo')
%
%
% CHANGELOG:
% 09/30/11: Changed sigmoid for gradient magnitude to be a local constrast
%           enhancement
% 09/30/11: Decided to reduce recommended size of blur used to compute gradients,
%           increase size of feature window to 16
% 10/01/11: Changed local maxima to 5x5 window
% 10/01/11: Reduced blur for each orientation bin when smoothing out
%           histograms
% 11/04/11: Made NMS stricter 0.2501 -> 3.0 / scale_factor + 0.01

% MAX_IMAGE_SIZE = 600;

if ischar(filename)
    I = sd_imread(filename);
else
    % otherwise assume it's a matrix
    I = filename;
end
    
% I = imresize(I, 0.5);

[h,w,d] = size(I);

ratio = 1.0;
if h > max_image_size || w > max_image_size
    ratio = max_image_size / max(w, h);
    fprintf('Image too large, resizing by %0.3f\n', ratio);
    I = imresize(I, ratio, 'lanczos3');
end

% Various fixed settings...
% num_scales = 40; % 32;
% nms_radius = 1; % 3
nms_radius = 3;
% nms_thresh = 1.0e-2;
% nms_thresh = 1.0e-1;
% nms_thresh = 3.0e-1;

feature_locs = [];
region_locs = [];

score_data = cell(num_scales+1);

% use_rotational_symmetry = 0;
downsample = 1;
% base_sym_scale = 6; % 12
% mag_factor = 2.0;

min_sym_scale = (base_sym_scale / 2.0) / mag_factor;
% min_sym_scale = 3.01;

fprintf('@@@ min_sym_scale: %0.3f\n', min_sym_scale);
fprintf('@@@ mag_factor   : %0.3f\n', mag_factor);

for round = 0:num_scales
    scale_round = mag_factor * (scale_factor^round);
    
    if downsample
        Is = imresize(I, scale_round, 'lanczos3');
    else
        Is = I;
    end
    pnts_scale = 1 ./ (scale_round * ratio);

    % horizontal symmetries
    if downsample
        sym_scale = base_sym_scale;
        feat_scale = base_sym_scale ./ (2.0 * scale_round * ratio);
        % feat_scale = min_sym_scale / (scale_factor^round * ratio);
    else
        sym_scale = base_sym_scale / (scale_round);
        feat_scale = sym_scale;
    end
    
    if debug
        imwrite(im2double(Is), sprintf('%s_%03d.jpg', output_prefix, round));
    end
        
    % Compute symmetry scores at current scale
    fprintf('@@@ using symmetry scale %0.3f (feature scale %0.3f)\n', sym_scale, feat_scale);    
    fprintf('@@@ using grad scale %0.3f\n', grad_scale); % default grad_scale: 3.0
    fprintf('@@@ computing current scale\n');

    % score_r = sd_compute_symmetry_orient_fast_bins_noah(Is, sym_scale, grad_scale, num_bins, 'rotational');

    % output_file = sprintf('%s_rot_g%0.3f_%03d.png', ...
    %                       output_prefix, base_sym_scale, round);
    % sd_imwrite_grayscale(score_r, [0 3], output_file);    
  
    if ~use_rotational_symmetry
        if ~generate_regions
            [ score_h, score_v, product ] = ...
                sd_symg_compute_horz_vert_symmetry_regions(Is, sym_scale, grad_scale, hist_scale, num_bins);

            % product = max(conv2(product, -fspecial('log', [21 21], 2.0), 'same'), 0.0);
            
            if debug
                % Write diagnostic images
                output_file = sprintf('%s_horz_s%0.3f_g%0.3f_h%0.3f_b%d_%03d.png', ...
                                      output_prefix, base_sym_scale, grad_scale, hist_scale, num_bins, round);
                sd_imwrite_grayscale(score_h, [0 sqrt(2.0 * nms_thresh)], output_file);
                output_file = sprintf('%s_vert_s%0.3f_g%0.3f_h%0.3f_b%d_%03d.png', ...
                                      output_prefix, base_sym_scale, grad_scale, hist_scale, num_bins, round);
                sd_imwrite_grayscale(score_v, [0 sqrt(2.0 * nms_thresh)], output_file);
                output_file = sprintf('%s_prod_s%0.3f_g%0.3f_h%0.3f_b%d_%03d_s%0.3f.png', ...
                                      output_prefix, base_sym_scale, grad_scale, hist_scale, num_bins, round, pnts_scale);
                sd_imwrite_grayscale(product, [0 2.0 * nms_thresh], output_file);
            end
        else
            % generate regions
            scores_h = sd_compute_symmetry_orient_fast_bins_noah(Is, sym_scale, grad_scale, hist_scale, num_bins, 0, 'horizontal', 2);

            if debug
                output_file = sprintf('%s_rhorz_s%0.3f_g%0.3f_h%0.3f_b%d_%03d.png', ...
                                      output_prefix, base_sym_scale, grad_scale, hist_scale, num_bins, round);
                sd_imwrite_grayscale(scores_h, [0 200.0 * nms_thresh], output_file);
            end
        end
    else
        product = sd_compute_symmetry_orient_fast_bins_noah(Is, sym_scale, grad_scale, hist_scale, num_bins, 0, 'rotational', 0);
        % output_file = sprintf('%s_rot_g%0.3f_h%0.3f_b%d_%03d_s%0.3f.png', ...
        %                       output_prefix, base_sym_scale, hist_scale, num_bins, round, pnts_scale);

        if debug
            output_file = sprintf('%s_rot_s%0.3f_g%0.3f_h%0.3f_b%d_%03d.png', ...
                                  output_prefix, base_sym_scale, grad_scale, hist_scale, num_bins, round);
            sd_imwrite_grayscale(product, [0 2.0 * nms_thresh], output_file);
        end
    end
        
    % test computation of region scores
    % region_locs_level = sd_compute_symmetry_regions(score_h, sym_scale * 0.5, pnts_scale, feat_scale, 16);
    % region_locs = [ region_locs; region_locs_level ];
    
    % Compute the scores at two neighboring scales, same image resolution
    % fprintf('@@@ computing higher scale\n');
    % [ score_h_prev, score_v_prev, product_prev ] = ...
    %     sd_compute_horz_vert_symmetries(Is, sym_scale * scale_factor, ...
    %                                     grad_scale, num_bins);

    % fprintf('@@@ computing lower scale\n');
    % [ score_h_next, score_v_next, product_next ] = ... 
    %     sd_compute_horz_vert_symmetries(Is, sym_scale / scale_factor, ...
    %                                     grad_scale, num_bins);

    if ~generate_regions
        product_prev = zeros(size(product));
        product_next = zeros(size(product));
    
        % Find local maxima in x-y-scale space, save the resulting features
        feature_locs_level = ...
            sd_symg_locate_features_scale(product_prev, product, product_next, ...
                                          pnts_scale, feat_scale, ...
                                          nms_radius, nms_thresh);
        feature_locs = [ feature_locs ; feature_locs_level ];
    else
        % compute regions
        [region_locs_level, region_scores] = sd_compute_symmetry_regions_dog(scores_h, base_sym_scale, pnts_scale, feat_scale, 40, nms_thresh);
        region_locs = [ region_locs; region_locs_level ];
    end 
end

if ~generate_regions
    numFeats = size(feature_locs, 1);
    fprintf('@@@ found %d features (before filtering)\n', numFeats);
    % feature_locs = sd_filter_features_scale(feature_locs, (3.0/scale_factor) + 0.01, min_sym_scale + 0.01, 0.25 * base_sym_scale); % base_sym_scale was 3 was 2 was 5
    feature_locs = sd_filter_features_scale2(feature_locs, 0.4); % base_sym_scale was 3 was 2 was 5
    numFeats = size(feature_locs, 1);
    fprintf('@@@ found %d features (after filtering)\n', numFeats);

    % Write keypoints to a file
    keypoints_fname = sprintf('%s_g%0.3f_h%0.3f_t%0.3f_b%d.ks', ...
                              output_prefix, base_sym_scale, hist_scale, nms_thresh, num_bins);
    f = fopen(keypoints_fname, 'w');
    fprintf(f, '# keys = %d, px = %f, py = %f\n', size(feature_locs,1), w/2, h/2);
    fclose(f);
    dlmwrite(keypoints_fname, feature_locs, '-append', 'delimiter', ' ');

    % write to another file
    f = fopen(output_fname, 'w');
    fprintf(f, '# keys = %d, px = %f, py = %f\n', size(feature_locs,1), w/2, h/2);
    fclose(f);
    dlmwrite(output_fname, feature_locs, '-append', 'delimiter', ' ');
else
    % write regions
    numRegions = size(region_locs, 1);
    fprintf('@@@ found %d regions (before filtering)\n', numRegions);
    region_locs = sd_filter_regions_scale(region_locs, 0.5); % 2 was 5
    numRegions = size(region_locs, 1);
    fprintf('@@@ found %d regions (after filtering)\n', numRegions);
    
    regions_fname = sprintf('%s_g%0.3f_h%0.3f_t%0.3f_b%d.rs', ...
                             output_prefix, base_sym_scale, hist_scale, nms_thresh, num_bins);
    f = fopen(regions_fname, 'w');
    fprintf(f, '# keys = %d, px = %f, py = %f\n', size(region_locs,1), w/2, h/2);
    fclose(f);
    dlmwrite(regions_fname, region_locs, '-append', 'delimiter', ' ');    

    % compute features from regions (by averaging scales)
    feature_locs = [ region_locs(:,1:2), sqrt(region_locs(:,3) .* region_locs(:,4)), region_locs(:,5:6) ];
    
    features_fname = sprintf('%s_g%0.3f_h%0.3f_t%0.3f_b%d.ks', ...
                             output_prefix, base_sym_scale, hist_scale, nms_thresh, num_bins);
    f = fopen(features_fname, 'w');
    fprintf(f, '# keys = %d, px = %f, py = %f\n', size(feature_locs,1), w/2, h/2);
    fclose(f);
    dlmwrite(features_fname, feature_locs, '-append', 'delimiter', ' ');    
end
