function score = sd_symg_compute_symmetry_orient_fast_bins(I, feature_scale, grad_scale, hist_scale, num_bins, ...
                                                           normalize_hist, orientation, smooth_type)
% Compute a 'symmetry score' at a given scale
% orientations: 
%
% Inputs:
%    I             : input image (could be color or grayscale)
%    feature_scale : scale of the filter used to detect symmetries
%    grad_scale    : how much to blur the image when computing 
%                    gradients (used to create Gabor filters) [0.1]
%    hist_scale    : how much to blur the histogram bins spatially [0.1]
%    num_bins      : number of orientation bins to use [4]
%    normalize_hist: soft-normalize the per-pixel histograms after
%                  : smoothing?
%    orientation   : symmetry type: 
%                     {'horizontal', 'vertical', (possibly one day
%                     'rotational')} ['horizontal']
%    smooth_type   : how to smooth per-pixel symmetry histograms. [0]
%                  :   0: Gaussian blur before dividing AND/SUM
%                  :   1: Gaussian blur *within scanline* before division
%                  :   2: no blur of histograms, Gaussian blur of scores
%
% Output:
%    score : matrix of symmetry scores
% 
% EXAMPLE: sd_symg_compute_symmetry_orient_fast_bins(image, 12.0, 1.0, 0.5, 8, 0,
% 'horizontal', 0);
% 
% CHANGELOG:
% 10/02/2011: added 'smooth_type' parameter -- changes behavior of how
%             per-pixel histograms are combined to form a final histogram
% 10/26/2011: Added LarryZ's gradient normalizer


if ~exist('grad_scale', 'var')
    grad_scale = 0.1;
end

if ~exist('hist_scale', 'var')
    hist_scale = 0.1;
end

if ~exist('num_bins', 'var')
    num_bins = 4;
end

if ~exist('normalize_hist', 'var')
    normalize_hist = 0;
end

if ~exist('orientation', 'var')
    orientation = 'horizontal';
end

if ~exist('smooth_type', 'var')
    smooth_type = 0;
end

% other constants
normalize_grad_mag_type = 2;  % -1: raw gradient magnitudes, 0: sigmoid, 1: contrast enhance, 2: larryz
normalize_by_max = 0; % 1
use_min = 0; % 1

soft_assign_orientations = 1;

use_harris = 0; % 1;
harris_threshold = 0.0002;

epsilon_hist = 5.0e-2;  % epsilon used for histogram normalization
% epsilon used for normalizing histogram scores

if normalize_grad_mag_type == -1
    epsilon_base = 5.0e-2;
elseif normalize_grad_mag_type == 0
    epsilon_base = 5.0e-2; % 4.0; % 1.0; % 1.0e-2;
elseif normalize_grad_mag_type == 1
    epsilon_base = 5.0e-2;
elseif normalize_grad_mag_type == 2
    % epsilon_base = 5.0e-1; % for larryz
    % epsilon_base = 1.0e-1; % for larryz
    epsilon_base = 5.0e-2; % for larryz
end

epsilon = epsilon_base / hist_scale;


fprintf('### grad_scale  = %0.3f\n', grad_scale);
fprintf('### hist_scale  = %0.3f\n', hist_scale);
fprintf('### num_bins    = %d\n',    num_bins);
fprintf('### orientation = %s\n',    orientation);
fprintf('### smooth_type = %d\n',    smooth_type);
fprintf('### normalizer  = %d\n',    normalize_grad_mag_type);
fprintf('### softorient  = %d\n',    soft_assign_orientations);

% Smooth the (grayscale) image
s = size(size(I));

if s(2) == 2
    Igray = im2double(I);
else
    Igray = im2double(rgb2gray(I));
end

% To detect vertical symmetries, simply flip the image
if strcmp(orientation, 'vertical')
    Igray = Igray';
end

% J = imsmooth(Igray, );
J = Igray;
[height, width] = size(J);



% COMPUTE GRADIENTS

% Gaussian kernel
fprintf('Initializing kernels...\n');

if grad_scale > 0.0
  G = fspecial('gaussian', [1 ceil(8 * grad_scale)+1], grad_scale);
  DG = conv(G, [-1 0 1], 'same');
else
  G = 1;  
  DG = [-1 0 1];
end

% x and y partial derivatives
fprintf('Computing derivatives...\n');
J_x = conv2(conv2(J, G', 'same'), DG, 'same');
J_y = conv2(conv2(J, G, 'same'), DG', 'same');
% J_x = conv2(J, DG, 'same');
% J_y = conv2(J, DG', 'same');

% compute gradient orientations, magnitude
J_mag = sqrt(J_x .* J_x + J_y .* J_y);
J_ori = atan2(J_y, J_x);

% normalize the gradient magnitudes
if normalize_grad_mag_type == -1
    % use the raw gradient magnitudes
    J_mag_sig = J_mag .* (J_mag > 0.1);
elseif normalize_grad_mag_type == 0
    % use a sigmoid
    alpha = 60.0;  beta = -0.1;  % sigmoid params
    J_mag_sig = 1 ./ (1 + exp(-(alpha * (J_mag + beta))));
    J_mag_sig = max(J_mag_sig, 1.0e-5) - 1.0e-5; % shouldn't have negative magnitudes
elseif normalize_grad_mag_type == 1
    % use my constrast enhancement on the gradient magnitude and threshold
    J_mag_sig = local_contrast_enhance(J_mag, 6, 0.1);
    J_mag_sig = max(J_mag_sig, 0.05) - 0.05; % shouldn't have negative magnitudes
    threshold = 0.75;
    J_mag_sig = min(J_mag_sig, threshold); % threshold gradients that are very large
elseif normalize_grad_mag_type == 2
    % use LarryZ's gradient magnitude normalization
    % [J_mag_sig, J_ori] = sd_normalized_gradient_zitnick(J, 6.0); % was 6.0
    % J_mag_sig = min(J_mag_sig, 1.2);
    
    % [J_mag_sig, J_ori] = sd_normalized_gradient_zitnick(J, 3.0, 2.0e-1); % was 6.0
    % J_mag_sig = min(J_mag_sig, 2.0);
    % J_mag_sig = max(J_mag_sig, 0.5) - 0.5;

    [J_mag_sig, J_ori] = sd_normalized_gradient_zitnick(J, 3.0, 2.0e-1); % was 6.0
    J_mag_sig = min(J_mag_sig, 1.5);
    J_mag_sig = max(J_mag_sig, 0.2) - 0.2;
else
    error('000', 'Unknown gradient magnitude normalizer');
end

% undo the 180-degree ambiguity
J_ori = J_ori .* (J_ori >= 0.0) + (J_ori+pi) .* (J_ori < 0.0);

% divide into num_bins orientation bins (counting theta and theta+180 as the same)
bin_centers = linspace(0,pi,num_bins+1);
bin_centers = bin_centers(1:num_bins);
bin_size = pi / num_bins;
J_hists = zeros(height, width, num_bins);

% Take care of rounding issues
if soft_assign_orientations
    if ~strcmp(orientation, 'rotational')
        bin_sigma = 0.5 * bin_size;
    else
        bin_sigma = 1.0 * bin_size;
    end
    bin_sigma_sq = bin_sigma * bin_sigma;
    for i = 1:num_bins
        bin_diffs = min(abs(J_ori - bin_centers(i)), ...
                        abs(J_ori - pi - bin_centers(i)));
        J_hists(:,:,i) = exp(-bin_diffs .* bin_diffs / bin_sigma_sq);
        
    end
    
    J_mags = sum(J_hists(:,:,:) .^ 2, 3);
    
    for i = 1:num_bins
        J_hists(:,:,i) = J_mag_sig .* J_hists(:,:,i); % (J_hists(:,:,i) ./ J_mags);
    end
else
    J_bins = round(J_ori ./ bin_size) + 1;
    J_bins = J_bins .* (J_bins <= num_bins) + 1 * (J_bins > num_bins);

    for i = 1:num_bins
        J_hists(:,:,i) = (J_bins == i) .* J_mag_sig;
    end
end
    
% Smooth out the histograms a bit spatially
Gsigma = hist_scale; % 1.0;
G = fspecial('gaussian', [1 ceil(8 * Gsigma) + 1], Gsigma);

for i = 1:num_bins
    J_hists(:,:,i) = conv2(conv2(J_hists(:,:,i), G, 'same'), G', 'same');
end

if normalize_hist
  % soft-normalize the histograms
  hist_mags = sum(J_hists, 3);
  J_hists = J_hists ./ (repmat(hist_mags + epsilon_hist, [1 1 num_bins]));
end

% threshold the histograms
% max_histogram_value = 0.2;
% J_hists = min(J_hists, max_histogram_value);

% Set close to zero near the boundary, pad with zeros
bdry_width = 4;
bdry_width = min(min(bdry_width, height-1), width-1);

J_hists_padded = zeros(height, 2*width+1, num_bins);

for i = 1:num_bins
    J_hists(:,:,i) = sd_set_zero_boundary(J_hists(:,:,i), bdry_width);
    J_hists_padded(:,:,i) = [J_hists(:,:,i) zeros(height, width+1)];
end

% COMPUTE A DISTANCE TRANSFORM FOR EACH DERIVATIVE ORIENTATION
% This could be done in several ways.  Simply put, we want to be able to
% compute, for any edge, whether there is a similar edge (possibly flipped)
% in some other part of the image.  This is like a soft AND operation
% (i.e., a product).  Why not just take a product?  We expect the
% alignment to be less exact further away from the center of the window.
% How to fix this?  By ANDing an edge magnitude not with the exact matching
% pixel, but with the largest one in a small window, weighted by spatial 
% distance (where "small" ideally gets bigger further away from the
% center).  This is not exactly a distance transform, but rather a
% "geometric blur(?)" -- actually, I believe it can be formulated as a
% max-convolution.  For now, let us just compute the AND operation and
% multiply by a Gaussian.

score = zeros(height, width);

% G_scale = fspecial('gaussian', [height 2 * width], feature_scale);
G_rad = 3*ceil(feature_scale);
G_size = 2*G_rad + 1;
% G_scale = fspecial('gaussian', [6*ceil(feature_scale)+1 2*width+1], feature_scale);
% G_scale = fspecial('gaussian', [G_size G_size], feature_scale);
% G_scale = fspecial('gaussian', [G_size 2*width+1], feature_scale);

use_center_surround = 0;
if ~use_center_surround
    G_1D = fspecial('gaussian', [1 G_size], feature_scale);
    G_2D = fspecial('gaussian', [G_size, G_size], feature_scale);
else
    % create sum of two Gaussians slightly offset
    G_1D = sum_of_offset_gaussians(feature_scale, G_size);
    G_2D = gaussian_ring_filter(feature_scale, G_rad);
end
    

orth_scale_factor = 0.5;
feature_scale_orth = orth_scale_factor * feature_scale;
G_1D_orth = fspecial('gaussian', [1 G_size], feature_scale_orth);

% G_scale2 = conv2(G_1D', G_1D, 'full');
% G_scale2 = G_scale2((end+1)/2-G_rad:(end+1)/2+G_rad,:);

if ~strcmp(orientation, 'rotational')
    iter_bounds = num_bins/2+1;
else
    iter_bounds = num_bins;
end

fprintf('Computing distance transforms...\n');
for x = 1:width
  if mod(x,100) == 0
    fprintf('processing column %d...\n', x);
  end

  % Compute the size of the intersection region
  % isect_width = min(2*x - 1, 2*(width-x)+1);
  isect_rad = min(x-1, width-x); %width-x+1
  conv_rad = min(isect_rad, G_rad);
  % conv_width = min(isect_width, G_size); % width of the convolution region

  % Slice out the right part of each matrix

  J_hist_slices = zeros(height, 2*conv_rad+1, num_bins/2+1);
  Jf_hist_slices = zeros(height, 2*conv_rad+1, num_bins/2+1);

  for i = 1:iter_bounds
      % compute the flipped index
      i_f = mod(-(i-1),num_bins)+1;
      
      J_hist_slices(:,:,i) = J_hists_padded(:,x-conv_rad:x+conv_rad,i);

      if ~strcmp(orientation, 'rotational')
          Jf_hist_slices(:,:,i) = J_hists_padded(:,x+conv_rad:-1:x-conv_rad,i_f);
      end
  end

  % compute
  if ~strcmp(orientation, 'rotational')
      if use_min
          AND = min(J_hist_slices, Jf_hist_slices);
          SUM = max(J_hist_slices, Jf_hist_slices);
      else
          % use the dot product
          AND = J_hist_slices .* Jf_hist_slices; % DOT
      end
  end
  
  % Grab the slices of the gaussian kernel
  G_slice_1D_x = G_1D(G_rad+1-conv_rad:G_rad+1+conv_rad);

  if width >= G_rad
      G_slice_1D_y = G_1D_orth';
  else
      G_slice_1D_y = fspecial('gaussian', [G_size 1], feature_scale_orth);
  end
      
  accum_num = zeros(height,iter_bounds);
  accum_den = zeros(height,iter_bounds);
  
  % if smooth_type == 2, first compute per-pixel symmetry score
  if smooth_type == 2
    % for i=1:num_bins/2+1
    %   per_pixel_scores = per_pixel_scores + AND(:,:,i) ./ (SUM(:,:,i) + epsilon);
    % end

    % per_pixel_scores = sum(AND,3) ./ (sum(SUM,3) + epsilon);
    % per_pixel_scores = sum(AND,3);
    per_pixel_scores = sum(AND,3);
    
    % now average per-pixel scores 
    per_pixel_scores_padded = [ zeros(G_rad, 2*conv_rad+1); per_pixel_scores; zeros(G_rad, 2*conv_rad+1) ];
    score(:,x) = conv2(conv2(per_pixel_scores_padded, G_slice_1D_x, 'valid'), G_slice_1D_y, 'valid');
    % score(:,x) = conv2(per_pixel_scores, G_slice_1D_x, 'valid');
  else % smooth_type == 0 or 1
      % do one big convolution, first padding the vectors to create
      % a correct-sized valid region.  What this convolution does is average the
      % masses of each AND or SUM'ed histogram bin spatially.  Alternatively,
      % we can smooth the scores vertically *AFTER* doing this horizontally...
      for i=1:iter_bounds
          if ~strcmp(orientation, 'rotational') 
              if smooth_type == 0
                  AND_padded = [ zeros(G_rad, 2*conv_rad+1); AND(:,:,i); zeros(G_rad, 2*conv_rad+1) ];
                  if normalize_by_max
                      SUM_padded = [ zeros(G_rad, 2*conv_rad+1); SUM(:,:,i); zeros(G_rad, 2*conv_rad+1) ];
                  end
              elseif smooth_type == 1
                  AND_padded = AND(:,:,i);
                  if normalize_by_max
                      SUM_padded = AND(:,:,i);
                  end
              end

              % convolution is separable...
              if smooth_type == 0
                  accum_num(:,i) = conv2(conv2(AND_padded, G_slice_1D_x, 'valid'), G_slice_1D_y, 'valid');
                  if normalize_by_max
                      accum_den(:,i) = conv2(conv2(SUM_padded, G_slice_1D_x, 'valid'), G_slice_1D_y, 'valid');
                  end
              elseif smooth_type == 1
                  % only average within scanlines
                  accum_num(:,i) = conv2(AND_padded, G_slice_1D_x, 'valid');
                  if normalize_by_max
                      accum_den(:,i) = conv2(SUM_padded, G_slice_1D_x, 'valid');
                  end
              end
          else % orientation == 'rotational'
              % J_hist_padded = [ zeros(G_rad, 2*conv_rad+1); J_hist_slices(:,:,i); zeros(G_rad, 2*conv_rad+1) ];
              
              for y = 1:height
                  isect_rad_y = min(y-1, height-y);
                  conv_rad_y = min(isect_rad_y, G_rad);
                  J_region = J_hist_slices(y-conv_rad_y:y+conv_rad_y,:,i);
                  
                  G_region = G_2D(G_rad+1-conv_rad_y:G_rad+1+conv_rad_y, G_rad+1-conv_rad:G_rad+1+conv_rad);
                  
                  % flip and flop the region
                  Jff_region = J_region(end:-1:1,end:-1:1);
                  
                  % compute min and max region with its flip-flopped self
                  if use_min
                      AND = min(J_region, Jff_region);
                      if normalize_by_max
                          SUM = max(J_region, Jff_region);
                      end
                  else
                      AND = J_region .* Jff_region;
                      if normalize_by_max
                      end
                  end
                  
                  % take dot product of the results with the gaussian kernel
                  accum_num(y,i) = sum(sum(G_region .* AND));
                  if normalize_by_max
                      accum_den(y,i) = sum(sum(G_region .* SUM));
                  end
              end
          end % are we handling rotation symmetries?
      end % loop over bins

       score(:,x) = zeros(height, 1);

       for i=1:iter_bounds
           if normalize_by_max == 1
              score(:,x) = score(:,x) + accum_num(:,i) ./ (accum_den(:,i) + epsilon);
           else
              score(:,x) = score(:,x) + accum_num(:,i);
           end
       end
       
      % score(:,x) = sum(accum_num,2);
      % score(:,x) = sum(accum_num,2) ./ (sum(accum_den,2) + epsilon);

  end % check for smooth_type  
end % loop over each column

% if smooth_type == 1 || smooth_type == 2
%   % optionally smooth scores vertically...
%   score = conv2(score, G_1D', 'same');
% end

% For vertical symmetries, reflip
if strcmp(orientation, 'vertical')
    score = score';
end

