function [ feature_locs, feature_desc ] = sd_symd(img, keypoints_fname, desc_fname, varargin)
% Extract symmetry descriptors.

sd_init;

% Params
params = sd_defaults();
for k = 1:2:length(varargin)
    
    switch(lower(varargin{k}))
        case 'min_img_size'
            params.min_img_size = varargin{k + 1};
        case 'max_img_size'
            params.max_img_size = varargin{k + 1};
        case 'n_iters'
            params.n_iters = varargin{k + 1};
        case 'level_scale_factor'
            params.level_scale_factor = varargin{k + 1};
        case 'k_width'
            params.k_width = varargin{k + 1};
        case 'geom_blur'
            params.geom_blur = varargin{k + 1};
        case 'uniform_blur'
            params.uniform_blur = varargin{k + 1};
        case 'n_ang_buckets'
            params.n_ang_buckets = varargin{k + 1};
        case 'n_rad_buckets'
            params.n_rad_buckets = varargin{k + 1};
        case 'r0'
            params.r0 = varargin{k + 1};
        case 'desc_support'
            params.desc_support = varargin{k + 1};
        case 'use_log_mapping'
            params.use_log_mapping = varargin{k + 1};
        case 'dist_func'
            params.dist_func = varargin{k + 1};
        case 'alpha'
            params.alpha = varargin{k + 1};
        case 'nms_radius'
            params.nms_radius = varargin{k + 1};
        case 'nms_thresh'
            params.nms_thresh = varargin{k + 1};
        case 'baseline_descriptor'
            params.baseline_descriptor = varargin{k + 1};
        case 'overlap_thresh'
            params.overlap_thresh = varargin{k + 1};
        case 'nms_feats_on'
            params.nms_feats_on = varargin{k + 1};
        case 'randomize_seed'
            params.randomize_seed = varargin{k + 1};
        case 'input_keypoints'
            params.input_keypoints = varargin{k + 1};
        case 'channels'            
            params.channels = [];
            if sum(varargin{k + 1} == 'r') ~= 0
                params.channels = [params.channels 1];
            end
            if sum(varargin{k + 1} == 'h') ~= 0
                params.channels = [params.channels 2];
            end
            if sum(varargin{k + 1} == 'v') ~= 0
                params.channels = [params.channels 3];
            end
        otherwise
            error(sprintf('Unrecognized option: %s', lower(varargin{k})));
    end
end

switch(params.dist_func)
    case 1
        symdist_rot  = @(img) sd_symmetry_distance(img, 'rotational', params.n_iters, params.k_width);
        symdist_hor  = @(img) sd_symmetry_distance(img, 'bilateral', params.n_iters, params.k_width, 'bilateral_angle', 0);
        symdist_vert = @(img) sd_symmetry_distance(img, 'bilateral', params.n_iters, params.k_width, 'bilateral_angle', 90);
    case 2
        symdist_rot  = @(img) sd_symmetry_distance2(img, 'rotational', params.n_iters, params.k_width, 'params.geom_blur', params.geom_blur);
        symdist_hor  = @(img) sd_symmetry_distance2(img, 'bilateral', params.n_iters, params.k_width, 'bilateral_angle', 0, 'params.geom_blur', params.geom_blur);
        symdist_vert = @(img) sd_symmetry_distance2(img, 'bilateral', params.n_iters, params.k_width, 'bilateral_angle', 90, 'params.geom_blur', params.geom_blur);
    case 3
        assert(params.geom_blur == 0);
        wmat = fspecial('gaussian', params.k_width, params.k_width / 6);
        symdist_rot  = @(img) sd_naive_symdist('r', img, wmat);
        symdist_hor  = @(img) sd_naive_symdist('h', img, wmat);
        symdist_vert = @(img) sd_naive_symdist('v', img, wmat);
end

% Print values
sd_log(1, sprintf('params.min_img_size = %d', params.min_img_size));
sd_log(1, sprintf('params.max_img_size = %d', params.max_img_size));
sd_log(1, sprintf('params.n_iters = %d', params.n_iters));
sd_log(1, sprintf('params.level_scale_factor = %f', params.level_scale_factor));
sd_log(1, sprintf('params.k_width = %d', params.k_width));
sd_log(1, sprintf('params.geom_blur = %d', params.geom_blur));
sd_log(1, sprintf('params.uniform_blur = %d', params.uniform_blur));
sd_log(1, sprintf('params.dist_func = %d', params.dist_func));
sd_log(1, sprintf('params.symscore_width = %d', params.symscore_width));
sd_log(1, sprintf('params.alpha = %f', params.alpha));
sd_log(1, sprintf('params.nms_radius = %d', params.nms_radius));
sd_log(1, sprintf('params.nms_thresh = %d', params.nms_thresh));
sd_log(1, sprintf('params.nms_feats_on = %d', params.nms_feats_on));
sd_log(1, sprintf('params.overlap_thresh = %d', params.overlap_thresh));
sd_log(1, sprintf('params.baseline_descriptor = %d', params.baseline_descriptor));
sd_log(1, sprintf('params.randomize_seed = %d', params.randomize_seed));
if ~isempty(params.input_keypoints)
    sd_log(1, sprintf('params.input_keypoints = %s', params.input_keypoints));
end

if params.randomize_seed
    RandStream.setDefaultStream(RandStream('mt19937ar','seed',sum(100*clock)));
else
    RandStream.setDefaultStream(RandStream('mt19937ar','seed',0));
end

params.channels_str = '';
params.channels_conv = 'rhv';
for i = 1:length(params.channels)
    params.channels_str = [params.channels_str params.channels_conv(params.channels(i))];
end
sd_log(1, sprintf('params.channels = %s', params.channels_str));

sd_log(1, sprintf('params.n_ang_buckets = %d', params.n_ang_buckets));
sd_log(1, sprintf('params.n_rad_buckets = %d', params.n_rad_buckets));
sd_log(1, sprintf('params.r0 = %d', params.r0));
sd_log(1, sprintf('params.desc_support = %d', params.desc_support));
sd_log(1, sprintf('params.use_log_mapping = %d', params.use_log_mapping));

% Load image
img_fname = '';
if isa(img, 'char')
    img_fname = img;
    img = sd_imread(img_fname);
end
img_orig = img;

% if ~isa(img, 'double')
%     img = im2double(img);
% end
% 
% if size(img,3) == 2
%     img = img(:,:,1);
% end
% 
% if size(img,3) >= 3
%     img = rgb2gray(img);
% end

% Build image pyramid
img_pyr = sd_build_image_pyramid2(img, params.level_scale_factor, params.min_img_size, params.max_img_size, params.uniform_blur);
n_levels = length(img_pyr.levels);

% Compute sym dist independently for each level
sd_log(1, 'Computing sym distance for each level');
symscore_pyr = struct('levels', {{}}, 'level_scale_factor', img_pyr.level_scale_factor);
for i = 1:n_levels
    sd_log(2, sprintf('Processing level %d', i));
    
    imgvar = sd_pointwise_image_variance(img_pyr.levels{i}, params.k_width);

    sd_log(3, 'Computing symmetry distance');    
    if params.baseline_descriptor
        sd_log(3, 'Baseline descriptor');
        [imgdx, imgdy] = sd_gradient(img_pyr.levels{i}, 5);
        
        rot_symdist  = 1 - sqrt(imgdx.^2 + imgdy.^2);
        hor_symdist  = 1 - sqrt(imgdy.^2);
        vert_symdist = 1 - sqrt(imgdx.^2);
    else
        rot_symdist = symdist_rot(img_pyr.levels{i});
        hor_symdist = symdist_hor(img_pyr.levels{i});
        vert_symdist = symdist_vert(img_pyr.levels{i});
    end
    
    
    if 0 % debug
        [p, fname, ext] = fileparts(desc_fname);
        debug_symdist = cat(2, mat2gray(rot_symdist), mat2gray(hor_symdist), mat2gray(vert_symdist));
        pyrlevel_fname = fullfile(p, sprintf('%s_symdist%04d%s', fname, i, '.tif'));
        imwrite(debug_symdist, pyrlevel_fname);
    end
        
    sd_log(3, 'Computing symmetry score');    
    rot_sym_score  = sd_symmetry_score(rot_symdist, imgvar, params.symscore_width, params.alpha, 'rotational');
    hor_sym_score  = sd_symmetry_score(hor_symdist, imgvar, params.symscore_width, params.alpha, 'bilateral', 0);
    vert_sym_score = sd_symmetry_score(vert_symdist, imgvar, params.symscore_width, params.alpha, 'bilateral', 90);
    
    symscore_pyr.levels{end + 1} = cat(3, rot_sym_score, hor_sym_score, vert_sym_score);
    
    if 0
        subplot(1,3,1);
        imshow(rot_sym_score, []);
        subplot(1,3,2);
        imshow(hor_sym_score, []);
        subplot(1,3,3);
        imshow(vert_sym_score, []);
        pause
    end
end

if ~isempty(params.input_keypoints)
    sd_log(1, sprintf('Reading keypoints from file %s', params.input_keypoints));
    feature_locs = sd_read_keypoints(params.input_keypoints);
    assert(size(feature_locs,2) == 5);
else
    % Locate the features on the pyramid
    sd_log(1, 'Locating features on pyramid');
    %[params.nms_thresh_ params.nms_radius ] = sd_nms_params(params.symscore_width);
    [ feature_locs ] = sd_locate_features(symscore_pyr, params.nms_thresh, params.nms_radius, params.channels, params.k_width);
    
    % Suppress repeated detections across scales
    if params.nms_feats_on
        sd_log(1, 'Suppressing repeated detections across scales')
        [ feature_locs ] = sd_nms_features2(feature_locs, params.overlap_thresh);
        sd_log(1, sprintf('Found a total of %d features\n', size(feature_locs,1)));
    end
end

% Debug
if 1 %strcmp('', img_fname) == 0
    sd_log(1, 'Saving symmetry score pyramid to file');    
    [p, fname, ext] = fileparts(desc_fname);
    b = (params.k_width + 1)/2;
    
    for i = 1:n_levels
        pyrlevel = symscore_pyr.levels{i};

%         for c = 1:size(pyrlevel,3)
%             pl = pyrlevel(:,:,c);
%             pyrlevel(:,:,c) = zeros(size(pyrlevel(:,:,c)));
%             
%             pl(b:end-b, b:end - b) = pl(b:end-b, b:end - b) - min(min(pl(b:end-b, b:end - b)));
%             pl(b:end-b, b:end - b) = pl(b:end-b, b:end - b) / max(max(pl(b:end-b, b:end - b)));
%             
%             pyrlevel(b:end-b, b:end - b,c) = pl(b:end-b, b:end - b);            
%         end
        
        pyrlevel = mat2gray( max(0, cat(2, ...
            pyrlevel(b:(end-b),b:(end-b),1), ...
            pyrlevel(b:(end-b),b:(end-b),2), ...
            pyrlevel(b:(end-b),b:(end-b),3))  ));
        
        pyrlevel_fname = fullfile(p, sprintf('%s_pyramid%02d%s', fname, i, '.tif'));
        imwrite(pyrlevel, pyrlevel_fname);
        sd_log(1, sprintf('saved file %s', pyrlevel_fname));
    end
end

% Extract the descriptors
sd_log(1, 'Extract descriptors on pyramid');
[ feature_desc, feature_locs ] = sd_extract_descriptors2(symscore_pyr, feature_locs,...
    params.k_width, params.desc_support, ...
    params.r0, params.n_rad_buckets, params.n_ang_buckets, params.use_log_mapping);

% % From original features, keep x, y, scale, angle, response
% feature_locs = cat(2, feature_locs(:, 1:4), feature_locs(:, 8));

% debug
if 0
    figure
    imshow(img_orig * 0.5, []);
    hold on
    
    for i = 1:size(feature_locs,1)
        r = feature_locs(i,3) * 5;
        rectangle('position', [feature_locs(i,1:2) r r], 'curvature', [1 1], 'edgecolor', 'r'); 
    end
    
    %plot(feature_locs(:,1), feature_locs(:,2), 'r.');
    error 'debug'
end


% Write descriptors and locations to file
if nargin > 1     
    if ~isempty(keypoints_fname)
        sd_log(1, 'Save keypoints to file');
        sd_write_keypoints(feature_locs, keypoints_fname, size(img));
    end

    if ~isempty(desc_fname)
        sd_log(1, 'Save descriptors to file');

        % write descriptors
        dlmwrite(desc_fname, size(feature_desc), ' ');
        dlmwrite(desc_fname, feature_desc, '-append', 'delimiter', ' ');
    end
end

end
