// Author: Daniel Cabrini Hauagge
// Date: 2011-02-04

#include "mex.h"
	   
#include <math.h>
#include <iostream>

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#define IDX(x, y, height) ((x) * (height) + (y))
#define PRINT_VAR(var) std::cout << __LINE__ << ": " << #var << " = " << (var) << std::endl

static int debug = 0;

static inline double min(double x, double y) { return (x <= y ? x : y); }
static inline double max(double x, double y) { return (x <= y ? y : x); }

typedef struct {
  int width, height;
} Size;

void
logpolar_mapping(double *feature_vector, const double *img, Size img_size, int (&pnt)[2], 
		 int window_size, int r0, int n_r, int n_theta, int log_mapping)
{
  int hws = (window_size - 1) / 2;

  // This controls the shape of the logarithmic curve that warps the
  // radius.
  double alpha = 10; // TODO allow user to specify this parameter

  double s_r = (hws - r0) / n_r;
  double s_theta = 2 * M_PI / n_theta;

  // Initialize the feature vector
  int n_f = n_r * n_theta;
  for (int i = 0; i < n_f; i++) feature_vector[i] = 0;
  
  // Mapping
  for(int x = -hws; x <= hws; x++) {
    for(int y = -hws; y <= hws; y++) {
            
      double r = sqrt((double) (x * x) + (y * y));
      if ( r > hws ) continue;
      
      double f = img[IDX(x + pnt[0], y + pnt[1], img_size.height)];
      
      // If we are within the central bucket, then skip the log polar remapping
      // just update the value in the feature vector.
      if (r < r0 ) {
	feature_vector[0] = max(f, feature_vector[0]);      
	continue;
      }

      double theta = atan2((double) y, (double) x) + M_PI;

      double r_prime = (r - r0) / (hws - r0);
      if(log_mapping) r_prime = log(alpha * r_prime + 1) / log(alpha + 1);

      int r_idx = min(floor(n_r * r_prime), n_r - 1);
      int theta_idx = min(floor(theta / s_theta), n_theta - 1);
      int fv_idx = r_idx * n_theta + theta_idx + 1;
      
      feature_vector[fv_idx] = max(f, feature_vector[fv_idx]);
    }
  }
}

// sd_logpolar_mapping(img, pnt, window_size, r0, n_r, n_theta, log_mapping = 1)
void 
mexFunction(int nlhs, mxArray *plhs[],
	    int nrhs, const mxArray *prhs[])
{
  // Make sure the number of parameters is correct
  if (nrhs != 6 && nrhs != 7) mexErrMsgTxt("6 or 7 input arguments are needed.");
  if (nlhs > 1) mexErrMsgTxt("Too many output arguments.");

  // Check dimensions of input arguments
  if (! ((mxGetN(prhs[1]) == 1 &&  mxGetM(prhs[1]) == 2) ||
	   ((mxGetN(prhs[1]) == 2 &&  mxGetM(prhs[1]) == 1)))) mexErrMsgTxt("pnt should be 1x2 or 2x1");
  if ((mxGetN(prhs[2]) != 1) || (mxGetM(prhs[2]) != 1)) mexErrMsgTxt("window_size must be a scalar");
  if ((mxGetN(prhs[3]) != 1) || (mxGetM(prhs[3]) != 1)) mexErrMsgTxt("r0 must be a scalar");
  if ((mxGetN(prhs[4]) != 1) || (mxGetM(prhs[4]) != 1)) mexErrMsgTxt("n_r must be a scalar");
  if ((mxGetN(prhs[5]) != 1) || (mxGetM(prhs[5]) != 1)) mexErrMsgTxt("n_theta must be a scalar");
  if ((nrhs == 6) && ((mxGetN(prhs[6]) != 1) || (mxGetM(prhs[6]) != 1))) mexErrMsgTxt("log_mapping must be a scalar");  
  
  // Get input arguments
  double *img = (double *) mxGetPr(prhs[0]);
  int img_size[2] = {mxGetM(prhs[0]), mxGetN(prhs[0])};
  double *pnt_aux = mxGetPr(prhs[1]);
  int pnt[2] = {pnt_aux[0], pnt_aux[1]};
  int window_size = (int) mxGetPr(prhs[2])[0];
  int r0 = (int) mxGetPr(prhs[3])[0];
  int n_r = (int) mxGetPr(prhs[4])[0];
  int n_theta = (int) mxGetPr(prhs[5])[0];
  int log_mapping = 1;
  if(nrhs == 7) log_mapping = (int) mxGetPr(prhs[6])[0];
 
  int hws = (window_size - 1) / 2;

#if 0
  PRINT_VAR(img_size[0]);
  PRINT_VAR(img_size[1]);
  PRINT_VAR(pnt[0]);
  PRINT_VAR(pnt[1]);
  PRINT_VAR(window_size);
  PRINT_VAR(r0);
  PRINT_VAR(n_r);
  PRINT_VAR(n_theta);
  PRINT_VAR(log_mapping);
#endif

  Size img_size2;
  img_size2.height = img_size[0];
  img_size2.width = img_size[1];

  // Validate input arguments
  if( (window_size % 2) == 0) mexErrMsgTxt("window_size must be an odd number");

  if( (pnt[0] - hws < 0) || (pnt[1] - hws < 0) || 
      (pnt[0] + hws >= img_size2.width) || (pnt[1] + hws >= img_size2.height) ) {
    PRINT_VAR(pnt[0]);
    PRINT_VAR(pnt[1]);
    PRINT_VAR(img_size2.width);
    PRINT_VAR(img_size2.height);
    PRINT_VAR(hws);
    PRINT_VAR(window_size);
    mexErrMsgTxt("ROI is not contained within the image");
  }
  if( n_theta <= 0 ) mexErrMsgTxt("Invalid value for n_theta");
  if( n_r <= 0) mexErrMsgTxt("Invalid value for n_r");

  // Allocate output matrix
  plhs[0] = mxCreateDoubleMatrix(n_r * n_theta + 1, 1, mxREAL);
  double *feature_vector = mxGetPr(plhs[0]);

  // Call computational routine
  logpolar_mapping(feature_vector, img, img_size2, pnt, window_size, r0, n_r, n_theta, log_mapping);  
}
