function [ img_pyr ] = sd_build_image_pyramid2(img, scale_factor, min_img_size, max_img_size, blur_k_size, magnif)
% SD_BUILD_IMAGE_PYRAMID2 - Build an image pyramid.
% scale_factor - how much smaller the next level will be, in range (0, 1)
% min_img_size - the smallest level of the pyramid will be no smaller that
%     this
% max_img_size - the base of the pyramid will be no larger than this
% blur_k_size - amount of blur applied to each pyramid level

if ~exist('magnif', 'var')
    magnif = 1;
end

% Build pyramid
sd_log(1, 'Building pyramid');

ori_size = size(img);

% Pad image with zeros to avoid problems when downsampling image
%img_ = zeros(sd_closest_odd_int_(max(size(img))));
img_ = zeros(max(size(img)));
img_(1:size(img,1), 1:size(img,2)) = img;
img = img_;

% Scale down image if larger than max allowed size
if max(size(img)) > max_img_size
    sd_log(2, 'Image larger than max allowed size, downsampling');    
    img = sd_special_scale_image_(img, max_img_size);    
end

% Magnify image if magnif ~= 1
if magnif ~= 1
    error 'not implemented'
    sd_log(2, sprintf('Magnify image by %f', magnif));
    img = sd_special_scale_image_(img, size(img,1) * magnif);        
end

img_pyr = struct('levels', {{}}, 'level_scale_factor', []);

curr_level = img;
while size(curr_level,1) > min_img_size
    img_pyr.levels{end + 1} = curr_level;    
    curr_level = sd_special_scale_image_(img,  size(curr_level,1) * scale_factor);
end

n_levels = length(img_pyr.levels);
sd_log(2, sprintf('n levels: %d', n_levels));

% Crop levels and save scale factor
for i = 1:n_levels
    img_pyr.level_scale_factor(end + 1) = size(img_pyr.levels{i},1) / max(ori_size);
    
    level = img_pyr.levels{i};
    %assert(all(mod(size(level),2) == 1));
    level_size = floor(ori_size * img_pyr.level_scale_factor(end));
    img_pyr.levels{i} = level(1:level_size(1), 1:level_size(2));
        
    sd_log(2, sprintf('level %d: scale factor = %f  size = %dx%d', i, img_pyr.level_scale_factor(end), ...
        size(img_pyr.levels{i},1), size(img_pyr.levels{i},2)));
end

% Uniform blur in each channel
if blur_k_size > 0
    assert(mod(blur_k_size,2) == 1);
    
    blur_kernel = fspecial('gaussian', [blur_k_size 1], blur_k_size / 6);
    
    for i = 1:n_levels
        img_pyr.levels{i} = conv2(blur_kernel, blur_kernel, img_pyr.levels{i});
    end
end

end


function [y] = sd_closest_odd_int_(x)
y = 2 * round(x / 2) + 1;
end

function [y] = sd_closest_even_int_(x)
y = 2 * round(x / 2);
end

function [ out_img ] = sd_special_scale_image_(img, target_size)
assert(size(img,1) == size(img,2));

d = sd_closest_even_int_(target_size - size(img,1));

out_img = imresize(img, size(img) + d);

assert(abs(size(out_img,1) - target_size) < 2);
end