package jif.lang;

/**
 * The Jif signature for the class Principal, which is the runtime
 representation
 * of principals of the decentralized label model.
 *
 * Objects that implement this interface can be coerced to values 
 * of the primitive type <code>principal</code>. For example,
 * <pre>
 jif.lang.Principal o = ...;
 principal p = o;
 * </pre>
 * <p>If the object <code>o</code> is <code>null</code>, then
 * the principal value is the special <code>NullPrincipal</code>
 * value, a principal for whom any other principal may act.
 *
 * <p>Any class that implements the <code>Principal</code> interface
 * implicitly has the authority of the principal "this", that is,
 * the principal represented by an object of the class.
 * 
 * <p>Notification of any change in delegation of authority should be
 * given to the Jif runtime system via calls to 
 * <code>PrincipalUtil.notifyNewDelegation</code>
 * and <code>PrincipalUtil.notifyRevokeDelegation</code>. These allow
 * the Jif runtime system to correctly cache results for acts-for queries. 
 */
public interface Principal {
    int __JIF_SIG_OF_JAVA_CLASS$20030619 = 0;
    
    /**
     * The name of this principal. Names should be unique within an
     * implementation class, that is, given two Principal objects, o and p, o
     * and p are considered to represent the same principal if
     * <code>o.getClass() == p.getClass()</code> and
     * <code>o.name().equals(p.name())</code>.
     */
    String name();
    
    /**
     * Does this principal delegate its authority to the principal p?
     */
    boolean delegatesTo(final Principal p);
    
    /**
     * Is this principal equal to the Principal p?
     */
    boolean equals(final Principal p);
    
    /**
     * Checks whether the <code>authPrf<code> provide sufficient proof of this
     * principal's authority to execute the closure <code>closure</code>, and
     * if so, returns true. In addition, if <code>executeNow</code> is true,
     * the closure will be executed immediately.
     *   For example, <code>authPrf<code> may be a password (or a password
     * digest, or a certificate, etc.); this method will return true
     * if that password matches the principal's password, and the closure
     * is one that the principal expects to be asked to authorize (typically
     * this is decided by inspecting the class of the closure).
     */
    boolean isAuthorized(final Object authPrf, final Closure closure,
                         final Label lb, final boolean executeNow);
    
    /**
     * Search for a proof that the principal p can act for this principal.
     * Return null if no such proof can be found.
     */
    ActsForProof findProofUpto(final Principal p, final Object searchState);
    
    /**
     * Search for a proof that this principal can act for the principal q.
     * Return null if no such proof can be found.
     */
    ActsForProof findProofDownto(final Principal q, final Object searchState);
    
    String jlc$CompilerVersion$jif = "3.5.0";
    long jlc$SourceLastModified$jif = 1466731418000L;
    String jlc$ClassType$jif =
      ("H4sIAAAAAAAAAN09C3gU1bmzs5t3hPAIJAi4hvAISgJReUXEGAwkRsiXAEV8" +
       "bCe7k2RhszOZnZBFqx94a8FH5eoFH7fip4i9iFSstrWVikpVQFusvfVZfBRt" +
       "SwXq1baibZV7/v+cmTOv3Wxy08d3+b49Z3L2/P/5z/8+j1l2nRCyEpowblW0" +
       "vVJfq8qJysZoe7OkJeRIsxJbu5Q0hcKf3//TyF2Xqe+KQvZKITeaWBZPSO1y" +
       "k5An9eidihbV1+pCUdMqaY1U1aNHY1VN0YRe0yQUhJV4QtekaFxPdAvXCb4m" +
       "oShKWqS4HpV0OVKvKV26cGaTSgbqiCl6lZzUq1RJk7qqkJSq5rqYlEgQTNnY" +
       "aiDJVTVlTTQia7pwRhMhnPWOSW1yrKqZfdcEf9UkNSFooGfzo5NDzHR2W86q" +
       "2nznVUWP+YWhK4Wh0XirLunRcJ0S1wk9K4XCLrmrTdYStZGIHFkpDIvLcqRV" +
       "1qJSLHo16ajEVwrDE9GOuKT3aHKiRU4osTXQcXiiRyUkwphGY5NQSFnSE9YV" +
       "zZhOdntUjkWMv7LaY1JHQhdGcbbQ6dVDO+FFPmGnrLVLYdkACayOxiPACweE" +
       "OcfyS0gHAprTJRN5mUMF4hJpEIZTycWkeEdVq65F4x2ka5bSowODx6REWgOC" +
       "kMKrpQ45pAslzn7N9CvSKw8ZASC6UOzshpiIlMY4pGSRz4nF5996TXxRXESa" +
       "I3I4BvTnEqDxDqAWuV3W5HhYpoCFU5vukEY9tVEUBNK52NGZ9nniax9fePb4" +
       "Zw7QPqd79FnStkoO66Hw9rYhr4ytq5jjpyqoJKIgfNvMUfmb2Tc1SZUY1igT" +
       "I3xZaXz5TMsLl63bKR8ThfwGITusxHq6iB4NCytdajQmawvluKyBiTQIeXI8" +
       "UoffNwg55LkpGpdp65L29oSsNwiBGDZlK/g3YVE7QQEsyiHP0Xi7Yjyrkt6J" +
       "z0lVEIQc8hHGkM/pghA4n9UVujCeKPO0hBauAstCpWgmOhGOqlKskjSpujC3" +
       "qlPpkqtWyZGqXkVbjR0jxOTxIT10EsYv6vX5CGvGOg0zRnR6kRIjxhsKb+65" +
       "6OKPHwm9JJqKyign+gq+ClBXmqgFnw9RjgRNppwmfFpNLI6QVVjRemXjVzeW" +
       "+YmI1d4ATDuJJlCCfyApNvdXD/bYgI4qTFQkfPW8397+ZXe5KPhXCkM6pURD" +
       "nMgQzJ94GaEgyv/ShWEWf9QMPgtMCR2TLox2+SrTRzkV2UHByqpd94ypu+AY" +
       "MiOPuBCd+FQYbrzTnGwWAJidTF4uaRxv9c6uP4tl2c+LQs5KoQhdE/HNy6VY" +
       "j9wqE9eXH03UscYm4TTb93a3QW2khpmnLox10mUZtsbwcSCCLKtwyTP0hud8" +
       "VJQh2GcYEVIBSO1c8hlLlDTB6kb4doQK5cgkyn8Yyh91o4F4yQ5ZG37kvu0n" +
       "12+YTVjXIGStAdIJV4p4v8U94N2/sWvLuILN790MbgCMQwCkE0CjnJIByue1" +
       "qlvfOPT7c0RB5A57qCUAEu7VWJwEIBuK7mAYV9Clmgxcfvuu5v/YcmLD5aid" +
       "pMcErwHLoawjzoGoFIkcNxzofvPdd7b/UjQ12q+TGNnTFouGyUMCA5hOKIvG" +
       "pRjV9WGnyD8f+XwJH5ghNEBNJFnH3E7Q9DsqsbNgKNTYUB9qbVgYWlIfaqxd" +
       "Xhuqa6ptbS2vnj79nOkzZ8wh1I7mhoPKLEeoW31wx65HagofehA5lIfaTsxB" +
       "B0rGEcYAhPE3FfVpUFQnCcrTOUqEIwE7SqN5KDzqk7IqtX7Be4g2nyhkO0lS" +
       "ouG1oHFO46ozv61BKyXqYHQe5+rcwL8GqxntpIGNH7gyGPkkWHa5KAQIzoic" +
       "CGtR1ZAw0d/8RLRLjUVJQI+wqWbrSiMRuZmpaFI8ESMcpra/FL+8OKlqECfX" +
       "SCSraIvRuF6RBG0xyUBnEgrPunmDpky4aSZIHi0EivIk2sA80B6EMAbhrOMD" +
       "hcJbi+/88fCHb6+lIe8MO4Sr9/nT674ROve7PxOZfo52uthFUqKT6PEbsddX" +
       "bnl76niK1aLn7PsnF9yw5Y4fPnEu9cKFRPGK5l8oCNQbEw/o5HiLLJFAQEUS" +
       "Cn9y71tyy3mf/4FanNIbdyZ/qhEKeLyBvFFDLMCdiwlVJS7VYuhnfvO+3Sfe" +
       "ab4QHaxFiBDdXQkm0xIL+6FcRCYxkeM36am8SNF1pcukKhSeN+GtVTVfvPI9" +
       "K/8dMJbetz50T85HZ39+H87c1KYJDm0yAdJqFJTnU5IJtRV2yTvotOpA9tQ9" +
       "pbf+at0SSrFTx7wg5leP3Hu0pPQa1BoVh29gA0N1qeol8q+QlQQXebCyad/T" +
       "OS0vWkSOciRc6MWOVKpQ1nMxtHqN5T28M/3Qol3EW69hyaO8cfNNpypv3Sxa" +
       "MuwJriTXCkOzbIc3OzPdKAhR/7vd1+7Zce0Gytzh9nzx4nhP13de++KnlXe9" +
       "d9AjFfITt4YZlY96ADq9KVieBUUljQ7wWKULuVIbrMfCui7kmasIJLjakhCh" +
       "PY5LlY8jzduv33xvZMmDMww3dAVBqCvqtJi8Ro5ZUOW4lpaX4gqEpyCz7l9Q" +
       "PnZf962ikEVSkGhigdwu9cT05rqLlJ44WUWNNJtaZLLAiqN/8lxa5qhhhGGO" +
       "t0Dj/bEJbM0xKScxD1266+DCSeHbSaZnpjKuVZUdqMaewOTTUZeaaYwmlLk0" +
       "QAnLEbJW5ONODUrfDz11bTmGFZ7fAYYzHNlSQbuidRFbA1Rs4vl6p6b08hZH" +
       "6oQiLSafckHIOsTqp6ypE2qQt+aIqDlQSExPNGES9+1kYRIjiR8JgInyZfEu" +
       "JUK8JvgbSLv/NnTijO8fv7WIqm2MtFDqNOHsvhHw9tKLhHUvXXVyPKLxhWFh" +
       "zPMs3o2utkZwzLWaJq0FOpLrfzHu7v3SVj9kgIEEydNphsd0HYjqwWlTP6E7" +
       "vuuFgugBShl7nGdnbRFj6UEv1nrlM0s7owmWT9zx6ciW67/y2TL0cWJYZxLP" +
       "be+JxRabhg7lXDVJFl6IIKi0B/VOOZhQ5TBZeAQNJx8s0wnmsmA0HnSvkADH" +
       "18wQZTrvOikeV3RX0M8OR79/oqr9C8O+5+N0ijzXXsiNYt2aU9NdBJKROxmK" +
       "DbOg6CLTKYpQyw6q4WDMsNOvY58LsKx15gnQuICCQ7mRTKjMK34uVVRLCL1q" +
       "1M/PGvvjy240ZtNIMaAq27hhBbOyY3Tx2788sGbRHwwEi6mq3MjwQHUzbWrh" +
       "xG3iYafLHYnsTbNovEAub86IAdU4xp0cYbV7DHtTCwe7h5NfbSPfo6nTru1T" +
       "yKdCELKXsXrGgB2J3eR8bC1uN7AKGLKaDRVi9XKHgY1xGlit1sHsa8dpz790" +
       "oqT+ADpWMRyF1N+5n6VE5FTOvEdVZc0aVMQ1UXgqoyYJxQ6ixysAZZeiqZ1R" +
       "pshgotRRByWto6dLjhMlh0a6AReMkJVSB9G4xFIlOKUNBpAjQalNWSMH29YG" +
       "rzly0/Yj37j92goY4GFkxnWp1QG3dLZR3kPxQDrOUpLx+SH6DOUuZOZ31P9X" +
       "+Nx7H8viq+Mkj6T5XWvBrp5/e2raG4ZNs5QNn59JEw+eg+J+HVZ/pgyhaZ8g" +
       "uN2dR5LLiGAKOrLq+NYlnx151KBiLp0cS5mvo9V+R6Mu+JjbsBvKucxAJC9D" +
       "0YVhhrslKhlsM7T6Zxm5m0NI3S+4kzjk9huH3J6Qgr3OXdIht5c6ZMnJ7ftw" +
       "TUpYivEsaekt+1+feffR2zETyIpZEztnLu+AjG2P7W/609pDBpeDdFyfgLx5" +
       "x9tj+eBxBnVWJsdXWBzk5STpVW1R0KqyUPyGqi2Uu6H4NTIcx4TiCPbck0bZ" +
       "fofKBsXzXqnHWCboBi9n7JV6NCpRQ/OG5b773gmtbKaxldKlKnGZZdMVpCWO" +
       "7otkS9T3Ue2078c5h59tHV5V3dsR5vDW4PqrV17/9rabCtp5roFpSp8pYqus" +
       "rxt3+yubvrVumQHa7aHFfmy7ZhDSp0+Y+aHnSZ0KefgtKPeCDnH587B/nIb9" +
       "zzKyw+OI96/ceo67Dcre1GLKDNYAsMmPslrE6ousMiOlTzDTIkdOtWAtCY3R" +
       "sCWvWvZF4+mfn7hkH83w4XTBax+uNhyWE4lm8i1dGzfaCSplhNR7EURomWpX" +
       "IScVVk26pa358UsPf9ljSdM8HDCnBxLxipKS3yz4UeU5OAeWgePRUmqwFkXR" +
       "z7st9sjj5/3+Crpod2cgZmclMKdm4Z5Fk0TbqRGMMhTl7stCdTK9qM9neNGk" +
       "p8fJaVOUmCzF0ybXNDreQBFcoHL12Za0+Ottbhe+ze3CEZevlKvXNrfGmU1Q" +
       "brYrOnd0P0jt6HxBp6OD4vcU7YfMSijkCVp95GXq3EpTGOGP3LR9RsUw5B/M" +
       "0OmDwNDvpmHorH8MQ7/3L8PQhf1iqKhO9uLot9NwdHEGHIV+zZSrqR15PzgN" +
       "+JYhtz3RQdtuj8l8hgQ3ZywG6HhZRvGH8vqKgYlI6oeI8Nmx9hxOPpUkROxl" +
       "9aPw7d9t7TnCSHFgqAOsfto6pCkuH+xmtPZ7FSh395CcoY8FoC9O047UgoEF" +
       "ka/DXBD5OtNNj1JrpRxKBTVXVf+Z+ODPfQh9XRorXAdFuy5kU+ahinutvKBh" +
       "Px3YvZiyNnosplDy45nEX/SSPJQvA6KNmViN70Yk/RYe5G90WY2jqYWD3Waa" +
       "iNHHYjWOpsO0iS5sfHd4m4NzYcMtxypBKP6TShFK2Iz23Y1TRtQIhD3XppHW" +
       "PSgtKNZzn1mdUl2g/BojiY+T3ocxzXKHEhAyZ2+Hm+Mdbo4jLt8Ozt4ON8c7" +
       "vGKziyt6Gq7s7j9Xev/5XNnTL67QmORiy+o0bNmXIVugeA5Zkyoo+jQ3AZnx" +
       "Cx4PZGTTlCkvDYyXP+8HL70i4Wnkc74g5D1G6hpS457YIETCAPYKOCLhKCMS" +
       "wpDPsnqPdUguo8MkEoYyioRwd7RZa7fEw2iilt4nvZoEwz6i4tsZRcXXeNR5" +
       "Pd1UKeXWWUD5Dlrcu2p/8MGf+xwFIjuaRvMhffS9qguFVg44hIC9L2Ty3u4l" +
       "d+z6oQlQ4AWgWAH6XgDkuFgCxR+RLSmN79dklcqE6wisqEjlTIEOeikSlBhY" +
       "P83ICE/igH/hFnfSbYQn3UZIwU5xizvpNsKT7sB6kgVWMdB3YIX+f8YO9qMx" +
       "elMOuUh5azIH72OSXEaYR5jyJqt/YWVOppYVjimJHk0emGWJeWksyxAvG8Eh" +
       "XpxBkFF+zGsGhnjFoZmIV8SYIY4wZUkanOJ1NLVwsBJTlkYfi3gdTYdpExPv" +
       "uIzEKw5xXW5o7WlL6JabyyumLH/266WJm+kuUYW5SS0n9Uq80c5AnXBvvuGv" +
       "LtDLN+HebKBNSuCJVG6TkJWAnroQTH1nHnHRnbYCu2RKyIf8Id7B6ht0Yazr" +
       "tm4dlSy76Tu7Xzd9LbAogQlc/42tUtYFr+q6OBcKD5vzrbb26y/cj7tlQ3Gy" +
       "uF/cSqc9ybK5yIHKW+397Jf+PXkdCh/d/c0DZ364fATe5jbYar3Ceamk1jiv" +
       "bMJVNtKelfPWsz8Z9dVX/IJYL+THFClSL8GF/gYhT+8kk+tUYpGkaqzse3OZ" +
       "9xYJslLHjT5+RjHz4E7/y8uv2EpvTFru+JbYJ229f8+uJY13Xjo2sFpuJAVN" +
       "XZgEVE1jOvAwq7eDvCai9k9KCtQMzvE2A3qZCIrpFoMg+WkT9nTsV5rnNOJk" +
       "+/mOm9Cy93M2NBxtfBDvWRfB1WPcmo9T0WJTQ9zeVBBNmDvB9CISjDRXwF1s" +
       "1xYu2ztmByD/9fJPnnhg9NJpfPcavJvfcdCBe9SjmS87wur37F75vAhFzDyy" +
       "JqtECYgjJi6WOFc4a8CLuEEJd4WDMFQw1qa67x9YCbRubH/46G1Hz3nupbtt" +
       "RyRnpN50xuOvGQ1jZp2/8gf7kJ15cXmNrC3uicWaBDEW9bg5bjsysx2Eo9fP" +
       "A8bOpwvtw3a/MpR8LiBc+Rur/2TlDnOp9X27VJKyx9qg72yvwFjIkH/uNYgu" +
       "rMwoMMbaBhgTL+krJkKnRnM1IS7kka45o0iHAGIrD2sL3ZFuoTvSUbAVPKwt" +
       "dEe6hZbVhFhvEipONxlcaqRl4Am2sfpOD49wZT89AuxACkeo8UNZA39/wPaO" +
       "fKJJAGgQ7rqAcb3D6resEkaA7JRM9OUCejypLErCqVG5V3RxBooZNeHF+5Q5" +
       "FegBsuH0r3w63i/zCDI1uLkrToMiz1QJcZ45hwlAzESWZt/I6g2D4yhgpFpa" +
       "1UERE1CWaDGpssj5gpBfyOosOxltGRmLnJTDPbq8WOkdoNF0951I5vNBvHLJ" +
       "Ukb/aK95mBbWm5GF4c0O8RpuTkm3hSXdFkbB1nFzSrotLOnOJZOG47shs1xy" +
       "DTBkCOfaIJxXi3Rvss/zatz++Iiq9y0Zrbo+Qp7w+22kwbV8sjfxQ+g5ApUl" +
       "mscDrN7qkKu42eEgEGoe632/FxR1ECp3BnfitLlst/BzVeR12k3j17w2jV+z" +
       "bhobK1z44lPADyt/ka6XYTtBzOfbCSK+4iIWIIsLcfjf8kUUNA6FYly3mTMj" +
       "hiYLhibEQH34YgMDi5fN3YZjn82BNQuwhsAJBNYNYIvpoRGhsiJn0DjfR3DP" +
       "7QoRj6VeheKYOaDvBD8JA20awNYC220x2ONicL/52R/2UW8ORezv6qj7LQ9x" +
       "IxqmobV8D1PsNRYVlusH4J3rzKvxoXDTn18dtWaK8gJ9scV+W8l5UTzHSHWJ" +
       "gPKnszroNDLvgyCLrcIpHe70AHgVq89y22ofwfwFtF/k1HP2lQN9BUCuTSTI" +
       "8iiqxEPhl2asf/OXN7bfj/f2/TG6WTqOPGrmo/nmzFSv97As2Cwv0Hjx52zy" +
       "qSUTuorVc1z8+dAOMIl1XOEF0LeZBP71zKS/SkwPHCyp0lQvRl5h5cvgWeBh" +
       "qwW6bhOlFn4ofPBp6Y1Pf3LfDnyFKelIT2ADHteGsIP5JKsfs4kWxn0/o9Tk" +
       "A2Tl73hC8YE7x/ggRTyFVE8YyQj4EasftxIC0McdNopQw1nvJ7ygXPH0o257" +
       "PD1hxFMoN8O7RVQRJ3NN7O4zIkKBfQvTZYm5TGUn23VWBN8k4mtf4ufd5iKj" +
       "AstK9/YLNJ/D/fz8AWsZQP+NalpKyVLtm29onz/WhtRfCUWYJowibDiIf0kb" +
       "8EnhRz4uTscgA3ummQBgRaHq6bAWcGOezK2520wR3kqdIvgLu/tOEaAf9a79" +
       "vCzjeSrhH9Hd16mEh4p6aaR/1EA10iW60n6KbjCyEBh3bEbaGbNqp6eSlA1c" +
       "SYy8BaXcnSJ3gS7PQLEvRW4B3z2POHAJJb4Ixc+4HokvUz2aOhA9csdWQFU1" +
       "aHpUPWh6dN4A9MglzNkDFGbSZDBVzcP/J9WsyUg1DzsdJ97PE98HKM8TcYho" +
       "iwShYDKpF5J0AsQyGCfiFk21HMZClownXTDkHFZPtw5pSsC/KOPtSevdMMK+" +
       "SLOmKO3LVL2vd4T8DThaH4fh/ovMw2t/XbpZUqKtE4CyEe3sEnXQ8EGxnOKE" +
       "8lKPi1jI5jLG3hovNkMJm0D+FZlkWv7LcGh+fZE0sCwmVVMLB+PXF40+lgTN" +
       "0XSYNtFNIH+7t+45NoH8XzH5uZxrLFc62KHF3Bm4UMfqed5Kt4ooXSQjpUvI" +
       "khbuhN+8kgeqfvRSz3WGGAssKB0CxSnMZKQv8ZqCKVAlI4GqKJkEl57qFqjq" +
       "FigFS3LpqW6Bqm6BqoZAr81MoHDPX8xBFuETjro0Tdq0HopaXTjNxn/o6fl6" +
       "GTTsp6bo3q+yNrLbY/4VqI0W5aK6YtUbKGPYuQvLFrtAUTTIAj6jJWlm9E2c" +
       "ERTXO9QZ+0yi7rqgxMtt0yhvAhR5ABSIVoAMbrRj2yDsrvq3cKXv620ghwdl" +
       "XLVKhJoJPN5mTMJyZ480QrmV6+ZtbnXlTZpwptd7X5afuAmFxSeql/3qgWuf" +
       "ojcTxnr1N7oqk0bUrjx4VzceTOd2SolOeGVWF0ZYfv+J/dhSTZMwZBUBrbO/" +
       "TYYGUc8n8iA2Tzx1cucDKQVFYwdf544xIi4IfTirCx2ew4/nIJZ1LkKNZ72L" +
       "vKAQwLLO9e/i1oHeY6fde5ioISzheQ+gHM3qkU6CvusgCKEmsN7FXlAWgjAV" +
       "8n9PQBUxwyNS9Zi5m82ucxbrQrGpgrVhPVGvaOhAhFS3Hk2acDNgFHPGVUYe" +
       "ZbMrGPKpjBzyXpz2s1wv97pV1d7EWYrH");
    String jlc$ClassType$jif$1 =
      ("gMWMgEpWT3Gy9AUHS83DQ+h9theUS8YHHTLe79jLoHdHB+ItSfgrtHjLyV7u" +
       "sj6Nu/xvp7uEx018vQP+AapXqaNLLYxUL5kwv+VMxN5C1+SJjk3LxhObv4Lq" +
       "7TTKYfou/zbuA94zfMB3jqT2AaZJ06j9GEJ+QOWPppHGWhi9P8QDPD9sRfif" +
       "9LAF1GwUO37lWFWAyyBpSiG8NbmYqNVRhzIO7qoCz+HL2JDTWF3u1GQqu49J" +
       "BLsiowSv2yutW6D0xvtO7Ngbr6m1DNcBx/g64Hi6eVKyrVOA8o+oP39SBw0f" +
       "FKcoTig/JY6z2+HxzAsPwOAqL0ZDCWlowJeJ1wuIABDIMj0KabBGDY+mFg6W" +
       "b3pGo4/FWTqaDtMmmoYGhnprnyMNDQgmP09xneVqhxfCzmZcmMVq+6vkBo8D" +
       "I4jatQ/GuiIjBQyM5EkWCDLNygInMZsRX+c1CVOkJRmJtBRlM5bLr9Qt0lK3" +
       "SCnYmVx+pW6RlrpFWmqIdFJmIh3dzVYWgZHdZmD5InVgCUyFDh/qwlCHBKBv" +
       "+rXFMa+1xTHb2qIbLQU10qJgVF+sugNlMU5zFIr0r661RaAEmcDn9FmaOc3E" +
       "OcHTWQ6VNtcWxGkX5ns5bxpRTYAiLwDbHbN/4NoicAFX+z7XFnYvyrhqlYgZ" +
       "qwNzPdYWpBHKeq6dc90Ka2vCKM7ALjGi+M7tqaO4I8vDnHwcY7WPsfpLh6UG" +
       "ljiyPIQywvApLyhnKhBoEXD+pv9Hk21Okc6fAfgnMqryWO13UrXCQRVCTWC9" +
       "c7ygLFRh7hm4nFsIEmRuNyVpJuWVpDh8HSprCfNxE4wUxaasgFnKyM+14cRk" +
       "Luw2t/ztTZxp8MFNOiDASFzGOpm2ysE0hCpmvYNeUC6mdTmYttqZsHdPHqAL" +
       "+tSdsLt80EdpfFCv0wfB42yesIPRQUXfkhxAws6cgSPDCVyH9u6Jjk3LxhOb" +
       "E4BqfRrlMB2C1dS/bpj6I7V9mTq1Myg3UtGjNaYxUEbqcjQxTNgDa1PZAmo2" +
       "ih2/4gk7GASmMcMJIfNYXZHpOwyOn7Lu1/sPHrDpFgf8uqykC9lxCX4N1HOV" +
       "YJ3yHWm+u4tyAh434TFJTl8DQ2IU2Jy0x8vRjGdTWF0Mnf4dteW2NKPfm+a7" +
       "+6C4WxdyO2T6fosjTpcY/hPGO9eQmWXcJI+pszOOqe6XrQLbeUQVczxeyMKv" +
       "i0l6xL9AetO9M0qZje+MBh7OyNciQOAR7sc2uXPKTe6ckoI9zv3xJreLNpvQ" +
       "NKDlfu6PHkAE8I5HYIcPHcJOJBr7EtbmmXkF7B6WuP7DDPqfPIQfuXdo7uh7" +
       "l72Ov6Zn/kcMeez3Kq0/4255zlY1uT2KnMqjF6pw8z2wh2iFkdfoQgAqIDjw" +
       "JP16ry74ydfw+LSqJmmCWmIVgbfG7bP/KDn84mwP/W9EQuH/qZ6xYO+BSfvZ" +
       "zxinfEWIQ+y+t3HxNR/PpD9jnhWOSVdfDYPmNgk5dL2DNMBvmp6ZEpuBK3tR" +
       "xV+HPJo30fbT3cMtjsI2O8tP8bpe9LD9Fyeh8Grh2puf2zB8PSFypZAXTSzV" +
       "ehI6/GcjeWHjXSFQe7p6mEvvvwZeJIgnO3/I2YLWehPKt+ruJU05p1aYr594" +
       "qjkuUYr+FwQJ2oNQZgAA");
    String jlc$CompilerVersion$jl = "2.7.1";
    long jlc$SourceLastModified$jl = 1466731418000L;
    String jlc$ClassType$jl =
      ("H4sIAAAAAAAAALW7a8w0WXoQ1jM7O3u/krXN2l4WeyBe1zLdda/K2BHV1VVd" +
       "t657V3UVl6HuVV33a1d1MDJXW1gYBGswEnaiZKOAsyGAsEgUGfGHm0BIIJTL" +
       "jwQURUoi4h/8SPiThFS/3zffN/PteIgRvFKdc/qc55zznOc811fP+c6vbj7e" +
       "d5uvN3WxJEU9vD0sTdS/rXpdH4V04fW9uXa8G/wcsP3Wn/ndX/wrH9t8wd18" +
       "IauMwRuygK6rIZoHd/PZMir9qOupMIxCd/OlKopCI+oyr8juK2BduZsv91lS" +
       "ecPYRb0e9XUxPQC/3I9N1D3t+V6ntPlsUFf90I3BUHf9sPmidPUmbzsOWbGV" +
       "sn54R9q8GWdREfbt5vdtXpM2H48LL1kBv0d67xTbpxW37KN/Bf90tqLZxV4Q" +
       "vTfljTyrwmHzm16d8eLEb4krwDr1E2U0pPWLrd6ovLVj8+VnKBVelWyNocuq" +
       "ZAX9eD2uuwybr/6ai65An2y8IPeS6N1h832vwqnPhlaoTz2R5TFl2HzlVbCn" +
       "leZu89VX7ux9t/Wr8o/97H9QcdXrTziHUVA88P/4Oulrr0zSozjqoiqInk38" +
       "7I9Kf9r7nl/56dc3mxX4K68AP4P5a7/3n//2b37tb/ydZzDf/yEwin+NguHd" +
       "4Nv+5//hD9DfID/2QOOTTd1nD1b4wMmfblV9PvLO3Ky8+D0vVnwMvv3e4N/Q" +
       "/5bzk78U/bPXN5/mN28GdTGWK1d9KajLJiui7hhVUecNUchvPhVVIf00zm8+" +
       "sbalrIqe9Spx3EcDv3mjeOp6s376vZIoXpd4kOiNtZ1Vcf1eu/GG9Kk9N5vN" +
       "5hPrt/nq+n3/ZvPGjz2vvzFsvray9m/ru2B7zeJnTKGuPBFkjVe8vXY1w+bf" +
       "26Z1GW2vUbi91V3+BBiuvPzU+OjZ82P/z91ee20lzQ+8KqbFytNcXYRR927w" +
       "rXHP/PO/+O7fe/0Foz7HfOXXdZ23H0u//WLpzWuvPS357zw4+RmlVzrlq/yt" +
       "aH32G8bvEn7PT//Qx9Yrbm5vPI49P4nAV55+PDTGq4zEPiSSX+XWW5np3cDd" +
       "fufPfZX+9//ZEzKfWgV68FaSr9LxtVfZ+QMc+ODrVw9ped3LdaFfKv/P13/o" +
       "zb/5+uYT7uaLT4rCqwbLK8bIiFZF9Omsp593SpvPfWD8g2L7jEffeS4ew+YH" +
       "XsXrfdu+856OeZDg9fcTd20/oB/tTzxd1GeeYD6/EukzD6oh6/cDK5P0z2vh" +
       "MfrF5lF+aX6i/5ee6P90N/yqpZKo+/L//B99+1/8/p8iVtLxm49PD9RXqnzx" +
       "JZw8PnTtH/nOz/3gZ771T3/mIYYP5tw8Fv3Bx42+ejMPzH/caH7hv/8H/zv8" +
       "+gP/9xTmF96nWVfqvfM+IX0s9tkncfzSSwYxu+hB5f/x59U/9XO/+lO/44k7" +
       "Vogf/rAN33qU9Cqc3iqVdfeH/077P/yT/+nb//j1Fxz1sWHzZjP6RRasjf7J" +
       "nAwrZlnlFc947fP/cv17bf3+38f3OOGj41GvN0k/F/uvv5D7ZuXzr7/7rsCz" +
       "7xr88V2FfVegLOpdWqIM4y1ot4N3GEh+COOuElGuR56ea8Dop7/1R//l2z/7" +
       "rdffZyZ++Ls09fvnPDMVT7f/6UfxzXnd5Td/1C5PM9j/9b/8if/mz//ETz1T" +
       "o1/+oNJjqrH8L/7b/+fvv/3z//Tvfog8f2w1aE9q4YlQ3/ds19deW+kHv42+" +
       "vXv8xp6Gfuip/C2P4keeUf3R/Maj+NFHATyh/c1h873XInjrPaJaqyVfeeCt" +
       "VWs8zf/y8H72e2bwPmT79diffwkm1atV/Jn/5U/8/T/+w/9kPYLwHis/oL++" +
       "2bx2/snXtD/x+PHjvz5Mv/rA1KjHLogkrx9OdZit3kD4Atnt8zt4VNCwqvr6" +
       "Q5Edvr7hkJ6n3vuTdyQNJ2fQL+89OsCpSvEClSHpRPFGIujR+cIcqVuu99mF" +
       "YizCpyM8h/S0vytu5e7ZdiYRj2A5NKpZDo+zBsnIjDVWN8a2sryxNHbsp9hy" +
       "470+GLdLC/pe5rsWvp2AoDSVCbbnZUeEMOAjpAtL0FbdovctAVzBAqp0emGa" +
       "+uYX9m6yhLglWmHp3TrClVyIdL6fBX9EtV0aiDG8Xf0DFrq0ESBqkyvyZuTZ" +
       "e1c1FHDJUaoioyzMUt8CgYY+h6fWc8Az6MmcA1ryztGj3aKdU8FCtCHKPXZW" +
       "Q2q60D5S1kYq2q2BcI21UyxxEi0IC46eaSR+m9EFVrj3cwvqQo63bS4YLdq1" +
       "TsEkmilXjpcdBMu0CtPUMg9vbnp6SivVVgwbXDqxdeMwYGLynMsGW9fH+F6j" +
       "8uHO6ZBNYC1U3SvrbKuAY6ciYqe7S9ZJY6UHws4AbzkpIhda6srODB1jcI88" +
       "x+bNgZ+koNUtsBUMiatBzLsy1q07MORRGUUT9jgx089I3ZsXLbhkmb6c81uT" +
       "Hh1U1gy2MhjSse5uqoTVcL7m9mnCpZVMAqel4THwfQ0LrYMCsKxaHqtOsiQn" +
       "zCnaW5Q5Uyn6FjFKEC/GLuGopjMFXUbL84lFUqFheNdcAjuPmNBqMZRFZij3" +
       "UN+ob8phxnTGThQ6P4ldoyFoQ/XFvcJlkp4M8Ua4lEFjZ+Xep03GFycHaY6G" +
       "fkcqFiVJzLzM+oUxPB7mkf1kVkx7mA8tZY8WbKNc6jQ1zZ7RacwZhxws5ryX" +
       "RRSSjy5ZVZk3k3Gk+oh7wzW1DXFyPf+Q1wqGcTeyc9IOWYYWvV6bpa0FkU9i" +
       "eTqd49KHzAA03LMDZonJw7PrXe6oeeqlAoN9USfMNL4deXPsZ67Fsj0i08sQ" +
       "elet9G5TMYrLPjM1PrrwxHl1weRbgqJS3hybsMFQCSJro5dv2f54cAtxm8CO" +
       "xeCaxvRLxqnr/ePOdenyARPuu8JieC0gUu6Yn7ZNklJIRZ6t8UpHoqrs0r03" +
       "Uv1AN3ZSKulRaQuxz69Gs47DbaE2eTvYi3tsL7nlX/XddVbMmTfBHdM7EzZd" +
       "UPo4HZVkd9crHct1/jqnvegDGIoposrdSlIQWnFZgxogaQG62F/KNmGrG6lf" +
       "E11umlsZs40s8ZhmWVc9Ki9X6x7ZgFPyYIhQ5mEUhIKm44Vq5Cm8HLa+k/BI" +
       "a7JaHqs2AZM44F4iVFTTy8oUWHPWD9Up0e4CiRw9JlmmrZHajB2T2klNhhOt" +
       "Q6ZUzxBgdKZ+zy3ozO2XtglJZGlGgHJOmaKOezoEWM3xu7tZHW82JWShySKj" +
       "p88TGWCjigwFFACS1qNivTI7s+XkBbOcK3lrGTK6cDv43AdOiZimT8IerNlQ" +
       "njZlte4L7sqzO/T+LTBoWhKH22Kqu3QMCUvQUugYOIzF71cdtN5VIRUHc4/l" +
       "PhLWmogGSb5UhLOkJFgjLLJoqHuE8XwadxUIAY5pXS/80RiokF+1UoNMxJlE" +
       "ZGjb25M08pKMk7uTn9An7UbnzI6+VgK/X8YIbIspVlQTD3b6HqVQvKXPFDPi" +
       "li3cGH4/inhJORZJF2UyB/E0jui0dYJSOqBeFjangOlunFR7t+u9PtYsnwfy" +
       "fK/jSQ1D3N+erxoItYB5lCuTrcfjcj7RFJoqIj8dG/1qkfaxWg5EDFxuAocm" +
       "qczsZYGtOHhlY/LQt70vJ6JsOuc92fpW4tgFH3vtHF7SsfWwZvANaKquOFkp" +
       "+d0lbinI42Vaq6LERLUNg7cZQzkvqeMopdDHVLOf9/tdd2QCh0pVBmZjqzmz" +
       "WkUHnamcW4a/MbjRyqh4uLc5zrfDBd9HSAJVS5oru2K1pUew5lgFwer7hIF7" +
       "cbm0DqdFKNAFN4k4tnM1NPxR5vBkOap5bnMeKlIaf6+zgD4OOuhSt124JXRJ" +
       "l67n9cZ54NbafmdkJ310Vw+Q5QnGOAVN521ZCSAUsWgZOr4YtNF5/KSPLKbv" +
       "j7t5Oe8rJunUsSoPTb4gnuHyXLZjsonPoZ4idnuMOVF+2imnQxoIfgZsCSKm" +
       "OR8DCj4iTgpidOS2MjEc0ZUpwtVZceMahcYhZc+taGIq4/dhePGrAseqTAQ9" +
       "qcLVvmLJ5YY6SKxWzEgGIqurRxEtiJqWNZ+5bWuUyyRJYCmNORt0RS44OCwL" +
       "XMZbhPUqmffDxd22KSNG0rKfWPYgnegwuxWWYOCWM6ZVPCkAbMSR5yMa581s" +
       "tj+EFy0vdzMvARgb6LQz59EJxrySMGQWbIIxF1MjkoHjsNrNle14UtwS4Unw" +
       "o+GO7kwC5VyYBDgCD0DREQ7lbqFCoWUFszqjvDDoFrSYUz0oDUUOGtn4MeAN" +
       "O8PoWHPGV2VbRCrMjbtG1TxwX1spRCbUTgbbqC6IFtAKTbtEPr144dZUjN1k" +
       "54YSZHqAnezV9DcaWJ3mgVsZyoSI0xbW6gJmbIxMF08hgliWm6WZYGP2TuNN" +
       "6DJz1V/K3ecDWLmroUil+hafQhDA3SDaHZfhnpLVfUaqVAXhMAIAsSouERmS" +
       "1JWNeXNAHLdwSIw6tKRoZdH+fC6uGn6zxbM0WrzfnWJ6zPC89HpeWMOZPKVT" +
       "1UGlwiGu93CcYHie3FN5ON1ucEhZ+5ae2kHICL1G3bNiI0XjXZCkZZqFvuMF" +
       "qamVG5MYlOWMZeTHNEvC28oT5m71Z6RCQBD7bMuyMHiuJFFZK2+JYvbQNk3T" +
       "41EZdofr1mcE/7SbRSOYDyCEYcCBNiLONzy3aq7iaLnnsD150U0xzbMbeZ5z" +
       "Kwsp32lUJTG+YKOKrBzOXXy4g6BvlkfWxWPKce6cXBiaP2ahzvoUqPfIVGAN" +
       "5u+I1OWwWsskw4uDnOOV3mwbbGkYqNUGwOPbQ6EZzKzKR8YgATB3VjuW6YSQ" +
       "8NXixswonEVCmBLfkKbtqs6yq0LEE27zbQmdBvbQygrXT+nqCwyilHLn6XKx" +
       "llUTzanNpnf11MP1Kl65e7tLtie0O1ZppC6+cyO+coB1JfOEZ5gllJu0D3ec" +
       "c21vdmGqGIOvKpdNxmRvpAVeHcPK19krdmYmsOqK067H61tpm+R9xoLMxvDt" +
       "slq3XWALjjtLqh8Bwqx7fprgxeVgCWyB8byY7fZbS1HcVEPNvRtRmizMRL+7" +
       "VWVfUjuuMqacOnAgcdNcs5uULI8sWL/sToSi4+S2bjuSJhQABeRddr1fLn0m" +
       "OwPknUk53AuirsylePCVBKMlU3QabblhLWIfTr2sTA5KSL04g2hSaSNP0xpe" +
       "6qZ4PYkwMlqWLtES2IKnrXqcpusCITt3kPmTR7UQnqZ7X1kc2JwhnCD6o69z" +
       "txuTYWULn643jTvffbtL5yOcOV4scU7Vm75cTww1+gTGU+EZmZtd4PgWfuAG" +
       "c/U98RSVY99Q4xqrzG4EzUUMaUYSL/2lOyZ7SIAjfY1dwFHGKfEoid5BtzLy" +
       "InKJA87HIhvAnus9Imzuq2GtWxOgDo190hcWlAG0wiwGZddgZfVf3WDnWScN" +
       "PF+Stl4cYdawbF56ynWjrHVBvenTpB50fb/YQ42F0jU9FenOwyg9OuWCZezJ" +
       "IzKzfKGIZ3nfUFfNK2TbOHEQ6Ry7e7Hsts4Vxq8klNQOcmTzubIbUDvCoeTq" +
       "cK7O2qFTzo00AwaHHZTz6CouDKrkBZBnnCOmrAVO+RUz0Ys97bsG3UZQ5x0R" +
       "V+5RTa5QUmBuQJQAAsBZ7uo4LqnnEskoQ6GOSlJ2pESlk1pAdopqvVZQ3GfS" +
       "SmcDLaeatAAdO42AcPX7/SSOwUwJNHnYOVaadn0hpkV/wk3MnUu03Mk8bABI" +
       "fr/mZDpysIBpSQcrl5BPBzTA3FMl4PFwM8+Nk01M70X5AIBAHMWxDt0Zc+Eb" +
       "l7kPlbQ6cNcjpOL2FtuRzWHfRIeQXeJEmtGZHK5AtJ/31era9OHF3t9z1DAb" +
       "ywb1ndguZ3U5thl0HjEOkU0+mBEzo1U/yHf+ahtjcd9f7pe24mzar/cZWeGC" +
       "hcl0o2tb5Yj6OA4n5eCL3lyEYiFWR9cNT1ZD2wpeD7lBmvA5gnY4EaApSg6c" +
       "hVpw2OvodLARte0sEN+SnefTQ4N2NdsYDdE47dh38wADy+hPnTWWq0XEPY6N" +
       "ywV304PR7AAH8FZpdXdiFCeDqEb0oeJQAt9vK94n73dxuQEKT2LbIKox/BB0" +
       "Ebgy8kF2qP5Uqn3RMVg99qu/W5j3g8nnoTLY1EA7sN7T9mRPLosAZVzhac4R" +
       "8A25mHtvBkY5XWwGKUoiA8PjPfZooylSWceZs16XdnPhjcCycghrPBglzr0a" +
       "k8XVCE8Ed6Cr/SzNWSMa8HJHztEZVajITiJVP5ftjc0v+h3vB7K/XtEtDNBd" +
       "U4dQkLhlUJzEsERclM13pt/qx9MpNYsVlA5NkWxPZBXJx7whYsGzlHNJ39nd" +
       "4l/kQxdpezaoOQmEXG8HKMPgpxFwj2CacaOpEKeL3MXH8gxqfud0ShXu1N11" +
       "wLh532DKAZmCM+7EJyO75rTVkOjx0sJMYx6VowpLOqn35xu63552Snm4yYep" +
       "Kbm644iThm9hiVC1NAazVkwVcpTqgZRvIcccOd4C7q7RtTqIqvdEBgQnHe6w" +
       "roXDeCUKIk002F1uR4xLXX06ruLZiolT8Fgd8YRczmQ6K2pzzlclW5CeesND" +
       "csDNijY4NmkppY9XDdLQ+cgBczmoLs4aKUyidFA7BFvTpLJD8HNSM2zubQ+X" +
       "/QFVBlm+gkLX6LO4N62LFQMuoBzUWEvJi153FypkxwXNz4CXZtU4TOI+VWfY" +
       "mdxskau6b4rDgqAVc9kahipoE4WJcmllMr5z8q3uROiR3NkOdG0d+wDbnpfJ" +
       "3GrabKXplV1moH3MhUiJyHcnOA3z0uxFFTONlBUh/g7BDmjqg8anB/jSNJyt" +
       "GiuPy+69AsvwXEDJdl9fMwcw4TqY76teZlJqcSGAK4FT6NfIVRoI5T4gikwE" +
       "+HQU79kyYMcW71dVeYmmLXpZjbiE4Vqp6NkC1DtfLi+8NSYRhbr5UtfAVYMp" +
       "t5rpGVHdo2kWdD4dnOWiAgQn4bsMsas8wJEEPYBHLG0MBMUdnjFpSnVEhZZS" +
       "kT4Zt+awes2tGp6TsSZOOVM7S1brtxDXXQ+PmRuUZcWAVJNbM7c7liFD7+rB" +
       "5RLJOqqG4FmMLrZ9PqCHogtGqNkSp2MMzDeWv66h6HRK49gg22RkrQRBUp4m" +
       "Jchc4BEpQuC+ujqFhlB2YlMcdkkO+epF0LMO6JJRrrFgmBdBy2C3rG+Za2Hy" +
       "jBJluVHsU0clYJ5mtlnCbVO2ig3f0Sr47PK3Mzpn96mRe9HE+zUsTSz7FO5h" +
       "h78AMeEoqyDOUXFSxnOKeneewyHFuDCeo6PXRWiHrWNbStM41jUrBS7fReXR" +
       "FiI775eZmo4Ek16aeZ+zLHA/9se0urK1W/IWYTi8g9zcc+5VigDKknXr94JZ" +
       "5E6cwUloOmGzumzTyN8Z8YbOFxutcbuQMMHzl2UsJW67Opa2UAL5eHZs1DB2" +
       "FXluiV2AUtVlP03jcZKGgyQDXOSLe9kYacE6YmYu2FTV1ZrJgRVXcXrlO/QI" +
       "6eLCxwQf4nssHXaOd++GEgkLY8Exc8Qv95CUfKpOtCLYItBwHhy2A1pM2Qfi" +
       "AZ3une9D2LINrvhQaPBxGdltYLQBufgiuvCcaAMwiQsxoF7ykE634TbYHVwY" +
       "Q2LKjNCr73gdTEdemYPXbPJPgiUPYlTEqt3fCZcMrcuWFxwJMDmWLow1FGW3" +
       "iZOaAJHnwHB2mmPutfc+ghWXnB2hLyTuaqKAfuXuCLM6jcQ4QUwzmWy32mvf" +
       "ESCIjAlgFBwQvlxrqGebUaAs+eImjMWAV2Wibi0eHSbay8icFOlBO90nBqLg" +
       "1qX5rmHrUwLYuUlWZm+hXSUrQg5sS6Uk6YWET5C8n4d4f5hsPd6DthqUNWnu" +
       "bMHUyHtk7fdyeZj2JM0b14N+ubFgD87OCN2Hcr8ctmQsqFWVJqQsNLEaeJMw" +
       "xUFI7yPSuJYGgd/GDsZZHJ2olL0flgfvMbSTXIUMsxU2QDyClkaGEcKHU8pq" +
       "N7D1MHwmHdrXR5/bItgR5O5wNTb+MOiLXjUyuptHZTpq4NVtoC2RxHgugGhw" +
       "KeWOhscssL39gbFJP0H9a7/0M8TUZKxShK80t2m4wd721OiElO0DQB+7LWQW" +
       "h0ppyAuEW4K4zKsE49Ay3dgrn3PbUXaSsTvd6tPJ9/ZpYkpA3gm5WOnVCeQN" +
       "PM89BK13GUcd+e4A9RLssmNtTsbZpyxVvA0hE2Gp61spm05M1ijyXAReqxcI" +
       "VUJ93KyGjby5JkFS40zReVQTZ9mlYHJ7");
    String jlc$ClassType$jl$1 =
      ("Uz0lFgWBOiYa3rVU4MMHrIt6JvcoXNBPpHKmo+vSSExHt9ZxdWD0eV/Xrhxx" +
       "LIBqKBsl+37kdh18w+Bse6cPPDhKeWfWp+wOakLgImh6K6g7Y9jusOWxBDog" +
       "S5o0bOOBZ0ZnQNr2Dsem769o0uzjRan71cwNR72UKsE8qH5L2Eoe1jK7I1GQ" +
       "Ryd5P3GScM79rSeSaLn37RI9skiwUwi9Gc28hFPjErK5vwcr9m509SApqxZx" +
       "lSOC9+nKVMvB3Oe0Enu8MF8yWJ/6fnv3hjsA8FlLLju382tAa/AbAEH5zbeI" +
       "/hrPGpCgVHMuyj44dDLe5uKIQBGTMMc6wmBoy846yNANvK/ibJCKftju71ak" +
       "g1AOX0H8Bl1XF8BPa18E7whSqNRhGba5h10cc9jfldinDRFAMt7m+V4K4mtM" +
       "5OezTGGsUyxwaZ6uaeZIaeYzGgcjTnI6ZGgsw457rQhoL2vjLe0pYS87ssNP" +
       "Ta0f0z4TqjXcjnX4oOJarV8xK0oj0VG8VcgOhEBBO+8mOzQE7WZIkbNA2qKQ" +
       "voXO9hrvRfcJa6+4pJEzP7dlvdLgAKq9rDtWMfYcIsDdXZzm+nIwK5AIB9sT" +
       "iUQg+8ACbKJAz+adMzhbSFSx6u9dLR+kKVloqQv4hovwHcUlVkuQCwTV8S2+" +
       "sQO7zUFOhQpNJvdnejQNkyeyrJ0ZNS0gr4+3ykXG/fi6jaGtecRss1XUY1fW" +
       "26q8s8ZE4lQP0bdpx11nFLovrXE2x7qzAvBOACGlnPgCvCgMu5Mid6/i4F00" +
       "tkZ+Uzl6aXaFNunF6p4HU2JH+3QNwDE/VQq2QupYXV1CB1RjWIoP1cKumpc3" +
       "oTVIHqJjz6uoIPZxj+T2XO22qyrLqVXGcTilCYfB522S6ObCWgzhLNoVYdZj" +
       "DDNP705BKNfNqoCqScCX5KJyuxpCcfUApjN7qmVCwXVL0wVYlnEFufO5uI+1" +
       "yXSdEjWP5wqCLXsvEzlSDjEyxiGaiVeqH0Ry60iysD1s4ztB2b2rqYDsFY24" +
       "WNJQQmRd3bd2OugxfbYdob5l/iHj413uE0yV6ndkNPsgoeCxLLV9ontci/mU" +
       "HRyV6C5CKLS7rTFlG/JTIo3bQWE8KiY9nJzo+zUyupQjuRDESiZFGszz9C3I" +
       "UOAus7FdhYn+lV6YfQr7jO8dfbltsMgoKDAwnIXYe5egMgQ4MXBVk5hJt5E9" +
       "fIdUhaMEwUf4OQAQrs6n8x3ylJQigz60NfYMelxzVu0UC2WEIk9JcZfzA3v0" +
       "Dfewt+6BuhVIu1/OCXi496FA4ipDEpx4IszASevsZqTOQWxYVNhRvbYc6Thj" +
       "cwjkTi5d1LfqYh5yjnXV4YY2Rk9vRUUdjGgKTycRMCSlFBVYsiRoChGTOJip" +
       "xyFxuq2n0w0athe/cc+tODDIeuYI7pnqohHMca4UkSEKUc3AK49epSOc1pOc" +
       "n4RydwBYt8jtCZeCYwrELrPr6VtpWSdpv3oELLIDisSUdxfDkQdW5jkAHwAE" +
       "lWuOuNw5dHTnTmIDDBI5Ft6XyPkUt1GIigNS1xdDuzbnic9Tf0zl1SFIojae" +
       "dOMO28cj1ffifkfPRnVyRW0i3JYWLargkWMrCjlToHjLtteWzRLWgKWmtbF9" +
       "W65f7VIdAZ5q7qBeiHnPjox57eqdGbY7rm68i3mV52pbY3GBra5z32B+DbfA" +
       "IneS72wRq+V3NuqJmDOcffGoOqM2SWw3UGfT5KkrdhXsoiQZKGB086KQmgvs" +
       "lmy1Scl8Gobdkq6mXUepvcFbNXXyO/0KLYe+m1o4EC8TBu2S9BYWF8kSU6EW" +
       "VBld1aawuAix07rSwgB38KtLYbrNrRlC/IJ68UiV7GxEbbGF28Qe4/oamHPV" +
       "gTJet2v8f+6uM7IyVGQqIWTUhRIdLZvduU48AcI2TA3TPd3c/Xnf+crWdJzj" +
       "OSAqIlu1wH0GgzE4iJrFOrcwRhFk5jOEj82so2couiFrqC64oJSvkQ0OSZNI" +
       "aTdJUshom4E1FzlYHEZoVvPGPY1Ww8Od4DY/Y/Q09gktMmUv4+ylWJ38K2UR" +
       "Falq+/N8W2SDdDO0JKAxmP1O645N0lmpah2nh/Ck601yOeQFBzlhOfVonZ3I" +
       "9U/dURWb6/6+XJo71IO3VZc2FhjijO103CMVgP31JR986SlN4kWS4DWLHwPv" +
       "fFh+wZtpcrJPCdPY5uoXk9SZMwj4oAEpOaTR2SAcjjOlCCevUs2F3B7gByNO" +
       "uoG7SouFj/0Vk8ftPDX4yMw7kJhWhUd5zaX1ZEly5rOE5rcJsE6zjNIgYAjX" +
       "HJbmVPEnRGpx0xSduB63d3+kSOQen/bwtV01u+cKO+xKHhkB0BRbXePPo51m" +
       "pEpoB/VA9nqcxaREseDxdlZgvLBZm1qWfWNkR0L2S6eVLT298wre1vQd4CRI" +
       "OJg2C4/gUWtHhBT2Up/VN2OClnGwscNungYFlxO/w0rwIAf71WZgCJdhkSUu" +
       "naVQRA3UKkwa+9suZsWgNrNkKzjYfQ135zvVqzeFaqWde6DtVXMJkQDeRp7a" +
       "H/jTdSd6/v54cICSTTDbh5tjjWzZ1W56h+FUlrWzxlpNVy0n+aAlpboj5yqy" +
       "2nSO6XvpBuXJagFsyvc1ceCRqzdo2+OY8de9tpxQJrfpNXRaEn5VHr1+AQKQ" +
       "tpRc0Eth2OvFCfWljtSGXSHCBl05pTUDjY3pxRoe2GXLh6xQ8ikc3i3HTlwu" +
       "dMsDyldWQp8X9pBu21tBc3txPwHdhT15ADibnA+1vC9pZ3DXConFAv2YyMXp" +
       "jN5Zvtt1GAbqHsWUwaEi2dJTAphRAs60NWyNlQJ7PuYXtOpRf25006Ey6Xbw" +
       "OV/eInKdBRR7Ae74ha2caenStnASt73JdzV07GbPiVGLhQG50hKaaPGEU9L1" +
       "mAEnfGg9BeGJ6VBxt8DBbXX1r6UUOx+La7aovLbcJqqRlrQS033NXA5AM5Gr" +
       "6IFKHmAxFSfVqe0WWwHhc3ehNd2iTZsmg9UXkFg9xdMSqdieZ0p9lBerEUbb" +
       "ag4pwRlNR8fyzCC38+x5fcJW4nICQCTQc2l8WJ/7YLpzQ5xYWckOlMHej0pT" +
       "QkvP7FK6KSdHjs40VoZm6QxD6/ZcqHItJQGDEfbSQOyYsIlZetrd4j0CLdHq" +
       "3+ja+RLn6UzdeixIDPVAWwtHjtgLWWKQfRvrXXHaQzQYGlu/uWI8bpGl2YKn" +
       "JYHPw/Eqe6swMn4bXLXZlJODB6ouVdZQlkJbJMnxI2WUp+Ak8LHsXFyuUW6L" +
       "UuHSCG57lesODZvExNguOU0WM31BZJBTLEeAF84ugREZFipFYMRkBbMgDhrE" +
       "nyFh2c3EpF+P4Sp7u1MVI+EQ2ZAsqnEJeDiA0Yg9h4sCmVhc91ezCZxtqh5X" +
       "5auV9iUQeKDp74rSRtil0yQ/OmtWcJ+j5SRO2a7AZomkokEhT7vWvUbAaZ/i" +
       "yiIHmaZQOt4H0HhRRTPCgrzmyfsEz0IXSXIzabJ4KOIRU7mcYgge5Rw+ULlA" +
       "UIR9hzODX5aXnMX3mD2q+OGgtvHKR/F4CWt8TPECh5QJc6ttHpyi6A6L7W6L" +
       "QCh6LxxwNCFxZagrizU34g7dUmh1Z8K41hKq5Q/Xm6CCWLfgyBLGN0AyWqEs" +
       "abvkrcSPj7ivbsVDRSDlTd3aJKON3LZ1sdlRgL0RT+59zqqYsXkNWIAB5mZZ" +
       "36rNgbxWqa0YVhgfHLWUkxwNJeggn/ZT2WAu04x7ER702QI4VajhXeybcHAa" +
       "rj0PKcSgEn4TkdKBZJcoJY5dCjVjDN33hGqwS+NSF9NYw6aUx7bH4BrZ3PVi" +
       "sFqEN6t3OU3TzY0EPqkUeMtDjmRHqSoOQDcK5x3agJ7ToSTSXh//8+kA9qYJ" +
       "UC1rbkJRD5Mj/vrs2G/4Ljv2FvjSks0ftdiw+aTn90PnBcOw+dSLzPpnK78v" +
       "SXjzSIP8wV8rR/0pBfLbf+Bbvxgq/yn4SIF8TITXBYe6+W1FNEXF+5b6xLrS" +
       "b3plpdNTXv7LxOC/cPrO3z3+1uBPvr752Iuc3u9K7//gpHc+mMn76S4axq4y" +
       "X+Tzdpsf+q4szjqIwrGLXu77o1/3fvndX/mJt15/LPEy0fmxwm98JW34M3Hd" +
       "lV7x2OC99wSfHtKuvr3seSWH+ImOX1m/tzabj/+D5/WvvD+H+KXn8V3X9fqL" +
       "u9efX063+a0vk3npuiii4JHr2791rsqnNErPL6JH/vf//YXfAv7y//GzX3yW" +
       "elqsPc+w6zbf/Fcv8LL/N+43P/n3fve/+NrTMq8FjxcaLxOOX4I9S/v/DS9X" +
       "prrOWx54zL//H/3gn/3b3i987JEK/Uaf3aNnqc7PGeyB1DPC/Z6nMnxl7Cnv" +
       "83cOL2/5nScw54P0/ZH1+8Zm8+b5eQ3+a9P3g5i89gT12jOJepTVR6D6tF02" +
       "bD4TRkWUeEPUm/WHpbN+wq/rIvKqDzvIl9fv7fUAf/15/Zf+bR3k/hEH+b2P" +
       "Yhw2b0bt6D0T4u7DkP3c+v3YZvOpv7LW76x1+m8I2TeeoN74kLzlZxn/Lwfe" +
       "exKxxsX941XS84HPvxiQPD96ejjTPe30Bz/i1D/zKH7fsPls1lPjqmO6lVXD" +
       "X/Psjwa32Xzm313r42bz6f/r39DZX38J9blH8YeeoP7kR+D9rUfxx4bN5+JV" +
       "Ha/arY7PzVC/R4qvvCAFtRKOrbsniA870RfXT9lsPvv9ay2vJ/vf/q2e6Bc+" +
       "4kT/4aP4+WHzhRcnOtS3anh60fOnX0H9+x6zvvlMcl7/8ef1auCI/1+vdV55" +
       "SvL8pc+/9tyPItNLM66vklV5j4cEH0qv91PiL3zE2H/+jBKP4j9+6vhD/6qN" +
       "gUfxn80fvPbvfU6zH3lef+Ux+p88Xfu3H8WffxS/9Ci+s7oNSTQ82f/32OsL" +
       "L2Xzqf/R/aSz/vK8OgIvXio9jMP3fdf7xGev6IK/+Itf+OT3/uL5v3syvy9e" +
       "ur0pbT4Zj0Xx/nc672u/2XRRnD2d+81nFrd5qn55RfI9ln+8HFirJ6T+6rPh" +
       "/2rYfOx5xP9fN838/wF7mxdWOzkAAA==");
}
