#include <cstdio>
#include <algorithm>
#include <unordered_map>
using namespace std;

/******************************************************************************
 * NOTE: I assume that you read the solution for the small bremuda triangle
 * before reading this one.
 *
 * The solution for the small bermuda triangle runs in time O((NM)^3), which
 * for NM <= 2000 is too slow (about 8 billion operations).
 *
 * This problem is probably the hardest one in warm up 2 (either this or "large
 * throne").
 *
 * To solve it we will use again the observation that there are no equilateral
 * triangles with integer vertices. This observation is important because it
 * guarantees that if two sides are equal than it is isosceles!
 *
 * Fix a particular integer point (x, y) in the grid of the input. We want to
 * count how many isosceles triangles can be formed where the two sides with
 * equal length meet at (x, y).
 *
 * For now assume that the points (1, 0), (0, 0) and (2, 0) form an isosceles
 * triangle (the distances are 1, 1, and 2... they only don't form one because
 * they are in the same line).
 *
 * Suppose that there are K points with distance exactly D of (x, y). Then there
 * are "K choose 2" possible isosceles triangles where two sides of length D
 * meet at (x,y).
 *
 * So the algorithm becomes:
 *   for all point (x, y):
 *      for all point (x1, y1):
 *         compute distance from (x, y) to (x1, y1). Let distance be D.
 *         register that we found one extra point at distance D
 *      for all distances D:
 *         If there are K points with distance D, add "K choose 2" to the answer
 *
 * This algorithm is O((NM)^2) which is fast enough.
 *
 * Look at the code to see how we discount collinear points. :)
 ******************************************************************************/

using ll = long long;

// We will work with squares of distances to avoid using square roots
ll compute(ll x, ll y, ll N, ll M) {
    unordered_map<ll, ll> points_per_distance;
    ll number_of_collinear = 0;
    for (ll x2 = 0; x2 <= N; x2++) {
        for (ll y2 = 0; y2 <= M; y2++) {
            ll dist2 = (x - x2) * (x - x2) + (y - y2) * (y - y2);
            if (dist2 == 0) continue; // don't consider (x2, y2) == (x, y)
            points_per_distance[dist2]++;
            // check if reflection of (x2, y2) across (x, y) is valid. If it is, that's a
            // collinear triple.
            
            // note that we are double counting the collinear triples. Try to
            // see why.
            ll reflectionx = x - (x2 - x), reflectiony = y - (y2 - y);
            if (reflectionx >= 0 && reflectionx <= N && reflectiony >= 0 && reflectiony <= M)
                number_of_collinear++;
        }
    }
    ll ans = 0;
    for (auto &entry : points_per_distance)
        ans += entry.second * (entry.second - 1) / 2;

    return ans - number_of_collinear / 2;
}

int main() {
    ll N, M;
    scanf("%lld %lld", &N, &M);
    ll ans = 0;
    for (int x = 0; x <= N; x++) {
        for (int y = 0; y <= M; y++) {
            ans += compute(x, y, N, M);
        }
    }
    printf("%lld\n", ans);
    return 0;
}
