
package cs465.brush;

import java.util.*;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import gl4java.*;
import gl4java.swing.GLJPanel;
import gl4java.awt.GLCanvas;
import gl4java.drawable.*;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;
import java.io.File;
import java.io.IOException;
import javax.imageio.ImageIO;

/**
 * class BrushFmwk
 *
 * One instance of this class handles the UI for the Brush assignment.  To add the
 * required functionality you should extend this class and override the
 * initializeBrush, applyBrush, and applyEraser methods.
 *
 * The architecture of this class is based entirely on the EventListener interfaces.  In
 * the constructor we create all the UI elements, asking some of them to inform us by
 * calling functions in these interfaces when events happen that we want to know about.
 */

public class BrushFmwk
    implements GLEventListener, GLEnum, GLUEnum,
	       MouseListener, MouseMotionListener, ActionListener, ChangeListener {
  
  
    public static void main(String args[]) {
	new BrushFmwk();
    }
  
  
    // These instance variables are UI elements that we need to retain access to after we
    // create them.  (Many others are created in the constructor but are not saved.)
  
    GLCanvas mainCanvas;
    JRadioButton paintButton, eraseButton, backButton, foreButton;
    JSlider shapeSlider, radiusSlider, opacitySlider;
    JColorChooser colorChooser;
  
  
    // The following variables are parameters that the user controls; they are maintained
    // by the stateChanged and actionPerformed methods.
  
    String selectedTool;
    int brushRad = 25;
    double brushExp = 1;
  
    // The size of the drawing canvas is constant, for the sake of simplicity.
  
    static final int canvWidth = 400, canvHeight = 300;
  
  
    // These three arrays are drawn into the canvas by the display() method.  The
    // checkered background image is drawn opaquely, then the paint images are layered on
    // top using an "over" operation.  They are represented in completely raw form, simply as
    // arrays of bytes with the pixels laid out row by row in that array.
    //
    // Starting from the beginning of these arrays the contents are:
    // row 0, column 0, red; row 0, column 0, green; row 0, column 0, blue; row 0, column 0, alpha;
    // row 0, column 1, red, .... etc.
  
    byte[] backgroundImage = new byte[canvWidth * canvHeight * 4];  // the fixed checkered backdrop
    byte[] forePaintImage = new byte[canvWidth * canvHeight * 4];   // the "foreground" paint image layer
    byte[] backPaintImage = new byte[canvWidth * canvHeight * 4];   // the "background" paint image layer
  
    // a pointer to the currently selected paint image layer
    byte[] currentPaintImage;
  
  
    /**
     * Creates the special canvas that allows us to draw using the OpenGL API, which
     * gives us access to fast drawing operations in graphics hardware.
     */
  
    private GLCanvas makeCanvas() {
	GLCapabilities cap = new GLCapabilities();
	cap.setDoubleBuffered(true);
	cap.setTrueColor(true);
	GLCanvas c = new GLCanvas(cap, canvWidth, canvHeight);
	return c;
    }
  
  
    /**
     * The constructor, which runs when the program starts up and an instance is created
     * in main().  Its main purpose is to create all the user interface elements and
     * arrange them in a reasonable way.  At the same time it registers this instance as
     * a listener for the different kinds of eventes that we need to know about.
     */
  
    BrushFmwk() {
	initializeBackground();
	loadDefaultBackPaintImage();
	recomputeBrush();
    
	mainCanvas = makeCanvas();
	mainCanvas.addGLEventListener(this);
	mainCanvas.addMouseListener(this);
	mainCanvas.addMouseMotionListener(this);
    
	JPanel canvPanel = new JPanel();
	canvPanel.add(mainCanvas);
	Dimension canvDim = new Dimension(canvWidth, canvHeight);
	canvPanel.setMinimumSize(canvDim);
	canvPanel.setMaximumSize(canvDim);
    
	// The tool selector panel
	paintButton = new JRadioButton("Airbrush");
	eraseButton = new JRadioButton("Eraser");
	paintButton.addActionListener(this);
	paintButton.setActionCommand("paint");
	eraseButton.addActionListener(this);
	eraseButton.setActionCommand("erase");
    
	ButtonGroup toolGroup = new ButtonGroup();
	toolGroup.add(paintButton);
	toolGroup.add(eraseButton);
	paintButton.setSelected(true);
	selectedTool = "paint";
    
	GridLayout grid = new GridLayout(2, 1);
	JPanel toolRadios = new JPanel(grid);
	toolRadios.add(paintButton);
	toolRadios.add(eraseButton);
	Border etchedBorder = BorderFactory.createEtchedBorder();
	Border titledBorder = BorderFactory.createTitledBorder(etchedBorder, "Tool");
	Border space = BorderFactory.createEmptyBorder(10, 10, 10, 10);
	toolRadios.setBorder(BorderFactory.createCompoundBorder(titledBorder, space));
    
	// The layer selector panel
	foreButton = new JRadioButton("Foreground");
	backButton = new JRadioButton("Background");
	backButton.addActionListener(this);
	backButton.setActionCommand("back");
	foreButton.addActionListener(this);
	foreButton.setActionCommand("fore");
    
	ButtonGroup layerGroup = new ButtonGroup();
	layerGroup.add(foreButton);
	layerGroup.add(backButton);
	foreButton.setSelected(true);  // NOTE this selection defaults to Foreground
	currentPaintImage = forePaintImage;
    
	grid = new GridLayout(2, 1);
	JPanel layerRadios = new JPanel(grid);
	layerRadios.add(foreButton);
	layerRadios.add(backButton);
	etchedBorder = BorderFactory.createEtchedBorder();
	titledBorder = BorderFactory.createTitledBorder(etchedBorder, "Layer");
	space = BorderFactory.createEmptyBorder(10, 10, 10, 10);
	layerRadios.setBorder(BorderFactory.createCompoundBorder(titledBorder, space));
    
	JButton loadButton = new JButton("Load image...");
	loadButton.setAlignmentX(JLabel.CENTER_ALIGNMENT);
	loadButton.addActionListener(this);
	loadButton.setActionCommand("load");
    
	Box controls = Box.createVerticalBox();
	controls.add(Box.createVerticalStrut(15));
	controls.add(toolRadios);
	controls.add(Box.createVerticalGlue());
	controls.add(layerRadios);
	controls.add(Box.createVerticalGlue());
	controls.add(loadButton);
	controls.add(Box.createVerticalStrut(15));
    
	Box canvBox = Box.createHorizontalBox();
	canvBox.add(canvPanel);
	canvBox.add(Box.createHorizontalStrut(15));
	canvBox.add(controls);
	canvBox.add(Box.createHorizontalStrut(15));
	canvBox.add(Box.createHorizontalGlue());
    
	shapeSlider = new JSlider(JSlider.HORIZONTAL, 1, 300, 100);
	shapeSlider.addChangeListener(this);
	radiusSlider = new JSlider(JSlider.HORIZONTAL, 1, 50, 25);
	radiusSlider.addChangeListener(this);
	opacitySlider = new JSlider(JSlider.HORIZONTAL, 0, 255, 128);
        opacitySlider.addChangeListener(this);
    
	colorChooser = new JColorChooser();
        colorChooser.getSelectionModel().addChangeListener(this);
	colorChooser.setPreviewPanel(new JPanel());
    
	Box row;
	Box column = Box.createVerticalBox();
	column.add(canvBox);
	column.add(Box.createVerticalStrut(15));
	row = Box.createHorizontalBox();
	row.add(Box.createHorizontalStrut(10));
	row.add(new JLabel("Shape exponent:"));
	row.add(Box.createHorizontalGlue());
	column.add(row);
	column.add(shapeSlider);
	row = Box.createHorizontalBox();
	row.add(Box.createHorizontalStrut(10));
	row.add(new JLabel("Radius:"));
	row.add(Box.createHorizontalGlue());
	column.add(row);
	column.add(radiusSlider);
	row = Box.createHorizontalBox();
	row.add(Box.createHorizontalStrut(10));
	row.add(new JLabel("Opacity:"));
	row.add(Box.createHorizontalGlue());
	column.add(row);
	column.add(opacitySlider);
	column.add(Box.createVerticalStrut(15));
	column.add(colorChooser);
	column.add(Box.createVerticalGlue());
    
	JFrame mainFrame = new JFrame("CS 465 Brush");
	mainFrame.getContentPane().add(column);
    
	// Handle closing the frame
	mainFrame.addWindowListener(new WindowAdapter() {
		public void windowClosing(WindowEvent e) {
		    System.exit(0);
		}
	    });
    
	mainFrame.pack();
	mainFrame.show();
    }
  

    String getCurrentDirectory() {
	String loc = this.getClass().getProtectionDomain().getCodeSource().getLocation().getFile();
	return loc.replaceAll("(%20)", " ");
    }

  
    void loadDefaultBackPaintImage() {
	String filename = getCurrentDirectory() + "cs465/brush/images/paper.jpg";
	filename = filename.replaceAll("(%20)", " ");
     
	try {
	    loadImageIntoLayer(new File(filename), backPaintImage);
	} catch (IOException e) {
	    System.out.println("Could not load default background layer image: " + filename);
	}
      
    }
  

    /**
     * Creates the checkered backdrop image you see if both layers are transparent.
     */

    void initializeBackground() {
	for (int iy = 0; iy < canvHeight; iy++)
	    for (int ix = 0; ix < canvWidth; ix++)
		if (((ix / 8) ^ (iy / 8)) % 2 == 1) {
		    backgroundImage[(iy * canvWidth + ix) * 4 + 0] = (byte) 156;
		    backgroundImage[(iy * canvWidth + ix) * 4 + 1] = (byte) 156;
		    backgroundImage[(iy * canvWidth + ix) * 4 + 2] = (byte) 156;
		    backgroundImage[(iy * canvWidth + ix) * 4 + 3] = (byte) 255;
		} else {
		    backgroundImage[(iy * canvWidth + ix) * 4 + 0] = (byte) 100;
		    backgroundImage[(iy * canvWidth + ix) * 4 + 1] = (byte) 100;
		    backgroundImage[(iy * canvWidth + ix) * 4 + 2] = (byte) 100;
		    backgroundImage[(iy * canvWidth + ix) * 4 + 3] = (byte) 255;
		}
    }


    Color getBrushColor() {
	return colorChooser.getColor();
    }


    int getBrushOpacity() {
	return opacitySlider.getValue();
    }
  
  
    /**
     * Override this to do whatever needs to be done when the brush paramters change.
     */
  
    void recomputeBrush() {
        System.out.println("Recomputing brush");
    }
  
  
    /**
     * Override this to do whatever needs to be done to draw into the paint image.
     */
  
    void applyBrush(int cx, int cy) {
	Color c = getBrushColor();
	int r = c.getRed();
	int g = c.getGreen();
	int b = c.getBlue();
	int a = getBrushOpacity();
	System.out.println("paint with color (" + r + ", " + g + ", " + b + ", " + a + ")");
    }
  
  
    /**
     * Override this to do whatever needs to be done to erase in the paint image.
     */
  
    void applyEraser(int cx, int cy) {
	int a = getBrushOpacity();
	System.out.println("erase with opacity " + a);
    }
  
  
    /**
     * Dispatches a mouse event to the active tool.
     */
  
    void applyTool(int cx, int cy) {
	if (selectedTool == "paint") applyBrush(cx, cy);
	else if (selectedTool == "erase") applyEraser(cx, cy);
    }
  
  
    // GLEventListener interface
  
    /**
     * Initialization callback, where we turn on whatever OpenGL features we need.  In
     * this case we turn on blending, which we use to composite the paint image over the
     * background.
     */
  
    public void init(GLDrawable d) {
	GLFunc gl = d.getGL();
	gl.glEnable(GL_BLEND);
    }
  
    /**
     * This function is called when the canvas size changes.  This should happen only
     * once at the beginning, since we do not allow changes to the size of the canvas.
     * We take the opportunity to set up the OpenGL coordinate system so that it
     * coincides with the pixels on the screen.
     */
  
    public void reshape(GLDrawable d, int wd, int ht) {
	GLFunc gl = d.getGL();
	gl.glMatrixMode(GL_PROJECTION);
	gl.glLoadIdentity();
	gl.glOrtho(0, wd, 0, ht, -1, 1);
	gl.glViewport(0, 0, wd, ht);
    }
  
    /**
     * This function is called whenever the canvas needs to be redrawn.  We do the
     * drawing in two operations: first, without blending (that is, result = 1 * source +
     * 0 * destination), we draw the checkered backgdrop image.  Then we draw the paint
     * layers with the correct rule for the over operation using premultiplied alpha:
     * result = 1 * source + (1 - alpha_source) * back.
     */
  
    public void display(GLDrawable d) {
	GLFunc gl = d.getGL();
	gl.glRasterPos2i(0, 0);
	gl.glBlendFunc(GL_ONE, GL_ZERO);
	gl.glDrawPixels(canvWidth, canvHeight, GL_RGBA, GL_UNSIGNED_BYTE, backgroundImage);
	gl.glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
	gl.glDrawPixels(canvWidth, canvHeight, GL_RGBA, GL_UNSIGNED_BYTE, backPaintImage);
	gl.glDrawPixels(canvWidth, canvHeight, GL_RGBA, GL_UNSIGNED_BYTE, forePaintImage);
    }
  
  
    // MouseListener interface
  
    /**
     * When the user clicks the mouse, this method is called.  We pass the message along
     * to applyTool.  Note that the y coordinate is being reflected because of a
     * difference between the OpenGL coordinate system in which the image is drawn (y
     * increases from bottom tol top, mathematician-wise) and the window coordinate
     * system in which the mouse coordinates are provided (y increases from top to
     * bottom, TV-wise).
     */
  
    public void mousePressed(MouseEvent e) {
	applyTool(e.getX(), canvHeight - e.getY() - 1);
	mainCanvas.repaint();
    }
  
  
    // MouseMotionListener interface
  
    /**
     * This is called whenever the mouse moves while the button is being held down.  It
     * is treated the same as a mouse-down.
     */

    public void mouseDragged(MouseEvent e) {
	applyTool(e.getX(), canvHeight - e.getY() - 1);
	mainCanvas.repaint();
    }
  
  
    // ActionListener interface
  
    /**
     * This listener function handles the tool selection buttons and the Load Image
     * button.  We can tell which by looking at what string we get; we gave the strings
     * to the buttons in the constructor.
     */

    public void actionPerformed(ActionEvent e) {
	String cmdString = e.getActionCommand();
	if (cmdString == "load") {
	    JFileChooser fc = new JFileChooser(getCurrentDirectory() + "cs465/brush/images/");
	    int returnVal = fc.showOpenDialog(mainCanvas);
	    if (returnVal == JFileChooser.APPROVE_OPTION) {
		File selectedFile = fc.getSelectedFile();
		try {
		    loadImageIntoLayer(selectedFile, currentPaintImage);
		}
		catch (IOException ex) {
		    throw new Error("Could not read file " + selectedFile);
		}
	    }
	}
	else if (cmdString == "back") {
	    currentPaintImage = backPaintImage;
	}
	else if (cmdString == "fore") {
	    currentPaintImage = forePaintImage;
	} else
	    selectedTool = e.getActionCommand();
    }

    /**
     * Read an image from a file into the currently selected layer.
     */

    private void loadImageIntoLayer(File imageFile, byte[] paintImage) throws IOException {
	BufferedImage loadedImage = ImageIO.read(imageFile);
	int imageWidth = loadedImage.getWidth();
	int imageHeight = loadedImage.getHeight();
	int pixelAlphaValue;
	initializeBackground();
	for (int iy = 0; iy < canvHeight && iy < imageHeight; iy++) {
	    for (int ix = 0; ix < canvWidth && ix < imageWidth; ix++) {
		int pixelValue = loadedImage.getRGB(ix,imageHeight-1-iy);
		pixelAlphaValue = 0xff & (pixelValue>>24);
		paintImage[(iy * canvWidth + ix) * 4 + 0] = 
		    (byte)((0xff & (pixelValue>>16)) * pixelAlphaValue / 255);
		paintImage[(iy * canvWidth + ix) * 4 + 1] = 
		    (byte)((0xff & (pixelValue>> 8)) * pixelAlphaValue / 255);
		paintImage[(iy * canvWidth + ix) * 4 + 2] =
		    (byte)((0xff & (pixelValue>> 0)) * pixelAlphaValue / 255);
		paintImage[(iy * canvWidth + ix) * 4 + 3] = (byte) pixelAlphaValue;
	    }
	}
    }
  

    // ChangeListener interface
  
    /**
     * This listener function handles the sliders that control brush parameters.  The
     * values are stored in instance variables.  (Except color and opacity, which are
     * accessible through getBrushColor and getBruchOpacity and whose events are still
     * caught here)
     */

    public void stateChanged(ChangeEvent e) {
	if (e.getSource() == radiusSlider)
	    brushRad = radiusSlider.getValue();
	else if (e.getSource() == shapeSlider)
	    brushExp = shapeSlider.getValue() / 100.0;
	recomputeBrush();
    }
  
    // unused bits of GLEventListener interface
    public void cleanup(GLDrawable d) {}
    public void postDisplay(GLDrawable d) {}
    public void preDisplay(GLDrawable d) {}
  
    // unused bits of MouseListener interface
    public void mouseClicked(MouseEvent e) {}
    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void mouseReleased(MouseEvent e) {}
  
    // unused bits of MouseMotionListener interface
    public void mouseMoved(MouseEvent e) {}
  
  
  
}
