package cs4621.lecture05;

import java.io.IOException;

import javax.media.opengl.GL2;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.glu.GLU;
import javax.vecmath.Matrix4f;

import cs4621.framework.GLFrame;
import cs4621.framework.GlslException;
import cs4621.framework.Program;
import cs4621.framework.Shader;
import cs4621.framework.Texture2D;
import cs4621.framework.TextureOneDim;
import cs4621.framework.TextureUnit;
import cs4621.framework.Uniform;

public class GlslTextureRenderingApp extends GLFrame {
	private static final long serialVersionUID = 1L;
	
	float angle;
	
	// Handler for the texture object
	private Texture2D texture = null;
	
	// Handler for the shader program
	private Program textureShaderProgram = null;
	
	// Handler for the texture uniform
	private Uniform samplerUniform = null;
	
	public GlslTextureRenderingApp() {
		super("GLSL + texturing!", 
				640, 480, 60);		
	}	

	public static void main(String args[]) {
		new GlslTextureRenderingApp().run();
	}
	
	public void init(GLAutoDrawable drawable) {
		super.init(drawable);
		
		final GL2 gl = drawable.getGL().getGL2();
		
		// We want to show the back face of the plane as well
		gl.glDisable(GL2.GL_CULL_FACE);			
		
		// Check whether GLSL is supported
		if ( !Shader.checkGlslSupport(gl) ) {
			System.exit(1);
		}
		
		// Load, compile, link and bind shader program
		if ( !loadShaders(gl, "texture_shader.vs", "texture_shader.fs") ) {
			System.exit(1);
		}
		
		// Get the sampler uniform
		samplerUniform = 
			textureShaderProgram.GetUniforms().get("sampler");
				
		if (samplerUniform == null) {
			System.out.println("Uniform sampler not used");
		}
		
		// Load the 2D texture
		try {
			texture = new Texture2D(gl, "data/textures/sample.jpg");
		}
		catch (IOException e) {
			System.out.print("Cannot load texture: ");
			System.out.println(e.getMessage());
			terminate();
		}
				
		angle = 0.0f;
	}
	
	protected void drawPlane(GL2 gl){		

		gl.glBegin(GL2.GL_QUADS);
		{			
			gl.glTexCoord2f(1, 1);
			gl.glVertex3f( 1.0f, 1.0f, 0.0f);
			gl.glTexCoord2f(0, 1);
			gl.glVertex3f(-1.0f, 1.0f, 0.0f);
			gl.glTexCoord2f(0, 0);
			gl.glVertex3f(-1.0f,-1.0f, 0.0f);
			gl.glTexCoord2f(1, 0);
			gl.glVertex3f( 1.0f,-1.0f, 0.0f);
		}
		gl.glEnd();

	}
	
	public void display(GLAutoDrawable drawable) {
		final GL2 gl = drawable.getGL().getGL2();
		final GLU glu = new GLU();
			
		gl.glClear(GL2.GL_COLOR_BUFFER_BIT | GL2.GL_DEPTH_BUFFER_BIT);
					
		gl.glLoadIdentity();
		glu.gluLookAt(0, 0, 5, 0, 0, 0, 0, 1, 0);
		
		angle += 1.0f;
		gl.glRotated(30, 0, 1, 0);
		gl.glRotated(angle, 1, 0, 0);
		
		texture.use(); // Make it the active texture unit
		textureShaderProgram.use(); // Activate the shader
		
		// Bind the active texture unit to the sampler uniform
		TextureUnit.getActiveTextureUnit(gl).bindToUniform(samplerUniform);
		
		drawPlane(gl);
		
		textureShaderProgram.unuse();
		texture.unuse();
	}
		
	public void reshape(GLAutoDrawable drawable, int x, int y, int width, int height) {
		final GL2 gl = drawable.getGL().getGL2();
		final GLU glu = new GLU();

		gl.glViewport(0, 0, width, height);
		gl.glMatrixMode(GL2.GL_PROJECTION);
		gl.glLoadIdentity();

		glu.gluPerspective(
				45.0f, 
				(double) width / (double) height, 
				0.1f,
				1000.0f);

		gl.glMatrixMode(GL2.GL_MODELVIEW);
		gl.glLoadIdentity();				
	}	
	
	private Boolean loadShaders(GL2 gl, 
			String vertexFileName, String fragmentFileName) {
				
		try {			
			// Load, compile and link the shaders
			this.textureShaderProgram = new Program(gl, 
					vertexFileName, fragmentFileName);
			
		} catch (GlslException e) {
			
			System.err.println(e.getMessage());
			return false;
			
		}	
		 
		return true;
	}			
}
