package cs2110.assignment3.swing;

import java.awt.Container;
import java.awt.Dimension;

import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.UIManager;

import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.io.IOException;


/**
 * Base class for implementing the species comparison GUI.
 * <p>
 * You will subclass ComparisonGUI to make your comparison GUI.
 * <p>
 * The GUI consists of a table with a picture of each species, and two larger pictures that are intended to show the currently selected species and its closest relative, as computed by your species distance algorithm.
 * <p>
 *   When the user clicks on a cell in the table, the method onSelectCell is called.  Your subclass should override <i>onSelectCell</i> to exhibit the behavior described in the assignment. 
 * <p>
 * You will refer to cells by their number, 0 through n-1, where n is the number of species given to the ComparisonGUI's constructor.
 * <p>
 * Read the documentation for each method to learn what you can do with this class.
 * @author Lonnie Princehouse
 *
 */
public class ComparisonGUI {

	private int n; 	// Number of species

	// Species table dimensions
	private int columns = 5;  // Hard-coded
	private int rows;  // Depends on n and columns

	// Allocate a cell object for each cell in the table
	private Cell cells[];

	// image and label for selected species
	private JLabel selectedLabel;
	private JLabel closestRelatedLabel;

	// image and label for closest related species
	private JLabel selectedImage;
	private JLabel closestRelatedImage;

	final static protected int CELL_WIDTH = 90;
	final static protected int CELL_HEIGHT = 90;


	/**
	 * The constructor takes a single argument: the number of species you plan on placing in the GUI's comparison table.
	 * <p>
	 * After constructing an instance of ComparisonGUI, you should call setCellImage for each cell (numbered 0 through n-1) to place images in the cells.
	 *
	 * @param n Number of species
	 */
	public ComparisonGUI(int n) {	  
	    try {
	        UIManager.setLookAndFeel(
	            UIManager.getCrossPlatformLookAndFeelClassName());
	    } catch (Exception e) { }
		if(n == 0)
			throw new RuntimeException("ComparisonGUI constructor must have n >= 0");
		this.n = n;
		setup();
	}

	private JFrame frame;  // Top-level window
	private JPanel mainPane; // Contains left and right
	private JPanel leftPanel;	// Contains the species grid
	private JPanel rightPanel;	// Contains the larger pictures and text


	// Configure the interface and start swing
	private void setup() {

		// Initial window dimensions, in pixels
		final int width = 800; 
		final int height = 500;

		frame = new JFrame("Species Comparison Interface");
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

		leftPanel = new JPanel();
		rightPanel = new JPanel();

		// Needs to be final so it can be used from the event handlers' anonymous inner classes
		final Container topPane = frame.getContentPane(); 

		rows = (int) Math.ceil( (double)n / (double) columns);

		topPane.setLayout(new BoxLayout(topPane, BoxLayout.Y_AXIS)); // Use absolute positioning because layout managers insist on resizing everything when labels' text changes
		mainPane = new JPanel();
		topPane.add(mainPane);
		mainPane.setLayout(null);
		mainPane.add(leftPanel);
		mainPane.add(rightPanel);

		leftPanel.setLayout(new GridLayout(rows,columns));   // Grid layout for the table
		rightPanel.setLayout(new BoxLayout(rightPanel,BoxLayout.Y_AXIS)); // Vertical box layout for the big images and labels

		updateManualLayout(width,height); // Initialize top-level layout

		createComparisonPanel(rightPanel);  // create the right-side GUI components

		cells = new Cell[n];

		// Add a resize listener using an anonymous inner class
		topPane.addComponentListener(new ComponentListener(){

			// This is the only event we're interested in
			public void componentResized(ComponentEvent e) {
				updateManualLayout(mainPane.getWidth(), mainPane.getHeight());
			}

			public void componentHidden(ComponentEvent e) {} // Ignore
			public void componentMoved(ComponentEvent e) {} // Ignore
			public void componentShown(ComponentEvent e) {} // Ignore
		});

		// Initialize cells.  Some variables final so they can be used from an anonymous inner class
		for(int i = 0; i < n; i++) {
			final ScalingImageButton imageButton = new ScalingImageButton();
			imageButton.setSize(CELL_WIDTH, CELL_HEIGHT);
			leftPanel.add(imageButton);
			cells[i] = new Cell(imageButton);

			final int index = i; 

			// Bind mouse events using an anonymous inner class
			imageButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent arg0) {
					onSelectCell(index);
				}
			});

			// Bind resize events
			imageButton.addComponentListener(new ComponentListener() {
				public void componentHidden(ComponentEvent arg0) {}
				public void componentMoved(ComponentEvent arg0) {}
				public void componentResized(ComponentEvent arg0) {
					imageButton.setScaledImage(width/columns,height/rows);
				}
				public void componentShown(ComponentEvent arg0) {}
			}); 
		}

		//Display the window.
		frame.setSize(width, height);
		frame.setMinimumSize(new Dimension(width, height));
		frame.pack();
		frame.setVisible(true);
	}

	private void updateManualLayout(int w, int h) {
		int bar = w-300; // Comparison panel width fixed at 300 pixels; table will fill the rest of the space

		// Absolute positioning.  Real programmers don't need layout managers.
		getComparisonPanel().setBounds(bar,0,bar+300,h);
		getGridPanel().setBounds(0,0,bar,h);

		// Force screen  update
		getComparisonPanel().updateUI();
		getGridPanel().updateUI();
	}

	/**
	 *  Optional: For extending the user interface.  Returns the top-level frame (A.K.A. top-level window).  The frame's content panel (getFrame().getContentPanel()) uses a vertical BoxLayout manager.  Components added will appear below the existing GUI.
	 * @return The top-level frame.
	 */
	public JFrame getFrame() {
		return frame;
	}

	/**
	 *  Optional: For extending the user interface.  The comparison panel is the panel containing the larger images and text.  It uses a vertical BoxLayout manager.
	 * @return The comparison panel JPanel 
	 */
	public JPanel getComparisonPanel() {
		return rightPanel;
	}

	/**
	 *  Optional: For extending the user interface.  The grid panel uses a GridLayout that contains buttons for each species..
	 * @return The grid panel JPanel 
	 */
	public JPanel getGridPanel() {
		return leftPanel;
	}

	/**
	 * Get the number of columns in the species table
	 * @return Number of columns in the species table
	 */
	public int getColumns() {
		return columns;
	}

	/**
	 * Get the number of rows in the species table
	 * @return Number of rows in the species table
	 */	
	public int getRows() {
		return rows;
	}

	private void createComparisonPanel(JPanel parent) {
		selectedLabel = new JLabel("Selected: ");
		parent.add(selectedLabel);
		selectedImage = new JLabel();
		parent.add(selectedImage);
		closestRelatedLabel = new JLabel("Closest relative: ");
		parent.add(closestRelatedLabel);
		closestRelatedImage = new JLabel();
		parent.add(closestRelatedImage);
	}

	private Cell getCell(int index) {
		return cells[index];
	}

	/**
	 * This method will be called when the user clicks a cell in the table.  Your subclass should override this method.
	 * @param cell
	 */
	public void onSelectCell(int  cell) { 
		System.out.println("onSelectCell: User clicked cell " + cell);
	}

	/**
	 * Load an image into the desired cell. 
	 * @param cell Cell number
	 * @param filename Path to the image file
	 * @throws InvalidCellNumberException 
	 */
	public void setCellImage(int cell, String filename) throws InvalidCellNumberException {
		checkCell(cell);
		try {
			getCell(cell).setImageFile(filename);
		} catch (IOException e) {
			System.err.println("Image not found: " + filename);
		}
	}

	private void checkCell(int cell) throws InvalidCellNumberException {
		if(cell < 0 || cell >= n) {
			throw new InvalidCellNumberException();
		}
	}
	/**
	 * Sets the background color of a cell.  Colors are specified as (r,g,b) triples between zero and one.
	 * @param cell Cell number
	 * @param red Red component. Must be between 0 and 1.
	 * @param green Green component. Must be between 0 and 1.
	 * @param blue Blue component. Must be between 0 and 1.
	 * @throws InvalidColorException 
	 * @throws InvalidCellNumberException 
	 */
	public void setCellColor(int cell, double red, double green, double blue) throws InvalidColorException, InvalidCellNumberException {
		checkCell(cell);
		if(red < 0 || red > 1) 
			throw new InvalidColorException();
		if(green < 0 || green > 1) 
			throw new InvalidColorException();
		if(blue < 0 || blue > 1) 
			throw new InvalidColorException();

		getCell(cell).setColor( (int) (red * 255), (int) (255 * green), (int) (255 * blue));
	}

	/**
	 * Sets the background color of a cell, specified as an instance of java.awt.Color
	 * @param cell Cell number
	 * @param color Desired background color
	 * @throws InvalidCellNumberException 
	 */
	public void setCellColor(int cell, java.awt.Color color) throws InvalidCellNumberException {
		checkCell(cell);
		getCell(cell).setColor(new Color(color));
	}

	/**
	 *  Set the name and image file of the currently selected species.  These will appear under "Selected:" at the right of the window.
	 * @param name Species name
	 * @param imageFile Image file path
	 */
	public void setSelectedInfo(String name, String imageFile) {
		selectedImage.setIcon(new ImageIcon(imageFile));
		selectedLabel.setText("Selected: " + name);
	}

	/**
	 *  Set the name and image file of the current most closely related species.  These will appear under "Closest relative:" at the right of the window.
	 * @param name Species name
	 * @param imageFile Image file path
	 */
	public void setClosestRelativeInfo(String name, String imageFile) {
		closestRelatedImage.setIcon(new ImageIcon(imageFile));
		closestRelatedLabel.setText("Closest relative: " + name);
	}

	/**
	 * Start the GUI's main event loop.  Call this after instantiating your GUI.
	 */
	private ColorThread colorthread;

	public void run() {
		colorthread = new ColorThread(cells);
		colorthread.start();		
	}

}
