package cs2110.assignment3;

import java.util.HashMap;

import org.eclipse.swt.*;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;

/**
 * Base class for implementing the species comparison GUI.
 * <p>
 * <b>BETA VERSION DOCUMENTATION</b>
 * <p>
 * You will subclass ComparisonGUI to make your comparison GUI.
 * <p>
 * The GUI consists of a table with a picture of each species, and two larger pictures that are intended to show the currently selected species and its closest relative, as computed by your species distance algorithm.
 * <p>
 *   When the user clicks on a cell in the table, the method onSelectCell is called.  Your subclass should override <i>onSelectCell</i> to exhibit the behavior described in the assignment. 
 * <p>
 * You will refer to cells by their number, 0 through n-1, where n is the number of species given to the ComparisonGUI's constructor.
 * <p>
 * Read the documentation for each method to learn what you can do with this class.
 * @author Lonnie Princehouse
 *
 */

public class ComparisonGUI {



	private Cell cells[];
	private HashMap<Cell,Integer> cellmap;

	private Shell shell;
	private Display display;

	private int rows;
	private int columns;
	private int n;

	private Label bigLabel1;
	private Label bigLabel2;
	private Label bigImage1;
	private Label bigImage2;

	final static protected int CELL_WIDTH = 90;
	final static protected int CELL_HEIGHT = 90;

	/**
	 * The constructor takes a single argument: the number of species you plan on placing in the GUI's comparison table.
	 * <p>
	 * After constructing an instance of ComparisonGUI, you should call setCellImage for each cell (numbered 0 through n-1) to place images in the cells.
	 *
	 * @param n Number of species
	 */

	private Canvas canvas;

	public ComparisonGUI(int n) {	  
		display = new Display();
		cellmap = new HashMap<Cell,Integer>();

		columns = 7;
		rows = (int) Math.ceil( (double)n / (double) columns);
		shell = new Shell(display);
		shell.setText("Species Comparison (beta)");
		shell.open();	

		FormLayout layout = new FormLayout();
		layout.marginWidth = 3;
		layout.marginHeight = 3;
		shell.setLayout(layout);

		createMenu();

		canvas = new Canvas(shell, SWT.BORDER);

		FormData data1 = new FormData();
		data1.left = new FormAttachment(0,0);
		data1.top = new FormAttachment(0,0);

		canvas.setLayoutData(data1);


		createComparisonPanel();

		this.n = rows * columns;
		cells = new Cell[this.n];
		canvas.setSize(columns*CELL_WIDTH, rows * CELL_HEIGHT);
		for(int row = 0; row < rows; row++) {
			for(int col = 0; col < columns; col++) {
				Cell cell = new Cell(this, canvas);
				Label img = cell.createImage();
				img.setBounds( col * CELL_WIDTH,
						row * CELL_HEIGHT, 
						CELL_WIDTH,
						CELL_HEIGHT);
				int index = row*columns+col;
				cells[index] = cell;
				cellmap.put(cell,index);
			}
		}
	}

	private void createMenu() {
		Menu menuBar = new Menu(shell, SWT.BAR);
		MenuItem fileMenuHeader = new MenuItem(menuBar, SWT.CASCADE);
		fileMenuHeader.setText("&File");

		Menu fileMenu = new Menu(shell, SWT.DROP_DOWN);
		fileMenuHeader.setMenu(fileMenu);

		MenuItem fileExitItem = new MenuItem(fileMenu, SWT.PUSH);
		fileExitItem.setText("E&xit");

		fileExitItem.addSelectionListener(new FileExitListener());


				// fileSaveItem.addSelectionListener(new fileSaveItemListener());
				// helpGetHelpItem.addSelectionListener(new helpGetHelpItemListener());

				shell.setMenuBar(menuBar);
	}

	private class FileExitListener implements SelectionListener {
		public void widgetSelected(SelectionEvent event) {
			shell.close();
			display.dispose();
		}

		public void widgetDefaultSelected(SelectionEvent arg0) {			
		} 	
	}

	private void createComparisonPanel() {
		// not at all happy with the layout of this thing...
		//Composite parent = new Composite(shell, SWT.BORDER);
		Composite parent = shell;
		//FormLayout l = new FormLayout();
		//l.type = SWT.VERTICAL;
		//parent.setLayout(l);

		bigLabel1 = new Label(parent,SWT.NONE);
		bigLabel1.setText("Selected: ");
		bigLabel1.setVisible(true);
		FormData l1data = new FormData();
		l1data.width = 250;
		l1data.left = new FormAttachment(canvas,10);
		bigLabel1.setLayoutData(l1data);

		bigImage1 = new Label(parent,SWT.NONE);
		bigImage1.setVisible(true);
		FormData i1data = new FormData();
		i1data.left = new FormAttachment(canvas,10);
		i1data.top = new FormAttachment(bigLabel1,10);
		bigImage1.setLayoutData(i1data);

		bigLabel2 = new Label(parent,SWT.NONE);
		bigLabel2.setText("Closest relative::");
		bigLabel2.setVisible(true);
		FormData l2data = new FormData();
		l2data.width = 250;
		l2data.left = new FormAttachment(canvas,10);
		l2data.top = new FormAttachment(bigImage1,10);
		bigLabel2.setLayoutData(l2data);

		bigImage2 = new Label(parent,SWT.NONE);
		bigImage2.setVisible(true);
		FormData i2data = new FormData();
		i2data.left = new FormAttachment(canvas,10);
		i2data.top = new FormAttachment(bigLabel2,10);
		bigImage2.setLayoutData(i2data);

		parent.setVisible(true);
	}

	private Cell getCell(int index) {
		return cells[index];
	}

	/**
	 * This method will be called when the user clicks a cell in the table.  Your subclass should override this method.
	 * @param cell
	 */
	public void onSelectCell(int  cell) { 
		System.out.println("onSelectCell: User clicked cell " + cell);
	}

	protected void selectCell(Cell cell) {
		int index = cellmap.get(cell);
		onSelectCell(index);
	}

	/**
	 * Load an image into the desired cell. 
	 * @param cell Cell number
	 * @param filename Path to the image file
	 * @throws InvalidCellNumberException 
	 */
	public void setCellImage(int cell, String filename) throws InvalidCellNumberException {
		checkCell(cell);
		getCell(cell).setImageFile(filename);
	}

	private void checkCell(int cell) throws InvalidCellNumberException {
		if(cell < 0 || cell >= n) {
			throw new InvalidCellNumberException();
		}
	}
	/**
	 * Sets the background color of a cell.  Colors are specified as (r,g,b) triples between zero and one.
	 * @param cell Cell number
	 * @param red Red component. Must be between 0 and 1.
	 * @param green Green component. Must be between 0 and 1.
	 * @param blue Blue component. Must be between 0 and 1.
	 * @throws InvalidColorException 
	 * @throws InvalidCellNumberException 
	 */
	public void setCellColor(int cell, double red, double green, double blue) throws InvalidColorException, InvalidCellNumberException {
		checkCell(cell);
		if(red < 0 || red > 1) 
			throw new InvalidColorException();
		if(green < 0 || green > 1) 
			throw new InvalidColorException();
		if(blue < 0 || blue > 1) 
			throw new InvalidColorException();

		getCell(cell).setColor( (int) (red * 255), (int) (255 * green), (int) (255 * blue));
	}

	/**
	 * Clears the background colors of all cells.
	 */
	/*
	public void clearColors() {
		for(Cell c : cells) {
			c.clearColor();
		}
	}
	 */
	/**
	 *  Set the name and image file of the currently selected species.  These will appear under "Selected:" at the right of the window.
	 * @param name
	 * @param imageFile
	 */
	public void setSelectedInfo(String name, String imageFile) {
		bigImage1.setImage(loadImage(imageFile));
		bigLabel1.setText("Selected: " + name);
	}

	/**
	 *  Set the name and image file of the current most closely related species.  These will appear under "Closest relative:" at the right of the window.
	 * @param name
	 * @param imageFile
	 */

	public void setClosestRelativeInfo(String name, String imageFile) {
		bigImage2.setImage(loadImage(imageFile));
		bigLabel2.setText("Closest relative: " + name);
	}


	protected Image loadImage(String filename) {
		return new Image(display,filename);
	}

	/**
	 * Start the GUI's main event loop.  Call this after instantiating your GUI.
	 */

	private ColorThread colorthread;

	public void run() {
		colorthread = new ColorThread(display, cells);
		shell.pack();
		shell.open();
		colorthread.start();		
		while (!shell.isDisposed ()) {
			if (!display.readAndDispatch ()) display.sleep ();
		}
		colorthread.running = false;
		try {
			colorthread.join();
		} catch (InterruptedException e) {
			// ignore
		}
		display.dispose();
	}	

}

