import java.awt.*;
import java.awt.event.*;

/**
 * Main GUI for playing FIAR.
 */
class MainWindow extends Frame {
	
	private IGame d_game;
	private GUIGameBoard d_gameBoardDisplay;
	
	private Button d_btnQuit;
	private Button d_btnNewGame;
	
	private Button d_btnPlayerOneMove;
	private Button d_btnPlayerTwoMove;
	
	private TextField d_txtPlayerOneColumn;
	private TextField d_txtPlayerTwoColumn;
	
	private TextField d_txtMessageDisplay;
	
	private Label d_gameTypeLabel;
		
	private NewGameDialog d_dlgNewGame;
	private int d_rows;
	private int d_cols;
    private int d_firstPlayer;
	private int d_gameType;           // invalid game type if < 0
	
	public MainWindow() {
		//set title
		super("Four in a Row - Main Window");

		//create gui for main window
        Panel panelRight = new Panel(new GridLayout(4,1));

        Panel p2 = new Panel();
		d_btnNewGame = new Button("New Game...");
		d_btnQuit = new Button("Quit");
        p2.add(d_btnNewGame);
        p2.add(d_btnQuit);
		panelRight.add(p2);

  		Panel p3 = new Panel(new GridLayout(2,1));
        d_btnPlayerOneMove = new Button("Player 1 Move");
		d_btnPlayerOneMove.setForeground(Color.red);
		d_btnPlayerTwoMove = new Button("Player 2 Move");
		d_btnPlayerTwoMove.setForeground(Color.blue);
		d_txtPlayerOneColumn = new TextField("0", 1);
		d_txtPlayerTwoColumn = new TextField("0", 1);
        Panel p4 = new Panel();
        p4.add(d_btnPlayerOneMove);
        p4.add(d_txtPlayerOneColumn);
        p3.add(p4);

        Panel p5 = new Panel();
        p5.add(d_btnPlayerTwoMove);
        p5.add(d_txtPlayerTwoColumn);
        p3.add(p5);
		panelRight.add(p3);

				
		d_txtMessageDisplay = new TextField("", 30);
		d_txtMessageDisplay.setBackground(Color.lightGray);
		d_txtMessageDisplay.setEditable(false);
        Panel p6 = new Panel();
		p6.add(d_txtMessageDisplay);
		panelRight.add(p6);
		
		d_gameTypeLabel = new Label("", Label.CENTER);
		panelRight.add(d_gameTypeLabel);		

        Panel panelLeft = new Panel(new BorderLayout());
		d_gameBoardDisplay = new GUIGameBoard();
		panelLeft.add(d_gameBoardDisplay, BorderLayout.CENTER);

        setLayout(new GridLayout(1,2));
        add(panelLeft);
        add(panelRight);

		disableMoves();

		addListeners();
		
		setBackground(Color.green);		
		pack();
		show();
	}

	/** Sets up and starts a new game.
	 */
	public void startNewGame(int rows, int cols, int firstPlayer,
	                         int gameType) {
		if (gameType >= 0) {
			d_rows = rows;
			d_cols = cols;
			d_firstPlayer = firstPlayer;
     		d_gameType = gameType;
			d_game = new GameImplementation(d_rows, d_cols, d_firstPlayer,
	                                        d_gameType);
			d_gameBoardDisplay.setNewGUIGameBoard(d_rows, d_cols);
			d_txtPlayerOneColumn.setText("0");
			d_txtPlayerTwoColumn.setText("0");
			d_txtMessageDisplay.setText("");
			switch (d_gameType) {
				case IGameBoard.FIAR_DIAGONALS_ONLY_GAMETYPE:
						d_gameTypeLabel.setText("Four in a Row - Diagonals only"); 
						break;
				case IGameBoard.NO_CHECKING_GAMETYPE:
						d_gameTypeLabel.setText("No checking");
						break;
				case IGameBoard.FIAR_NORMAL_GAMETYPE:
				default:
						d_gameTypeLabel.setText("Four in a Row - All directions");
			}
			togglePlayers();
			validate();
		}
	}
	
	/** Add listener objects to appropriate controls.
	 */
	private void addListeners() {
		d_btnQuit.addActionListener( new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				setVisible(false);
				dispose();
				System.exit(0);
			}
		});

		d_btnNewGame.addActionListener( new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				new NewGameDialog(MainWindow.this);
			}
		});

		d_btnPlayerOneMove.addActionListener( new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
                int col;
        		try {
        		    col = Integer.parseInt(d_txtPlayerOneColumn.getText());
                    if (col < 0 || col >= d_cols) throw new NumberFormatException();
        		} catch (NumberFormatException e) {
        			d_txtMessageDisplay.setText("Invalid column no.");
                    return;
        		}
				doMove(1, col);
			}
		});

		d_btnPlayerTwoMove.addActionListener( new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
                int col;
        		try {
        		    col = Integer.parseInt(d_txtPlayerTwoColumn.getText());
                    if (col < 0 || col >= d_cols) throw new NumberFormatException();
        		} catch (NumberFormatException e) {
        			d_txtMessageDisplay.setText("Invalid column no.");
                    return;
        		}
				doMove(2, col);
			}
		});
		
	}
	
	/** Add the move to the GUI.
	 */
	private void doMove(int playerNum, int col) {
		int rc = d_game.makeMove(col, playerNum);
		if (rc < 0) {
			d_txtMessageDisplay.setText("Invalid move.");
		} else {
            d_gameBoardDisplay.addPiece(rc, col, d_game.getPlayer(playerNum).getPlayerColor());
            if (d_game.isGameOver()) {
                setWinner(d_game.getWinner());
            } else {
    			togglePlayers();
	    	}
            repaint();
        }
	}
	
	/** Set the player button enablings to their correct values.
	 */
	private void togglePlayers() {
		d_txtMessageDisplay.setText("");
		switch(d_game.getCurrentPlayer().getPlayerNumber()) {
            case 1:
    			d_btnPlayerOneMove.setEnabled(true);
    			d_btnPlayerTwoMove.setEnabled(false);
    			d_txtPlayerOneColumn.setEnabled(true);
    			d_txtPlayerTwoColumn.setEnabled(false);
                break;
		    case 2:
    			d_btnPlayerOneMove.setEnabled(false);
    			d_btnPlayerTwoMove.setEnabled(true);
			    d_txtPlayerOneColumn.setEnabled(false);
			    d_txtPlayerTwoColumn.setEnabled(true);
		}
	}
		
	/** Update the screen to its "I Have a Winner" state.
	 */
	private void setWinner(IPlayer p) {
		d_txtMessageDisplay.setText("Player " + p.getPlayerNumber() + " wins.");
        d_txtPlayerOneColumn.setText("");
        d_txtPlayerTwoColumn.setText("");
        disableMoves();
	}

	/** Disable all the "move" controls.
	 */
	private void disableMoves() {
		d_btnPlayerOneMove.setEnabled(false);
		d_btnPlayerTwoMove.setEnabled(false);
		d_txtPlayerOneColumn.setEnabled(false);
		d_txtPlayerTwoColumn.setEnabled(false);
	}
}
