/* Copyright (c) 2010, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */

package com.cburch.logisim.gui.log;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import javax.swing.JButton;
import javax.swing.JRadioButton;
import javax.swing.ButtonGroup;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;

import com.cburch.logisim.data.Value;
import com.cburch.logisim.data.TestVector;
import com.cburch.logisim.util.JFileChoosers;
import com.cburch.logisim.util.StringUtil;

class FilePanel extends LogPanel {
	private class Listener implements ActionListener, ModelListener {
		public void selectionChanged(ModelEvent event) { }

		public void entryAdded(ModelEvent event, Value[] values) { }
		
		public void filePropertyChanged(ModelEvent event) {
			Model model = getModel();
			
			File file = model.getFile();
			fileField.setText(file == null ? "" : file.getPath());
			enableButton.setEnabled(file != null);
			disableButton.setEnabled(file != null);
			if (model.isFileEnabled())
				enableButton.setSelected(true);
			else
				disableButton.setSelected(true);
			
			headerCheckBox.setSelected(model.getFileHeader());
			keepWritingCheckBox.setSelected(model.isFileKeepWriting());
			if (event != null && event.fileError != null) {
				JOptionPane.showMessageDialog(getLogFrame(),
				StringUtil.format(Strings.get("fileCannotWriteMessage"), event.file == null ? "???" : event.file.getName()),
				Strings.get("fileCannotWriteTitle"),
				JOptionPane.OK_OPTION);
			}
		}
		

		public void actionPerformed(ActionEvent event) {
			Object src = event.getSource();
			if (src == enableButton) {
				getModel().setFileEnabled(true);
			} else if(src == disableButton) {
				getModel().setFileEnabled(false);
			} else if (src == selectButton) {
				int result = chooser.showSaveDialog(getLogFrame());
				if (result != JFileChooser.APPROVE_OPTION) return;
				File file = chooser.getSelectedFile();
				if (file.exists() && (!file.canWrite() || file.isDirectory())) {
					JOptionPane.showMessageDialog(getLogFrame(),
						StringUtil.format(Strings.get("fileCannotWriteMessage"), file.getName()),
						Strings.get("fileCannotWriteTitle"),
						JOptionPane.OK_OPTION);
					return;
				}
				if (file.exists() && file.length() > 0) {
					String[] options = {
							Strings.get("fileOverwriteOption"),
							Strings.get("fileAppendOption"),
							Strings.get("fileCancelOption"),
					};
					int option = JOptionPane.showOptionDialog(getLogFrame(),
						StringUtil.format(Strings.get("fileExistsMessage"), file.getName()),
						Strings.get("fileExistsTitle"),
						0, JOptionPane.QUESTION_MESSAGE, null,
						options, options[0]);
					if (option == 0) {
						try {
							FileWriter delete = new FileWriter(file);
							delete.close();
						} catch (IOException e) { }
					} else if (option == 1) {
						// do nothing
					} else {
						return;
					}
				}
				getModel().setFile(file);
			} else if (src == headerCheckBox) {
				getModel().setFileHeader(headerCheckBox.isSelected());
			} else if(src == keepWritingCheckBox) {
				getModel().setFileKeepWriting(keepWritingCheckBox.isSelected());
			}
		}
	}
	
	private Listener listener = new Listener();
	private JRadioButton enableButton = new JRadioButton();
	private JRadioButton disableButton = new JRadioButton();
	private JLabel fileLabel = new JLabel();
	private JTextField fileField = new JTextField();
	private JButton selectButton = new JButton();
	private JCheckBox headerCheckBox = new JCheckBox();
	private JCheckBox keepWritingCheckBox = new JCheckBox();
	private JFileChooser chooser = JFileChoosers.create();
	
	public FilePanel(LogFrame frame) {
		super(frame);

		chooser.addChoosableFileFilter(chooser.getAcceptAllFileFilter());
		chooser.addChoosableFileFilter(TestVector.FILE_FILTER);
		chooser.setFileFilter(TestVector.FILE_FILTER);

		JPanel filePanel = new JPanel(new GridBagLayout());
		GridBagLayout gb = (GridBagLayout) filePanel.getLayout();
		GridBagConstraints gc = new GridBagConstraints();
		  gc.fill = GridBagConstraints.HORIZONTAL;
		gb.setConstraints(fileLabel, gc);
		filePanel.add(fileLabel);
		  gc.weightx = 1.0;
		gb.setConstraints(fileField, gc);
		filePanel.add(fileField);
		  gc.weightx = 0.0;
		gb.setConstraints(selectButton, gc);
		filePanel.add(selectButton);
		fileField.setEditable(false);
		fileField.setEnabled(false);
		
		setLayout(new GridBagLayout());
		gb = (GridBagLayout) getLayout();
		gc = new GridBagConstraints();
		  gc.gridx = 0;
		  gc.weightx = 1.0;
		  gc.gridy = GridBagConstraints.RELATIVE;
		JComponent glue, panel;
		glue = new JPanel(); gc.weighty = 1.0; gb.setConstraints(glue, gc); add(glue); gc.weighty = 0.0;
		ButtonGroup group = new ButtonGroup();
		group.add(enableButton);
		group.add(disableButton);
		panel = new JPanel();
		panel.setLayout(new java.awt.GridLayout(2, 1));
		panel.add(enableButton);
		panel.add(disableButton);
		gb.setConstraints(panel, gc); add(panel);

		glue = new JPanel(); gc.weighty = 1.0; gb.setConstraints(glue, gc); add(glue); gc.weighty = 0.0;
		  gc.fill = GridBagConstraints.HORIZONTAL;
		gb.setConstraints(filePanel, gc);      add(filePanel);
		  gc.fill = GridBagConstraints.NONE;
		glue = new JPanel(); gc.weighty = 1.0; gb.setConstraints(glue, gc); add(glue); gc.weighty = 0.0;
		panel = new JPanel();
		panel.setLayout(new java.awt.GridLayout(2, 1));
		panel.add(headerCheckBox);
		panel.add(keepWritingCheckBox);
		gb.setConstraints(panel, gc); add(panel);

		glue = new JPanel(); gc.weighty = 1.0; gb.setConstraints(glue, gc); add(glue); gc.weighty = 0.0;
		
		enableButton.addActionListener(listener);
	        disableButton.addActionListener(listener);
		selectButton.addActionListener(listener);
		headerCheckBox.addActionListener(listener);
	        keepWritingCheckBox.addActionListener(listener);
		modelChanged(null, getModel());
		localeChanged();
	}

	@Override
	public String getTitle() {
		return Strings.get("fileTab");
	}

	@Override
	public String getHelpText() {
		return Strings.get("fileHelp");
	}

	@Override
	public void localeChanged() {
		enableButton.setText(Strings.get("fileEnableButton"));
		disableButton.setText(Strings.get("fileDisableButton"));
		fileLabel.setText(Strings.get("fileLabel") + " ");
		selectButton.setText(Strings.get("fileSelectButton"));
		headerCheckBox.setText(Strings.get("fileHeaderCheck"));
	        keepWritingCheckBox.setText(Strings.get("fileKeepWritingCheck"));	
	}

	@Override
	public void modelChanged(Model oldModel, Model newModel) {
		if (oldModel != null) oldModel.removeModelListener(listener);
		if (newModel != null) {
			newModel.addModelListener(listener);
			listener.filePropertyChanged(null);
		}
	}

    public void resetLog() {
	getModel().setFile((File)null);
	headerCheckBox.setSelected(true);
	keepWritingCheckBox.setSelected(true);
	getModel().setFileHeader(true);
    }
}
