package eventgen;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;

/** TreeGenerator is an efficient implementation of the EventGenerator
 * interface in the case where the probability of a given random variable being
 * true is small.  The implementation uses Fenwick trees containing
 * probabilities of the subtrees *not* generating an event.
 *
 * This implementation uses assertions aggressively and therefore runs much
 * more slowly when assertions are enabled.
 *
 * @author Andrew Myers
 *
 * @param <Variable> The kind of variables used.
 * @param <Alloc> An allocator for arrays of events.
 */
public class TreeGenerator<Variable, Alloc extends TreeGenerator.ArrayAllocator<Variable>>
		implements EventGenerator<Variable>
{
	static final int INIT_SIZE = 16;
	Random rnd;
	Alloc alloc;
	double[] ftree = new double[INIT_SIZE];
	Variable[] events; // Events corresponding to each of the entries in ftree.
	                   // Non-null only for leaves.
	int entries = 0;   // The number of entries in the tree. Must be odd. If
	                   // there are n events, entries = 2*n+1
	HashMap<Variable, Integer> ev_index; // maps an event to its index in ftree

	/** Check the representation invariant for this class. Throw
	 * Error if it is not satisfied. */
	protected boolean RI() throws Error {
		assert entries >= 0;
		assert entries == 0 || entries % 2 == 1;
		int n = size();
		for (int i = entries - n; i < entries; i++)
			assert(events[i] != null);
		for (int i = 0; i < entries-n; i++)
			assert ftree[i] == ftree[2*i+1] * ftree[2*i+2];
		for (Entry<Variable, Integer> entry : ev_index.entrySet()) {
			int v = entry.getValue();
			assert entries - n <= v && v < entries;
			assert events[v] == entry.getKey();
		}
		assert ftree.length == events.length;
		return true;
	}

	public interface ArrayAllocator<T> {
		T[] newArray(int n);
	}
	
	public TreeGenerator(Alloc a, Random rnd_) {
		events = a.newArray(INIT_SIZE);
		alloc = a;
		assert rnd_ != null;
		rnd = rnd_;
		ev_index = new HashMap<Variable, Integer>();
		assert RI();
	}

	public int size() {
		return (entries+1)/2;	
	}
	
	public boolean contains(Variable e) {
		assert RI();
		return ev_index.containsKey(e);
	}

	public void removeVariable(Variable e) {
		assert RI();
		assert contains(e);
		int n = ev_index.get(e);
		ev_index.remove(e);
		if (entries == 1) {
			entries = 0;
			assert RI(); return;
		}
		if (n >= entries-2) { // removing one of the last two.
			int kept = (entries-1) + (entries-2) - n;
			int parent = (n-1)>>1;
			ftree[parent] = ftree[kept];
			events[parent] = events[kept];
			ev_index.put(events[parent], parent);
			entries -= 2;
			fixProbabilities(parent);
			assert RI(); return;
		}
		int p2 = (entries-2)>>1;
		Variable eL = events[entries-1];
		Variable eR = events[entries-2];
		events[n] = eL;
		ftree[n] = ftree[entries-1];
		fixProbabilities(n);
		events[p2] = eR;
		ftree[p2] = ftree[entries-2];
		ev_index.put(eL, n);
		ev_index.put(eR, p2);
		fixProbabilities(p2);
		entries -= 2;
		assert RI();
	}

	private void grow(int n) {
		int len = ftree.length;
		if (entries + n < len) return;
		double[] t = new double[len * 2];
		Variable[] ne = alloc.newArray(len*2);
		for (int i = 0; i < entries; i++) {
			t[i] = ftree[i];
			ne[i] = events[i];
		}
		ftree = t;
		events = ne;
		assert RI();
	}

	public void addVariable(Variable e, double p) {
		assert RI();
		assert !contains(e);
		int parent = (entries-1)>>1;
		int n = entries;
		grow(2);
		ev_index.put(e, n);
		events[n] = e; ftree[n] = 1.0-p;
		ev_index.put(e, n);
		if (++entries == 1) { assert RI(); return; }
		Variable ep = events[parent];
		entries++;
		events[n+1] = ep;
		ev_index.put(ep, n+1);
		ftree[n+1] = ftree[parent];
		events[parent] = null;
		fixProbabilities(n);
		assert RI();
	}
	
	/** Fix the probabilities in the Fenwick tree for all the ancestors of n */
	protected void fixProbabilities(int n) {
		int parent = (n-1)>>1;
		while (parent >= 0) {
			ftree[parent] = ftree[parent * 2 + 1] * ftree[parent * 2 + 2];
			parent = (parent-1)>>1;
		}
	}

	public void setProbability(Variable e, double p) {
		assert RI();
		int n = ev_index.get(e);
		ftree[n] = 1.0-p;
		fixProbabilities(n);
		assert RI();
	}

	public Collection<Variable> sample() {
		return sample(0, 1.0);
	}
	/** return a set of events under a particular tree node
	 *  that sampled to be true, under the condition that at 
	 *  either least one event is generated or n == 0.
	 * @param n  The index of the tree node. 
	 */
	private Collection<Variable> sample(int n, double t) {
		assert RI();
		double s = rnd.nextDouble();
		double p = ftree[n];
		if (t != 1.0) p = Math.pow(p, t);
		if (n==0) {
			if (s < p) return null;
			s -= p;
		} else {
			s *= (1.0-p); // condition on an event being required.
		}
		if (2*n+2 >= entries) { // leaf: generate this event.
			Collection<Variable> ret = new HashSet<Variable>();
			ret.add(events[n]);
			return ret;
		}
		double l = ftree[2*n+1];
		double r = ftree[2*n+2];
		if (t != 1.0)  {
			l = Math.pow(l,t);
			r = Math.pow(r, t);
		}
		if (s < l*(1.0-r)) return sample(2*n+2, t);
		s -= l*(1.0-r);
		Collection<Variable> le = sample(2*n+1, t);
		if (s < r*(1.0-l)) return le;
		Collection<Variable> re = sample(2*n+2, t);
		if (le == null) le = re;
		else if (re != null) le.addAll(re);
		return le;
	}

	public Collection<Variable> sample(double t) {
		return sample(0, t);
	}
	
	/* Create a copy of this object. Does not copy the contained Variables. */
	public TreeGenerator(TreeGenerator<Variable, Alloc> gen) {
		rnd = gen.rnd;
		alloc = gen.alloc;
		ftree = Arrays.copyOf(gen.ftree, gen.ftree.length);
		events = Arrays.copyOf(gen.events, gen.events.length);
		entries = gen.entries;
		ev_index = (HashMap<Variable, Integer>)gen.ev_index.clone();
	}
}
// vim: ts=4
