/*
 * Copyright (c) 1988, 1989, 1991, 1994, 1995, 1996, 1997
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static const char copyright[] =
    "@(#) Copyright (c) 1988, 1989, 1991, 1994, 1995, 1996, 1997\n\
The Regents of the University of California.  All rights reserved.\n";
static const char rcsid[] =
    "@(#)$Header: /fsys/sim1/z/estan/argus/eye/ftraceroute.c,v 1.3 1999/04/18 21:24:25 haye Exp $ (LBL)";
#endif

/*
 * ftraceroute host  - trace the route ip packets follow going to "host".
 *
 * Attempt to trace the route an ip packet would follow to some
 * internet host.  We find out intermediate hops by launching probe
 * packets with a small ttl (time to live) then listening for an
 * icmp "time exceeded" reply from a gateway.  We start our probes
 * with a ttl of one and increase by one until we get an icmp "port
 * unreachable" (which means we got to "host") or hit a max (which
 * defaults to 30 hops & can be changed with the -m flag).  Three
 * probes (change with -q flag) are sent at each ttl setting and a
 * line is printed showing the ttl, address of the gateway and
 * round trip time of each probe.  If the probe answers come from
 * different gateways, the address of each responding system will
 * be printed.  If there is no response within a 5 sec. timeout
 * interval (changed with the -w flag), a "*" is printed for that
 * probe.
 *
 * Probe packets are UDP format.  We don't want the destination
 * host to process them so the destination port is set to an
 * unlikely value (if some clod on the destination is using that
 * value, it can be changed with the -p flag).
 *
 * A sample use might be:
 *
 *     [yak 71]% traceroute nis.nsf.net.
 *     traceroute to nis.nsf.net (35.1.1.48), 30 hops max, 56 byte packet
 *      1  helios.ee.lbl.gov (128.3.112.1)
 *      2  lilac-dmc.Berkeley.EDU (128.32.216.1)
 *      3  lilac-dmc.Berkeley.EDU (128.32.216.1)
 *      4  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)
 *      5  ccn-nerif22.Berkeley.EDU (128.32.168.22)
 *      6  128.32.197.4 (128.32.197.4)
 *      7  131.119.2.5 (131.119.2.5)
 *      8  129.140.70.13 (129.140.70.13)
 *      9  129.140.71.6 (129.140.71.6)
 *     10  129.140.81.7 (129.140.81.7)
 *     11  nic.merit.edu (35.1.1.48)
 *
 * Other possible annotations after the time are !H, !N, !P (got a host,
 * network or protocol unreachable, respectively), !S or !F (source
 * route failed or fragmentation needed -- neither of these should
 * ever occur and the associated gateway is busted if you see one).  If
 * almost all the probes result in some kind of unreachable, traceroute
 * will give up and exit.
 *
 * Notes
 * -----
 * This program must be run by root or be setuid.  (I suggest that
 * you *don't* make it setuid -- casual use could result in a lot
 * of unnecessary traffic on our poor, congested nets.)
 *
 * This program requires a kernel mod that does not appear in any
 * system available from Berkeley:  A raw ip socket using proto
 * IPPROTO_RAW must interpret the data sent as an ip datagram (as
 * opposed to data to be wrapped in a ip datagram).  See the README
 * file that came with the source to this program for a description
 * of the mods I made to /sys/netinet/raw_ip.c.  Your mileage may
 * vary.  But, again, ANY 4.x (x < 4) BSD KERNEL WILL HAVE TO BE
 * MODIFIED TO RUN THIS PROGRAM.
 *
 * The udp port usage may appear bizarre (well, ok, it is bizarre).
 * The problem is that an icmp message only contains 8 bytes of
 * data from the original datagram.  8 bytes is the size of a udp
 * header so, if we want to associate replies with the original
 * datagram, the necessary information must be encoded into the
 * udp header (the ip id could be used but there's no way to
 * interlock with the kernel's assignment of ip id's and, anyway,
 * it would have taken a lot more kernel hacking to allow this
 * code to set the ip id).  So, to allow two or more users to
 * use traceroute simultaneously, we use this task's pid as the
 * source port (the high bit is set to move the port number out
 * of the "likely" range).  To keep track of which probe is being
 * replied to (so times and/or hop counts don't get confused by a
 * reply that was delayed in transit), we increment the destination
 * port number before each probe.
 *
 * Don't use this as a coding example.  I was trying to find a
 * routing problem and this code sort-of popped out after 48 hours
 * without sleep.  I was amazed it ever compiled, much less ran.
 *
 * I stole the idea for this program from Steve Deering.  Since
 * the first release, I've learned that had I attended the right
 * IETF working group meetings, I also could have stolen it from Guy
 * Almes or Matt Mathis.  I don't know (or care) who came up with
 * the idea first.  I envy the originators' perspicacity and I'm
 * glad they didn't keep the idea a secret.
 *
 * Tim Seaver, Ken Adelman and C. Philip Wood provided bug fixes and/or
 * enhancements to the original distribution.
 *
 * I've hacked up a round-trip-route version of this that works by
 * sending a loose-source-routed udp datagram through the destination
 * back to yourself.  Unfortunately, SO many gateways botch source
 * routing, the thing is almost worthless.  Maybe one day...
 *
 *  -- Van Jacobson (van@ee.lbl.gov)
 *
 * Optimization thru batching:
 * 
 * The idea is to send the traceroute probes in batches so as to get 
 * rid of most of the timeout.
 * For example, in the original traceroute by Van Jacobson, probes with
 * ttl 1 is sent, and then the program waits until the reply to it is
 * received from the first router on the route. Then the next probe with
 * ttl 2 is sent, and then wait, and then tll 3, and then wait...
 *
 * Instead of doing that sequentially, this ftraceroute send a batch of
 * 10 probes with ttl from 1 to 10. Then it waits for replies from all
 * the routers on the route. Whoever replies will be processed right 
 * away.
 * The resulting timeout for the first batch is greatly reduced and 
 * approaches rtt(10), where rtt(i) denotes the round-trip-time from 
 * the i'th router.
 *
 * Of course, subsequent batches will be sent if the destination is not
 * reached in the first batch. The program will returns when the 
 * destination is reached. 
 *
 * For the same reason as Van Jacobson's, you should try not to read
 * the code if possible. Maybe the most annoying part would be the my
 * naming style is quite different from Van Jacobson's. I tend
 * to use long names with cases.
 *
 * Similarly, I cannot steal the honor of this idea from 
 * Prof. S. Keshav. 
 *
 *  -- Haye Chan (haye@cs.cornell.edu)
 *     Tue Dec 20 03:50:13 PST 1988
 */

#include <sys/param.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#include <sys/socket.h>
#include <sys/time.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/ip_icmp.h>
#include <netinet/udp.h>
#include <netinet/udp_var.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <memory.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include "ifaddrlist.h"
#include "savestr.h"
#include "util.h"

/* Maximum number of gateways (include room for one noop) */
#define NGATEWAYS ((int)((MAX_IPOPTLEN - IPOPT_MINOFF - 1) / sizeof(u_int32_t)))

/* check whether a is within [base, base+range) */
#define VALID(a, base, range) (((a) >= (base)) && ((a) < ((base)+(range))))
#define min(a, b) ((a) < (b)? a:b)

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN	64
#endif

/* Data section of the probe packet */
struct outdata {
	u_char seq;		/* sequence number of this packet */
	u_char ttl;		/* ttl packet left with */
	struct timeval tv;	/* time packet left */
};

u_char	traceroute_packet[512];		/* last inbound (icmp) packet */

struct ip *outip;		/* last output (udp) packet */
struct udphdr *outudp;		/* last output (udp) packet */
struct outdata *outdata;	/* last output (udp) packet */

struct icmp *outicmp;		/* last output (icmp) packet */

/* loose source route gateway list (including room for final destination) */
u_int32_t gwlist[NGATEWAYS + 1];

int s;				/* receive (icmp) socket file descriptor */
int sndsock;			/* send (udp/icmp) socket file descriptor */

struct sockaddr whereto;	/* Who to try to reach */
struct sockaddr_in wherefrom;	/* Who we are */
int packlen;			/* total length of packet */
int minpacket;			/* min ip packet size */
int maxpacket;	/* max ip packet size */

char *g_prog;
char *source;
char *hostname;
char *device;

int nprobes;
int nSent;
int nReceived;
int max_ttl;
int ttlPerBatch;
int first_ttl;
u_short ident;
u_short port;	/* start udp dest port # for probe packets */

int options;			/* socket options */
int verbose;
int waittime;		/* time to wait for response (in us) */
int nflag;			/* print addresses numerically */
int useicmp;			/* use icmp echo instead of udp packets */
int docksum;		/* don't calculate checksums */
int optlen;			/* length of ip options */

char *g_answer;
int g_nAnswer;
struct in_addr *g_addrs;

extern int optind;
extern int opterr;
extern char *optarg;

/* Forwards */
int	main(int, char **);
int	packet_ok(u_char *, int, struct sockaddr_in *, int, int);
int	process(u_char *, int, struct sockaddr_in *, 
		struct in_addr *, int);
int	send_probe(int, int);
__dead	void ftraceroute_usage(void);
int	wait_for_reply(int, struct sockaddr_in *);
void    SendNProbes(int, int);
int     WaitForBatch(int, struct sockaddr_in *, int, int, int *, int *);
void    WrapUp();
void    PrintBatch(int startTTL, int minDestSeq);
int     BatchDone(int nRepliesInBatch, int startTTL, int minDestSeq);

int ftraceroute(int argc, char **argv, char *answer, int nAnswer)
{
	register int op, code, n;
	register char *cp;
	register u_char *outp;
	register u_int32_t *ap;
	register struct sockaddr_in *from = &wherefrom;
	register struct sockaddr_in *to = (struct sockaddr_in *)&whereto;
	register struct hostinfo *hi;
	int on = 1;
	register struct protoent *pe;
	register int ttl, probe, i;
	register int seq = 0;
	int tos = 0, settos = 0;
	register int lsrr = 0;
	register u_short off = 0;
	struct ifaddrlist *al;
	char errbuf[132];
	int reached = 0;
	time_t timeid;
	int len;
	int minDestSeq;
	int nRepliesInThisBatch;

	nSent = 0;
	nReceived = 0;
	s = 0;
	sndsock = 0;
	maxpacket = 32 * 1024;	/* max ip packet size */
	nprobes = 1;
	max_ttl = 30;
	ttlPerBatch = 10;
	first_ttl = 1;
	port = 32768 + 666;	/* start udp dest port # for probe packets */
	waittime = 1000000;		/* time to wait for response (in us) */

	source = NULL;
	*answer = (char) NULL;
	g_answer = answer;
	g_nAnswer = nAnswer;
	g_addrs = NULL;

#ifdef CANT_HACK_CKSUM
	docksum = 0;		/* don't calculate checksums */
#else
	docksum = 1;		/* calculate checksums */
#endif

	if ((cp = strrchr(argv[0], '/')) != NULL)
		g_prog = cp + 1;
	else
		g_prog = argv[0];

	opterr = 0;

	optind = 1;

	while ((op = getopt(argc, argv, "dFInrvxf:g:i:m:b:p:q:s:t:w:")) != EOF)
		switch (op) {

		case 'd':
			options |= SO_DEBUG;
			break;

		case 'f':
			first_ttl = str2val(optarg, "first ttl", 1, 255);
			break;

		case 'F':
			off = IP_DF;
			break;

		case 'g':
			if (lsrr >= NGATEWAYS) {
				PrintError2(
				    "%s: No more than %d gateways\n",
				    g_prog, NGATEWAYS);
				return 1;
			}
			getaddr(gwlist + lsrr, optarg);
			++lsrr;
			break;

		case 'i':
			device = optarg;
			break;

		case 'I':
			++useicmp;
			break;

		case 'm':
			max_ttl = str2val(optarg, "max ttl", 1, 255);
			break;

		case 'b':
		        ttlPerBatch = str2val(optarg, "batch size", 1, 255);
			break;

		case 'n':
			++nflag;
			break;

		case 'p':
			port = str2val(optarg, "port", 1, -1);
			break;

		case 'q':
			nprobes = str2val(optarg, "nprobes", 1, -1);
			break;

		case 'r':
			options |= SO_DONTROUTE;
			break;

		case 's':
			/*
			 * set the ip source address of the outbound
			 * probe (e.g., on a multi-homed host).
			 */
			source = optarg;
			break;

		case 't':
			tos = str2val(optarg, "tos", 0, 255);
			++settos;
			break;

		case 'v':
			++verbose;
			break;

		case 'x':
			docksum = (docksum == 0);
			break;

		case 'w':
			waittime = 1000 * str2val(optarg, "wait time", 1, -1);
			break;

		default:
			ftraceroute_usage();
			return 1;
		}

	if (first_ttl > max_ttl) {
		PrintError3(
		    "%s: first ttl (%d) may not be greater than max ttl (%d)\n",
		    g_prog, first_ttl, max_ttl);
		return 1;
	}

	if (!docksum)
		PrintError1("%s: Warning: ckecksums disabled\n", g_prog);

	if (lsrr > 0)
		optlen = (lsrr + 1) * sizeof(gwlist[0]);
	minpacket = sizeof(*outip) + sizeof(*outdata) + optlen;

	if (useicmp)
		minpacket += 8;			/* XXX magic number */
	else
		minpacket += sizeof(*outudp);

	if (packlen == 0)
		packlen = minpacket;		/* minimum sized packet */
	else if (minpacket > packlen || packlen > maxpacket) {
		PrintError3("%s: packet size must be %d <= s <= %d\n",
		    g_prog, minpacket, maxpacket);
		return 1;
	}

	/* Process destination and optional packet size */
	switch (argc - optind) {

	case 2:
		packlen = str2val(argv[optind + 1],
		    "packet length", minpacket, -1);
		/* Fall through */

	case 1:
		hostname = argv[optind];
		hi = gethostinfo(hostname);

		if (!hi)
		  return 1;
		setsin(to, hi->addrs[0]);
		if (hi->n > 1)
		  PrintError3(
		    "%s: Warning: %s has multiple addresses; using %s\n",
		    g_prog, hostname, inet_ntoa(to->sin_addr));
		hostname = hi->name;
		hi->name = NULL;
		freehostinfo(hi);
		break;

	default:
		ftraceroute_usage();
		return 1;
	}

	outip = (struct ip *)malloc((unsigned)packlen);
	if (outip == NULL) {
		PrintError2("%s: malloc: %s\n", g_prog, strerror(errno));
		return 1;
	}
	memset((char *)outip, 0, packlen);

	outip->ip_v = IPVERSION;
	if (settos)
		outip->ip_tos = tos;
#ifdef BYTESWAP_IP_LEN
	outip->ip_len = htons(packlen);
#else
	outip->ip_len = packlen;
#endif

	outip->ip_off = off;
	outp = (u_char *)(outip + 1);
#ifdef HAVE_RAW_OPTIONS
	if (lsrr > 0) {
		register u_char *optlist;

		optlist = outp;
		outp += optlen;

		/* final hop */
		gwlist[lsrr] = to->sin_addr.s_addr;

		outip->ip_dst.s_addr = gwlist[0];

		/* force 4 byte alignment */
		optlist[0] = IPOPT_NOP;
		/* loose source route option */
		optlist[1] = IPOPT_LSRR;
		i = lsrr * sizeof(gwlist[0]);
		optlist[2] = i + 3;
		/* Pointer to LSRR addresses */
		optlist[3] = IPOPT_MINOFF;
		memcpy(optlist + 4, gwlist + 1, i);
	} else
#endif
		outip->ip_dst = to->sin_addr;

	outip->ip_hl = (outp - (u_char *)outip) >> 2;
	ident = (getpid() & 0xffff) | 0x8000;
	if (useicmp) {
		outip->ip_p = IPPROTO_ICMP;

		outicmp = (struct icmp *)outp;
		outicmp->icmp_type = ICMP_ECHO;
		outicmp->icmp_id = htons(ident);

		outdata = (struct outdata *)(outp + 8);	/* XXX magic number */
	} else {
		outip->ip_p = IPPROTO_UDP;

		outudp = (struct udphdr *)outp;
		outudp->uh_sport = htons(ident);
		outudp->uh_ulen =
		    htons((u_short)(packlen - (sizeof(*outip) + optlen)));
		outdata = (struct outdata *)(outudp + 1);
	}

	cp = "icmp";
	if ((pe = getprotobyname(cp)) == NULL) {
		PrintError2("%s: unknown protocol %s\n", g_prog, cp);
		return 1;
	}
	if ((s = socket(AF_INET, SOCK_RAW, pe->p_proto)) < 0) {
		PrintError2("%s: icmp socket: %s\n", g_prog, strerror(errno));
		return 1;
	}
	if (options & SO_DEBUG)
		(void)setsockopt(s, SOL_SOCKET, SO_DEBUG, (char *)&on,
		    sizeof(on));
	if (options & SO_DONTROUTE)
		(void)setsockopt(s, SOL_SOCKET, SO_DONTROUTE, (char *)&on,
		    sizeof(on));

#ifndef __hpux
	sndsock = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
#else
	sndsock = socket(AF_INET, SOCK_RAW,
	    useicmp ? IPPROTO_ICMP : IPPROTO_UDP);
#endif
	if (sndsock < 0) {
		PrintError2("%s: raw socket: %s\n", g_prog, strerror(errno));
		WrapUp();
		return 1;
	}

	/* Revert to non-privileged user after opening sockets */
	setuid(getuid());

#if defined(IP_OPTIONS) && !defined(HAVE_RAW_OPTIONS)
	if (lsrr > 0) {
		u_char optlist[MAX_IPOPTLEN];

		cp = "ip";
		if ((pe = getprotobyname(cp)) == NULL) {
			PrintError2("%s: unknown protocol %s\n", g_prog, cp);
			WrapUp();
			return 1;
		}

		/* final hop */
		gwlist[lsrr] = to->sin_addr.s_addr;
		++lsrr;

		/* force 4 byte alignment */
		optlist[0] = IPOPT_NOP;
		/* loose source route option */
		optlist[1] = IPOPT_LSRR;
		i = lsrr * sizeof(gwlist[0]);
		optlist[2] = i + 3;
		/* Pointer to LSRR addresses */
		optlist[3] = IPOPT_MINOFF;
		memcpy(optlist + 4, gwlist, i);

		if ((setsockopt(sndsock, pe->p_proto, IP_OPTIONS, optlist,
		    i + sizeof(gwlist[0]))) < 0) {
			PrintError2("%s: IP_OPTIONS: %s\n",
			    g_prog, strerror(errno));
			WrapUp();
			return 1;
		    }
	}
#endif

#ifdef SO_SNDBUF
	if (setsockopt(sndsock, SOL_SOCKET, SO_SNDBUF, (char *)&packlen,
	    sizeof(packlen)) < 0) {
		PrintError2("%s: SO_SNDBUF: %s\n", g_prog, strerror(errno));
		WrapUp();
		return 1;
	}
#endif
#ifdef IP_HDRINCL
	if (setsockopt(sndsock, IPPROTO_IP, IP_HDRINCL, (char *)&on,
	    sizeof(on)) < 0) {
		PrintError2("%s: IP_HDRINCL: %s\n", g_prog, strerror(errno));
		WrapUp();
		return 1;
	}
#else
#ifdef IP_TOS
	if (settos && setsockopt(sndsock, IPPROTO_IP, IP_TOS,
	    (char *)&tos, sizeof(tos)) < 0) {
		PrintError3("%s: setsockopt tos %d: %s\n",
		    g_prog, tos, strerror(errno));
		WrapUp();
		return 1;
	}
#endif
#endif
	if (options & SO_DEBUG)
		(void)setsockopt(sndsock, SOL_SOCKET, SO_DEBUG, (char *)&on,
		    sizeof(on));
	if (options & SO_DONTROUTE)
		(void)setsockopt(sndsock, SOL_SOCKET, SO_DONTROUTE, (char *)&on,
		    sizeof(on));

	/* Get the interface address list */
	n = ifaddrlist(&al, errbuf);
	if (n < 0) {
		PrintError2("%s: ifaddrlist: %s\n", g_prog, errbuf);
		WrapUp();
		return 1;
	}
	if (n == 0) {
		PrintError1(
		    "%s: Can't find any network interfaces\n", g_prog);
		WrapUp();
		return 1;
	}

	/* Look for a specific device */
	if (device != NULL) {
		for (i = n; i > 0; --i, ++al)
			if (strcmp(device, al->device) == 0)
				break;
		if (i <= 0) {
			PrintError2("%s: Can't find interface %s\n",
			    g_prog, device);
			WrapUp();
			return 1;
		}
	}

	/* Determine our source address */
	if (source == NULL) {
		/*
		 * If a device was specified, use the interface address.
		 * Otherwise, use the first interface found.
		 * Warn if there are more than one.
		 */
		setsin(from, al->addr);
		if (n > 1 && device == NULL) {
			PrintError3(
		    "%s: Warning: Multiple interfaces found; using %s @ %s\n",
			    g_prog, inet_ntoa(from->sin_addr), al->device);
		}
	} else {
		hi = gethostinfo(source);
		if (!hi)
		  return 1;
		source = hi->name;
		hi->name = NULL;
		if (device == NULL) {
			/*
			 * Use the first interface found.
			 * Warn if there are more than one.
			 */
			setsin(from, hi->addrs[0]);
			if (hi->n > 1)
				PrintError3(
			"%s: Warning: %s has multiple addresses; using %s\n",
				    g_prog, source, inet_ntoa(from->sin_addr));
		} else {
			/*
			 * Make sure the source specified matches the
			 * interface address.
			 */
			for (i = hi->n, ap = hi->addrs; i > 0; --i, ++ap)
				if (*ap == al->addr)
					break;
			if (i <= 0) {
				PrintError3(
				    "%s: %s is not on interface %s\n",
				    g_prog, source, device);
				WrapUp();
				return 1;
			}
			setsin(from, *ap);
		}
		freehostinfo(hi);
	}
	outip->ip_src = from->sin_addr;
#ifndef IP_HDRINCL
	if (bind(sndsock, (struct sockaddr *)from, sizeof(*from)) < 0) {
		PrintError2("%s: bind: %s\n",
		    g_prog, strerror(errno));
		WrapUp();
		return 1;
	}
#endif

	PrintError3("%s to %s (%s)",
		    g_prog, hostname, inet_ntoa(to->sin_addr));
	if (source)
	  PrintError1(" from %s", source);
	PrintError2(", %d hops max, %d byte packets\n", max_ttl, packlen);

	/* init the array that stores addresses of routers that replied */
	len = sizeof(struct in_addr) * ttlPerBatch;
	g_addrs = malloc(len);
	if (!g_addrs)
	  return -1;

	for (ttl = first_ttl; ttl < max_ttl; ttl += ttlPerBatch) {
	  bzero(g_addrs, len);
	  nRepliesInThisBatch = 0;
	  for (i=0; i<nprobes && !reached; i++) {
	    SendNProbes(ttl, ttlPerBatch);
	    reached = WaitForBatch(s, from, ttl, ttlPerBatch, 
				   &nRepliesInThisBatch, &minDestSeq);
	    MyPrintf1("reached = %d\n", reached);
	  }
	  PrintBatch(ttl, minDestSeq);
	  /* end the traceroute when either host reached or no replies in this
	     batch */
	  if (reached || !nRepliesInThisBatch)
	    break;
	  /* waittime should be double as we go further in the next batch */
	  /* maybe not! */
	  /* waittime *= 2; */
	}

	WrapUp();
	return 0;
}

/* send n probes from ttl=startTTL */
void SendNProbes(int startTTL, int n)
{
  int ttl;
  int i;

  ttl = startTTL;
  for (i=0; i<n; i++) {
    if (EMPTY_IN_ADDR(g_addrs+i)) {
      send_probe(ttl, ttl);
    }
    ttl++;
  }
  MyPrintf2("##probes %d - %d sent\n", startTTL, ttl - 1);
}

/* helper procedure for WaitForBatch
 * process the code returned by packet_ok */
void ProcessCode(int i, int *got_there, int *unreachable)
{
    int code;
    struct ip *ip;

    code = i - 1;
    switch (code) {

    case ICMP_UNREACH_PORT:
#ifndef ARCHAIC
      ip = (struct ip *)traceroute_packet;
      if (ip->ip_ttl <= 1)
	Printf(" !");
#endif
      ++(*got_there);
      break;
      
    case ICMP_UNREACH_NET:
      ++(*unreachable);
      Printf(" !N");
      break;
      
    case ICMP_UNREACH_HOST:
      ++(*unreachable);
      Printf(" !H");
      break;
      
    case ICMP_UNREACH_PROTOCOL:
      ++(*got_there);
      Printf(" !P");
      break;
      
    case ICMP_UNREACH_NEEDFRAG:
      ++(*unreachable);
      Printf(" !F");
      break;

    case ICMP_UNREACH_SRCFAIL:
      ++(*unreachable);
      Printf(" !S");
      break;

/* rfc1716 */
#ifndef ICMP_UNREACH_FILTER_PROHIB
#define ICMP_UNREACH_FILTER_PROHIB	13	/* admin prohibited filter */
#endif
    case ICMP_UNREACH_FILTER_PROHIB:
      ++(*unreachable);
      Printf(" !X");
      break;
      
    default:
      ++(*unreachable);
      Printf1(" !<%d>", code);
      break;
    }
}

/*
 * startTTL - ttl of the first probe in this batch
 * batchSize - number of probes to be sent in this batch
 * nRepliesInBatch - number of replies from current batch so far
 * minDestSeq - minimum destination sequence number so far:
 *              we would only print the routes up to this sequence number
 * return true iff reached destination
 */
int WaitForBatch(int socket, struct sockaddr_in * from, 
		 int startTTL, int batchSize,
		 int *nRepliesInBatch, int *minDestSeq)
     /* minDestSeq - smallest sequence number reaching destination */
     /* this is for knowing which is the last packet we should print */
{
  int cc, i, code, len;
  int seq;
  int got_there;
  int unreachable;
  struct ip *ip;

  got_there = 0; /* flag that will be set whenever the dest reply */
  unreachable = 0; /* flag that will be set when someone say unreachable */
  *minDestSeq = startTTL + batchSize - 1;

  /* keep reading packets until wait_for_reply return <= 0,
   *   or when we have received enough replies
   * nRepliesExpected is initially the batchSize,
   * then it would be maintained at maxDestSeq - startTTL + 1
   */
  while (!BatchDone(*nRepliesInBatch, startTTL, *minDestSeq)) {
    cc = wait_for_reply(socket, from);
    if (cc <= 0)
      break;
    nReceived++;

    i = packet_ok(traceroute_packet, cc, from, startTTL, batchSize);
    if (!i)
      continue;

    seq = process(traceroute_packet, cc, from, g_addrs, startTTL);
    if (!seq)
      continue;

    (*nRepliesInBatch)++;
    MyPrintf4("packet received: i = %d, seq = %d, nRepliesInBatch = %d/%d\n",
	     i, seq, *nRepliesInBatch, *minDestSeq);

    if (i == -2) {
#ifndef ARCHAIC
      ip = (struct ip *)traceroute_packet;
      if (ip->ip_ttl <= 1)
	Printf(" !");
#endif
      got_there++;
      *minDestSeq = min(seq, *minDestSeq);
      continue;
    }
    /* time exceeded in transit */
    if (i == -1) {
      continue;
    }

    ProcessCode(i, &got_there, &unreachable);

    if (got_there ||
	(unreachable > 0 && unreachable >= nprobes - 1))
      *minDestSeq = min(seq, *minDestSeq);
  }

  return got_there;
}

int
wait_for_reply(register int sock, register struct sockaddr_in *fromp)
{
	fd_set fds;
	struct timeval now, wait;
	struct timezone tz;
	register int cc = 0;
	int fromlen = sizeof(*fromp);

	FD_ZERO(&fds);
	FD_SET(sock, &fds);

	wait.tv_sec = waittime / 1000000;
	wait.tv_usec = waittime % 1000000;

	if (select(sock + 1, &fds, NULL, NULL, &wait) > 0)
	  cc = recvfrom(sock, (char *)traceroute_packet, 
			sizeof(traceroute_packet), 0,
			(struct sockaddr *)fromp, &fromlen);
	return(cc);
}

int
send_probe(register int seq, int ttl)
{
	register int cc;
	register struct udpiphdr * ui;
	struct ip tip;
	struct timezone tz;

	outip->ip_ttl = ttl;
#ifndef __hpux
	outip->ip_id = htons(ident + seq);
#endif

	/*
	 * In most cases, the kernel will recalculate the ip checksum.
	 * But we must do it anyway so that the udp checksum comes out
	 * right.
	 */
	if (docksum) {
		outip->ip_sum =
		    in_cksum((u_short *)outip, sizeof(*outip) + optlen);
		if (outip->ip_sum == 0)
			outip->ip_sum = 0xffff;
	}

	/* Payload */
	outdata->seq = seq;
	outdata->ttl = ttl;
	gettimeofday(&(outdata->tv), &tz);

	if (useicmp)
		outicmp->icmp_seq = htons(seq);
	else
		outudp->uh_dport = htons(port + seq);

	/* (We can only do the checksum if we know our ip address) */
	if (docksum) {
		if (useicmp) {
			outicmp->icmp_cksum = 0;
			outicmp->icmp_cksum = in_cksum((u_short *)outicmp,
			    packlen - (sizeof(*outip) + optlen));
			if (outicmp->icmp_cksum == 0)
				outicmp->icmp_cksum = 0xffff;
		} else {
			/* Checksum (must save and restore ip header) */
			tip = *outip;
			ui = (struct udpiphdr *)outip;
			ui->ui_next = 0;
			ui->ui_prev = 0;
			ui->ui_x1 = 0;
			ui->ui_len = outudp->uh_ulen;
			outudp->uh_sum = 0;
			outudp->uh_sum = in_cksum((u_short *)ui, packlen);
			if (outudp->uh_sum == 0)
				outudp->uh_sum = 0xffff;
			*outip = tip;
		}
	}

	/* XXX undocumented debugging hack */
	if (verbose > 1) {
		register const u_short *sp;
		register int nshorts, i;

		sp = (u_short *)outip;
		nshorts = (u_int)packlen / sizeof(u_short);
		i = 0;
		Printf1("[ %d bytes", packlen);
		while (--nshorts >= 0) {
			if ((i++ % 8) == 0)
				Printf("\n\t");
			Printf1(" %04x", ntohs(*sp++));
		}
		if (packlen & 1) {
			if ((i % 8) == 0)
				Printf("\n\t");
			Printf1(" %02x", *(u_char *)sp);
		}
		Printf("]\n");
	}

#if !defined(IP_HDRINCL) && defined(IP_TTL)
	if (setsockopt(sndsock, IPPROTO_IP, IP_TTL,
	    (char *)&ttl, sizeof(ttl)) < 0) {
		PrintError3("%s: setsockopt ttl %d: %s\n",
		    g_prog, ttl, strerror(errno));
		return 1;
	}
#endif

#ifdef __hpux
	cc = sendto(sndsock, useicmp ? (char *)outicmp : (char *)outudp,
	    packlen - (sizeof(*outip) + optlen), 0, &whereto, sizeof(whereto));
	if (cc > 0)
		cc += sizeof(*outip) + optlen;
#else
	cc = sendto(sndsock, (char *)outip,
	    packlen, 0, &whereto, sizeof(whereto));
#endif
	if (cc < 0 || cc != packlen)  {
		if (cc < 0)
			PrintError2("%s: sendto: %s\n",
			    g_prog, strerror(errno));
		Printf4("%s: wrote %s %d chars, ret=%d\n",
		    g_prog, hostname, packlen, cc);
		(void)fflush(stdout);
	}
	nSent++;
	return 0;
}

int
packet_ok(register u_char *buf, int cc, register struct sockaddr_in *from,
    register int start, int range)
{
	register struct icmp *icp;
	register u_char type, code;
	register int hlen;
#ifndef ARCHAIC
	register struct ip *ip;
	register struct ip *hip;
	register struct udphdr *up;
	register struct icmp *hicmp;

	ip = (struct ip *) buf;
	
	hlen = ip->ip_hl << 2;
	if (cc < hlen + ICMP_MINLEN) {
		if (verbose)
			Printf2("packet too short (%d bytes) from %s\n", cc,
				inet_ntoa(from->sin_addr));
		return (0);
	}
	cc -= hlen;
	icp = (struct icmp *)(buf + hlen);
#else
	icp = (struct icmp *)buf;
#endif

	type = icp->icmp_type;
	code = icp->icmp_code;
	if ((type == ICMP_TIMXCEED && code == ICMP_TIMXCEED_INTRANS) ||
	    type == ICMP_UNREACH || type == ICMP_ECHOREPLY) {

		hip = &icp->icmp_ip;
		hlen = hip->ip_hl << 2;
		if (useicmp) {
		        if (icp->icmp_id != htons(ident))
			  return 0;
			/* XXX */
			if (type == ICMP_ECHOREPLY &&
			    icp->icmp_id == htons(ident) &&
			    VALID(ntohs(icp->icmp_seq), start, range))
				return (-2);

			hicmp = (struct icmp *)((u_char *)hip + hlen);
			/* XXX 8 is a magic number */
			if (hlen + 8 <= cc &&
			    hip->ip_p == IPPROTO_ICMP &&
			    hicmp->icmp_id == htons(ident) &&
			    VALID(ntohs(hicmp->icmp_seq), start, range))
				return (type == ICMP_TIMXCEED ? -1 : code + 1);
		} else {
			up = (struct udphdr *)((u_char *)hip + hlen);
			if (up->uh_sport != htons(ident))
			  return 0;

			/* XXX 8 is a magic number */

			MyPrintf4("port = %d, start = %d, range = %d, dport = %d\n",
				  port, start, range, ntohs(up->uh_dport));
			if (hlen + 12 <= cc &&
			    hip->ip_p == IPPROTO_UDP &&
			    up->uh_sport == htons(ident) &&
			    VALID(ntohs(up->uh_dport), port + start, range))
				return (type == ICMP_TIMXCEED ? -1 : code + 1);
		}
	}
#ifndef ARCHAIC
	if (verbose) {
		register int i;
		u_int32_t *lp = (u_int32_t *)&icp->icmp_ip;

		Printf2("\n%d bytes from %s to ", cc, inet_ntoa(from->sin_addr));
		Printf4("%s: icmp type %d (%s) code %d\n",
		    inet_ntoa(ip->ip_dst), type, pr_type(type), icp->icmp_code);
		for (i = 4; i < cc ; i += sizeof(*lp))
			Printf2("%2d: x%8.8x\n", i, *lp++);
	}
#endif
	return(0);
}

// return the sequence number of the packet if we got a new sequence number
// otherwise return 0
int
process(register u_char *buf, register int cc, 
	register struct sockaddr_in *from,
	struct in_addr *addrs, int startTTL)
{
	register struct ip *ip;
	struct icmp *icp;
	struct ip *hip;
	struct udphdr *up;
	int hlen, hlen2;
	int seq;

	ip = (struct ip *) buf;
	hlen = ip->ip_hl << 2;
	cc -= hlen;
	icp = (struct icmp *)(buf + hlen);
	hip = &icp->icmp_ip;
	hlen2 = hip->ip_hl << 2;
	up = (struct udphdr *)((u_char *) hip + hlen2);
	seq = ntohs(up->uh_dport) - port;

	if (verbose)
		Printf2(" %d bytes to %s", cc, inet_ntoa (ip->ip_dst));

	if (!EMPTY_IN_ADDR(addrs+seq-startTTL))
	  return 0;
	else {
	  addrs[seq-startTTL] = from->sin_addr;
	  return seq;
	}
}

__dead void
ftraceroute_usage(void)
{
	extern char version[];

	Printf1("Version %s\n", version);
	Printf1("Usage: %s [-dFInrvx] [-g gateway] [-i iface] \
[-f first_ttl]\n\t[-m max_ttl] [ -b batch_size] [ -p port] [-q nqueries] \
[-s src_addr] \n\t[-t tos] [-w waittime] host [packetlen]\n",
	    g_prog);
	return;
}

/* close opened sockets, free memory, print statistics */
void WrapUp()
{
  if (s) {
    close(s);
    s = NULL;
  }
  if (sndsock) {
    close(sndsock);
    sndsock = NULL;
  }
  if (g_addrs) {
    free(g_addrs);
    g_addrs = NULL;
  }
  Printf("-- Tracerotue Statistics --\n");
  Printf3("%d packets transmitted, %d packets received, %d%% packet loss\n",
	 nSent, nReceived, (nSent - nReceived) * 100 / nSent);
}

void PrintBatch(int startTTL, int minDestSeq)
{
  struct in_addr *curBatchAddr;
  int seq;
  
  curBatchAddr = g_addrs;
  for (seq = startTTL; seq <= minDestSeq; seq++, curBatchAddr++)
    if (!EMPTY_IN_ADDR(curBatchAddr)) {
      if (nflag)
	Printf2("%2d %s\n", seq, inet_ntoa(*(curBatchAddr)));
      else
	Printf3("%2d %s (%s)\n", seq, 
		inetname(*(curBatchAddr)), inet_ntoa(*curBatchAddr));
    } else {
      Printf1("%2d *\n", seq);
    }
}

int BatchDone(int nRepliesInBatch, int startTTL, int minDestSeq)
{
  int i;
  int nExpected = minDestSeq - startTTL + 1;

  MyPrintf1("In BatchDone : nRepliesInBatch = %d\n", nRepliesInBatch);
  /* if number of replies is less than expected, for sure the batch is 
   * not done yet */
  if (nRepliesInBatch < nExpected)
    return 0;
  else {
    /* otherwise, the batch might be done already. need to verify it */
    for (i = 0; i < nExpected; i++)
      if (EMPTY_IN_ADDR(g_addrs+i))
	return 0;
  }

  return 1;
}
