/*
 *			P I N G . C
 *
 * Using the InterNet Control Message Protocol (ICMP) "ECHO" facility,
 * measure round-trip-delays and packet loss across network paths.
 *
 * Author -
 *	Mike Muuss
 *	U. S. Army Ballistic Research Laboratory
 *	December, 1983
 * Modified at Uc Berkeley
 * the batching of it is done by
 *     Haye Chan at Cornell University
 *
 * Changed argument to inet_ntoa() to be struct in_addr instead of u_long
 * DFM BRL 1992
 *
 * Status -
 *	Public Domain.  Distribution Unlimited.
 *
 * Bugs -
 *	More statistics could always be gathered.
 *	This program has to run SUID to ROOT to access the ICMP socket.
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/time.h>

#include <signal.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netdb.h>

#include "util.h"

#define	MAXWAIT		3	/* max time to wait for response, sec. */
#define DEFAULT_WAIT    1000000    /* default wait time in us */
#define	MAXPACKET	4096	/* max packet size */
#define VERBOSE		1	/* verbose flag */
#define QUIET		2	/* quiet flag */
#define FLOOD		4	/* floodping flag */
#define MAXNUMOFHOST   40       /* maximum number of hosts */
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN	64
#endif

u_char	packet[MAXPACKET];
int	i, pingflags, options;
extern	int errno;

int s;			/* Socket file descriptor */
struct hostent *hp;	/* Pointer to host info */
struct timezone tz;	/* leftover */

//struct sockaddr ping_whereto;/* Who to ping */
int datalen;		/* How much data */

char usage[] =
"Usage:  bping [-dfqrv] host [packetsize [count [timeout [preload]]]]\n";

char *hostname;
char hnamebuf[MAXHOSTNAMELEN];

int npackets;
int preload;		/* number of packets to "preload" */
int nBatchSent;
int ntransmitted;
u_short ident;
int ping_waittime = DEFAULT_WAIT;

char *g_prog;
int nreplied;		/* # of hosts that replied */
int nreceived;
int finished;
int timing;
int tmin;
int tmax;
int tsum;			/* sum of all times, for doing average */

int nhost;
struct sockaddr *ping_whereto[MAXNUMOFHOST];
int g_pingttl[MAXNUMOFHOST];
int g_pingreplied[MAXNUMOFHOST];
int g_TripTime[MAXNUMOFHOST];

char *g_answer;
int g_nAnswer;

void finish(), catcher();
char *inet_ntoa();
int gethost(struct sockaddr_in *sa, char *host);
int parse(char *hostfile);
int pinger(int seq);
void batchpinger();

/*
 * 			M A I N
 */
int bping(int argc, char **argv, char *answer, int nAnswer)
{
	struct sockaddr_in from;
	char **av = argv;
	char *cp;
	struct sockaddr_in *to = (struct sockaddr_in *) &ping_whereto;
	int on = 1;
	struct protoent *proto;
	int len = sizeof (packet);
	int fromlen = sizeof (from);
	int cc;
	struct timeval timeout;
	fd_set fdmask;
	time_t timeid;

	s = 0;
	npackets = 3;
	preload = 0;		/* number of packets to "preload" */
	nBatchSent = 0;
	ntransmitted = 0;
	ping_waittime = DEFAULT_WAIT;
	nreplied = 0;		/* # of packets we got back */
	nreceived = 0;
	finished = 0;
	timing = 0;
	tmin = 999999999;
	tmax = 0;
	tsum = 0;	      /* sum of all times, for doing average */
	nhost = 0;

	bzero((char *) ping_whereto, sizeof(ping_whereto));
	bzero((char *) g_pingttl, sizeof(g_pingttl));
	bzero((char *) g_TripTime, sizeof(g_TripTime));	
	*answer = (char)NULL;
	g_nAnswer = nAnswer;
	g_answer = answer;

	if ((cp = strchr(av[0], '/')) != NULL)
	  g_prog = cp + 1;
	else
	  g_prog = av[0];

	argc--, av++;
	while (argc > 0 && *av[0] == '-') {
		while (*++av[0]) switch (*av[0]) {
			case 'd':
				options |= SO_DEBUG;
				break;
			case 'r':
				options |= SO_DONTROUTE;
				break;
			case 'v':
				pingflags |= VERBOSE;
				break;
			case 'q':
				pingflags |= QUIET;
				break;
			case 'f':
				pingflags |= FLOOD;
				break;
		}
		argc--, av++;
	}
	if(argc < 1 || argc > 4)  {
		Printf1("%s", usage);
		return 1;
	}

	if (parse(av[0]))
	  return 1;
	/*	hostname = av[0]; */

	if( argc >= 2 )
		datalen = atoi( av[1] );
	else
		datalen = 64-8;
	if (datalen > MAXPACKET) {
		PrintError("ping: packet size too large\n");
		finish();
		return 1;
	}
	if (datalen >= sizeof(struct timeval))	/* can we time 'em? */
		timing = 1;

	if (argc >= 3)
		npackets = atoi(av[2]);

	if (argc == 4)
		ping_waittime = 1000 * atoi(av[3]);   // convert us to ms

	if (argc == 5)
		preload = atoi(av[4]);

	ident = time(&timeid) & 0xFFFF;

	if ((proto = getprotobyname("icmp")) == NULL) {
	        PrintError("icmp: unknown protocol\n");
		finish();
		return 10;
	}

	if ((s = socket(AF_INET, SOCK_RAW, proto->p_proto)) < 0) {
		PError("ping: socket");
		finish();
		return 5;
	}

	if (options & SO_DEBUG) {
	  if(pingflags & VERBOSE) {
	    Printf("...debug on.\n");
	  }
	  setsockopt(s, SOL_SOCKET, SO_DEBUG, 
		     (void*)&on, sizeof(on));
	}
	if (options & SO_DONTROUTE) {
	  if(pingflags & VERBOSE) {
	    Printf("...no routing.\n");
	  }
	  setsockopt(s, SOL_SOCKET, SO_DONTROUTE, 
		     (void*)&on, sizeof(on));
	}

//	if(to->sin_family == AF_INET) {
//		printf("PING %s (%s): %d data bytes\n", hostname,
//		  inet_ntoa(to->sin_addr), datalen);	/* DFM */
//	} else {
//		printf("PING %s: %d data bytes\n", hostname, datalen );
//	}
//	setlinebuf( stdout );

	signal( SIGINT, finish );
	signal(SIGALRM, catcher);

	/* fire off them quickies */
	for(i=0; i < preload; i++)
		batchpinger();

	if(!(pingflags & FLOOD))
		catcher();	/* start things going */

	while (!finished) {
	        timeout.tv_sec = 0;
        	timeout.tv_usec = ping_waittime;
		fromlen = sizeof (from);
		FD_ZERO(&fdmask);
		FD_SET(s, &fdmask);
		
		if(pingflags & FLOOD) {
			batchpinger();
			if( select(32, (fd_set *)&fdmask, 0, 0, &timeout) == 0)
				continue;
		}

		if ( (cc=recvfrom(s, packet, len, 0,
				  (struct sockaddr *) &from, 
				  &fromlen)) < 0) {
		  if( errno == EINTR ) {
		    continue;
		  }
		  PError("ping: recvfrom");
		  continue;
		}

		pr_pack( packet, cc, &from );
		if (nreplied >= nhost) {
			finished = 1;
		}
	}

	sigignore(SIGALRM);
	finish();
	return 0;
}

/*
 * 			C A T C H E R
 * 
 * This routine causes another PING to be transmitted, and then
 * schedules another SIGALRM for 1 second from now.
 * 
 */
void catcher()
{
  //  printf("before: finished = %d, nBatchSent = %d\n", finished, nBatchSent);

  /* if finished, don't do anything */
  if (finished)
    return;

  if (!npackets || nBatchSent < npackets) {
    /* always send batches if npackets == 0 or */
    /* send a batch also when number of batches sent < npackets */
    batchpinger();
    signal(SIGALRM, catcher);
    ualarm(ping_waittime);
  } else {
    /* if I've sent enough batches, I can finish it up too */
    finished = 1;
  }
  
  //  printf("after: finished = %d, nBatchSent = %d\n", finished, nBatchSent);

}

/*
 * 			P I N G E R
 * 
 * Compose and transmit an ICMP ECHO REQUEST packet.  The IP packet
 * will be added on by the kernel.  The ID field is our UNIX process ID,
 * and the sequence number is an ascending integer.  The first 8 bytes
 * of the data portion are used to hold a UNIX "timeval" struct in VAX
 * byte-order, to compute the round-trip time.
 */
int pinger(int seq)
{
	static u_char outpack[MAXPACKET];
	register struct icmp *icp = (struct icmp *) outpack;
	int i, cc;
	register struct timeval *tp = (struct timeval *) &outpack[8];
	register u_char *datap = &outpack[8+sizeof(struct timeval)];

	icp->icmp_type = ICMP_ECHO;
	icp->icmp_code = 0;
	icp->icmp_cksum = 0;
	icp->icmp_seq = seq;
	icp->icmp_id = ident;		/* ID */

	cc = datalen+8;			/* skips ICMP portion */

	if (timing)
		gettimeofday( tp, &tz );

	for( i=8; i<datalen; i++)	/* skip 8 for time */
		*datap++ = i;

	/* Compute ICMP checksum here */
	icp->icmp_cksum = in_cksum( (u_short *)icp, cc );

	i = sendto( s, outpack, cc, 0, 
		    ping_whereto[seq], sizeof(struct sockaddr) );

	if( i < 0 || i != cc )  {
		if( i<0 )  PError("sendto");
		Printf2("ping: wrote %d chars, ret=%d\n", cc, i );
	}
	if(pingflags == FLOOD) {
		Printf(".");
	}

	ntransmitted++;
}

/*
 *			P R _ P A C K
 *
 * Print out the packet, if it came from us.  This logic is necessary
 * because ALL readers of the ICMP socket get a copy of ALL ICMP packets
 * which arrive ('tis only fair).  This permits multiple copies of this
 * program to be run without having intermingled output (or statistics!).
 */
pr_pack( buf, cc, from )
char *buf;
int cc;
struct sockaddr_in *from;
{
	struct ip *ip;
	register struct icmp *icp;
	register long *lp = (long *) packet;
	register int i;
	struct timeval tv;
	struct timeval *tp;
	int hlen, triptime;

	from->sin_addr.s_addr = ntohl( (in_addr_t) from->sin_addr.s_addr );
	gettimeofday( &tv, &tz );

	ip = (struct ip *) buf;
	hlen = ip->ip_hl << 2;
	if (cc < hlen + ICMP_MINLEN) {
	  if (pingflags & VERBOSE) {
	    Printf2("packet too short (%d bytes) from %s\n", cc,
		    inet_ntoa(from->sin_addr)); /* DFM */
	  }
	  return;
	}
	cc -= hlen;
	icp = (struct icmp *)(buf + hlen);
	if( (!(pingflags & QUIET)) && 
	    (icp->icmp_type != ICMP_ECHOREPLY) &&
	    (icp->icmp_type != ICMP_UNREACH) )  {
		return;
	}
	if( icp->icmp_id != ident )
		return;			/* 'Twas not our ECHO */

	if (timing) {
		tp = (struct timeval *)&icp->icmp_data[0];
		tvsub( &tv, tp );
		triptime = tv.tv_sec*1000+(tv.tv_usec/1000);
		tsum += triptime;
		if( triptime < tmin )
			tmin = triptime;
		if( triptime > tmax )
			tmax = triptime;
	}

	i = icp->icmp_seq;
	if (g_pingreplied[i] == 0) {
	  nreplied++;
	  if (icp->icmp_type == ICMP_UNREACH)
	    g_pingreplied[i] = -2;
	  else
	    g_pingreplied[i] = 1;
	  g_pingttl[i] = ip->ip_ttl;
	  g_TripTime[i] = tv.tv_sec*1000000 + tv.tv_usec;
	}

	nreceived++;
}

int gethost(struct sockaddr_in *sa, char *host)
{
//  bzero((char *)sa, sizeof(struct sockaddr) );

  sa->sin_family = AF_INET;
  sa->sin_addr.s_addr = inet_addr(host);
  if (sa->sin_addr.s_addr == (unsigned)-1) {
    hp = gethostbyname(host);
    if (hp) {
      sa->sin_family = hp->h_addrtype;
      bcopy(hp->h_addr, (caddr_t)&sa->sin_addr, hp->h_length);
    } else {
      return -1;
    }
  }

  return 0;
}

/* parse the hostlist which is in "www.yahoo.com,128.232.22.232,..." format
 * into array 
 */
int parse(char *hostlist)
{
  char *host = NULL;
  char *comma;

  host = hostlist;
  while ((nhost < MAXNUMOFHOST) && *host) {
    ping_whereto[nhost] = (struct sockaddr *) malloc(sizeof(struct sockaddr));
    comma = (char *) strchr(host, ',');
    if (comma)
      *comma = (char) NULL;
    if (gethost((struct sockaddr_in *)ping_whereto[nhost], host) < 0) {
      Printf1("ping: unknown host %s\n", host);
      g_pingreplied[nhost] = -1;
    }
    else 
      g_pingreplied[nhost] = 0;
    nhost++;
    if (comma)
      host = comma+1;  /* go to next char */
    else
      *host = (char) NULL;    /* to end the loop */
  }

  return 0;
}

/*
 *			F I N I S H
 *
 * Print out statistics, and give up.
 * Heavily buffered STDIO is used here, so that all the statistics
 * will be written with 1 sys-write call.  This is nice when more
 * than one copy of the program is running on a terminal;  it prevents
 * the statistics output from becomming intermingled.
 */
void finish()
{
  int i, hop;

  Printf("PRT - Ping Result Table :\n");

  for (i=0; i<nhost; i++) {
    if (g_pingreplied[i] == 0)
      Printf1("%2d: dead, ", i);
    else if (g_answer, g_pingreplied[i] > 0)
      Printf1("%2d: alive, ", i);
    else if (g_pingreplied[i] == -1)
      Printf1("%2d: unknown host, ", i);
    else if (g_pingreplied[i] == -2)
      Printf1("%2d: unreachable, ", i);
    else
      Printf1("%2d: error, ", i);
    
    if (g_pingttl[i])
      hop = (g_pingttl[i]>128 ?255 :
	     (g_pingttl[i]>64 ? 128:
	      g_pingttl[i]>32 ? 64:32)) - g_pingttl[i];
    else
      hop = -1;

    Printf1("hop count = %0d, ", hop);
    Printf1("rtt = %0.3f ms\n", g_TripTime[i] / 1000.0);

    free(ping_whereto[i]);
  }

  Printf("-- PING Statistics --\n");
  Printf1("%d packets transmitted, ", ntransmitted );
  Printf1("%d packets received, ", nreceived );
  if (nBatchSent)
    if( nreceived > ntransmitted) {
      Printf("-- somebody's printing up packets!");
    } else {
      Printf1("%d%% packet loss",
	      (int) (((ntransmitted-nreceived)*100) /
		     ntransmitted));
    }
  
  Printf("\n");
  
  if (nreplied && timing) {
    Printf3("round-trip (ms)  min/avg/max = %d/%d/%d\n",
	    tmin, tsum / nreplied, tmax );
  }
  
  if (s) {
    close(s);
    s = 0;
  }
}

void batchpinger()
{
  int i;

  nBatchSent++;
  for (i=0; i<nhost; i++) {
    if (g_pingreplied[i] == 0)
      pinger(i);
  }
}
