#Routines for handling the remote interactions between the workers and
#the central process
use Socket;

#Initializes the connection of a worker to the central process or dies
sub init_worker_connection {
  socket(DBCONNECTION, PF_UNIX, SOCK_STREAM,0) or 
    die "Worker cannot create socket: $!";
  return if connect(DBCONNECTION, sockaddr_un($UNIXSOCKET));
  my $i;
  for ($i=0;$i<5;$i++) {
    return if connect(DBCONNECTION, sockaddr_un($UNIXSOCKET));
    sleep 3;
  }
  die "Cannot connect to the database via $UNIXSOCKET: $!"; 
}

#Initializes the central socket that's waiting for connections from
#workers and waits for the first connection
sub init_server_connection {
  my $name=$UNIXSOCKET;
  my $uaddr=sockaddr_un($name);
  my $proto=getprotobyname('tcp');

  unlink($name);
  socket(WORKERCONNECTION, PF_UNIX, SOCK_STREAM,0) or 
    die "Central process cannot create socket: $!";
  bind(WORKERCONNECTION, $uaddr) or 
    die "Central process cannot bind to Unix socket $name: $!";
  listen(WORKERCONNECTION,5) or 
    die "Cannot listen for worker connections: $!";
  #These are global variables
  $Workercounter=0; #It's not decremented when the workers quit. 
  @Workerfiledescriptors=(); 
  @Idleworkers=();
  %fd2handle=();
  $listenfd=fileno('WORKERCONNECTION');
  &accept_new_worker;
}

#Accepts a connection (to the central process) from a new worker
sub accept_new_worker {
  my ($handle,$paddr);

  $Workercounter++;
  $handle="CONNECTION$Workercounter";
  accept($handle,WORKERCONNECTION) or
    die "Error accepting connections from workers: $!";
  push (@Workerfiledescriptors,fileno($handle));
  $fd2handle{fileno($handle)}=$handle;
  log_message (4, "Accepted worker connection $handle");
}

#Creates a bitmask for select with all the active worker connections +
#the listen socket
sub fhbits {
  my ($bitmask,$fh);
  $bitmask='';
  $zeromask='';#this is a global
  foreach $fh ($listenfd,@Workerfiledescriptors){
    vec($bitmask,$fh,1)=1;
    vec($zeromask,$fh,1)=0;
  }
  return $bitmask;
}

#Input: A file handle
#Output: true on success, false on failure
#Blocking, unbuffered send, prepending the number of bytes
sub senddata
  {my ($handle,$data)=@_;
   my ($offset,$len);

   $data=sprintf("%09u",length($data)).$data;
   $len=length($data);
   $offset=0;
#log_message(5,"sending data >$data< length $len to $handle.");
   while($len)
     {my($x)=syswrite $handle,$data,$len,$offset;
      unless(defined $x)
	{log_message (1,"Error writing to $handle: $!");
	 return '';}
      $len -= $x;
      $offset+= $x;
#      print ">$offset";
    }
#   print "\n";
#log_message(5,"sent data >$data< length $len to $handle.");
   return 1;
 }

#Input: A file handle
#Output: The received data or undef on error
#Blocking unbuffered read of a chunk of data - length in the first 9 bytes
sub recvdata
  {my ($handle)=@_;
   my ($data,$len);

   $len=readnbytes($handle,9);
   unless (defined $len) {return undef;}
   $data=readnbytes($handle,$len);
#log_message(5,"received data >$data< on $handle.");
#print "received data >$data< on $handle";

   return $data;
 }

#Input: A file handle and the umber of bytes to read
#Output: The data, or undef on error
sub readnbytes {
  my ($handle,$len)=@_;
  my ($data,$offset);
  
  $data='';
  for($offset=0;$offset<$len;) {
    my($x)=sysread($handle,$data,$len-$offset,$offset);
    if(!$x) {
      if(!defined $x) {
	next if ($! =~ /^Interrupted/);
	log_message(0, "Read error on $handle at $offset of $len".
		    "data: >$data< error: >$!<");
	return undef;
	} else {#EOF
	  log_message(0, "End of file for $handle at $offset instead of $len".
		     "data: >$data<");
	    return undef;
	}
    }
    $offset+=$x;
#      print "<$offset";
  }
#   print "\n";
  return $data;
}

#Input: a handle, a function name and the argument list
#Output: return values
#args and retval are limited to lists of lists of lists of strings that do not
#contain ' '  ',' '{' '}' and ';' 
sub remoteinvoke
  {my ($handle,@stuff)=@_;
   my $answer;

#log_message(5,"Remoteinvoke got data ".(join(' ',@stuff)));
   senddata($handle,fastserialize(@stuff));
   $answer=recvdata($handle);
#log_message(5,"Remoteinvoke got answer $answer");
   if(defined $answer)
     {return fastparse($answer)}
   else
     {log_message (1, "No answer on connection $handle");
      exit;
    }
 }

#Fast serializer for lists of lists of lists (at most)
sub fastserialize
  {my (@stuff)=@_;
   my ($i);

#log_message(5,"Fastserialize got message ".(join ' ',@_));

   for ($i=0;$i<@stuff;$i++)
     {if (ref($stuff[$i]))
	{if (ref($stuff[$i]) ne 'ARRAY')
	   {die "Bad data $stuff[$i] for fastserializer 1";}
	 my($j,$jlp);
	 $jlp=\@{$stuff[$i]};
	 for($j=0;$j<@$jlp;$j++)
           {if (ref($$jlp[$j]))
	      {if (ref($$jlp[$j]) ne 'ARRAY')
		 {die "Bad data $$jlp[$j] for fastserializer 2";}
	       $$jlp[$j]='{'.(join(';',@{$$jlp[$j]})).'}';
	     }
	  }
	 $stuff[$i]='{'.(join(',',@$jlp)).'}';
       }
    }
   return join(' ',@stuff);
 }

#Fast parser for what fastserialize did
#The function name has to be extracted before this to take care of special
#cases e.g. sending over normally serialized SNMPdata
sub fastparse
  {my ($data)=@_;
   my $i;
   my @result=split (/ /,$data);

   for ($i=0;$i<@result;$i++)
     {if($result[$i] =~ /^\{(.*)\}$/)
	{my (@il)=split (',',$1);
	 my $j;

	 for ($j=0;$j<@il;$j++)
	   {if($il[$j] =~ /^\{(.*)\}$/)
	      {my (@jl)=split (';',$1);
	       $il[$j]=\@jl;
	     }
	  }
	 $result[$i]=\@il;
       }
    }
   return @result;
 }

return 1;
