use Socket;

#Initializes dns statistics counters and empties caches
sub dns_init
  {%NS_CACHE = ();
   %NS_INV_CACHE = ();
   $NS_LOOKUP = 0;
   $NS_INV_LOOKUP = 0;
   $NS_CACHE_MISS = 0;
   $NS_INV_CACHE_MISS = 0;
#   $NSLOOKUP = './nslookup';
   $NSLOOKUP = '/usr/local/bin/nslookup 2>&1';
   return 1;
 }

#Input: a string with a computer name
#Output: the list of its addresses in the dotted format
sub dns_name2addrs
  {my($name)=@_;
   my($aliases, $addrtype, $length, @addrs, $cachepointer);

   $NS_LOOKUP++;
   if (defined ($cachepointer = $NS_CACHE{$name}))
     {return @$cachepointer;}
   $NS_CACHE_MISS++;
   ($name, $aliases, $addrtype, $length, @addrs) = gethostbyname($name);
   if ($#addrs>=0)
     {@addrs=map {&inet_ntoa($_)} @addrs; #from Socket.pm
      @addrs=grep {&routable_address($_)} @addrs;}
   else
     {@addrs=()}
   $NS_CACHE{$name} = \@addrs;
   return @addrs;
 }

#Input: an address in the dotted format
#Output: the name that's returned by DNS from inverse lookup
#(Checks cache first)
sub dns_addr2name
  {my($addr)=@_;
   my($name);

   $NS_INV_LOOKUP++;
   if (!&routable_address($addr))
     {&log_message(2,"Unroutable address $addr looked up");
      return '';}
   $name = $NS_INV_CACHE{$addr};
   return $name if ($name);
   $NS_INV_CACHE_MISS++;
   $name = (gethostbyaddr(&inet_aton($addr), AF_INET) or '');
   $name =~ tr/[A-Z]/[a-z]/;
   $NS_INV_CACHE{$addr} = $name;
   return $name;
 }

#Input: a domain (with no dot at the end or begining)
#Output: a pair with the list of names and the list of subdomains
#(repetitions possible but unlikely in both, addresses in dotted format)
#As a side effect it adds to the DNS (direct) cache
#This function works with both RedHat Linux 5.1 and SunOS 5.6 nslookup
sub dns_ls 
  {my ($domain)=@_;
   my ($line,$server,$serverlist);

   $domain =~ tr/A-Z/a-z/;
   log_message (4,"DNS listing ---$domain---");
   $serverlist=&get_DNS_servers($domain);
   if($#$serverlist <0) #out of business
     {return ([],[]);}

   open(DNSFOO,">foo.$domain");
   print DNSFOO "set timeout=15\nset retry=2\nls -d $domain\n";
   close(DNSFOO);

   $server=shift @$serverlist;
   log_message(4, "Using +++$server to list $domain+++");
   open(DNSDATA,"$NSLOOKUP - $server < foo.$domain|") or
     log_message(1,"Cannot nslookup domain $domain at $server: $!");

   my($names,$subdomains,$name) = ([],[],'');
   while($line=<DNSDATA>) {
     if ($line =~ /\*\*\*/)
	#errors, try another server
	{if($#$serverlist <0)
	   {last;}#time to return
	 $server=shift @$serverlist;
	 close DNSDATA;
	 log_message(4, "Using +++$server to list $domain+++");
	 open(DNSDATA,"$NSLOOKUP - $server < foo.$domain|") or
	   log_message(1,"Cannot nslookup domain $domain at $server: $!");
	 $name='';
       }

      $line =~ s/^\s?(.*)/$1/; #cut one leading space
      if ($line =~ s/^(\S+)(.*)/$2/) #have a name?
	{my($n)=$1;
	 if ($n !~ /^[a-zA-Z0-9\-\.]+$/)
	   {$name='';} #no
	 else
	   {$n =~ tr/A-Z/a-z/;
	    if ($n =~ /(.*)\.$/) {$n=$1} #need to add domain?
	    elsif ($n !~ /$domain$/) {$n.='.'.$domain;}
	    $name=$n;
	  }
       }
      if (!$name)
	{next;} #initial garbage
      my(@word)=split /\s+/, $line;
      if (!$word[0])
	{shift @word;}
      if($word[0] !~ /^(A|NS)$/)# or /^(A|NS|HINFO|MX|CNAME)$/
	{shift @word;
	 if($word[0] !~ /^(A|NS)$/)
	   {shift @word;
	    if($word[0] !~ /^(A|NS)$/) {next;}
	  }
       }
      if ($word[0] eq 'A') #have an address
	{my ($addr)=$word[1];
	 if (valid_address($addr) && routable_address($addr))
	   {if($name ne $$names[$#$names])
	      {push @$names, $name;}
	    &add_to_cache($name,$addr);}
       }
      else #have a domain
	{if($name ne $$subdomains[$#$subdomains])
	   {push @$subdomains, $name;}
       }
    }
   close(DNSDATA);
   unlink "foo.$domain";
   my (@result)=($names,$subdomains);
   return @result;
 }

#Input: a domain
#Output: the list of nameservers for that domain (repetitions very unlikely)
#This function works with both RedHat Linux 5.1 and SunOS 5.6 nslookup
sub get_DNS_servers
  {my ($domain)=@_;
   my ($line,%servers,@addrs,$result);

   open(DNSBAR,">bar.$domain");
   print DNSBAR "set timeout=15\nset retry=2\nset type=NS\n$domain\n";
   close(DNSBAR);
   open(DNSDATA,"$NSLOOKUP <bar.$domain|");
   %servers=();
   @addrs=();
   while($line=<DNSDATA>)#get all nameservers
     {if($line =~ /$domain\s+nameserver\s+=\s+([a-zA-Z0-9\-\.]+)/)
	{my $name=$1;
	 $name =~ tr/A-Z/a-z/;
	 $servers{$name}='1';
       }
      if($line =~ /([a-zA-Z0-9\-\.]+)\s+internet\s+address\s+=\s+([0-9\.]+)/)
	{my ($name,$address)=($1,$2);
	 $name =~ tr/A-Z/a-z/;
	 if ($servers{$name} and valid_address($address) and
	     routable_address($address)) {
	   push @addrs,$address;
	   $NS_INV_CACHE{$address} = $name;
	 }
       }
    }
   close DNSDATA;
   unlink "bar.$domain";
log_message(5, "Nameservers for $domain ".(join (' ',@addrs)));
   if ($#addrs>=0)
     {($result) = &simple_bping_alive($g_nDefaultPingTimeout,@addrs);}
   else {$result=[];}
   return $result;
 }

#Input: a domain 
#Output: the list of mail exchangers for that domain (repetitions very
#unlikely)
#This function works with both RedHat Linux 5.1 and SunOS 5.6 nslookup
sub get_mail_exchangers
  {my ($domain)=@_;
   my ($line,%servers,@addrs,$result);

   open(DNSBAR,">bar.$domain");
   print DNSBAR "set timeout=15\nset retry=2\nset type=MX\n$domain\n";
   close(DNSBAR);
   open(DNSDATA,"$NSLOOKUP <bar.$domain|");
   %servers=();
   @addrs=();
   while($line=<DNSDATA>)#get all mail exchangers
     {if($line =~ /$domain\s+preference\s+=\s+\d+,\s+mail exchanger\s+=\s+([a-zA-Z0-9\-\.]+)/)
	{my $name=$1;
	 $name =~ tr/A-Z/a-z/;
	 $servers{$name}='1';
       }
      if($line =~ /([a-zA-Z0-9\-\.]+)\s+internet\s+address\s+=\s+([0-9\.]+)/)
	{my ($name,$address)=($1,$2);
	 $name =~ tr/A-Z/a-z/;
	 if ($servers{$name} and valid_address($address) and
	     routable_address($address)) {
	   push @addrs,$address;
	   $NS_INV_CACHE{$address} = $name;
	 }
       }
    }
   close DNSDATA;
   unlink "bar.$domain";
   log_message(5, "Mail exchangers for $domain ".(join (' ',@addrs)));
   if ($#addrs>=0)
     {($result) = &simple_bping_alive($g_nDefaultPingTimeout,@addrs);}
   else {$result=[];}
   return $result;
 }

#Input: a domain (with no dot at the end or begining)
#Output: a list of the names in the domain and its subdomains (repetitions
#possible)
#As a side effect it fills up the DNS (direct) cache
#We don't normally use this anymore
sub rec_dns
  {my ($domain)=@_;
   my ($names,$doms)=&dns_ls($domain);
   
   if ($doms && $#$doms>-1)
     {my ($subdom);
      foreach $subdom (@$doms)
	{my $sdnames;
	 if($sdnames=&rec_dns($subdom))
	   {push @$names, @$sdnames;}
       }
    }
   return $names;
 }

#Input: a name and an IP addres it maps to.
#Adds the address to the entry in the cache that belongs to the name (unless
#it's already there)
sub add_to_cache
  {my ($name,$addr)=@_;
   my $ptr=$NS_CACHE{$name};

#print "$name $addr\n";   
   if (!defined ($ptr))
    {$NS_CACHE{$name} = [$addr];}
   else
     {if (grep {$addr eq $_} @$ptr)
	{return}
      push @$ptr,$addr;}
 }

#Prints DNS statistics
sub dns_print_stats
  {my($hit, $ihit);
   $hit = $NS_LOOKUP - $NS_CACHE_MISS;
   $ihit = $NS_INV_LOOKUP - $NS_INV_CACHE_MISS;
   log_message (4, "ns forward lookup cache hits/total/cache size:$hit/$NS_LOOKUP/".
     ((keys %NS_CACHE)+0));
   log_message(4, "ns inverse lookup cache hits/total/cache size:$ihit/$NS_INV_LOOKUP/".
     ((keys %NS_INV_CACHE)+0));
 }

return 1;

