/*
 *			P I N G . C
 *
 * Using the InterNet Control Message Protocol (ICMP) "ECHO" facility,
 * measure round-trip-delays and packet loss across network paths.
 *
 * Author -
 *	Mike Muuss
 *	U. S. Army Ballistic Research Laboratory
 *	December, 1983
 * Modified at Uc Berkeley
 *
 * Changed argument to inet_ntoa() to be struct in_addr instead of u_long
 * DFM BRL 1992
 *
 * Status -
 *	Public Domain.  Distribution Unlimited.
 *
 * Bugs -
 *	More statistics could always be gathered.
 *	This program has to run SUID to ROOT to access the ICMP socket.
 */

#include <stdio.h>
#include <errno.h>
#include <sys/time.h>

#include <signal.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netdb.h>

#define	MAXWAIT		3	/* max time to wait for response, sec. */
#define DEFAULT_WAIT    1000000  /* default wait time in us */
#define	MAXPACKET	4096	/* max packet size */
#define VERBOSE		1	/* verbose flag */
#define QUIET		2	/* quiet flag */
#define FLOOD		4	/* floodping flag */
#define MAXNUMOFHOST   40       /* maximum number of hosts */
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN	64
#endif

u_char	packet[MAXPACKET];
int	i, pingflags, options;
extern	int errno;

int s = 0;			/* Socket file descriptor */
struct hostent *hp;	/* Pointer to host info */
struct timezone tz;	/* leftover */

//struct sockaddr ping_whereto;/* Who to ping */
int datalen;		/* How much data */

char usage[] =
"Usage:  bping [-dfqrv] host [packetsize [count [timeout [preload]]]]\n";

char *hostname;
char hnamebuf[MAXHOSTNAMELEN];

int npackets;
int preload;		/* number of packets to "preload" */
int nBatchSent;
int ntransmitted;
u_short ident;
int ping_waittime = DEFAULT_WAIT;

int nreplied;		/* # of packets we got back */
int nreceived;
int finished;
int timing;
int tmin;
int tmax;
int tsum;			/* sum of all times, for doing average */

int nhost;
struct sockaddr *ping_whereto[MAXNUMOFHOST];
int replied[MAXNUMOFHOST];

void finish(), catcher();
char *inet_ntoa();
int gethost(struct sockaddr_in *sa, char *host);
int parse(char *hostlist);
void FreeSA();
void batchpinger();

/*
 * 			M A I N
 */
//int ping(int argc, char **argv, int out)
int main(int argc, char **argv)
{
	struct sockaddr_in from;
	char **av = argv;
	struct sockaddr_in *to = (struct sockaddr_in *) &ping_whereto;
	int on = 1;
	struct protoent *proto;
	int len = sizeof (packet);
	int fromlen = sizeof (from);
	int cc;
	struct timeval timeout;
	int fdmask = 1 << s;
	time_t timeid;

	npackets = 0;
	preload = 0;		/* number of packets to "preload" */
	nBatchSent = 0;
	ntransmitted = 0;
	ping_waittime = DEFAULT_WAIT;
	nreplied = 0;		/* # of packets we got back */
	nreceived = 0;
	finished = 0;
	timing = 0;
	tmin = 999999999;
	tmax = 0;
	tsum = 0;	      /* sum of all times, for doing average */
	nhost = 0;

	argc--, av++;
	while (argc > 0 && *av[0] == '-') {
		while (*++av[0]) switch (*av[0]) {
			case 'd':
				options |= SO_DEBUG;
				break;
			case 'r':
				options |= SO_DONTROUTE;
				break;
			case 'v':
				pingflags |= VERBOSE;
				break;
			case 'q':
				pingflags |= QUIET;
				break;
			case 'f':
				pingflags |= FLOOD;
				break;
		}
		argc--, av++;
	}
	if(argc < 1 || argc > 4)  {
	        printf( usage);
		return 1;
	}

	parse(av[0]);

	if( argc >= 2 )
		datalen = atoi( av[1] );
	else
		datalen = 64-8;
	if (datalen > MAXPACKET) {
		fprintf(stderr, "ping: packet size too large\n");
		FreeSA();
		return 1;
	}
	if (datalen >= sizeof(struct timeval))	/* can we time 'em? */
		timing = 1;

	if (argc >= 3)
		npackets = atoi(av[2]);

	if (argc == 4)
		ping_waittime = 1000 * atoi(av[3]);   // convert us to ms

	if (argc == 5)
		preload = atoi(av[4]);

	ident = time(&timeid) & 0xFFFF;

	if ((proto = getprotobyname("icmp")) == NULL) {
		fprintf(stderr, "icmp: unknown protocol\n");
		FreeSA();
		return 10;
	}

	if ((s = socket(AF_INET, SOCK_RAW, proto->p_proto)) < 0) {
		perror("ping: socket");
		FreeSA();
		return 5;
	}
	if (options & SO_DEBUG) {
	  if(pingflags & VERBOSE) {
	    printf("...debug on.\n");
	  }
	  setsockopt(s, SOL_SOCKET, SO_DEBUG, 
		     (void*)&on, sizeof(on));
	}
	if (options & SO_DONTROUTE) {
	  if(pingflags & VERBOSE) {
	    printf("...no routing.\n");
	  }
	  setsockopt(s, SOL_SOCKET, SO_DONTROUTE, 
		     (void*)&on, sizeof(on));
	}

//	if(to->sin_family == AF_INET) {
//		printf("PING %s (%s): %d data bytes\n", hostname,
//		  inet_ntoa(to->sin_addr), datalen);	/* DFM */
//	} else {
//		printf("PING %s: %d data bytes\n", hostname, datalen );
//	}
	setlinebuf( stdout );

	signal(SIGINT, finish );
	signal(SIGALRM, catcher);

	/* fire off them quickies */
	for(i=0; i < preload; i++)
		batchpinger();

	if(!(pingflags & FLOOD))
		catcher();	/* start things going */

	while (!finished) {
		fromlen = sizeof (from);
		fdmask = 1 << s;

		timeout.tv_sec = 0;
		timeout.tv_usec = 10000;

		if(pingflags & FLOOD) {
			batchpinger();
			if( select(32, (fd_set *)&fdmask, 0, 0, &timeout) == 0)
				continue;
		}
		if ( (cc=recvfrom(s, packet, len, 0,
				  (struct sockaddr *) &from, &fromlen)) < 0) {
			if( errno == EINTR )
				continue;
			perror("ping: recvfrom");
			continue;
		}
		pr_pack( packet, cc, &from );
		if (npackets && (nreplied >= (nhost * npackets))) {
			finish();
		}
	}
	signal(SIGALRM,SIG_IGN);

	return 0;
}

struct itimerval alarm_time;
struct itimerval dummy;
void us_alarm(int waittime)
{alarm_time.it_interval.tv_sec=waittime/1000000;
 alarm_time.it_interval.tv_usec=waittime%1000000;
 alarm_time.it_value.tv_sec=waittime/1000000;
 alarm_time.it_value.tv_usec=waittime%1000000;
 setitimer(ITIMER_REAL,&alarm_time,&dummy);
}

/*
 * 			C A T C H E R
 * 
 * This routine causes another PING to be transmitted, and then
 * schedules another SIGALRM for 1 second from now.
 * 
 * Bug -
 * 	Our sense of time will slowly skew (ie, packets will not be launched
 * 	exactly at 1-second intervals).  This does not affect the quality
 *	of the delay and loss statistics.
 */
void catcher()
{
//	int ping_waittime;

	batchpinger();
	if (npackets == 0 || nBatchSent < npackets)
		us_alarm(ping_waittime);
	else {
	  /*
		if (nreplied) {
			ping_waittime = 2 * tmax / 1000;
			if (ping_waittime == 0)
				ping_waittime = 1;
		} else
			ping_waittime = MAXWAIT;
			*/
		signal(SIGALRM, finish);
		us_alarm(ping_waittime);
	}
}

/*
 * 			P I N G E R
 * 
 * Compose and transmit an ICMP ECHO REQUEST packet.  The IP packet
 * will be added on by the kernel.  The ID field is our UNIX process ID,
 * and the sequence number is an ascending integer.  The first 8 bytes
 * of the data portion are used to hold a UNIX "timeval" struct in VAX
 * byte-order, to compute the round-trip time.
 */
pinger(int seq)
{

	static u_char outpack[MAXPACKET];
	register struct icmp *icp = (struct icmp *) outpack;
	int i, cc;
	register struct timeval *tp = (struct timeval *) &outpack[8];
	register u_char *datap = &outpack[8+sizeof(struct timeval)];

	icp->icmp_type = ICMP_ECHO;
	icp->icmp_code = 0;
	icp->icmp_cksum = 0;
	icp->icmp_seq = seq;
	icp->icmp_id = ident;		/* ID */

	cc = datalen+8;			/* skips ICMP portion */

	if (timing)
		gettimeofday( tp, &tz );

	for( i=8; i<datalen; i++)	/* skip 8 for time */
		*datap++ = i;

	/* Compute ICMP checksum here */
	icp->icmp_cksum = in_cksum( icp, cc );

	/* cc = sendto(s, msg, len, flags, to, tolen) */
	i = sendto( s, outpack, cc, 0, ping_whereto[seq], sizeof(struct sockaddr) );

	if( i < 0 || i != cc )  {
		if( i<0 )  perror("sendto");
		printf("ping: wrote %d chars, ret=%d\n", cc, i );
	}
	if(pingflags == FLOOD) {
		putchar('.');
		fflush(stdout);
	}

	ntransmitted++;
}

/*
 *			P R _ P A C K
 *
 * Print out the packet, if it came from us.  This logic is necessary
 * because ALL readers of the ICMP socket get a copy of ALL ICMP packets
 * which arrive ('tis only fair).  This permits multiple copies of this
 * program to be run without having intermingled output (or statistics!).
 */
pr_pack( buf, cc, from )
char *buf;
int cc;
struct sockaddr_in *from;
{
	struct ip *ip;
	register struct icmp *icp;
	register long *lp = (long *) packet;
	register int i;
	struct timeval tv;
	struct timeval *tp;
	int hlen, triptime;

	from->sin_addr.s_addr = ntohl( from->sin_addr.s_addr );
	gettimeofday( &tv, &tz );

	ip = (struct ip *) buf;
	hlen = ip->ip_hl << 2;
	if (cc < hlen + ICMP_MINLEN) {
	  if (pingflags & VERBOSE) {
	    printf("packet too short (%d bytes) from %s\n", cc,
		    inet_ntoa(from->sin_addr)); /* DFM */
	  }
	  return;
	}
	cc -= hlen;
	icp = (struct icmp *)(buf + hlen);
	if( (!(pingflags & QUIET)) && icp->icmp_type != ICMP_ECHOREPLY )  {
//	  printf("host is dead\n");
//		printf("%d bytes from %s: icmp_type=%d (%s) icmp_code=%d\n",
//		  cc, inet_ntoa(from->sin_addr),
//		  icp->icmp_type, pr_type(icp->icmp_type), icp->icmp_code);/*DFM*/
//		if (pingflags & VERBOSE) {
//			for( i=0; i<12; i++)
//				printf("x%2.2x: x%8.8x\n", i*sizeof(long),
//				  *lp++);
//		}
		return;
	}
	if( icp->icmp_id != ident )
		return;			/* 'Twas not our ECHO */

	if (timing) {
		tp = (struct timeval *)&icp->icmp_data[0];
		tvsub( &tv, tp );
		triptime = tv.tv_sec*1000+(tv.tv_usec/1000);
		tsum += triptime;
		if( triptime < tmin )
			tmin = triptime;
		if( triptime > tmax )
			tmax = triptime;
	}

	if(!(pingflags & QUIET)) {
		if(pingflags != FLOOD) {
		  if (replied[icp->icmp_seq] == 0) {
		    nreplied++;
		    replied[icp->icmp_seq]++;
		  }
//		  printf("host is alive\n");

//			printf("%d bytes from %s: icmp_seq=%d", cc,
//		  inet_ntoa(from->sin_addr),
//			  icp->icmp_seq );	/* DFM */
//                    if (timing) 
//				printf(" time=%d ms\n", triptime );
			//else
//				putchar('\n');
		} else {
			putchar('\b');
			fflush(stdout);
		}
	}

	nreceived++;
}

/*
 *			F I N I S H
 *
 * Print out statistics, and give up.
 * Heavily buffered STDIO is used here, so that all the statistics
 * will be written with 1 sys-write call.  This is nice when more
 * than one copy of the program is running on a terminal;  it prevents
 * the statistics output from becomming intermingled.
 */
void finish()
{
	printf("---PING Statistics----\n");
	printf("%d packets transmitted, ", ntransmitted );
	printf("%d packets received, ", nreceived );
	if (nBatchSent)
	  if( nreceived > ntransmitted) {
	    printf("-- somebody's printing up packets!");
	  } else {
	    printf("%d%% packet loss", 
		    (int) (((ntransmitted-nreceived)*100) /
			   ntransmitted));
	  }

	if (nreplied && timing) {
	    printf("round-trip (ms)  min/avg/max = %d/%d/%d\n",
		tmin,
		tsum / nreplied,
		tmax );

	}
	fflush(stdout);
	FreeSA();
	finished = 1;
}

int gethost(struct sockaddr_in *sa, char *host)
{
  bzero((char *)sa, sizeof(struct sockaddr) );

  sa->sin_family = AF_INET;
  sa->sin_addr.s_addr = inet_addr(host);
  if (sa->sin_addr.s_addr == (unsigned)-1) {
    hp = gethostbyname(host);
    if (hp) {
      sa->sin_family = hp->h_addrtype;
      bcopy(hp->h_addr, (caddr_t)&sa->sin_addr, hp->h_length);
    } else {
      return -1;
    }
  }

  return 0;
}

int parse(char *hostlist)
{
  char *host = NULL;
  char *comma;

  host = hostlist;
  while ((nhost < MAXNUMOFHOST) && *host) {
    ping_whereto[nhost] = (struct sockaddr *) malloc(sizeof(struct sockaddr));
    comma = (char *) strchr(host, ',');
    if (comma)
      *comma = (char) NULL;
    if (gethost((struct sockaddr_in *)ping_whereto[nhost], host) < 0) {
      printf("ping: unknown host %s\n", host);
      replied[nhost] = -1;
    }
    else 
      replied[nhost] = 0;
    nhost++;
    if (comma)
      host = comma+1;  /* go to next char */
    else
      *host = (char) NULL;    /* to end the loop */
  }

  return 0;
}

void FreeSA()
{
  int i;

  printf("PRT - Ping Result Table :\n");


  for (i=0; i<nhost; i++) {
    if (replied[i] == 0)
      printf("%2d: dead\n", i);
    else if ( replied[i] > 0)
      printf("%2d: alive\n", i);
    else if (replied[i] == -1)
      printf("%2d: unknown host\n", i);
    else
      printf("%2d: error\n", i);

    free(ping_whereto[i]);
  }
  if (s) {
    close(s);
    s = 0;
  }
}

void batchpinger()
{
  int i;

  nBatchSent++;
  for (i=0; i<nhost; i++) {
    if (replied[i] == 0)
      pinger(i);
  }
}
