// Donna Bergmark - November 2000 - the Reference Linking project

// Context encapsulates locales in which references have been made
// It is specific to a reference string, not to a sentence in the text

// It also contains a static reconstruct routine that manufactures
// a Context[] out of XML for <context-list>

// Updates:
// 11-22-00:  Added another attribute so we have both anchor and norm.
// 11-30-00:  Add the reconstruct routine
// 12-12-00:  Add reference index (0-based) as another local field
//            Add context ordinal (1-based) as another local field
// 04-17-01:  Context is stored as trimmed with white space removed
// 05-15-01:  Have to trim the stuff reconstructed from a text node, too

package Linkable.Utility;

import uk.ac.soton.harvester.Utils;
import uk.ac.soton.harvester.EntityDecode;

import org.w3c.dom.*;              // for reconstruct

import java.util.Vector;

/**
 * This class encapsulates a single context.  A context
 * contains 1 or more references.  One separate context object
 * is created for each one of these references.
 */

public class Context {

   private static final String ME = "Context: ";
   private static final boolean DEBUG = CONFIG.DEBUG;

   private static EntityDecode ed = new EntityDecode();

   private String anchor;         // e.g. "According to Bray (1996)"
   private String normalizedRef;  // e.g. "[Bray, 1996]"
   private String context;        // the sentence containing that anchor
   private int myRef;             // 0-based index of reference for this anchor
   // ordinal is the 1-based index of where this context came in the text
   private int ordinal;           

   /**
    * constructor -
    * @param - String which is an anchor in this context
    * @param - String which is the normalized reference
    * @param - String which is the sentence containing this anchor
    * @param - integer which is the 0-base index of the reference
    * @param - integer which is the 1-base position of this context in 
    * the text.
    * Note: the caller guarantees that the context string is trimmed
    * and free of newlines.  Reconstructed surrogates have contexts
    * that are already cleaned.  Analysis on the fly has to clean it
    * before calling this constructor.
    */

   public Context (String a, String b, String c, int d, int e) {
      super();
      anchor = a;     
      normalizedRef = b;
      context = c;
      myRef = d;
      ordinal = e;
   }

   // accessor functions
   public String getAnchor() { return anchor; }
   public String getNormalizedRef() { return normalizedRef; }
   public String getContext() { return context; }
   public int getRefIndex() { return myRef; }
   public int getOrd() { return ordinal; }

   // dump functions
   public String toXML ( String pad ) {
      return pad + "<context ord=\""+ordinal+"\""
		 +" anchor=\"" + Utils.PCDATA(anchor) 
		 + "\" normalization=\""
		 + Utils.PCDATA(normalizedRef) + "\">\n"
                 + Utils.PCDATA(context.replace('\n',' ')) + "\n" +
       	     pad + "</context>\n";
                 //+ context + "\n" +
   }

   public String toString () {
      return "" + ordinal + " " + anchor + " " + context;
   }

   // reconstruct -
   /** rebuilds a Context[] array from XML for a <context-list> tree.
   * @param 0-based index of the <reference> element containing this 
   * context-list.
   * @param The root node of the tree
   * @returns an array of contexts, even if it is an array of size 0.
   */
   public static Context[] reconstruct ( int r, Element context_list ) {
      if ( context_list == null ) {
	 System.err.println(ME+"reconstruct called with a null context_list");
	 return new Context[0];
      }
      Vector results = new Vector();
      System.err.println(ME+context_list.getNodeName());
      NodeList nl = context_list.getElementsByTagName("context");
      if (nl == null) return new Context[0];

      for (int i=0; i<nl.getLength(); i++) {
	 Node n = nl.item(i);        // <context>
	 if ( n.getNodeType() != Node.ELEMENT_NODE ) {
	    System.err.println(ME+"reconstruct expecting a <context> element"
	    + " but got a node of type " + n.getNodeType() +">");
	    break;
	 }
	 if ( !n.getNodeName().equals("context") ) {
	    System.err.println(ME+"reconstruct expecting a <context> node"
	    + " but this element is named " + n.getNodeName());
	    break;
	 }
	 NamedNodeMap attrs = n.getAttributes();
         String a = (attrs.getNamedItem("anchor")).getNodeValue();
         String b = (attrs.getNamedItem("normalization")).getNodeValue();
	 String d = (attrs.getNamedItem("ord")).getNodeValue();
	 System.err.println(ME+"<context ord=\""+d+"\" anchor=\""+a+"\" normalization=\""
	 +b+"\"");
	 // piece together the context string
	 NodeList strings = n.getChildNodes();
	 String c = "";
	 if ( strings == null ) break;
	 for ( int j=0; j<strings.getLength(); j++) {
	    Node text = strings.item(j);
	    switch ( text.getNodeType() ) {
	    case Node.TEXT_NODE: 
	       c+=text;
	       break;
	    case Node.ENTITY_REFERENCE_NODE:
	       c+=(char)ed.lookup(text.getNodeName());
	       break;
	    default:
	    }
	 }
	 System.err.println(ME+c);
	 results.addElement(new Context(a,b,c.trim(),r,Integer.valueOf(d).intValue()));
      }

      Context[] x = new Context[results.size()];
      if ( results.size() == 0 ) return x;
      return (Context[])results.toArray(x);
   }

} // Context

