// bergmark - June 2000 - Reference Linking project

/**
 * AuthorDatabase is a collection of static routines suitable for
 * maintaining a database of authors and their works.  It is kept
 * separate from Author, because one might want to use a different
 * kind of database, and still deal with Author objects.
 */

// AuthorDatabase - keep track of author names so that no one
// author gets created multiple times.

// At run time, the Author Database is a hash table.  In storage, it
// is a flat file.  

// Key: last name  (as in "first von last jr")
// Value: a Vector of Author objects: all the authors with this last name

// 2000-10-03: Add factility for persistent storage: save and reload
// 2000-10-18: Added routine to delete hash table entries

package Linkable.Utility;

import java.util.*;
import java.io.*;

public class AuthorDatabase {

   private static final String ME = "AuthorDatabase: ";
   private static final boolean DEBUG = false;

   private static Hashtable authors = new Hashtable();

   /**
    * fetchAuthor - returns a vector of Author objects with this last name
    * or returns null if lookup fails
    * @param - last name of author to be fetched
    */

   public static Vector fetchAuthor ( String lastName ) {
      return (Vector)authors.get(lastName);
   }

   /**
    * fetchAuthor - this form returns the particular element in the
    * database that matches Author a ( or null )
    */
   public static Author fetchSpecificAuthor ( Author a ) {
       String last = a.getLastName();
       if ( last == null || last.equals("") ) return null;
      Vector authorList = (Vector)authors.get(last);
      if ( authorList == null ) return null;
      for (int i=0; i<authorList.size(); i++) {
          Author b = (Author)authorList.elementAt(i);
          if (a.equals(b)) return b;
      }
      return null;     // Author not in hash table
   }

   /**
    * stashAuthor - adds a specified Author to the hash table. 
    * As a side-effect, if there is a hash conflict, but the
    * values are consistent with each other, then fields get
    * completed and the merged Author stays at this key and 
    * at this Vector index.  We assume the Author given to this
    * routine will be discarded during garbage collection.
    *
    * If the two entries are inconsistent, then assume that we
    * have a new author with the same last name as another one.
    * @param the Author that should be inserted into the table
    */

    public static void stashAuthor ( Author a ) {

       // Check to see if there already is an entry with this last name
       String last = a.getLastName();
       if ( last == null || last.equals("") ) {
	  System.err.println(ME+"given author with null last name!");
	  return;
       }
       if (DEBUG) System.err.println(ME+"in stashAuthor, name = <"
	  + last + ">");
       Vector authorList;
       if ( (Vector)authors.get(last) == null ) {  // new entry
	  Vector v = new Vector();
	  v.addElement ( a );      
	  authors.put(last, v);
	  if (DEBUG)
	     System.err.println(ME+"started new list for authors with"
	       +" last name " + last);
       } else {            // one or more authors with same last name
	  authorList = (Vector)authors.get(last);
	  for (int i=0; i<authorList.size(); i++) {
	     Author b = (Author)authorList.elementAt(i);
	     if (a.equals(b)) return;
		/* we are done - author is in the list */
	  }
	  authorList.add(a);  // increase value at this key by one
	  if (DEBUG) System.err.println(ME+"added this Author "
	     + "to the list of authors with last name " + last);
       }
    }

    /**
     * isInDatabase - return true if this author is already there
     * (flesh out author data in the process)
     * @param the Author object to be looked up
     */
    public static boolean isInDatabase ( Author a ) {
       // Check to see if there is an entry with this last name
       String last = a.getLastName();
       Vector authorList;
       if ( (Vector)authors.get(last) == null ) return false;
       authorList = (Vector)authors.get(last);
       for (int i=0; i<authorList.size(); i++) {
          Author b = (Author)authorList.elementAt(i);
          if (a.equals(b)) return true;
       }
       return false;
    }

    /**
     * delete - remove the entire entry from the hashtable
     *
     */
    public static void delete ( String key ) {
       authors.remove ( key );
    }

    /**
     * updateAuthor - The author vector at the specified key is
     * replaced by the new Vector (good for deleting Authors)
     */
    public static void updateAuthor ( String key, Vector v ) {
       authors.put( key, v );
    }

   /**
     * reload - initialize the hash table from storage (partially) from
     * the file specified in CONFIG.java.
     * format for the  record in file is
     *   (author-last-name, Vector of Author objects)
     */
   public static void reload ( ) {
      String filename = CONFIG.AuthorDB;
      System.err.println(ME+"reloading Author hashtable from " + filename);
      try {
         FileInputStream fis = new FileInputStream ( filename );
         ObjectInputStream ois = new ObjectInputStream ( fis );
         while ( true ) {
            // read next record on the input stream
            authors = (Hashtable)ois.readObject();
         }
      } catch (Exception e) {}
      System.err.println(ME+authors.size()+" authors reloaded from disk");
   }


    // This routine saves the author database to the
    // file whose name is given in CONFIG
    public static void save () {
       String filename = CONFIG.AuthorDB;
       System.err.println(ME+"saving the author database to " + filename);
       int n=0;
       ObjectOutputStream oos = null;
       try {
	  FileOutputStream fos = new FileOutputStream ( filename );
	  oos = new ObjectOutputStream ( fos );
          oos.writeObject(authors);   // write out the whole hashtable
       } catch (Exception e) {
	  System.err.println(ME+"caught exception e.toString() while "
	  + "writing the Authors hashtable to disk");
       }
       try { oos.flush(); oos.close(); } catch (Exception e){}
       System.err.println(ME+"wrote a hashtable of size " 
       + authors.size() + " records to file");
    }

    /**
     * dump - Print out current contents of the Author database
     */
    public static void dump () {
       System.err.println("\n Key (last name)             Authors");
       Enumeration w = authors.elements();
       for ( Enumeration e = authors.keys(); e.hasMoreElements(); ) {
	  String result = (String)e.nextElement() + "     "; 
	  Vector v = (Vector)w.nextElement();
	  for (int i=0; i<v.size(); i++ ) {
	     result += ((Author)v.elementAt(i)).toString();
	  }
	  System.err.println(result);
       } // enumeration
       System.err.println("Number of authors in the database: " + 
	       authors.size());
    }

    /**
     * dump - Write current contents of the Author database to file
     */
    public static void dump (String fname) {
       System.err.println(ME+"dumping the author database to " + fname);
       try {
          FileOutputStream out = new FileOutputStream(fname);
          String line="<?xml version=\"1.0\" ?>\n<author-database>\n";
          out.write(line.getBytes());
          Enumeration w = authors.elements();
          for ( Enumeration e = authors.keys(); e.hasMoreElements(); ) {
	     out.write("<author>\n".getBytes());
	     String result = (String)e.nextElement() + "     "; 
	     Vector v = (Vector)w.nextElement();
	     for (int i=0; i<v.size(); i++ ) {
	        result += ((Author)v.elementAt(i)).toString();
	     }
	     out.write(result.getBytes());
	     out.write("</author>\n".getBytes());
          } // enumeration
	  out.write("</author-database>\n".getBytes());
         out.close();
      } catch (Exception e) {
         System.err.println(ME+"failed to dump the Author Database."
         +" Exception is " + e.toString() );
      }

    }

}
