// bergmark - April 2000 - The Reference Linking Project

// Reference encapsulates data about a reference contained in a Creation,
// but encountered while analyzing this item

// Modifications:
// 2000-07-28:    Reference objects have nothing to do with BibData
//                objects, only with Creations.  Constructor changed.
// 2000-08-01: Output original reference and contexts as PCDATA
// 2000-08-03: If a Creation cannot be constructed because the reference
//             string could not be parsed, at least include literal and
//             context and ordinal number
// 2000-09-29: Output the tag along with the literal reference string.
// 2000-10-03: Add "save()" routine to update Creation Database
// 2000-10-18: Fix NullPointerException in "save()"
// 2000-11-16: Context objects used to hold contexts
// 2000-11-21: Static routine to reconstruct a reference list
// 2000-11-30: Finish the reconstruct routine
// 2000-12-04: Add the "reconstruct(Document,int)" routine
// 2000-12-14: Be sure to entify the tag on the <literal tag=""> element
// 2001-04-04: Remove "\n" when resurrecting literals
// 2001-04-16: Add exceptions for error handling

package Linkable.API;

import uk.ac.soton.harvester.Utils;

import Linkable.Utility.Context;
import Linkable.Utility.MetaData;
import Linkable.Utility.CONFIG;

import java.util.StringTokenizer;

import org.w3c.dom.*;             // For reconstruct

/**
 * This class encapsulates a single reference in a given document.
 * Usually this class is instantiated once sufficient bibliographic
 * data has been extracted from a reference string.  Instances of
 * this class make up the Reference[] array that is part of the local
 * data within a Surrogate object.
 */

public class Reference {
   private static final String ME = "Reference: ";
   private static final boolean DEBUG = CONFIG.DEBUG;

   private Creation referenceData;  // URI and URN of the reference
   private int ordinalNumber;      // Which reference is this 1..N
   private String origRef;         // How the reference appeared in the text
   private Context context[];      // Context(s) for this reference
   // tag is "10." or null or (Brady, 1963)
   private String tag;             // Reference tag


   // Constructor returns a new Reference Object
   public Reference (Creation bd, int n, String _tag, String ref, Context[] c)
   throws SurrogateException {

      if ( bd == null ) {
	 throw new SurrogateException (
	 SurrogateException.setInternal(
	 ME+" constructor given a null Creation!"));
      }
      else referenceData = bd;
      ordinalNumber = n;
      origRef = ref;
      context = new Context[c.length];
      tag = _tag;
      System.arraycopy( c, 0, context, 0, c.length );
      if (DEBUG) {
         System.err.println(ME+" constructed a new reference:");
         System.err.println(this.toString());
      }
   }

   /** returns the Creation (i.e. doc id) of this reference */
   public Creation getDocID() { return referenceData; }

   public String getTag() { return tag; }

   /** returns a copy of the Contexts in which this reference occurred */
   public Context[] getContexts () {
      Context[] result = new Context[context.length];
      System.arraycopy ( context, 0, result, 0, context.length );
      return result;
   }

   /** getLiteral - return the literal */
   public String getLiteral () { return origRef; }

   public String toString() {
      return toXML ("");
   }

   public String toXML(String pad) {
      String bd;
      if ( referenceData == null ) bd = "<work></work>\n";
      else bd = referenceData.toXML(pad);
      return "<reference ord=\"" + ordinalNumber + "\">\n" +
	     pad + bd +
             pad + "<literal tag=\""+
	     (tag==null?"":Utils.PCDATA(tag))+"\">\n"
	     + Utils.PCDATA(origRef.replace('\n',' ')) + "\n" +
	     pad + "</literal>\n" +
             pad + "<context-list>\n" +
             listContexts("    ") +
             pad + "</context-list>\n" +
             "</reference>";
   }

   // PROTECTED METHODS

   // Called by Surrogate to update the fileIndex hashtable in Creation
   // Database.  Conditions of call: Surrogate has just written itself
   // out to storage, including this reference object.
   protected void save ( String doi) {
      if ( referenceData != null )
         referenceData.save( doi);
   }

   // reconstruct -
   /** Returns one specified reference from the DOM tree given it.
   * The DOM tree corresponds to a Surrogate/refList file.
   * @param Document for the file
   * @param index of which reference is desired
   * @returns one Reference object
   */
   protected static Reference reconstruct ( Document doc) {
      if (DEBUG) System.err.println(ME+"in reconstruct for document "
      + MetaData.getValue(doc,"dc:title"));
      Reference result = null;
      return result;
   }

   // reconstruct -
   /**
     * Reconstructs an entire reference list from inside a saved Surrogate.
     * @param Name of the refList file that contains the info
     * @returns A fixed sized array of References
     * Note: called by Surrogate when reconstructing a Surrogate
     * Also called by Creation when reconstructing a creation
     */

   protected static Reference[] reconstruct ( String filename )
   throws SurrogateException {
      Reference[] result = null;
      Document doc = MetaData.getDOM ( MetaData.getData ( filename ) );
      int nref = Integer.valueOf(
	 MetaData.getAttr ( doc, "referencelist", "length" )).intValue();
      if ( DEBUG )
      System.err.println(ME+"reconstructing a list of " + nref + " references");
      Reference[] refList = new Reference [nref];
      NodeList nl = doc.getElementsByTagName("reference");
      if ( DEBUG )
      System.err.println("a "+(nl==null?"null":"got non-null nl"
	+" ("+nl.getLength()+" elements)"));
      if ( nl == null ) return null;

      result = new Reference[ nl.getLength() ];

      for ( int i = 0; i<nl.getLength(); i++ ) {
	 Element ref = (Element)nl.item(i);  // reconstruct i-th reference
	 Element work = (Element)ref.getElementsByTagName("work").item(0);
	 Element literal = (Element)ref.getElementsByTagName("literal").item(0);
	 Element context_list=(Element)ref.getElementsByTagName("context-list").item(0);
	 if ( DEBUG )
	 System.err.println("Got work, literal,context_list for ref " + i);

	 // Make a Creation "referenceData" out of the "work" element
         Creation c = Creation.doWork ( work );

         // Process the <literal> Element into tag and refString
         String refString=""; String tag="";
         if ( literal != null ) {
            // Get the tag, which is an attribute
            NamedNodeMap litAttrs = literal.getAttributes();
            Node tagNode = litAttrs.getNamedItem ( "tag" );
            tag = tagNode.getNodeValue();
	    refString = (
	       ( MetaData.getValue(literal) ).replace('\n',' ') ).trim();
if ( DEBUG )
System.err.println(ME+"reconstructed literal:\n"+refString);
         } // literal not null

	 // Finally, reconstruct the context list
        Context[] contexts = Context.reconstruct (i,(Element)context_list);

	 result[i] = new Reference ( c, i+1, tag, refString, contexts );
      }
      return result;
   }

   // PRIVATE METHODS

   private String listContexts(String pad) {
      String result = "";
      if ( context == null ) return result;
      for ( int i=0; i < context.length; i++ )
         result = result + context[i].toXML(pad);
      return result;
   }

}
