// bergmark - April 2000 - The Reference Linking Project

// BibData for this item can be used to get the associated Creation
// BibData objects belong to Surrogates.
// There are two important fields in the BibData object:
//     myURN     -- the synthetic URN which is the CreationDatabase key
//     myDOI     -- if this is an archive item, the FullID
// URNs are used to identify works, or creations and they are constructed
//    out of the work's first author, publication date, and title.
// DOIs belong with Surrogates and have the archive id and the within-archive
//    ID.
//
// The BibData object holds its Creation just for convenience, both in
// order to avoid multiple lookups, and to serve as a template during
// Surrogate construction.  Several different BibData objects might contain
// the same creation.

// Modifications:
// 2000-07-28:     Did away with DIDs.  BibData gets myDOI and myURN.
//                 Did away with MIMEfile.
// 2000-12-14:     When saving my URN, entify any "&" in it

package Linkable.API;

import Linkable.Utility.*;            // for database lookup

import java.util.Vector;

import org.w3c.dom.*;             // For reconstruct

/**
 * This class represents an instance of a work.  The primary job of this
 * class is to keep trans of work identification elements, such as its
 * DOI, synthetic URN, and most importantly, its "key" in the Creation
 * database, which should have a unique record for each work encountered.
 * Different BibData object instances may well point to the same Creation
 * record.  Methods in this class are mainly construct, reconstruct, get
 * and put.
 */
public class BibData {

   private static final String ME = "BibData: ";
   private static final boolean DEBUG = false;

   private String myDOI;
   private String myURN;
   private Creation creationPointer;  // used to access the creation

  /** constructor
   *  Creations can be made with or without its URN initialized
   *  It is possible to set the URN later on.  So, URN can be null.
   *  Invariant: DOI != null because DOI points to where the repository is.
   * @param The String which is the DOI for this article.  May be our own
   * invention, but may not be null.
   * @param An optional string containing the synthetic urn or partial
   * urns for a work.  If a urn is provided, then there must be a record
   * in the database at that key.  If the urn is null, then a dummy
   * creation is instantiated, to be filled in as the document is parsed.
   * @throws SurrogateException
   * @returns a new BibData object
   */
   public BibData ( String DOI, String URN ) throws SurrogateException {
      myDOI = DOI;
      myURN = URN;
      if ( DEBUG )
      System.err.println(ME+"constructing bibdata for doi "
      + DOI + ", urn = " + (URN==null?"null":URN));
      if ( URN != null ) {  // fetch all creations matching this URN
         Vector creations = Creation.fetchURN ( myURN );
         if (DEBUG)
            System.err.println(ME+"in constructor found " +
	       (creations==null?0: creations.size()) +
	       " creations for URN already in the Creation Database");
         if ( !creations.isEmpty() )
	      // TBD: merge all the creations first, updating the DB's
	      creationPointer = (Creation)creations.elementAt(0);
         else {
            System.err.println (ME+"Severe error: attempt to construct"
	      + " a BibData with a URN that has no associated Creation");
	    throw new SurrogateException(SurrogateException.setInternal(
	    "Attempt to construct a BibData with a URN that has no associated Creation"));
         }
      } else creationPointer = new Creation ( );  // template
   }

   // accessor functions

   // Reconstruct a BibData object from its XML file.
   // Read in myDOI and myURN.  Then retrieve the Creation
   // associated with myURN.
   // Invariants: DOI, URN, and Creation are all non-null, although
   // the URN might have some '*' in it and the Creation might have
   // some empty fields.  Also, retrieving the Creation implies that
   // the CreationDatabase has to be rebuilt before any BibData
   // objects are reconstructed.
   /* TBD - needed?
   public static BibData reconstruct (Document doc) {
      return new BibData ( docID );
   }
   */

   /**
    * getURN - return the URN of the work associated with this surrogate
    * @returns the String giving the URN
    */
   public String getURN() { return myURN; }

   /**
    * getCreation - table lookup to find URL and BibData
    * @returns the Creation object corresponding to this archive Item
    */
   public Creation getCreation() {
      return creationPointer;
   }

   /**
    * cook - document has been parsed, and local text fragments are here
    * @param - the String containing the local metadata.  Cook it.
    * Upon return, the Author database will have been updated by this
    * items authors, and the creation database by the work in creationPointer
    * and this BibData object *will* have a URN.
    */
   public void cook ( String mf ) {
      // Fire up an XML parser and compare data with the Creation database
      MetaData md = new MetaData ( creationPointer );
      md.setFile ( mf );
      myURN = creationPointer.getURN();
   }

   public String toXML(String pad) {
      String result = "<?xml version=\"1.0\" ?>\n";
      if ( creationPointer == null ) return
	 result + "<work></work>\n";
      return result + creationPointer.toXML(pad+"  ");
   }

   public String toString() {
      //System.err.println(ME+"in toString");
      if ( creationPointer == null ) return "<work></work>\n";
      return creationPointer.toString();
   }

   /**
     * save - called by Surrogate to update CreationDatabase
     * with a fileIndex record.
     */
   protected void save ( String doi ) {
      if ( myURN == null ) creationPointer.getURN();
      if ( myURN.indexOf("&")!=-1) {
	 int k = 0; int i;
	 while ( (i = myURN.indexOf("&", k)) != -1 ) {
	    myURN = myURN.substring(0,i)+"amp;"+myURN.substring(i+1);
	    k = i+4;
	 }
      }
      if (DEBUG) System.err.println ( ME +
      "saving myURN="+myURN+", doi="+doi+" to fileIndex");
      CreationDatabase.save ( myURN, doi );
   }

   // dump routines are used to write out an XML fragment for
   // saving this object in permanent storage for later restoration.
   // We do not need to save the CreationPointer because the URN
   // can be used to look up the Creation.
   public String dump() {
      if ( myURN == null ) {     // complain loudly
	 System.err.println(ME+"invariant has been violated: "
	    +"dumping a BibData without an associated Work!!"
	    + "No dump for item " + myDOI);
	 return "";
      }
      return "<DOI>"+myDOI+"</DOI>\n" + "<URN>"+myURN+"</URN>";
   }

}
