/*
 * This file is part of the Polyglot extensible compiler framework.
 *
 * Copyright (c) 2000-2006 Polyglot project group, Cornell University
 * 
 */

package coffer.ast;

import coffer.types.*;
import coffer.extension.*;
import polyglot.ast.*;
import polyglot.types.*;
import polyglot.visit.*;
import polyglot.util.*;
import java.util.*;

/** An implementation of the <code>CofferConstructorDecl</code> interface.
 * <code>ConstructorDecl</code> is extended with pre- and post-conditions.
 */
public class CofferConstructorDecl_c extends ConstructorDecl_c implements CofferConstructorDecl
{
    protected KeySetNode entryKeys;
    protected KeySetNode returnKeys;
    protected List throwConstraints;

    public CofferConstructorDecl_c(Position pos, Flags flags, Id name, List formals, KeySetNode entryKeys, KeySetNode returnKeys, List throwConstraints, Block body) {
	super(pos, flags, name, formals, Collections.EMPTY_LIST, body);
	this.entryKeys = entryKeys;
        this.returnKeys = returnKeys;
	this.throwConstraints = TypedList.copyAndCheck(throwConstraints, ThrowConstraintNode.class, true);
    }

    public KeySetNode entryKeys() {
	return this.entryKeys;
    }

    public CofferConstructorDecl entryKeys(KeySetNode entryKeys) {
	CofferConstructorDecl_c n = (CofferConstructorDecl_c) copy();
	n.entryKeys = entryKeys;
	return n;
    }

    public KeySetNode returnKeys() {
        return this.returnKeys;
    }

    public CofferConstructorDecl returnKeys(KeySetNode returnKeys) {
	CofferConstructorDecl_c n = (CofferConstructorDecl_c) copy();
	n.returnKeys = returnKeys;
	return n;
    }

    public List throwConstraints() {
	return this.throwConstraints;
    }

    public List throwTypes() {
        return new CachingTransformingList(throwConstraints, new GetType());
    }

    public class GetType implements Transformation {
        public Object transform(Object o) {
            return ((ThrowConstraintNode) o).type();
        }
    }

    public ConstructorDecl throwTypes(List l) {
        throw new InternalCompilerError("unimplemented");
    }

    public CofferConstructorDecl throwConstraints(List throwConstraints) {
	CofferConstructorDecl_c n = (CofferConstructorDecl_c) copy();
	n.throwConstraints = TypedList.copyAndCheck(throwConstraints, ThrowConstraintNode.class, true);
	return n;
    }

    /*
    public Context enterScope(Context context) {
        CofferContext c = (CofferContext) super.enterScope(context);
        c = (CofferContext) c.pushBlock();

        if (entryKeys != null) {
            for (Iterator i = entryKeys.keys().iterator(); i.hasNext(); ) {
                Key key = (Key) i.next();
                c.addHeldKey(key);
            }
        }

        return c;
    }
    */

    protected CofferConstructorDecl_c reconstruct(Id name, List formals, KeySetNode entryKeys, KeySetNode returnKeys, List throwConstraints, Block body) {
	if (entryKeys != this.entryKeys || returnKeys != this.returnKeys || !CollectionUtil.equals(throwConstraints, this.throwConstraints)) {
	    CofferConstructorDecl_c n = (CofferConstructorDecl_c) copy();
	    n.entryKeys = entryKeys;
	    n.returnKeys = returnKeys;
	    n.throwConstraints = TypedList.copyAndCheck(throwConstraints, ThrowConstraintNode.class, true);
	    return (CofferConstructorDecl_c) n.reconstruct(name, formals, Collections.EMPTY_LIST, body);
	}

	return (CofferConstructorDecl_c) super.reconstruct(name, formals, Collections.EMPTY_LIST, body);
    }

    public Node visitChildren(NodeVisitor v) {
	Id name = (Id) visitChild(this.name, v);
        List formals = visitList(this.formals, v);
        KeySetNode entryKeys = (KeySetNode) visitChild(this.entryKeys, v);
        KeySetNode returnKeys = (KeySetNode) visitChild(this.returnKeys, v);
	List throwConstraints = visitList(this.throwConstraints, v);
	Block body = (Block) visitChild(this.body, v);
	return reconstruct(name, formals, entryKeys, returnKeys, throwConstraints, body);
    }

    public Node buildTypes(TypeBuilder tb) throws SemanticException {
        CofferNodeFactory nf = (CofferNodeFactory) tb.nodeFactory();
        CofferConstructorDecl n = (CofferConstructorDecl) super.buildTypes(tb);
        CofferConstructorInstance ci =
            (CofferConstructorInstance) n.constructorInstance();

        if (n.entryKeys() == null) {
            n = n.entryKeys(nf.CanonicalKeySetNode(position(),
                                                   ci.entryKeys()));
        }

        if (n.returnKeys() == null) {
            n = n.returnKeys(nf.CanonicalKeySetNode(position(),
                                                    ci.returnKeys()));
        }

        List l = new LinkedList();
        boolean changed = false;

        for (Iterator i = n.throwConstraints().iterator(); i.hasNext(); ) {
            ThrowConstraintNode cn = (ThrowConstraintNode) i.next();
            if (cn.keys() == null) {
                cn = cn.keys(n.entryKeys());
                changed = true;
            }
            l.add(cn);
        }

        if (changed) {
            n = n.throwConstraints(l);
        }
        
        CofferTypeSystem vts = (CofferTypeSystem) tb.typeSystem();
        ClassType ct = tb.currentClass();
        
        KeySet entryKeys;
        KeySet returnKeys;
        
        if (n.entryKeys() == null) {
            entryKeys = vts.emptyKeySet(position());
        }
        else {
            entryKeys = n.entryKeys().keys();
        }
        
        if (n.returnKeys() == null) {
            returnKeys = vts.emptyKeySet(position());
            
            if (ct instanceof CofferClassType) {
                CofferClassType vct = (CofferClassType) ct;
                if (vct.key() != null)
                    returnKeys = returnKeys.add(vct.key());
            }
        }
        else {
            returnKeys = n.returnKeys().keys();
        }
        
        ci.setEntryKeys(entryKeys);
        ci.setReturnKeys(returnKeys);

        return n;
    }

    public Node typeCheck(TypeChecker tc) throws SemanticException {
        CofferClassType ct = (CofferClassType) tc.context().currentClass();

        CofferConstructorInstance ci =
            (CofferConstructorInstance) this.constructorInstance();

        if (ct.key() != null) {
            if (ci.entryKeys().contains(ct.key())) {
                throw new SemanticException("Constructor cannot hold key \"" +
                                            ct.key() + "\" (associated with " +
                                            "this) on entry.", position());
            }

            if (! ci.returnKeys().contains(ct.key())) {
                throw new SemanticException("Constructor must hold key \"" +
                                            ct.key() + "\" (associated with " +
                                            "this) on exit.", position());
            }
        }

        return super.typeCheck(tc);
    }

    public Node disambiguate(AmbiguityRemover ar) throws SemanticException {
        if (this.ci.isCanonical()) {
            return this;
        }
        
        if (this.entryKeys != null && ! this.entryKeys.keys().isCanonical()) {
            return this;
        }

        if (this.returnKeys != null && ! this.returnKeys.keys().isCanonical()) {
            return this;
        }

        CofferConstructorDecl_c n = (CofferConstructorDecl_c) super.disambiguate(ar);

	CofferTypeSystem vts = (CofferTypeSystem) ar.typeSystem();
	ClassType ct = ar.context().currentClass();

        KeySet entryKeys;
        KeySet returnKeys;

	if (n.entryKeys == null) {
            entryKeys = vts.emptyKeySet(position());
        }
        else {
            entryKeys = n.entryKeys.keys();
        }

	if (n.returnKeys == null) {
            returnKeys = vts.emptyKeySet(position());

            if (ct instanceof CofferClassType) {
                CofferClassType vct = (CofferClassType) ct;
                if (vct.key() != null)
                    returnKeys = returnKeys.add(vct.key());
            }
        }
        else {
            returnKeys = n.returnKeys.keys();
        }

	CofferConstructorInstance ci = (CofferConstructorInstance) n.ci;
        ci.setEntryKeys(entryKeys);
        ci.setReturnKeys(returnKeys);

	List throwConstraints = new ArrayList(n.throwConstraints.size());
	for (Iterator i = n.throwConstraints.iterator(); i.hasNext(); ) {
	    ThrowConstraintNode cn = (ThrowConstraintNode) i.next();

            if (cn.constraint().keys() != null) {
                throwConstraints.add(cn.constraint());
            }
            else {
                ThrowConstraint c = (ThrowConstraint) cn.constraint().copy();
                c.setKeys(entryKeys);
                throwConstraints.add(c);
            }
	}

	ci.setThrowConstraints(throwConstraints);
    
	return n;
    }

    /** Write the constructor to an output file. */
    public void prettyPrintHeader(CodeWriter w, PrettyPrinter tr) {
	w.begin(0);
	w.write(flags().translate());

        print(name, w, tr);
	w.write("(");

	w.begin(0);

	for (Iterator i = formals.iterator(); i.hasNext(); ) {
	    Formal f = (Formal) i.next();
	    print(f, w, tr);

	    if (i.hasNext()) {
		w.write(",");
		w.allowBreak(0, " ");
	    }
	}

	w.end();
	w.write(")");
    
	if (! (tr instanceof Translator)) {
	    if (entryKeys != null) {
	        w.allowBreak(6, " ");
	        print(entryKeys, w, tr);
	    }
	    if (returnKeys != null) {
	        w.write(" -> ");
	        print(returnKeys, w, tr);
	    }
	}

	if (! throwConstraints.isEmpty()) {
	    w.allowBreak(6);
	    w.write("throws ");

	    for (Iterator i = throwConstraints.iterator(); i.hasNext(); ) {
	        ThrowConstraintNode cn = (ThrowConstraintNode) i.next();
		print(cn, w, tr);

		if (i.hasNext()) {
		    w.write(",");
		    w.allowBreak(4, " ");
		}
	    }
	}

	w.end();
    }
}
