/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2002-2010 Oracle.  All rights reserved.
 *
 */

package com.sleepycat.je.cleaner;

import java.io.File;

import junit.framework.TestCase;

import com.sleepycat.bind.tuple.IntegerBinding;
import com.sleepycat.je.CheckpointConfig;
import com.sleepycat.je.Cursor;
import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseConfig;
import com.sleepycat.je.DatabaseEntry;
import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.DbInternal;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.OperationStatus;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.junit.JUnitThread;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.util.TestUtils;

/**
 * Test concurrent checkpoint and Database.sync operations.
 *
 * Prior to the fix for [#18567], checkpoint and Database.sync both acquired a
 * shared latch on the parent IN while logging a child IN.  During concurrent
 * logging of provisional child BINs for same parent IN, problems can occur
 * when the utilization info in the parent IN is being updated by both threads.
 *
 * For example, the following calls could be concurrent in the two threads for
 * the same parent IN.
 *
 *  BIN.afterLog (two different BINs)
 *   |--IN.trackProvisionalObsolete (same parent)
 *      |-- PackedObsoleteInfo.copyObsoleteInfo (array copy)
 *
 * Concurrent array copies could corrupt the utilization info, and this could
 * result in the exception reported by the OTN user later on, when the parent
 * IN is logged and the obsolete info is copied to the UtilizationTracker:
 *
 *    java.lang.IndexOutOfBoundsException
 *    at com.sleepycat.bind.tuple.TupleInput.readBoolean(TupleInput.java:186)
 *    at com.sleepycat.je.cleaner.PackedObsoleteInfo.countObsoleteInfo(PackedObsoleteInfo.java:60)
 *    at com.sleepycat.je.log.LogManager.serialLogInternal(LogManager.java:671)
 *    at com.sleepycat.je.log.SyncedLogManager.serialLog(SyncedLogManager.java:40)
 *    at com.sleepycat.je.log.LogManager.multiLog(LogManager.java:388)
 *    at com.sleepycat.je.recovery.Checkpointer.logSiblings(Checkpointer.java:1285)
 *
 * Although we have not reproduced this particular exception, the test case
 * below, along with assertions in IN.beforeLog and afterLog, demonstrate that
 * we do allow concurrent logging of two child BINs for the same parent IN.
 * Since both threads are modifying the parent IN, it seems obvious that using
 * a shared latch could cause more than one type of problem and should be
 * disallowed.
 */
public class SR18567Test extends TestCase {

    private static final String DB_NAME = "foo";

    private static final CheckpointConfig forceConfig = new CheckpointConfig();
    static {
        forceConfig.setForce(true);
    }

    private File envHome;
    private Environment env;
    private Database db;
    private JUnitThread junitThread1;
    private JUnitThread junitThread2;

    public SR18567Test() {
        envHome = new File(System.getProperty(TestUtils.DEST_DIR));
    }

    @Override
    public void setUp() {
        TestUtils.removeLogFiles("Setup", envHome, false);
        TestUtils.removeFiles("Setup", envHome, FileManager.DEL_SUFFIX);
    }

    @Override
    public void tearDown() {
        if (junitThread1 != null) {
            while (junitThread1.isAlive()) {
                junitThread1.interrupt();
                Thread.yield();
            }
            junitThread1 = null;
        }
        if (junitThread2 != null) {
            while (junitThread2.isAlive()) {
                junitThread2.interrupt();
                Thread.yield();
            }
            junitThread2 = null;
        }

        try {
            if (env != null) {
                env.close();
            }
        } catch (Throwable e) {
            System.out.println("tearDown: " + e);
        }

        try {
            TestUtils.removeLogFiles("tearDown", envHome, true);
            TestUtils.removeFiles("tearDown", envHome, FileManager.DEL_SUFFIX);
        } catch (Throwable e) {
            System.out.println("tearDown: " + e);
        }

        db = null;
        env = null;
        envHome = null;
    }

    /**
     * Opens the environment and database.
     */
    private void openEnv()
        throws DatabaseException {

        EnvironmentConfig config = TestUtils.initEnvConfig();
        DbInternal.disableParameterValidation(config);
        config.setAllowCreate(true);
        /* Do not run the daemons. */
        config.setConfigParam
            (EnvironmentParams.ENV_RUN_CLEANER.getName(), "false");
        config.setConfigParam
            (EnvironmentParams.ENV_RUN_EVICTOR.getName(), "false");
        config.setConfigParam
            (EnvironmentParams.ENV_RUN_CHECKPOINTER.getName(), "false");
        config.setConfigParam
            (EnvironmentParams.ENV_RUN_INCOMPRESSOR.getName(), "false");

        env = new Environment(envHome, config);

        openDb();
    }

    /**
     * Opens that database.
     */
    private void openDb()
        throws DatabaseException {

        DatabaseConfig dbConfig = new DatabaseConfig();
        dbConfig.setAllowCreate(true);
        dbConfig.setDeferredWrite(true);
        db = env.openDatabase(null, DB_NAME, dbConfig);
    }

    /**
     * Closes the environment and database.
     */
    private void closeEnv()
        throws DatabaseException {

        if (db != null) {
            db.close();
            db = null;
        }
        if (env != null) {
            env.close();
            env = null;
        }
    }

    public void testSR18567()
        throws Throwable {

        openEnv();

        DatabaseEntry key = new DatabaseEntry();
        DatabaseEntry data = new DatabaseEntry(new byte[0]);

        for (int i = 0; i < 100; i += 1) {
            for (int j = 0; j < 1000; j += 1) {
                IntegerBinding.intToEntry(j, key);
                db.put(null, key, data);
            }
            junitThread1 = new JUnitThread("Checkpoint") {
                @Override
                public void testBody() {
                    env.checkpoint(forceConfig);
                }
            };
            junitThread2 = new JUnitThread("Database.sync") {
                @Override
                public void testBody() {
                    db.sync();
                }
            };
            junitThread1.start();
            junitThread2.start();
            junitThread1.finishTest();
            junitThread2.finishTest();
        }

        closeEnv();
    }
}
