/**
 * Copyright (C) 2010 Fabric project group, Cornell University
 *
 * This file is part of Fabric.
 *
 * Fabric is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Fabric is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 */
package fabric.worker.remote.messages;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import fabric.worker.Worker;
import fabric.worker.Store;
import fabric.worker.remote.RemoteWorker;
import fabric.common.exceptions.InternalError;
import fabric.lang.Object._Proxy;
import fabric.messages.Message;
import fabric.messages.Message.Response;

public abstract class InterWorkerMessage<R extends Response> extends
    Message<RemoteWorker, R> {

  public InterWorkerMessage(MessageType messageType) {
    super(messageType);
  }

  /**
   * Used for passing object references between workers.
   */
  public static void writeRef(_Proxy ref, DataOutput out) throws IOException {
    out.writeUTF(ref.$getStore().name());
    out.writeLong(ref.$getOnum());
  }

  /**
   * Used for passing object references between workers.
   * 
   * @param type
   *          The type of the reference being read. This must be the interface
   *          corresponding to the Fabric type, and not the _Proxy or _Impl
   *          classes.
   */
  @SuppressWarnings("unchecked")
  public static _Proxy readRef(Class<?> type, DataInput in) throws IOException {
    Store store = Worker.getWorker().getStore(in.readUTF());
    Class<? extends _Proxy> proxyType = null;
    for (Class<?> c : type.getClasses()) {
      if (c.getSimpleName().equals("_Proxy")) {
        proxyType = (Class<? extends _Proxy>) c;
        break;
      }
    }

    if (proxyType == null)
      throw new InternalError("Unable to find proxy class for " + type);

    try {
      Constructor<? extends _Proxy> constructor =
          proxyType.getConstructor(Store.class, long.class);

      return constructor.newInstance(store, in.readLong());
    } catch (SecurityException e) {
      throw new InternalError(e);
    } catch (NoSuchMethodException e) {
      throw new InternalError(e);
    } catch (IllegalArgumentException e) {
      throw new InternalError(e);
    } catch (InstantiationException e) {
      throw new InternalError(e);
    } catch (IllegalAccessException e) {
      throw new InternalError(e);
    } catch (InvocationTargetException e) {
      throw new InternalError(e);
    }
  }
}
